/*	$OpenBSD: pfctl.h,v 1.42 2007/12/05 12:01:47 chl Exp $ */

/*-
 * SPDX-License-Identifier: BSD-2-Clause
 *
 * Copyright (c) 2001 Daniel Hartmeier
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above
 *      copyright notice, this list of conditions and the following
 *      disclaimer in the documentation and/or other materials provided
 *      with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _PFCTL_H_
#define _PFCTL_H_

#include <libpfctl.h>

#ifdef PFCTL_DEBUG
#define DBGPRINT(...)	fprintf(stderr, __VA_ARGS__)
#else
#define DBGPRINT(...)	(void)(0)
#endif

extern struct pfctl_handle	*pfh;

struct pfctl;

enum pfctl_show { PFCTL_SHOW_RULES, PFCTL_SHOW_LABELS, PFCTL_SHOW_NOTHING };

enum {	PFRB_TABLES = 1, PFRB_TSTATS, PFRB_ADDRS, PFRB_ASTATS,
	PFRB_IFACES, PFRB_TRANS, PFRB_MAX };
struct pfr_buffer {
	int	 pfrb_type;	/* type of content, see enum above */
	int	 pfrb_size;	/* number of objects in buffer */
	int	 pfrb_msize;	/* maximum number of objects in buffer */
	void	*pfrb_caddr;	/* malloc'ated memory area */
};
#define PFRB_FOREACH(var, buf)				\
	for ((var) = pfr_buf_next((buf), NULL);		\
	    (var) != NULL;				\
	    (var) = pfr_buf_next((buf), (var)))

RB_HEAD(pfr_ktablehead, pfr_ktable);
struct pfr_ktable {
	struct pfr_tstats	 pfrkt_ts;
	RB_ENTRY(pfr_ktable)	 pfrkt_tree;
	SLIST_ENTRY(pfr_ktable)	 pfrkt_workq;
	struct radix_node_head	*pfrkt_ip4;
	struct radix_node_head	*pfrkt_ip6;
	struct pfr_ktable	*pfrkt_shadow;
	struct pfr_ktable	*pfrkt_root;
	struct pf_kruleset	*pfrkt_rs;
	long			 pfrkt_larg;
	int			 pfrkt_nflags;
};
#define pfrkt_t		pfrkt_ts.pfrts_t
#define pfrkt_name	pfrkt_t.pfrt_name
#define pfrkt_anchor	pfrkt_t.pfrt_anchor
#define pfrkt_ruleset	pfrkt_t.pfrt_ruleset
#define pfrkt_flags	pfrkt_t.pfrt_flags
#define pfrkt_cnt	pfrkt_kts.pfrkts_cnt
#define pfrkt_refcnt	pfrkt_kts.pfrkts_refcnt
#define pfrkt_tzero	pfrkt_kts.pfrkts_tzero

struct pfr_uktable {
	struct pfr_ktable	pfrukt_kt;
	struct pfr_buffer	pfrukt_addrs;
	int			pfrukt_init_addr;
	SLIST_ENTRY(pfr_uktable) pfrukt_entry;
};

#define pfrukt_t	pfrukt_kt.pfrkt_ts.pfrts_t
#define pfrukt_name	pfrukt_kt.pfrkt_t.pfrt_name
#define pfrukt_anchor	pfrukt_kt.pfrkt_t.pfrt_anchor

extern struct pfr_ktablehead pfr_ktables;

struct pfr_anchoritem {
	SLIST_ENTRY(pfr_anchoritem)	 pfra_sle;
	char				*pfra_anchorname;
};

SLIST_HEAD(pfr_anchors, pfr_anchoritem);

int	 pfr_add_table(struct pfr_table *, int *, int);
int	 pfr_del_table(struct pfr_table *, int *, int);
int	 pfr_get_tables(struct pfr_table *, struct pfr_table *, int *, int);
int	 pfr_clr_astats(struct pfr_table *, struct pfr_addr *, int, int *, int);
int	 pfr_clr_addrs(struct pfr_table *, int *, int);
int	 pfr_add_addrs(struct pfr_table *, struct pfr_addr *, int, int *, int);
int	 pfr_del_addrs(struct pfr_table *, struct pfr_addr *, int, int *, int);
int	 pfr_set_addrs(struct pfr_table *, struct pfr_addr *, int,
	    int *, int *, int *, int);
int	 pfr_get_addrs(struct pfr_table *, struct pfr_addr *, int *, int);
int	 pfr_get_astats(struct pfr_table *, struct pfr_astats *, int *, int);
int	 pfr_tst_addrs(struct pfr_table *, struct pfr_addr *, int, int *, int);
int	 pfr_ina_define(struct pfr_table *, struct pfr_addr *, int, int *,
	    int *, int, int);
void	 pfr_buf_clear(struct pfr_buffer *);
int	 pfr_buf_add(struct pfr_buffer *, const void *);
void	*pfr_buf_next(struct pfr_buffer *, const void *);
int	 pfr_buf_grow(struct pfr_buffer *, int);
int	 pfr_buf_load(struct pfr_buffer *, char *, int,
	    int (*)(struct pfr_buffer *, char *, int, int), int);
char	*pf_strerror(int);
int	 pfi_get_ifaces(const char *, struct pfi_kif *, int *);

void	 pfctl_print_title(char *);
int	 pfctl_do_clear_tables(const char *, int);
void	 pfctl_show_tables(const char *, int);
int	 pfctl_table(int, char *[], char *, const char *, char *,
	    const char *, int);
int	 pfctl_show_altq(int, const char *, int, int);
void	 warn_duplicate_tables(const char *, const char *);
void	 pfctl_show_ifaces(const char *, int);
void	pfctl_show_creators(int);
FILE	*pfctl_fopen(const char *, const char *);

#ifdef __FreeBSD__
extern int altqsupport;
extern int dummynetsupport;
#define	HTONL(x)		(x) = htonl((__uint32_t)(x))
#endif

#ifndef DEFAULT_PRIORITY
#define DEFAULT_PRIORITY	1
#endif

#ifndef DEFAULT_QLIMIT
#define DEFAULT_QLIMIT		50
#endif

/*
 * generalized service curve used for admission control
 */
struct segment {
	LIST_ENTRY(segment)	_next;
	double			x, y, d, m;
};

extern	int loadopt;

int		 check_commit_altq(int, int);
void		 pfaltq_store(struct pf_altq *);
char		*rate2str(double);

void	 print_addr(struct pf_addr_wrap *, sa_family_t, int);
void	 print_addr_str(sa_family_t, struct pf_addr *);
void	 print_host(struct pf_addr *, u_int16_t p, sa_family_t, int);
void	 print_seq(struct pfctl_state_peer *);
void	 print_state(struct pfctl_state *, int);

int	 pfctl_cmdline_symset(char *);
int	 pfctl_add_trans(struct pfr_buffer *, int, const char *);
u_int32_t
	 pfctl_get_ticket(struct pfr_buffer *, int, const char *);
int	 pfctl_trans(int, struct pfr_buffer *, u_long, int);

int	 pf_get_ruleset_number(u_int8_t);
void	 pf_init_ruleset(struct pfctl_ruleset *);
int	 pfctl_anchor_setup(struct pfctl_rule *,
	    const struct pfctl_ruleset *, const char *);
void	 pf_remove_if_empty_ruleset(struct pfctl_ruleset *);
struct pfctl_ruleset	*pf_find_ruleset(const char *);
struct pfctl_ruleset	*pf_find_or_create_ruleset(const char *);
void			 pf_init_eth_ruleset(struct pfctl_eth_ruleset *);
int			 pfctl_eth_anchor_setup(struct pfctl *,
			    struct pfctl_eth_rule *,
			    const struct pfctl_eth_ruleset *, const char *);
struct pfctl_eth_ruleset	*pf_find_or_create_eth_ruleset(const char *);
void			 pf_remove_if_empty_eth_ruleset(
			    struct pfctl_eth_ruleset *);

void		 expand_label(char *, size_t, struct pfctl_rule *);

const char *pfctl_proto2name(int);

void	 pfctl_err(int, int, const char *, ...);
void	 pfctl_errx(int, int, const char *, ...);

#endif /* _PFCTL_H_ */
