/* $Id: directory-handle.cc,v 1.10 2004/02/13 18:22:29 murrayc Exp $ */               

/* Copyright 2003 gnome-vfsmm Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <libgnomevfsmm/directory-handle.h>
#include <libgnomevfsmm/private.h>

#include <libgnomevfs/gnome-vfs-directory.h>

namespace
{

//SignalProxy_Visit:

//This Signal Proxy allows the C++ coder to specify a sigc::slot instead of a static function.
class SignalProxy_Visit
{
public:
  typedef Gnome::Vfs::DirectoryHandle::SlotVisit SlotType;

  SignalProxy_Visit(const SlotType& slot);
  ~SignalProxy_Visit();

  static gboolean c_callback(const gchar* rel_path, GnomeVFSFileInfo* info, gboolean recursing_will_loop, gpointer data, gboolean* recurse);
              
protected:
  SlotType slot_;
};

SignalProxy_Visit::SignalProxy_Visit(const SlotType& slot)
:
  slot_ (slot)
{}

SignalProxy_Visit::~SignalProxy_Visit()
{}

gboolean SignalProxy_Visit::c_callback(const gchar* rel_path, GnomeVFSFileInfo* info, gboolean recursing_will_loop, gpointer data, gboolean* recurse)
{
  SignalProxy_Visit *const self = static_cast<SignalProxy_Visit*>(data);
  gboolean bResult = false;
  try
  {
    bool bRecurse = false;
    Glib::ustring strTemp = Glib::convert_const_gchar_ptr_to_ustring(rel_path);
    bResult = (self->slot_)(strTemp, Glib::wrap(info, true /* take_copy */), recursing_will_loop, bRecurse);
    *recurse = bRecurse;
  }
  catch(...)
  {
    Glib::exception_handlers_invoke();
  }

  return bResult;
}

} //anonymous namespace

namespace Gnome
{

namespace Vfs
{

GnomeVFSDirectoryHandle* DirectoryHandle::gobj()
{
  return gobj_;
}

const GnomeVFSDirectoryHandle* DirectoryHandle::gobj() const
{
  return gobj_;
}

DirectoryHandle::DirectoryHandle()
: gobj_(0)
{
}

DirectoryHandle::~DirectoryHandle()
{
  if(gobj())
    close();
}

void DirectoryHandle::open(const Glib::ustring& text_uri, FileInfoOptions options) throw(exception)
{
  GnomeVFSResult result = gnome_vfs_directory_open(&gobj_, text_uri.c_str(), static_cast<GnomeVFSFileInfoOptions>(options) );
  handle_result(result);
}

void DirectoryHandle::open(const Glib::RefPtr<const Uri>& uri, FileInfoOptions options) throw(exception)
{
  GnomeVFSResult result = gnome_vfs_directory_open_from_uri(&gobj_, const_cast<GnomeVFSURI*>(uri->gobj()), static_cast<GnomeVFSFileInfoOptions>(options) );
  handle_result(result);
}

void DirectoryHandle::close() throw(exception)
{
  if(gobj_)
  {
    GnomeVFSResult result = gnome_vfs_directory_close(gobj());
    handle_result(result);
  }
}
  
Glib::RefPtr<FileInfo> DirectoryHandle::read_next(bool& file_exists) throw(exception)
{
  GnomeVFSFileInfo* fileinfo = gnome_vfs_file_info_new();
  GnomeVFSResult result = gnome_vfs_directory_read_next(gobj(), fileinfo);
  Glib::RefPtr<FileInfo> cppFileInfo = Glib::wrap(fileinfo); //Takes ownership, so we don't need to unref fileinfo.
  
  if(result == GNOME_VFS_ERROR_EOF) //there are no more files.
    file_exists = false;  //This isn't really an error. It's just the end of the file list.
  else
    handle_result(result);
    
  return cppFileInfo;
}

void DirectoryHandle::visit(const Glib::ustring& uri, FileInfoOptions info_options, DirectoryVisitOptions visit_options, const SlotVisit& slot) throw(exception)
{
  SignalProxy_Visit proxy(slot);
  GnomeVFSResult result = gnome_vfs_directory_visit(uri.c_str(), static_cast<GnomeVFSFileInfoOptions>(info_options), static_cast<GnomeVFSDirectoryVisitOptions>(visit_options), &SignalProxy_Visit::c_callback, &proxy);
  handle_result(result);
}

void DirectoryHandle::visit(const Glib::RefPtr<const Uri>& uri, FileInfoOptions info_options, DirectoryVisitOptions visit_options, const SlotVisit& slot) throw(exception)
{
  SignalProxy_Visit proxy(slot);
  GnomeVFSResult result = gnome_vfs_directory_visit_uri(const_cast<GnomeVFSURI*>(uri->gobj()), static_cast<GnomeVFSFileInfoOptions>(info_options), static_cast<GnomeVFSDirectoryVisitOptions>(visit_options), &SignalProxy_Visit::c_callback, &proxy);
  handle_result(result);
}
 
void DirectoryHandle::visit_files(const Glib::ustring& uri, const Glib::ListHandle<Glib::ustring>& file_list, FileInfoOptions info_options, DirectoryVisitOptions visit_options, const SlotVisit& slot) throw(exception)
{
  SignalProxy_Visit proxy(slot);
  GnomeVFSResult result = gnome_vfs_directory_visit_files(uri.c_str(), file_list.data(), static_cast<GnomeVFSFileInfoOptions>(info_options), static_cast<GnomeVFSDirectoryVisitOptions>(visit_options), &SignalProxy_Visit::c_callback, &proxy);
  handle_result(result);
}

void DirectoryHandle::visit_files(const Glib::RefPtr<const Uri>& uri, const Glib::ListHandle<Glib::ustring> & file_list, FileInfoOptions info_options, DirectoryVisitOptions visit_options, const SlotVisit& slot) throw(exception)
{
  SignalProxy_Visit proxy(slot);
   GnomeVFSResult result = gnome_vfs_directory_visit_files_at_uri(const_cast<GnomeVFSURI*>(uri->gobj()), file_list.data(), static_cast<GnomeVFSFileInfoOptions>(info_options), static_cast<GnomeVFSDirectoryVisitOptions>(visit_options), &SignalProxy_Visit::c_callback, &proxy);
  handle_result(result);
}

void DirectoryHandle::list_load(const Glib::ListHandle<Glib::ustring>& list, const Glib::ustring& text_uri, FileInfoOptions info_options) throw(exception)
{
  GList* temp_list = list.data();
  GnomeVFSResult result = gnome_vfs_directory_list_load(&temp_list, text_uri.c_str(), static_cast<GnomeVFSFileInfoOptions>(info_options));
  handle_result(result);
}
  
} //namespace Vfs

} //namespace Gnome



