/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtk;

import org.gnu.glib.GObject;
import org.gnu.glib.Handle;
import org.gnu.glib.Type;

/**
 * ToolTips are the messages that appear next to a widget when the mouse
 * pointer is held over it for a short period of time.
 */
public class ToolTips extends GtkObject {
	/**
	 * Construct a new ToolTips object.
	 */
	public ToolTips() {
		super(gtk_tooltips_new());
	}
	
	public ToolTips(Handle handle) {
		super(handle);
	}
    
    /**
     * Internal static factory method to be used by Java-Gnome only.
     */
    public static ToolTips getToolTips(Handle handle) {
        if (handle == null)
            return null;
        
        ToolTips obj = (ToolTips) GObject.getGObjectFromHandle(handle);
        
        if (obj == null)
            obj = new ToolTips(handle);
        
        return obj;
    }

	/**
	 * Adds a tooltip to a Widget.
	 * 
	 * @param widget The Widget to add to tooltips.
	 * @param tipText The text to display for the tooltip.
	 * @param additionalInfo Additional information that might be useful to
	 * the user if they get stuck.
	 */
	public void setTip(Widget widget, String tipText, String additionalInfo) {
		ToolTips.gtk_tooltips_set_tip(getHandle(), widget.getHandle(), tipText, additionalInfo);
	}
	
	/**
	 * Enables the ToolTips.
	 */
	public void enable() {
		ToolTips.gtk_tooltips_enable(getHandle());
	}
	
	/**
	 * Disables the ToolTips.
	 */
	public void disable() {
		ToolTips.gtk_tooltips_disable(getHandle());
	}
	
	/**
	 * Retrieve the ToolTipsData associated with the provided widget
	 * @param widget
	 */
	public static ToolTipsData getData(Widget widget) {
        Handle handle = gtk_tooltips_data_get(widget.getHandle());
        return ToolTipsData.getToolTipsData(handle);
    }
	
	/**
	 * Retrieve the runtime type used by the GLib library.
	 */
	public static Type getType() {
		return new Type(gtk_tooltips_get_type());
	}

	/****************************************
	 * BEGINNING OF JNI CODE
	 ****************************************/
	native static final protected int gtk_tooltips_get_type();
	native static final protected Handle gtk_tooltips_new();
	native static final protected void gtk_tooltips_enable(Handle tooltips);
	native static final protected void gtk_tooltips_disable(Handle tooltips);
	native static final protected void gtk_tooltips_set_tip(Handle tooltips, Handle widget, String tipText, String tipPrivate);
	native static final protected void gtk_tooltips_force_window(Handle tooltips);
	native static final protected Handle gtk_tooltips_data_get(Handle widget);
	/****************************************
	 * END OF JNI CODE
	 ****************************************/
}
