/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtk;

import java.util.Vector;

import org.gnu.glib.EventMap;
import org.gnu.gtk.event.GtkEventType;
import org.gnu.gtk.event.ToggleToolButtonEvent;
import org.gnu.gtk.event.ToggleToolButtonListener;

/**
 */
public class ToggleToolButton extends ToolButton {

	public ToggleToolButton() {
		super(gtk_toggle_tool_button_new());
		initializeEventHandlers();
	}
	
	public ToggleToolButton(String stockId) {
		super(gtk_toggle_tool_button_new_from_stock(stockId));
		initializeEventHandlers();
	}
	
	public ToggleToolButton(int hndl) {
		super(hndl);
		initializeEventHandlers();
	}
	
	public void setActive(boolean active) {
		gtk_toggle_tool_button_set_active(getHandle(), active);
	}
	
	public boolean getActive() {
		return gtk_toggle_tool_button_get_active(getHandle());
	}
	
	/***************************************
	 * EVENT HANDLING
	 ****************************************/
	/**
	 * Listeners for handling events
	 */
	private Vector tbListeners = null;

	/**
	 * Register an object to handle dialog events.
	 * @see ToggleToolButtonListener
	 */
	public void addListener(ToggleToolButtonListener listener) {
		// Don't add the listener a second time if it is in the Vector.
		int i = findListener(tbListeners, listener);
		if (i == -1) {
			if (null == tbListeners) {
				tbListeners = new Vector();
			}
			tbListeners.addElement(listener);
		}
	}
	
	/**
	 * Removes a listener
	 * @see #addListener(ToggleToolButtonListener)
	 */
	public void removeListener(ToggleToolButtonListener listener) {
		int i = findListener(tbListeners, listener);
		if (i > -1) {
			tbListeners.remove(i);
		}
		if (0 == tbListeners.size()) {
			tbListeners = null;
		}
	}

	protected void fireToggleToolButtonEvent(ToggleToolButtonEvent event) {
		if (null == tbListeners) {
			return;
		}
		int size = tbListeners.size();
		int i = 0;
		while (i < size) {
			ToggleToolButtonListener tbl = (ToggleToolButtonListener)tbListeners.elementAt(i);
			tbl.toggleToolButtonEvent(event);
			i++;
		}
	}

	private void handleToggled() {
		ToggleToolButtonEvent evt = new ToggleToolButtonEvent(this, ToggleToolButtonEvent.Type.TOGGLED);
		fireToggleToolButtonEvent(evt);
	}

	protected void initializeEventHandlers() {
		if (!eventsInitialized) {
			super.initializeEventHandlers();
			evtMap.initialize(this);
			eventsInitialized = true;
		}
	}

	public Class getEventListenerClass(String signal) {
		Class cls = evtMap.getEventListenerClass(signal);
		if (cls == null) cls = super.getEventListenerClass(signal);
		return cls;
	}

	public GtkEventType getEventType(String signal) {
		GtkEventType et = evtMap.getEventType(signal);
		if (et == null) et = super.getEventType(signal);
		return et;
	}

	private static EventMap evtMap = new EventMap();
	static {
		addEvents(evtMap);
	}

	/**
	 * Implementation method to build an EventMap for this widget class.
	 * Not useful (or supported) for application use.
	 */
	private static void addEvents(EventMap anEvtMap) {
		anEvtMap.addEvent("toggled", "handleToggled", ToggleToolButtonEvent.Type.TOGGLED, ToggleToolButtonListener.class);
	}
	
	/****************************************
	 * BEGINNING OF JNI CODE
	 ****************************************/
	native static final protected int gtk_toggle_tool_button_get_type();
	native static final protected int gtk_toggle_tool_button_new();
	native static final protected int gtk_toggle_tool_button_new_from_stock(String stockId);
	native static final protected void gtk_toggle_tool_button_set_active(int button, boolean active);
	native static final protected boolean gtk_toggle_tool_button_get_active(int button);
	/****************************************
	 * END OF JNI CODE
	 ****************************************/
}
