/*
 * A GNOME front end for LinuxConf
 *
 * (C) 1997 the Free Software Foundation
 *
 * Author: Miguel de Icaza (miguel@gnu.org)
 *
 */
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include "gnome.h"

#define BSIZE 512
#define NFORM_STACK 16
#define GNOME_WINDOW "GNOME-Window"

enum form_type {
	FORM_HBOX,
	FORM_TABLE
};

static struct {
	GtkWidget  *table;
	int        column;
	int        line;
	int        type;
	GHashTable *hash;
} form_stack [NFORM_STACK];

int stack_top = 0;

#define current_table  form_stack [stack_top].table
#define current_column form_stack [stack_top].column
#define current_line   form_stack [stack_top].line
#define current_hash   form_stack [stack_top].hash
#define current_type   form_stack [stack_top].type

GHashTable *toplevels;
GtkWidget *current_window, *current_book;
GtkWidget *last_widget, *last_menu;

char *reg = NULL;
int  reg_size = 0;

#define bind_sym(x,y) g_hash_table_insert (current_hash, g_strdup (x), y);

char **
split_string (char *str, int items)
{
	char **result;
	int  i;

	result = g_malloc (sizeof (char *) * items + 1);

	for (i = 0; i < items; i++){
		result [i] = str;

		while (*str && (!isspace (*str)))
			str++;
		
		if (*str){
			*str = 0;
			str++;
		}

		while (isspace (*str))
			str++;
	}
	result [i] = str;
	return result;
}

GtkWidget *
new_table (void)
{
	GtkWidget *table;

	table = gtk_table_new (100, 100, 0);
	gtk_container_border_width (GTK_CONTAINER (table), 6);
#if 0
	gtk_table_set_row_spacings (GTK_TABLE(table), 4);
	gtk_table_set_col_spacings (GTK_TABLE(table), 4);
#endif
	return table;
}

GtkWidget *
new_hbox (void)
{
	GtkWidget *hbox;

	hbox = gtk_hbox_new (1, 0);
	gtk_container_border_width (GTK_CONTAINER (hbox), 6);
	return hbox;
}

void
maybe_skipit (void *data, void *data2)
{
	GtkTableChild *child = data;

	if (!((current_line >= child->top_attach) &&
	      (current_line < child->bottom_attach)))
		return;
	
	if ((current_column >= child->left_attach) &&
	    (current_column < child->right_attach))
		current_column = child->right_attach;
}


void
walk_over_allocated_space ()
{
	GtkTable *table = GTK_TABLE(current_table);
	GList *p;

	p = table->children;
	if (!p)
		return;

	g_list_foreach (table->children, maybe_skipit, NULL);
	return;
}

void
place (GtkWidget *widget)
{
	if (current_type == FORM_TABLE){
		walk_over_allocated_space ();
		last_widget = widget;
		gtk_table_attach (GTK_TABLE (current_table), widget,
				  current_column, current_column + 1,
				  current_line,   current_line + 1,
				  GTK_SHRINK|GTK_FILL, 
				  GTK_SHRINK|GTK_FILL,
				  0, 0);
		current_column++;
	} else
		gtk_box_pack_start (GTK_BOX (current_table), widget, 0, 0, 0);
}

void
create_form_type (enum form_type ft, char *str)
{
	GtkWidget  *window, *table;
	GHashTable *wtab;
	char **strs;

	if (!form_stack [0].table){
		strs = split_string (str, 1);
		window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
		gtk_window_set_title (GTK_WINDOW(window), strs [1]);
		if (ft == FORM_TABLE)
			table = new_table ();
		else
			table = new_hbox ();
		gtk_widget_show (table);
		gtk_container_add (GTK_CONTAINER(window), table);
		wtab = g_hash_table_new (g_string_hash, g_string_equal);
		g_hash_table_insert (toplevels, strs [0], wtab);
		g_hash_table_insert (wtab, GNOME_WINDOW, window);
		
		current_hash   = wtab;
		current_window = window;
	} else {
		strs = split_string (str, 0);
		if (ft == FORM_TABLE)
			table = new_table ();
		else
			table = new_hbox ();
		
		gtk_widget_show (table);
		bind_sym (strs [0], table);
		place (table);

		/* nest */
		wtab = g_hash_table_new (g_string_hash, g_string_equal);
		gtk_object_set_user_data (GTK_OBJECT (table), wtab);
		stack_top++;
	}
	current_table  = table;
	current_column = current_line = 0;
	current_type = ft;
	free (strs);
}

void
create_form (char *str)
{
	create_form_type (FORM_TABLE, str);
}

void
create_form_button (char *str)
{
	create_form_type (FORM_HBOX, str);
}
	
void
create_book (char *str)
{
	GtkWidget *book;
	char **strs;

	strs = split_string (str, 1);
	book = gtk_notebook_new ();
	gtk_widget_show (book);
	bind_sym (strs [0], book);
	place (book);
	current_book = book;
	free (strs);
}

void
button_dump_form (GtkWidget *widget, gpointer data)
{
	printf ("Should dump all\n");
}

void
report_action (GtkWidget *widget, gpointer data)
{
	printf ("Should report action\n");
}

void
generic_button_setup (char *sym, GtkWidget *widget, int should_dump)
{
	bind_sym (sym, widget);
	place (widget);

	if (should_dump)
		gtk_signal_connect (GTK_OBJECT (widget), "clicked", (GtkSignalFunc) button_dump_form, 0);
	else
		gtk_signal_connect (GTK_OBJECT (widget), "clicked", (GtkSignalFunc) report_action, 0);
}

		      
void
create_button (char *str)
{
	GtkWidget *button;
	char **strs;

	strs = split_string (str, 2);
	button = gtk_button_new_with_label (strs [2]);
	gtk_widget_show (button);
	generic_button_setup (strs [0], button, strs [1][0] == '1');
	free (strs);
}

void
create_button_fill (char *str)
{
	GtkWidget *button, *label;
	char **strs;

	strs = split_string (str, 1);
	button = gtk_button_new ();
	label = gtk_label_new (strs [1]);
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_container_add (GTK_CONTAINER(button), label);
	gtk_widget_show (label);
	gtk_widget_show (button);
	generic_button_setup (strs [0], button, 0);
	free (strs);
}

void
internal_create_button_xpm (char *id, char *fname, char *text)
{
	GtkWidget *button, *pixmap, *box, *container, *label;

	button = gtk_button_new ();
	/* We have a text */
	if (text [0]){
		container = box = gtk_vbox_new (0, 0);
		label = gtk_label_new (text);
		gtk_widget_show (label);
	} else {
		container = button;
		label = NULL;
		box = NULL;
	}
	pixmap = gnome_create_pixmap_widget (current_window, container, fname);
	unlink (fname);
	if (label){
		gtk_box_pack_start_defaults (GTK_BOX(box), pixmap);
		gtk_box_pack_start_defaults (GTK_BOX(box), label);
		gtk_container_add (GTK_CONTAINER(button), box);
		gtk_widget_show (container);
	} else
		gtk_container_add (GTK_CONTAINER(button), pixmap);
	gtk_widget_show (pixmap);
	gtk_widget_show (button);
	generic_button_setup (id, button, 0);
}

void
create_button_xpm (char *str)
{
	char **strs;
	char *fname;
	
	strs = split_string (str, 2);
	fname = g_hash_table_lookup (current_hash, strs [1]);
	if (!fname){
		printf ("Pixmap id %s not defined\n", strs [1]);
		exit (1);
	}
	internal_create_button_xpm (strs [0], fname, strs [2]);
	free (strs);
}

void
create_button_xpmf (char *str)
{
	char **strs;

	strs = split_string (str, 2);
	internal_create_button_xpm (strs [0], strs [1], strs [2]);
	free (strs);
}

void
create_combo (char *str)
{
	GtkWidget *entry;
	char **strs;
	int maxlen;
	
	strs = split_string (str, 3);
	maxlen = atoi (strs [1]);
	entry = gtk_entry_new ();
	gtk_widget_show (entry);
	gtk_entry_set_text (GTK_ENTRY (entry), strs [2]);
	place (entry);
	bind_sym (strs [0], entry);
	free (strs);
}

void
combo_item (char *str)
{
	/* FIXME: not handled for now */
}

void
resize_option_menu(GtkWidget *widget, gpointer data)
{
	GtkOptionMenu *omenu;

	omenu = GTK_OPTION_MENU (data);
	gtk_option_menu_set_menu (omenu, widget);
}

void
create_choice (char *str)
{
	GtkWidget *choice, *menu;
	char **strs;
	
	strs = split_string (str, 1);
	choice = gtk_option_menu_new ();
	menu   = gtk_menu_new ();
	gtk_option_menu_set_menu (GTK_OPTION_MENU (choice), menu);
	place (choice);
	bind_sym (strs [0], menu);
	printf ("binding: %s to %p\n", strs [0], menu);
	gtk_widget_show (choice);

	/* FIXME: This is a hack to work around a bug in Gtk: When
	 * menu items are added to a menu which is attached to an
	 * option_menu, *after* the attachment has been made, the
	 * option_menu is not resized.  This should be done inside
	 * GtkOptionMenu.
	 */
	
	gtk_signal_connect(GTK_OBJECT(menu), "need_resize",
			   (GtkSignalFunc) resize_option_menu,
			   choice);
	free (strs);
}

void
choice_item (char *str)
{
	GtkWidget *menu, *item;
	char **strs;

	strs = split_string (str, 1);
	menu = g_hash_table_lookup (current_hash, strs [0]);
	if (!menu){
		printf ("Can not find %s\n", strs [0]);
		return;
	}
	printf ("Anado: %s\n", strs [1]);
	item = gtk_menu_item_new_with_label (strs [1]);
	gtk_menu_append (GTK_MENU (menu), item);
	gtk_widget_show (item);
	free (strs);
}

void
create_checkbox (char *str)
{
	GtkWidget *check;
	char **strs;

	strs = split_string (str, 2);
	check = gtk_check_button_new_with_label (strs [2]);
	gtk_widget_show (check);
	gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON(check), *strs [1] == '0' ? 0 : 1);
	bind_sym (strs [0], check);
	place (check);
	free (strs);
}

void
create_group (char *str)
{
	GtkWidget *group, *table;
	GHashTable *wtab;
	char **strs;

	strs = split_string (str, 1);
	group = gtk_frame_new (strs [1][0] ? strs [1] : NULL);
	gtk_widget_show (group);
	table = new_table ();
	gtk_widget_show (table);
	bind_sym (strs [0], table);
	place (group);
	gtk_container_add (GTK_CONTAINER(group), table);

	/* Nest */
	wtab = g_hash_table_new (g_string_hash, g_string_equal);
	gtk_object_set_user_data (GTK_OBJECT(table), wtab);
	stack_top++;
	current_table = table;
	current_column = current_line = 0;
	current_type = FORM_TABLE;

	free (strs);
}

void
create_label (char *str)
{
	GtkWidget *label;

	label = gtk_label_new (str);
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_widget_show (label);
	place (label);
}

void
create_page (char *str)
{
	GtkWidget *label, *table;
	GHashTable *wtab; 
	char **strs;

	strs = split_string (str, 1);
	label = gtk_label_new (strs [1]);
	gtk_widget_show (label);
	table = new_table ();
	gtk_widget_show (table);
	bind_sym (strs [0], table);
	place (table);
	gtk_notebook_append_page (GTK_NOTEBOOK(current_book), table, label);

	/* Nest */
	wtab = g_hash_table_new (g_string_hash, g_string_equal);
	gtk_object_set_user_data (GTK_OBJECT (table), wtab);
	stack_top++;
	current_table = table;
	current_column = current_line = 0;
	current_type = FORM_TABLE;
	current_hash = wtab;

	free (strs);
}

void
create_string (char *str)
{
	GtkWidget *entry;
	char **strs;
	int  maxlen;
	
	strs = split_string (str, 2);
	maxlen = atoi (strs [1]);
	entry = gtk_entry_new ();
	gtk_widget_show (entry);
	gtk_entry_set_text (GTK_ENTRY(entry), strs [2]);
	place (entry);
	bind_sym (strs [0], entry);
	free (strs);
}

void
create_hline (char *str)
{
	GtkWidget *sep;
	
	sep = gtk_hseparator_new ();
	gtk_widget_show (sep);
	place (sep);
}

void
newline (char *str)
{
	current_line++;
	current_column = 0;
}

void
end (char *str)
{
	if (stack_top)
		stack_top--;
	else {
		gtk_widget_show (current_window);
		/* bad hack: force MainForm  :-) */
		form_stack [0].table = 0;
	}
}

void
skip (char *str)
{
	current_column += atoi (str);
}

float
h_align (char *str)
{
	if (*str == 'l')
		return 0.0;
	if (*str == 'r')
		return 1.0;
	return 0.5;
}

float
v_align (char *str)
{
	if (*str == 't')
		return 0.0;
	if (*str == 'b')
		return 1.0;
	return 0.5;
}

/*
 * When we receive this command, it means that we have to change the layout of
 * the last inserted widget.
 * This also provides alignement contraints.
 */
void
dispolast (char *str)
{
	GtkWidget *align;
	char **strs;
	int  column;

	column = current_column - 1;
	strs = split_string (str, 4);
	gtk_container_remove (GTK_CONTAINER(current_table), last_widget);
	
	align = gtk_alignment_new (h_align (strs [0]), v_align (strs [2]), 1.0, 1.0);
	gtk_widget_show (align);
	gtk_table_attach_defaults (GTK_TABLE (current_table), align,
				   column,       column + atoi (strs [1]),
				   current_line, current_line + atoi (strs [3]));
	gtk_container_add (GTK_CONTAINER(align), last_widget);
	free (strs);
}

void
dump (char *str)
{
	
}

void
delete (char *str)
{
	printf ("delete invoked\n");
}

void
unimplemented (char *str)
{
}

void
str (char *str)
{
	if (reg){
		reg_size += strlen (str) + 1;
		reg = realloc (reg, reg_size);
		strcat (reg, str);
	} else {
		reg = strdup (str);
		reg_size = strlen (reg + 1);
	}
}

void
xfer_xpm (char *str)
{
	FILE *f;
	char *fname = tmpnam (NULL);

	f = fopen (fname, "w");
	if (f == NULL)
		return;
	fwrite (reg, 1, strlen (reg), f);
	fclose (f);

	bind_sym (str, g_strdup (fname));
	free (reg);
	reg = 0;
	reg_size = 0;
}

void
sidetitle (char *str)
{
	/* nothing */
}

struct {
	char *name;
	void (*cmd)(char *str);
} cmds [] = {
	{ "Book",        create_book },
	{ "Button",      create_button },
	{ "ButtonFill",  create_button_fill },
	{ "Checkbox",    create_checkbox },
	{ "Choice",      create_choice },
	{ "Choice_item", choice_item }, 
	{ "Combo",       create_combo },
	{ "Combo_item",  combo_item },
	{ "Delete",      delete },
	{ "Dispolast",   dispolast },
	{ "Dump",        dump },
	{ "End",         end },
	{ "Form",        create_form },
	{ "FormButton",  create_form_button },
	{ "Group",       create_group },
	{ "Hline",       create_hline },
	{ "Label",       create_label },
	{ "MainForm",    create_form  },
	{ "Newline",     newline },
	{ "Page",        create_page },
	{ "Sidetitle",   sidetitle },
	{ "Skip",        skip },
	{ "Str",         str },
	{ "String",      create_string },
	{ "Xfer_xpm",    xfer_xpm },
	{ NULL, NULL }
};

struct {
	void (*cmd)(char *str);
} cmds_bynumber [] = {
	{ unimplemented },	/* 0. indexing starts at 1 */
	{ str },		/* 1 str */
	{ newline },		/* 2 */
	{ skip },		/* 3 */
	{ create_hline },	/* 4 */
	{ dispolast },		/* 5 */
        { create_label },	/* 6 */
	{ create_string },	/* 7 */
	{ create_checkbox },	/* 8 */
	{ unimplemented },	/* 9 radio */
	{ create_button },	/* 10 */
	{ create_button_xpm },	/* 11 button xpm */
	{ create_button_xpmf },	/* 12 button xpmf */
	{ create_choice },	/* 13 */
	{ choice_item },	/* 14 */
	{ unimplemented },	/* 15 */
	{ unimplemented },	/* 16 */
	{ create_combo },	/* 17 */
	{ combo_item },		/* 18 */
	{ create_group },	/* 19 */
	{ create_form },	/* 20 */
	{ create_page },	/* 21 */
	{ create_book },	/* 22 */
	{ create_form },	/* 23 */
	{ end },		/* 24 */
	{ delete },		/* 25 */
	{ dump },		/* 26 */
	{ unimplemented },	/* 27 icon xpm */
	{ xfer_xpm },		/* 28 xfer xpm */
	{ sidetitle },		/* 29 sidetitle */
	{ unimplemented },	/* 30 fill */
	{ create_form_button },	/* 31 formbutton */
	{ unimplemented },	/* 32 groupfit */
	{ unimplemented },	/* 33 setweightlast */
	{ create_label },	/* 34 FIXME: richtext */
	{ create_string },	/* 35 FIXME: password */
	{ create_button_fill },	/* 36 buttonfill */
	{ unimplemented },	/* 37 enteraction */
	{ unimplemented },	/* 38 curfield */
	{ unimplemented }	/* 39 last */
};
	
void
process (char *str)
{
	char *space;
	int i;
		
	while (*str && (*str == ' ' || *str == '\t'))
		str++;

	space = strchr (str, ' ');
	if (space)
		*space = 0;

	if (isdigit (str [0])){
		i = atoi (str);
		if (cmds_bynumber [i].cmd == unimplemented)
			printf ("unimp: %d\n", i);
		(*cmds_bynumber [i].cmd) (space+1);
	} else 
		for (i = 0; cmds [i].name; i++){
			if (strcmp (cmds [i].name, str))
				continue;
			
			printf ("comando: %s\n", cmds [i].name);
			(*cmds [i].cmd) (space+1);
		}
}

int
main (int argc, char *argv [])
{
	char buffer [BSIZE];
	char *s;
	
	gnome_init (&argc, &argv);
	toplevels = g_hash_table_new (g_string_hash, g_string_equal);
	
	while (fgets (buffer, BSIZE, stdin)){
		if ((s = strchr (buffer, '\n')))
			*s = 0;
		process (buffer);
	}
	gtk_main ();
	return 0;
}
