#include <gnome.h>
#include <stdio.h>
#include <string.h>

#include "details.h"
#include "procview.h"
#include "graph.h"

#define NUM_INFO 14
#define MAP_COLS 7

GtkWidget *dwin = NULL;

static GtkWidget *nb;

static GtkWidget *icl = NULL;
static GtkWidget *ils [NUM_INFO];

static GtkWidget *gswin = NULL;
static Graph *gg;

static GtkWidget *mcl = NULL;

static ProcMapRow **rows = NULL;

static void display_details (gint pid);

static void add_info (gint pid);
static void add_mem_map (gint pid);
static void add_mem_graph (gint pid);

static int  delete_event_handler (GtkWidget *widget, gpointer gp);

void
procview_details (gint pid)
{
	GtkWidget *b;
	GtkWidget *vb, *hb;
	gchar *title;
	gchar *form;

	if (!dwin) {
		dwin = gtk_window_new (GTK_WINDOW_TOPLEVEL);
		gtk_widget_set_name (dwin, "GTopDetails");

		b = gtk_button_new_with_label (_("Close"));
		vb = gtk_vbox_new (FALSE, 0);
		hb = gtk_hbox_new (FALSE, 0);
		nb = gtk_notebook_new ();

		gtk_widget_set_usize (dwin, 780, 360);
		gtk_window_set_policy (GTK_WINDOW (dwin), TRUE, TRUE, FALSE);
		gtk_container_border_width (GTK_CONTAINER (hb), GNOME_PAD);

		gtk_box_pack_start_defaults (GTK_BOX (vb), nb);
		gtk_box_pack_end (GTK_BOX (hb), b, FALSE, FALSE, 0);
		gtk_box_pack_start (GTK_BOX (vb), hb, FALSE, FALSE, 0);
		gtk_container_add (GTK_CONTAINER (dwin), vb);

		gtk_signal_connect (GTK_OBJECT (dwin), "delete_event",
				    GTK_SIGNAL_FUNC (delete_event_handler), NULL);
		gtk_signal_connect (GTK_OBJECT (b), "clicked",
				    GTK_SIGNAL_FUNC (delete_event_handler), NULL);

		gtk_widget_show (nb);
		gtk_widget_show (b);
		gtk_widget_show (hb);
		gtk_widget_show (vb);
	}

	form = _("Process %d details");
	title = g_new (char, strlen (form)+64);
	sprintf (title, form, pid);
	gtk_window_set_title (GTK_WINDOW (dwin), title);
	g_free (title);

	display_details (pid);

	gtk_widget_show (dwin);
	gdk_window_raise (dwin->window);
}

static int
delete_event_handler (GtkWidget *widget, gpointer gp)
{
	/* printf ("details: delete_event\n"); */

        gtk_widget_destroy (dwin);

	dwin = icl = mcl = gswin = NULL;

        return TRUE;
}

static void
display_details (gint pid)
{
	add_info (pid);
	add_mem_map (pid);
	add_mem_graph (pid);
}

static void
set_info (gint pid)
{
	ProcProcData d;
	PROCTAB *pt;
	proc_t *t;
	proc_field *f = p_fields;
	gint i;
	pid_t p [2];
	gchar *v;

	p [0] = pid;
	p [1] = 0;
	pt = openproc (PROC_FILLMEM |
		       PROC_FILLCMD |
		       PROC_FILLENV |
		       PROC_FILLTTY |
		       PROC_FILLUSR |
		       PROC_PID, p);


	t = (pt) ? readproc (pt, NULL) : NULL;
	d.p = t;

	/* FIXME! */
	d.pcpu = d.pmem = 0;

	gtk_clist_freeze (GTK_CLIST (icl));
	for (i = 0; i < NUM_INFO; i++) {
		if (f->label) {
			v = (t) ? sprint_fmt (&d, f->fmt) : "";
			gtk_clist_set_text (GTK_CLIST (icl),
					    i, 1, v);
			/* printf ("clist set %s\n", v); */
			if (t)
				g_free (v);
			
			f++;
		}
	}
	gtk_clist_thaw (GTK_CLIST (icl));

	closeproc (pt);
	if (t)
		freeproc (t);
}

static void
add_info (gint pid)
{
	if (!icl) {
		GtkWidget *l;
		gint i;
		gchar *t [3];

		l = gtk_label_new _("Process info");

		t [0] = _("Attribute");
		t [1] = _("Value");
		t [2] = "";
		icl = gtk_clist_new_with_titles (3, t);

		gtk_clist_column_titles_show (GTK_CLIST (icl));
		gtk_clist_set_column_justification (GTK_CLIST (icl),
						    0, GTK_JUSTIFY_RIGHT);
		gtk_clist_set_column_justification (GTK_CLIST (icl),
						    1, GTK_JUSTIFY_RIGHT);

		for (i = 0; i < NUM_INFO; i++) {
			t [0] = p_fields [i].long_info;
			t [1] = t [2] = "";
			
			gtk_clist_append (GTK_CLIST (icl), t);
			/* printf ("clist set %s\n", p_fields [i].long_info); */
		}

		gtk_clist_set_policy (GTK_CLIST (icl),
				      GTK_POLICY_AUTOMATIC,
				      GTK_POLICY_AUTOMATIC);

		gtk_clist_set_column_width (GTK_CLIST (icl), 0, 220);
		gtk_clist_set_column_width (GTK_CLIST (icl), 1, 140);

		gtk_container_border_width (GTK_CONTAINER (icl), GNOME_PAD);

		gtk_widget_show (icl);
		gtk_widget_show (l);

		gtk_notebook_append_page (GTK_NOTEBOOK (nb), icl, l);
	}

	set_info (pid);
}

static ProcMapRow **
get_map_rows (pid_t pid, ProcMapRow **rs)
{
	FILE *maps;
	gchar *form = "/proc/%d/maps";
	gchar *f;
	ProcMapRow *row;
	ProcMapRow **rows = NULL;
	GList *c, *list = NULL;
	gint rv;
	gint i, n = 0;
	gchar fn [4096];

	/* free old rows */

	if (rs) {
		ProcMapRow **c = rs;
		/* i = 0; */
		while (*c) {
			if ((*c)->filename)
				g_free ((*c)->filename);
			g_free (*c);
			c++;
			/* i++; */
		}

		/* printf ("freed: %d\n", i); */
		g_free (rs);
	}

	/* now we sample new ones */

	f = g_new (char, strlen (form)+64);
	sprintf (f, form, pid);
	maps = fopen (f, "r");
	g_free (f);

	if (maps) {
		/* i = 0; */
		do {
			row = g_new (ProcMapRow, 1);
			/* i++; */
			list = g_list_append (list, row);

			rv = fscanf (maps,
				     "%08lx-%08lx %4c\n %08lx "
				     "%02hx:%02hx %ld",
				     &row->VMstart, &row->VMend,
				     &row->flags, &row->VMoffset,
				     &row->dev_major, &row->dev_minor,
				     &row->inode);

			row->flags [4] = 0;

			fn [0] = fgetc (maps);
			/* printf ("fn [0] %c %x\n", fn [0], fn [0]); */

			if (fn [0] != '\n' && fn [0] != EOF) {

				fscanf (maps, "%*[ ]%[^\n]\n", &fn);
				row->filename = g_strdup (fn);

			} else row->filename = NULL;

			/* if (rv != EOF)
				printf ("%08lx-%08lx %s\n",
					row->VMstart, row->VMend,
					row->filename); */

			n++;

		} while (rv != EOF && rv && fn [0] != EOF);

		/* printf ("allocated: %d\n", i); */

		g_free (row);

		rows = g_new (ProcMapRow *, n);
		rows [n-1] = NULL;
		n--;

		for (i = 0, c = list; i < n; i++) {
			rows [i] = c->data;
			c = c->next;

			/* row = rows [i];
			   printf ("%08lx-%08lx %s\n",
			   row->VMstart, row->VMend,
			   row->filename); */
		}
		g_list_free (list);
	}

	fclose (maps);

	return rows;
}

static void
set_mem_map (ProcMapRow **rows)
{
	gchar *t [MAP_COLS];
	gchar buf [6][64];

	gtk_clist_freeze (GTK_CLIST (mcl));
	gtk_clist_clear (GTK_CLIST (mcl));

	if (rows)
		for (; *rows; rows++) {
			sprintf (buf [0], "%08lx", (*rows)->VMstart);
			t [0] = buf [0];
			sprintf (buf [1], "%08lx", (*rows)->VMend);
			t [1] = buf [1];
			t [2] = (*rows)->flags;
			sprintf (buf [2], "%08lx", (*rows)->VMoffset);
			t [3] = buf [2];
			sprintf (buf [3], "%02hx:%02hx",
				 (*rows)->dev_major,
				 (*rows)->dev_minor);
			t [4] = buf [3];
			sprintf (buf [4], "%ld", (*rows)->inode);
			t [5] = buf [4];
			t [6] = (*rows)->filename;
			
			gtk_clist_append (GTK_CLIST (mcl), t);
		}
	
	gtk_clist_thaw (GTK_CLIST (mcl));
}

static void
add_mem_map (gint pid)
{
	if (!mcl) {
		GtkWidget *l;
		gchar *t [MAP_COLS];

		l = gtk_label_new _("Raw memory map");

		t [0] = _("VM start");
		t [1] = _("VM end");
		t [2] = _("Flags");
		t [3] = _("VM offset");
		t [4] = _("Device");
		t [5] = _("Inode");
		t [6] = _("Filename");
		mcl = gtk_clist_new_with_titles (MAP_COLS, t);

		gtk_clist_set_policy (GTK_CLIST (mcl),
				      GTK_POLICY_AUTOMATIC,
				      GTK_POLICY_AUTOMATIC);

		gtk_clist_set_column_width (GTK_CLIST (mcl), 0, 90);
		gtk_clist_set_column_width (GTK_CLIST (mcl), 1, 90);
		gtk_clist_set_column_width (GTK_CLIST (mcl), 2, 50);
		gtk_clist_set_column_width (GTK_CLIST (mcl), 3, 90);
		gtk_clist_set_column_width (GTK_CLIST (mcl), 4, 60);
		gtk_clist_set_column_width (GTK_CLIST (mcl), 5, 90);
		gtk_clist_set_column_width (GTK_CLIST (mcl), 6, 200);

		gtk_clist_set_column_justification (GTK_CLIST (mcl),
						    0, GTK_JUSTIFY_RIGHT);
		gtk_clist_set_column_justification (GTK_CLIST (mcl),
						    1, GTK_JUSTIFY_RIGHT);
		gtk_clist_set_column_justification (GTK_CLIST (mcl),
						    2, GTK_JUSTIFY_RIGHT);
		gtk_clist_set_column_justification (GTK_CLIST (mcl),
						    3, GTK_JUSTIFY_RIGHT);
		gtk_clist_set_column_justification (GTK_CLIST (mcl),
						    4, GTK_JUSTIFY_RIGHT);
		gtk_clist_set_column_justification (GTK_CLIST (mcl),
						    5, GTK_JUSTIFY_RIGHT);

		gtk_container_border_width (GTK_CONTAINER (mcl), GNOME_PAD);

		gtk_widget_show (mcl);
		gtk_widget_show (l);

		gtk_notebook_append_page (GTK_NOTEBOOK (nb), mcl, l);		
	}

	rows = get_map_rows (pid, rows);

	set_mem_map (rows);
}

static gchar *
mem_seg (ProcMapRow **rs)
{
	ProcMapRow *r = *rs;
	gchar *f = r->flags;

	if (r->filename) {
		if (!strcmp (f, "rw-p"))
			return "Data";
		else if (!strcmp (f, "r-xp"))
			return "Text";
		else if (f[0] == 'r')
			return (f[1]=='w') ? "RW" : "RO";

	} else {
		if (!strcmp (f, "rwxp") && !*(rs+1))
			return "Stack";
		if (!strcmp (f, "---p"))
			return "Hole";
		else if (f[0]=='r')
			return (f[1]=='w') ? "RW" : "RO";
	}

	return "?????";
}

gpointer
mem_graph_data_fn (GraphCmd cmd, gpointer data)
{
	ProcMapRow **rs = data;
	static gchar buf [256];

	switch (cmd) {
	case GRAPH_FIRST:
		return (*rows) ? rows : NULL;
	case GRAPH_NEXT:
		return (*(rs+1)) ? rs+1 : NULL;
	case GRAPH_VALUE:
		return (gpointer)((*rs)->VMend - (*rs)->VMstart);
	case GRAPH_LABEL:
		sprintf (buf, "(%5s) : %8dk  %s",
			 mem_seg (rs),
			 ((*rs)->VMend - (*rs)->VMstart) >> 10,
			 ((*rs)->filename) ? strrchr ((*rs)->filename, '/')+1 : "");
		
		return buf;
	default:
		return NULL;
	}
}

#define GRAPH_COLORS 4

static GdkColor graph_colors [GRAPH_COLORS] = {
	{0, 0xefff, 0xefff, 0x5fff},
	{0, 0x8fff, 0xefff, 0x8fff},
	{0, 0x8fff, 0xafff, 0xefff},
	{0, 0xefff, 0xafff, 0xafff},
};


static void
add_mem_graph (gint pid)
{
	if (!gswin) {
		GtkWidget *l;
		GtkWidget *da;
		GtkWidget *f;

		l = gtk_label_new _("Graphical memory map");
		gswin = gtk_scrolled_window_new (NULL, NULL);
		gtk_widget_set_name (gswin, "GraphMap");
		gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (gswin),
						GTK_POLICY_AUTOMATIC,
						GTK_POLICY_AUTOMATIC);

		f = gtk_frame_new (NULL);
		gtk_frame_set_shadow_type (GTK_FRAME (f), GTK_SHADOW_NONE);
		gg = graph_new (mem_graph_data_fn);
		da = graph_widget (gg);
		graph_colors_set (gg, graph_colors, GRAPH_COLORS);

		gtk_container_border_width (GTK_CONTAINER (gswin), GNOME_PAD);
		gtk_container_border_width (GTK_CONTAINER (f), GNOME_PAD << 1);

		gtk_container_add (GTK_CONTAINER (f), da);
		gtk_container_add (GTK_CONTAINER (gswin), f);

		gtk_widget_show (l);
		gtk_widget_show (da);
		gtk_widget_show (f);
		gtk_widget_show (gswin);

		gtk_notebook_append_page (GTK_NOTEBOOK (nb), gswin, l);		
	}

	graph_update (gg);
	gtk_widget_queue_resize (gswin);
}
