/*
 * GXSNMP -- An snmp mangament application
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <time.h>
#include <stdio.h>   
#include <getopt.h>
#include <strings.h>
#include <signal.h>
#include <errno.h>
#include <malloc.h>
#include <sys/time.h>
#include <sys/socket.h>

#include "collector.h" 

/* The 3 queues representing the 3 states an SNMP variable can be in:
 * - w_queue: Currently waiting until a PDU slot gets available.
 * - s_queue: Currently waiting for an answer from the SNMP agent. Variables
 *            are grouped to _struct pdu, so they can be found easier when
 *            a response arrives.
 * - i_queue: Currently idle, maybe waiting for the next time for a request.
 */

GSList *w_queue, *i_queue, *s_queue;

/* Global counter of active PDUs waiting for a reply */

int pducnt=0;

/* Global SNMP ID (used together with the IP address to identify a request */

int id = 0;

/* Global fd we use for all our requests */

int fd = 0;

/* Encode a request to the SNMP agent. Multiple variables can be queried. */

static void insert_oid(gpointer data, gpointer lptr) {
  SNMP_OBJECT  *objs;
  struct _oid  *myoid = (struct _oid*) data;
  GSList      **olist = (GSList **) lptr;

  objs = g_malloc(sizeof(SNMP_OBJECT));

  objs->request   = 0;
  memcpy(&objs->id, &myoid->name, myoid->name_length * sizeof(glong));
  objs->id_len     = myoid->name_length;
  objs->type       = SNMP_NULL;
  objs->syntax_len = 0;

  *olist = g_slist_append(*olist, objs);
}

static struct _pdu *build_pdu(GSList *oids, struct _host *host) {
  struct _pdu  *mypdu;
  struct _oid  *myoid;
  GSList       *olist;
  SNMP_PDU      spdu;
  SNMP_AUTH     auth;
  gint          len;
  guchar        buffer[MAX_DGRAM_SIZE], *ptr;

  mypdu = (struct _pdu *) g_malloc(sizeof(struct _pdu));

  mypdu->oids = oids;
  mypdu->host = host;
  mypdu->retry_count = retry;
  mypdu->id   = ++id;

  ptr = buffer;
  olist = NULL;

  g_slist_foreach(oids, insert_oid, &olist);

  spdu.request.type = SNMP_PDU_GET;
  spdu.request.id   = mypdu->id;
  spdu.request.error_status = 0;
  spdu.request.error_index  = 0;

  len = sizeof(buffer);

  myoid = oids->data;

  strcpy(auth.name, host->community);
  auth.nlen = strlen(host->community);
  auth.type = AUTH_COMMUNITY;
  g_snmp_encode(&ptr, &len, &spdu, 
                &auth,
                SNMP_V1,
                olist);

  g_slist_free(olist);
  mypdu->buffer = malloc(len);
  memcpy(mypdu->buffer, ptr, len);
  mypdu->length = len;

  return mypdu;
}

/* Move variables with duration=0 back to w_queue. Used for SIGHUP */
void reget_static() {
  GSList      *id;
  struct _oid *myoid;

  id = i_queue;
  while(id) {
    myoid = id->data;
    id    = g_slist_next(id);
    if (!myoid->time) {
      s_queue = g_slist_append(s_queue, myoid);
      i_queue = g_slist_remove(i_queue, myoid);
    }
  }
}

static void set_vars(gpointer data, gpointer lptr) {
  struct _pdu * mypdu = (struct _pdu *) lptr;
  SNMP_OBJECT * obj   = (SNMP_OBJECT *) data;

  GSList      * id  = mypdu->oids;

  struct _oid *myoid;

  while (id) {
    myoid = id->data;
    id = g_slist_next(id);
    if (!memcmp(obj->id,myoid->name,myoid->name_length*sizeof(glong))) {
      output(myoid,obj,mypdu->now);
      if (myoid->duration) myoid->time = myoid->duration + mypdu->now;
        else myoid->time = 0;
      return;
    }
  }
}

void insert_oids(GSList *oids) {
  while (oids) {
    i_queue = g_slist_append(i_queue, oids->data);
    oids =    g_slist_remove(oids, oids->data);
  }
}

static void snmp_receive() 
{
  unsigned char buffer[MAX_DGRAM_SIZE];
  int adrsize, len, objlen, comsize, i, version;
  struct sockaddr_in address;
  GSList       *objs;
  SNMP_PDU      spdu;
  SNMP_AUTH     auth;
  GSList       *id;
  struct _pdu  *mypdu;

  /*
   * need to init adrsize properly 
   */
  adrsize = sizeof (address);
  len = recvfrom(fd, buffer, sizeof(buffer), 0, (struct sockaddr *)&address, 
                 &adrsize);
  /*
   * Need to check the return val of recvfrom
   */
  if (debug)
    g_print ("recvfrom len %d\n", len);

  g_snmp_decode(buffer, len, &spdu, &auth, &version, 
                &objs);

  if (debug) 
    printf("Response: reqid = %d, op=%d\n", spdu.request.id, 
	   spdu.request.type);
  
  id = s_queue;

  while (id) 
    {
      mypdu = id->data;
      if (mypdu->id == spdu.request.id) 
	break;
      id = g_slist_next(id);
    }
  mypdu->now = time(NULL);
  if (!id) 
    {
      if (debug) 
	printf("No corresponding request outstanding\n");
      g_slist_free(objs);
      return;
    }
  if (memcmp(auth.name, mypdu->host->community, auth.nlen)) 
    {
      if (debug) 
	printf("Wrong community\n");
      g_slist_free(objs);
      return;
    }
  /*
   * On my machine the &address and mypdu->host->address never matches
   * but by looking at the code it appears you just want to compare 
   * IP addresses.. The modifications I've made do just that.
   * Before I altered it I was getting the following
   * Strange Sender IP Address 130.205.65.21 130.205.65.21
   * hmm...
   */
  if (memcmp(&address.sin_addr, &mypdu->host->address.sin_addr, 
	     sizeof (struct in_addr))) 
    {
      if (debug) 
	printf ("Strange Sender IP Address %s %s\n", 
		inet_ntoa(address.sin_addr),
		inet_ntoa(mypdu->host->address.sin_addr));
      g_slist_free(objs);
      return;
    }
  if (debug) printf("Success!\n");
  g_slist_foreach(objs, set_vars, mypdu);

  insert_oids(mypdu->oids); 
  g_slist_free(objs);
  s_queue = g_slist_remove(s_queue, mypdu);
  return;
}

static send_get(struct _pdu *pdu) {
  sendto(fd, pdu->buffer, pdu->length, 0, &pdu->host->address, sizeof(pdu->host->address));
  pdu->time = time(NULL) + timeout;
}

static get(GSList *oid, struct _host *host) {
  struct _pdu *mypdu;

  mypdu = build_pdu(oid, host);
  send_get(mypdu);
  s_queue   = g_slist_append(s_queue, mypdu);
}

void process_table() {
  int numfds;
  fd_set fdset;
  int count, block;
  struct timeval to, *tv;
  time_t now;
  GSList *tmplist;
  GSList *oidlist;
  struct _oid *myoid, *myoid2;
  struct _pdu *mypdu;

/* This is the main loop of this collector. In short words, do the following:
   - check for i_queue entries with expired time and move them to w_queue.
   - check if there are any entries in w_queue.
     - If yes, check if an SNMP slot if free (maxpdu)
       - If yes, build PDU and send it
     - If no, get time of nearest i_queue entry and prepare select with block=0
   - call select()
   - start all over again
*/

  i_queue = NULL;
  s_queue = NULL;
  fd = socket(AF_INET, SOCK_DGRAM, 0);
  while (1) {
    block = 1;
    now = time(NULL);

/* Search through all i_queue entries */

    tmplist = i_queue;
    while(tmplist) {
      myoid = tmplist->data;
      tmplist = g_slist_next(tmplist);

      if (myoid->time) {
        if (myoid->time <= now) {
          w_queue = g_slist_append(w_queue, myoid);
          i_queue = g_slist_remove(i_queue, myoid);
        } else {
          if (block) {
            block = 0;
            to.tv_sec = MAX(myoid->time - now, 0);
          } else {
            if (myoid->time - now < to.tv_sec) to.tv_sec = MAX(myoid->time - now, 0);
          }
        }
      }
    } 

/* Search through all s_queue entries */

    tmplist = s_queue;
    while(tmplist) {
      mypdu = tmplist->data;
      tmplist = g_slist_next(tmplist);

      if (mypdu->time <= now) {
        if (mypdu->retry_count) {
          mypdu->retry_count--;
          send_get(mypdu);
        } else {
          oidlist = mypdu->oids;
          while (oidlist) {
            myoid = oidlist->data;
            myoid->time = now + defer;
            oidlist = g_slist_next(oidlist);
          }
          insert_oids(mypdu->oids);
          s_queue = g_slist_remove(s_queue, mypdu);
          g_free(mypdu);
        }
      } else {
        if (block) {
          block = 0;
          to.tv_sec = MAX(mypdu->time - now, 0);
        } else {
          if (mypdu->time - now < to.tv_sec) to.tv_sec = MAX(mypdu->time - now, 0);
        }
      }
    } 

    while(w_queue && (pducnt < maxreq)) {
      oidlist = NULL;
      myoid = w_queue->data;

      oidlist = g_slist_append(oidlist, myoid);
      w_queue = g_slist_remove(w_queue, myoid);
      count = 1;

      tmplist = w_queue;
      while(tmplist && count < pdusize) {
        myoid2 = tmplist->data;
        tmplist = g_slist_next(tmplist);
        if (!strcmp(myoid2->host->name,myoid->host->name)) {
          w_queue = g_slist_remove(w_queue, myoid2);
          oidlist = g_slist_append(oidlist, myoid2);
          count ++;
        }
      }
      if (debug) printf("Getting %d variables\n",count);
      get(oidlist, myoid->host);
    }

    numfds = fd+1;
    FD_ZERO(&fdset);
    FD_SET(fd, &fdset);
    to.tv_usec = 0;
    if (block) to.tv_sec = 10;

    if (debug) printf("Select sleeps for %d seconds\n", to.tv_sec);
    count = select(numfds, &fdset, 0, 0, &to);
    if (debug) printf("Select returned with count=%d\n", count);
    if (count > 0){
      snmp_receive();
    } else switch(count){
      case 0:
        break;
      case -1:
        if (errno == EINTR){
          continue;
        } else {
          exit(1);
        }
      default:
        exit(2);
    }
  }
}
