/*
 * GXSNMP - An snmp managment application
 * Copyright (C) 1998 Gregory McLean
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc.,  59 Temple Place - Suite 330, Cambridge, MA 02139, USA.
 *
 * Network Map panel. (New and Improved to use the gnome-canvas)
 */

#include "config.h"
#include <gnome.h>
#include <gdk/gdkkeysyms.h>
#include "main.h"
#include "net_map.h"
#include "net.h"

/*
 * Forward declaration
 */

static void                create_map_panel     (void);
static void                delete_map_panel     (GtkWidget        *widget,
						 gpointer         data);
static void                close_panel_cb       (GtkWidget        *widget,
						 gpointer         data);
static gint                canvas_key_press     (GnomeCanvas      *widget,
						 GdkEventKey      *e,
						 gpointer         data);
static gint                canvas_item_event    (GnomeCanvasItem  *item,
						 GdkEvent         *e,
						 gpointer         data);
static gint                canvas_event         (GtkWidget        *widget,
						 GdkEvent         *e,
						 gpointer         data);
static gint                update_mouse_stat    (GtkWidget        *widget,
						 GdkEventMotion   *e,
						 gpointer         data);
static void                setup_item           (GnomeCanvasItem  *item,
						 gpointer         data);
static void                free_imlib_image     (GtkObject        *object,
						 gpointer         data);
static void                host_pixmap          (GnomeCanvasGroup *group,
						 double           x,
						 double           y);

/*
 * Menu callbacks 
 */
static void         quit_app_cb                 (GtkWidget        *widget,
						 gpointer         data);
/*
 * Toolbar callbacks
 */
static void         add_host_cb                 (GtkWidget        *widget,
						 gpointer         data);
static void         host_list_cb                (GtkWidget        *widget,
						 gpointer         data);
/*
 * Global variables needed for this panel
 */


/*
 * Local variables for the network map panel
 */

static GtkWidget      *map_panel = NULL;

static GnomeUIInfo map_file_menu[] = {
  { GNOME_APP_UI_ITEM, N_("New"), NULL, NULL, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_NEW, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Open..."), NULL, NULL, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_OPEN, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Save"), NULL, NULL, NULL, NULL, 
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Save As..."), NULL, NULL, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE_AS, 0, 0, NULL },
  { GNOME_APP_UI_SEPARATOR },
  { GNOME_APP_UI_ITEM, N_("Print..."), NULL, NULL, NULL, NULL, 
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PRINT, 'P', GDK_CONTROL_MASK,
    NULL },
  { GNOME_APP_UI_SEPARATOR },
  { GNOME_APP_UI_ITEM, N_("Close"), NULL, NULL, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CLOSE, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Exit"), NULL, quit_app_cb, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_EXIT, 'X', GDK_CONTROL_MASK, 
    NULL },
  { GNOME_APP_UI_ENDOFINFO }
};

static GnomeUIInfo map_edit_menu[] = {
  { GNOME_APP_UI_ITEM, N_("Undo"), NULL, NULL, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0, 0, NULL },
  { GNOME_APP_UI_ENDOFINFO }
};

static GnomeUIInfo map_panel_menu[] = {
  { GNOME_APP_UI_SUBTREE, N_("File"), NULL, map_file_menu, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
  { GNOME_APP_UI_SUBTREE, N_("Edit"), NULL, map_edit_menu, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
  { GNOME_APP_UI_ENDOFINFO }
};

/*
 * Toolbar.. (gnome style)
 */
static GnomeUIInfo map_toolbar[] = {
  { GNOME_APP_UI_ITEM, N_("Add Host"), 
    N_("Add a new host to the database."), add_host_cb, NULL, NULL,
    GNOME_APP_PIXMAP_FILENAME, "add_host.xpm", 'A', GDK_MOD1_MASK, 
    NULL },
  { GNOME_APP_UI_ITEM, N_("Delete Host"), 
    N_("Delete a host from the database."), NULL, NULL, NULL,
    GNOME_APP_PIXMAP_FILENAME, "del_host.xpm", 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Host list"), 
    N_("Open the list of known hosts."), host_list_cb, NULL, NULL,
    GNOME_APP_PIXMAP_FILENAME, "host_list.xpm", 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Network list"), NULL, NULL, NULL, NULL,
    GNOME_APP_PIXMAP_FILENAME, "net_list.xpm", 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Mib Browser"), NULL, NULL, NULL, NULL, 
    GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Event Browser"), NULL, NULL, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
  { GNOME_APP_UI_ENDOFINFO }
};
  
/*
 * Local functions 
 */
static void
create_map_panel ()
{
  GtkWidget        *window_frame;
  GtkWidget        *map_table;
  GtkWidget        *separator_bar;
  GtkWidget        *logo_widget;
  GtkWidget        *mouse_stat;              /* The mouse x y on the map */
  GtkWidget        *map_stat;                /* Any stat info for the map */
  GtkWidget        *canvas;
  GtkWidget        *scrolled_window;
  GtkWidget        *frame;
  GtkWidget        *led_bar;
  GtkWidget        *s_bar;
  GnomeCanvasGroup *root;
  gchar            *logo_pixmap_filename;

  map_panel = gnome_app_new ("GXSNMP", _("New Network map"));
  gtk_signal_connect (GTK_OBJECT (map_panel), "delete_event",
		      (GtkSignalFunc) delete_map_panel,
		      NULL);

  window_frame = gtk_frame_new (NULL);
  gtk_container_border_width (GTK_CONTAINER (window_frame), 4);

  map_table = gtk_table_new (8, 7, FALSE);
  gtk_object_set_data (GTK_OBJECT (map_panel), "map_table", map_table);
  gtk_widget_set_name (GTK_WIDGET (map_table), "net_map_table");
  gtk_container_border_width (GTK_CONTAINER (map_table), 2);
  gtk_container_add (GTK_CONTAINER (window_frame), map_table);

  /*
   * set up some separator bars
   */
  /*  separator_bar = gtk_hseparator_new ();
  gtk_table_attach (GTK_TABLE (map_table), separator_bar,
		    0, 7, 6, 7,
		    GTK_FILL, GTK_FILL, 3, 3); */

  separator_bar = gtk_vseparator_new ();
  gtk_table_attach (GTK_TABLE (map_table), separator_bar, 
		    1, 2, 3, 6,
		    GTK_FILL, GTK_FILL, 3, 0);

  /*  separator_bar = gtk_hseparator_new ();
  gtk_table_attach (GTK_TABLE (map_table), separator_bar,
		    0, 7, 2, 3,
		    GTK_FILL, GTK_FILL, 3, 3); */

  /*
   * The logo
   */
  logo_pixmap_filename = gnome_unconditional_pixmap_file ("logo_v.xpm");
  logo_widget = gnome_pixmap_new_from_file (logo_pixmap_filename);
  gtk_table_attach (GTK_TABLE (map_table), logo_widget,
		    0, 1, 3, 6,
		    GTK_FILL, GTK_FILL, 0, 0);
  /*
   * The canvas for the map.
   */
  gtk_widget_push_visual (gdk_imlib_get_visual ());
  gtk_widget_push_colormap (gdk_imlib_get_colormap ());
  canvas = gnome_canvas_new ();
  gtk_widget_pop_colormap ();
  gtk_widget_pop_visual ();
  gnome_canvas_set_size (GNOME_CANVAS (canvas), 1600, 1024);
  gnome_canvas_set_scroll_region (GNOME_CANVAS (canvas), 
				  0, 0,
				  1600, 1024);
  gtk_signal_connect (GTK_OBJECT (canvas), "key_press_event",
		      (GtkSignalFunc) canvas_key_press,
		      map_panel);
  gtk_signal_connect (GTK_OBJECT (canvas), "motion_notify_event",
		      (GtkSignalFunc) update_mouse_stat,
		      map_panel);
  gtk_object_set_data (GTK_OBJECT (map_panel), "map_canvas", canvas);
  gtk_widget_set_name (GTK_WIDGET (canvas), "map_canvas");
  /*
   * Scrolled area for the canvas...
   */
  scrolled_window = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (scrolled_window), GTK_SHADOW_IN);
  gtk_table_attach (GTK_TABLE (map_table), scrolled_window, 
		    3, 6, 4, 5,
		    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_widget_set_usize (scrolled_window, 600, 400);
  s_bar = gtk_hscrollbar_new (GTK_LAYOUT (canvas)->hadjustment);
  gtk_table_attach (GTK_TABLE (map_table), s_bar,
		    2, 7, 5, 6,
		    GTK_FILL, GTK_FILL, 0, 3);
  s_bar = gtk_hruler_new ();
  gtk_signal_connect_object (GTK_OBJECT (canvas), "motion_notify_event",
			     (GtkSignalFunc) GTK_WIDGET_CLASS (GTK_OBJECT (s_bar)->klass)->motion_notify_event,
			     GTK_OBJECT (s_bar));
  gtk_table_attach (GTK_TABLE (map_table), s_bar,
		    3, 6, 3, 4,
		    GTK_FILL, GTK_FILL, 0, 0);
  gtk_ruler_set_range (GTK_RULER (s_bar), 0, 10, 5, 10);
  s_bar = gtk_vscrollbar_new (GTK_LAYOUT (canvas)->vadjustment);
  gtk_table_attach (GTK_TABLE (map_table), s_bar,
		    6, 7, 3, 5,
		    GTK_FILL, GTK_FILL, 3, 0);
  s_bar = gtk_vruler_new ();
  gtk_signal_connect_object (GTK_OBJECT (canvas), "motion_notify_event",
			     (GtkSignalFunc) GTK_WIDGET_CLASS (GTK_OBJECT (s_bar)->klass)->motion_notify_event,
			     GTK_OBJECT (s_bar));
  gtk_ruler_set_range (GTK_RULER (s_bar), 0, 10, 5, 10);
  gtk_table_attach (GTK_TABLE (map_table), s_bar,
		    2, 3, 3, 5,
		    GTK_FILL, GTK_FILL, 0, 0);
  gtk_container_add (GTK_CONTAINER (scrolled_window), canvas);
  root = GNOME_CANVAS_GROUP (gnome_canvas_root (GNOME_CANVAS (canvas)));
  gtk_object_set_data (GTK_OBJECT (map_panel), "map_canvas_root", root);
  GTK_WIDGET_SET_FLAGS (canvas, GTK_CAN_FOCUS);
  gtk_widget_grab_focus (canvas);
  /*
   * Status areas
   */
  mouse_stat = gtk_statusbar_new ();
  gtk_table_attach (GTK_TABLE (map_table), mouse_stat,
		    0, 1, 7, 8,
		    GTK_FILL, GTK_FILL, 2, 0);
  gtk_object_set_data (GTK_OBJECT (map_panel), "mouse_stat", mouse_stat);
  gtk_object_set_data (GTK_OBJECT (map_panel), "mouse_stat_id", (int *)
		       gtk_statusbar_get_context_id (GTK_STATUSBAR 
						     (mouse_stat), 
						     "mouse_stat"));
						     
  map_stat = gtk_statusbar_new ();
  gtk_table_attach (GTK_TABLE (map_table), map_stat,
		    1, 7, 7, 8,
		    GTK_FILL, GTK_FILL, 2, 0);  
  gtk_object_set_data (GTK_OBJECT (map_panel), "map_stat", mouse_stat);
  gtk_object_set_data (GTK_OBJECT (map_panel), "map_stat_id", (int *)
		       gtk_statusbar_get_context_id (GTK_STATUSBAR 
						     (mouse_stat), 
						     "map_stat"));
  /*
  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
  gtk_table_attach (GTK_TABLE (map_table), frame, 
		    6, 7, 7, 8,
		    GTK_FILL, GTK_FILL, 2, 0);
  led_bar = (GtkWidget *)led_bar_new (5);
  gtk_container_add (GTK_CONTAINER (frame), led_bar);
  
  -- This screws up the window layout...
  */
  /*
   * Set the gnome_app contents...
   */
  gnome_app_set_contents (GNOME_APP (map_panel), window_frame);
  gnome_app_create_menus (GNOME_APP (map_panel), map_panel_menu);
  gnome_app_create_toolbar (GNOME_APP (map_panel), map_toolbar);
  gtk_widget_show_all (map_panel);
}


/*
 * Callback functions
 */

/*
 * Window Manager delete signal
 */
static void
delete_map_panel (GtkWidget *widget, gpointer data)
{
}

/*
 * Close button callback
 */
static void
close_panel_cb (GtkWidget *widget, gpointer data)
{
  destroy_map_panel ();
}

/*
 * Keypress on the canvas
 */
static gint
canvas_key_press (GnomeCanvas *widget, GdkEventKey *e, gpointer data)
{
  int x, y;

  gnome_canvas_get_scroll_offsets (widget, &x, &y);
  switch (e->keyval)
    {
    case GDK_Up:
      gnome_canvas_scroll_to (widget, x, y - 20);
      break;
    case GDK_Down:
      gnome_canvas_scroll_to (widget, x, y + 20);
      break;
    case GDK_Left:
      gnome_canvas_scroll_to (widget, x - 10, y);
      break;
    case GDK_Right:
      gnome_canvas_scroll_to (widget, x + 10, y);
      break;
    default:
      break;
    }
  return FALSE;
}

/*
 * Events on the items in the canvas
 */
static gint
canvas_item_event (GnomeCanvasItem *item, GdkEvent *event, gpointer data)
{
  static double     x, y;
  double            new_x, new_y;
  GdkCursor         *fleur;
  gboolean          is_network;
  static int        dragging;
  GtkWidget         *canvas;
  hosts             *host = NULL;
  net_entry         *network = NULL;
  GSList            *host_list;
  GnomeCanvasPoints *points;

  is_network = gtk_object_get_data (GTK_OBJECT (item), "is_network");
  switch (event->type) 
    {
    case GDK_BUTTON_PRESS:
      switch (event->button.button)
	{
	case 1:
	  x = event->button.x;
	  y = event->button.y;
	  fleur = gdk_cursor_new (GDK_FLEUR);
	  gnome_canvas_item_grab (item,
				  GDK_POINTER_MOTION_MASK |
				  GDK_BUTTON_RELEASE_MASK,
				  fleur,
				  event->button.time);
	  gdk_cursor_destroy (fleur);
	  dragging = TRUE;
	  break;
	case 2:
	  /* do something nifty */
	  break;
	case 3:
	  g_print ("Menu popup? \n");
	  /* Do a popup menu? */
	  break;
	default:
	  break;
	}
    case GDK_MOTION_NOTIFY:
      if (dragging && (event->motion.state & GDK_BUTTON1_MASK))
	{
	  new_x = event->motion.x;
	  new_y = event->motion.y;
	  /* 
	   * TODO: Auto scroll the canvas when moving beyond the bounds of 
	   * visible window? And the BIG CAD style cross-hairs.
	   * Also need to adjust this code to deal with network drags.
	   */
	  gnome_canvas_item_move (item, new_x - x, new_y - y);
	  if (is_network)
	    {
	      network = (net_entry *)data;
	      host_list = network->nl_hosts;
	      while (host_list)
		{
		  host = (hosts *)host_list->data;
		  setup_host_net_connection (host, network);
		  host_list = g_slist_next (host_list);
		}
	    }
	  else
	    {
	      host = (hosts *)data;
	      if (host)
		{
		  setup_host_net_connection (host, host->hl_network);
		}
	    }
	  x = new_x;
	  y = new_y;
	}
      break;
    case GDK_BUTTON_RELEASE:
      gnome_canvas_item_ungrab (item, event->button.time);
      dragging = FALSE;
      break;
    case GDK_KEY_PRESS:
      g_print ("Item event got a key press..\n");
      break;
    case GDK_KEY_RELEASE:
      g_print ("Item event got a key release...\n");
    default:
      break;
    }
  return FALSE;
}

static gint 
canvas_event (GtkWidget *widget, GdkEvent *e, gpointer data)
{
  switch (e->type)
    {
    case GDK_BUTTON_PRESS:
      switch (e->button.button)
	{
	case GDK_BUTTON3_MASK:
	  g_print ("Main popup menu?\n");
	  break;
	default:
	  break;
	}
    }
  return FALSE;
}
  
static gint
update_mouse_stat (GtkWidget *widget, GdkEventMotion *e, gpointer data)
{
  gint              x, y, statusid;
  gchar             buf[80];
  GdkModifierType   state;
  GtkWidget         *stat_widget;

  if (e->is_hint)
    gdk_window_get_pointer (e->window, &x, &y, &state);
  else
    {
      x     = e->x;
      y     = e->y;
      state = e->state;
    }
  snprintf (buf, sizeof (buf), "%d x %d", x, y);
  stat_widget = get_widget (map_panel, "mouse_stat");
  if (stat_widget)
    {
      statusid = gtk_statusbar_get_context_id (GTK_STATUSBAR (stat_widget),
					       "mouse_stat");
      gtk_statusbar_push (GTK_STATUSBAR (stat_widget), statusid, buf);
      /*      gtk_statusbar_pop (GTK_STATUSBAR (stat_widget), statusid); */
    }
  return FALSE;
}

/* 
 * Menu callback functions
 */
static void
quit_app_cb (GtkWidget *widget, gpointer data)
{
  quit_app ();
}

/*
 * Toolbar callback functions
 */
static void
add_host_cb (GtkWidget *widget, gpointer data)
{
  open_add_host_panel ();
}

static void
host_list_cb (GtkWidget *widget, gpointer data)
{
  hlist_panel_open ();
}

/*
 * Exported global functions
 * (Standard panel funcs)
 *
 */

/*
 * Open
 */
net_entry *test;

void
open_map_panel ()
{

  if (map_panel)
    {
      if (!GTK_WIDGET_VISIBLE (map_panel))
 	gtk_widget_show (map_panel);
      return;
    }
  else
    {
      create_map_panel ();
      reset_map_panel ();
      test = (net_entry *) g_malloc (sizeof (net_entry));
      test->nl_name = g_strdup ("Test Network");
      map_add_network (test, 100, 50);
    }
}

/*
 * Close/destroy
 */
void
destroy_map_panel ()
{
  g_return_if_fail (map_panel != NULL);

  if (map_panel)
    {
      gtk_widget_destroy (map_panel);
      map_panel = NULL;
    }
}

/*
 * Hide
 */
void
hide_map_panel ()
{
  g_return_if_fail (map_panel != NULL);

  if (!GTK_WIDGET_VISIBLE (map_panel))
    gtk_widget_show (map_panel);
}

/*
 * reset
 */
void 
reset_map_panel ()
{

  g_return_if_fail (map_panel != NULL);
 
}

/*
 * Support functions for the map
 */

static void
setup_item (GnomeCanvasItem *item, gpointer data)
{
  g_return_if_fail (item != NULL);
  gtk_signal_connect (GTK_OBJECT (item), "event",
		      (GtkSignalFunc) canvas_item_event,
		      data);
}
static void
free_imlib_image (GtkObject *object, gpointer data)
{
  gdk_imlib_destroy_image (data);
}

static void
host_pixmap (GnomeCanvasGroup *group, double x, double y)
{
  GdkImlibImage    *im;
  GnomeCanvasItem  *image;
  char            *fn;

  fn = gnome_unconditional_pixmap_file ("desktop.xpm");
  im = gdk_imlib_load_image (fn);
  y -= (im->rgb_height + 3);
  image = gnome_canvas_item_new (group,
				 gnome_canvas_image_get_type (),
				 "image", im,
				 "x", x,
				 "y", y,
				 "width", (double) im->rgb_width,
				 "height", (double) im->rgb_height,
				 "anchor", GTK_ANCHOR_N,
				 NULL);
  gtk_signal_connect (GTK_OBJECT (image), "destroy",
		      (GtkSignalFunc) free_imlib_image,
		      im);
}

/*
 * Function   : map_add_host 
 * Description: This will add a host to the network map at a given x/y 
 *              position. Passing x/y as -1 will invoke the auto placement
 *              routine.
 * Arguments  : host  -- The hosts structure of the host to add.
 *              x     -- The X position on the map to add it to.
 *              y     -- The Y position on the map to add it to.
 * Returns    : TRUE  -- Host was suscessfully added to the map.
 *              FALSE -- Host was not added to the map. Check app_errno 
 *                       for further info.
 */
gboolean
map_add_host (hosts *host, gdouble x, gdouble y)
{
  GnomeCanvasGroup    *root;
  GnomeCanvasGroup    *host_group;
  GnomeCanvasItem     *item;
  /*
   * FIXME: Re-do this when I do the error handling stuff.
   */
  g_return_val_if_fail (map_panel != NULL, FALSE);
  root = gtk_object_get_data (GTK_OBJECT (map_panel), "map_canvas_root");
  if (root)
    {
      host->hl_mgroup = GNOME_CANVAS_GROUP (gnome_canvas_item_new 
					    (root, 
					     gnome_canvas_group_get_type(),
					     "x", 20.0,
					     "y", 10.0,
					     NULL));
      setup_item (GNOME_CANVAS_ITEM (host->hl_mgroup), host);
      gnome_canvas_item_new (host->hl_mgroup, 
			     gnome_canvas_text_get_type (),
			     "text", host->hl_disp,
			     "x", 0.0,
			     "y", 0.0,
			     "fill_color", "firebrick",
			     "anchor", GTK_ANCHOR_N,
			     NULL);
      host_pixmap (host->hl_mgroup, 0.0, 0.0);
      map_connect_host_net (host, test);
      return TRUE;
    }
  return FALSE;
}

/*
 * Function   : map_del_host
 * Description: This function will delete a host from the map.
 * Arguments  : host  -- The hosts structure describing the host to 
 *                       remove from the map.
 * Returns    : TRUE  -- Host was deleted from the map.
 *              FALSE -- Host was not deleted from the map. Check app_errno
 *                       for further info.
 */
gboolean
map_del_host (hosts *host)
{
  
}

/*
 * Function   : map_add_network
 * Description: This function shall add a network to the map at a given
 *              x/y position. Passing -1 for x & y shall invoke the auto
 *              placement routines.
 * Arguments  : network  -- The network structure of the network to add
 *                          to the map.
 *              X        -- The X position on the map.
 *              Y        -- The Y position on the map.
 * Returns    : TRUE     -- Network was added to the map.
 *              FALSE    -- Network was not added to the map. Check app_errno
 *                          for further info.
 */
gboolean
map_add_network (net_entry *network, gint x, gint y)
{
  GnomeCanvasGroup   *root;
  GnomeCanvasGroup   *net_group;
  GnomeCanvasItem    *item;
  GnomeCanvasPoints  *points;

  g_return_val_if_fail (map_panel != NULL, FALSE);
  root = gtk_object_get_data (GTK_OBJECT (map_panel), "map_canvas_root");
  if (root)
    {
      network->nl_mgroup = GNOME_CANVAS_GROUP (gnome_canvas_item_new 
				       (root, gnome_canvas_group_get_type (),
					"x", (double) x,
					"y", (double) y,
					NULL));
      gtk_object_set_data (GTK_OBJECT (network->nl_mgroup), 
			   "is_network", TRUE);
      setup_item (GNOME_CANVAS_ITEM (network->nl_mgroup), network);
      gnome_canvas_item_new (network->nl_mgroup,
			     gnome_canvas_text_get_type (),
			     "text", network->nl_name,
			     "x", 0.0,
			     "y", 0.0,
			     "fill_color", "black",
			     "anchor", GTK_ANCHOR_N,
			     NULL);
      points = gnome_canvas_points_new (2);
      points->coords[0] = -90.0;
      points->coords[1] = -2.0;
      points->coords[2] = 90.0;
      points->coords[3] = -2.0;
      gnome_canvas_item_new (network->nl_mgroup,
			     gnome_canvas_line_get_type (),
			     "points", points,
			     "fill_color", "green",
			     "width_units", 2.0,
			     "first_arrowhead", TRUE,
			     "last_arrowhead", TRUE,
			     NULL);
      gnome_canvas_points_free (points);
      return TRUE;
    }
  return FALSE;
}

/*
 * Function   : map_del_network
 * Description: This function shall delete a network from the map's display.
 * Arguments  : network  -- The network structure of the network to delete
 *                          from the map.
 * Returns    : TRUE     -- Network was deleted from the map.
 *              FALSE    -- Network was not deleted from the map. Check
 *                          app_errno for further info.
 */
gboolean
map_del_network (net_entry *network)
{

}

/*
 * Function   : map_connect_host_net
 * Description: This function will connect a host to a network.
 * Arguments  ; host    -- The host entry to connect.
 *              network -- The network to connect it to.
 * Returns    : TRUE    -- Network sucessfully connected.
 *              FALSE   -- Network Not connected.
 */
gboolean
map_connect_host_net (hosts *host, net_entry *network)
{
  GnomeCanvasGroup  *host_group;
  GnomeCanvasGroup  *net_group;
  GnomeCanvasPoints *points;

  net_group = network->nl_mgroup;
  host_group = host->hl_mgroup;
  points = gnome_canvas_points_new (2);
  points->coords[0] = net_group->xpos;
  points->coords[1] = net_group->ypos - 2;
  points->coords[2] = host_group->xpos;
  points->coords[3] = host_group->ypos;

  host->hl_mconnection = gnome_canvas_item_new (GNOME_CANVAS_GROUP (GNOME_CANVAS_ITEM (host_group)->canvas->root),
			 gnome_canvas_line_get_type (),
			 "points", points,
			 "fill_color", "black",
			 "width_units", 0.0,
			 NULL);
  host->hl_ngroup = network->nl_mgroup;
  host->hl_network = network;
  add_host_to_network (network, host);
  gnome_canvas_points_free (points);
  return TRUE;

}

void
setup_host_net_connection (hosts *host, net_entry *network)
{
  GnomeCanvasPoints   *points;

  g_return_if_fail (host != NULL);
  g_return_if_fail (network != NULL);

  points = gnome_canvas_points_new (2);
  points->coords[0] = host->hl_ngroup->xpos;
  points->coords[1] = host->hl_ngroup->ypos - 2;
  points->coords[2] = host->hl_mgroup->xpos;
  points->coords[3] = host->hl_mgroup->ypos;
  gnome_canvas_item_set ((GnomeCanvasItem *)host->hl_mconnection,
			 "points", points,
			 NULL);
  gnome_canvas_points_free (points);
}

/* EOF */
