/* GAIL - The GNOME Accessibility Implementation Library
 * Copyright 2001 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include "gailtoplevel.h"

static void             gail_toplevel_class_init        (GailToplevelClass      *klass);
static void             gail_toplevel_object_init       (GailToplevel           *toplevel);
static void             gail_toplevel_object_finalize   (GObject                *obj);

/* atkobject.h */

static gint             gail_toplevel_get_n_children    (AtkObject              *obj);
static AtkObject*       gail_toplevel_ref_child         (AtkObject              *obj,
                                                        gint                    i);

/* Callbacks */


static void             gail_toplevel_window_destroyed  (GtkWindow              *window,
                                                        GailToplevel            *text);
static gboolean         gail_toplevel_hide_event_watcher (GSignalInvocationHint *ihint,
                                                        guint                   n_param_values,
                                                        const GValue            *param_values,
                                                        gpointer                data);
static gboolean         gail_toplevel_show_event_watcher (GSignalInvocationHint *ihint,
                                                        guint                   n_param_values,
                                                        const GValue            *param_values,
                                                        gpointer                data);

/* Misc */

static void      _gail_toplevel_remove_child            (GailToplevel           *toplevel,
                                                        GtkWindow               *window);

static gpointer parent_class = NULL;

GType
gail_toplevel_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      static const GTypeInfo tinfo =
        {
          sizeof (GailToplevelClass),
          (GBaseInitFunc) NULL, /* base init */
          (GBaseFinalizeFunc) NULL, /* base finalize */
          (GClassInitFunc) gail_toplevel_class_init, /* class init */
          (GClassFinalizeFunc) NULL, /* class finalize */
          NULL, /* class data */
          sizeof (GailToplevel), /* instance size */
          0, /* nb preallocs */
          (GInstanceInitFunc) gail_toplevel_object_init, /* instance init */
          NULL /* value table */
        };

      type = g_type_register_static (ATK_TYPE_OBJECT,
                                   "GailToplevel", &tinfo, 0);
    }

  return type;
}

AtkObject*
gail_toplevel_new (void)
{
  GObject *object;
  AtkObject *accessible;

  object = g_object_new(GAIL_TYPE_TOPLEVEL, NULL);
  g_return_val_if_fail ((object != NULL), NULL);

  accessible = ATK_OBJECT (object);
  accessible->role = ATK_ROLE_INVALID;
  accessible->name = g_get_prgname();

  return accessible;
}

static void
gail_toplevel_class_init (GailToplevelClass *klass)
{
  AtkObjectClass *class = ATK_OBJECT_CLASS(klass);
  GObjectClass *g_object_class = G_OBJECT_CLASS(klass);

  parent_class = g_type_class_peek_parent (klass);

  class->get_n_children = gail_toplevel_get_n_children;
  class->ref_child = gail_toplevel_ref_child;

  g_object_class->finalize = gail_toplevel_object_finalize;
}

static void
gail_toplevel_object_init (GailToplevel *toplevel)
{
  GtkWindow *window;
  GList *l;
  guint signal_id;

  l = toplevel->window_list = gtk_window_list_toplevels ();

  while (l)
    {
      window = GTK_WINDOW (l->data);
      if (!window || GTK_WIDGET (window)->parent)
        {
          GList *temp_l  = l->next;

          toplevel->window_list = g_list_delete_link (toplevel->window_list, l);
          l = temp_l;
        }
      else
        {
          g_signal_connect (G_OBJECT (window), 
                            "destroy",
                            G_CALLBACK (gail_toplevel_window_destroyed),
                            toplevel);
        }
      l = l->next;
    }

  gtk_type_class (GTK_TYPE_WINDOW);

  signal_id  = g_signal_lookup ("show", GTK_TYPE_WINDOW);
  g_signal_add_emission_hook (signal_id, 0,
    gail_toplevel_show_event_watcher, toplevel, (GDestroyNotify) NULL);

  signal_id  = g_signal_lookup ("hide", GTK_TYPE_WINDOW);
  g_signal_add_emission_hook (signal_id, 0,
    gail_toplevel_hide_event_watcher, toplevel, (GDestroyNotify) NULL);
}

static void
gail_toplevel_object_finalize (GObject *obj)
{
  GailToplevel *toplevel = GAIL_TOPLEVEL (obj);

  if (toplevel->window_list)
    g_list_free (toplevel->window_list);

  G_OBJECT_CLASS (parent_class)->finalize (obj);
}

static gint
gail_toplevel_get_n_children (AtkObject *obj)
{
  GailToplevel *toplevel = GAIL_TOPLEVEL (obj);

  gint rc = g_list_length (toplevel->window_list);
  return rc;
}

static AtkObject*
gail_toplevel_ref_child (AtkObject *obj,
                         gint      i)
{
  GailToplevel *toplevel;
  gpointer ptr;
  GtkWidget *widget;
  AtkObject *atk_obj;

  toplevel = GAIL_TOPLEVEL (obj);
  ptr = g_list_nth_data (toplevel->window_list, i);
  widget = GTK_WIDGET (ptr);
  atk_obj = gtk_widget_get_accessible (widget);

  g_object_ref (atk_obj);
  return atk_obj;
}

/*
 * Window destroy events on GtkWindow cause a child to be removed
 * from the toplevel
 */
static void
gail_toplevel_window_destroyed (GtkWindow    *window,
                                GailToplevel *toplevel)
{
  _gail_toplevel_remove_child (toplevel, window);
}

/*
 * Show events cause a child to be added to the toplevel
 */
static gboolean
gail_toplevel_show_event_watcher (GSignalInvocationHint *ihint,
                                  guint                  n_param_values,
                                  const GValue          *param_values,
                                  gpointer               data)
{
  GailToplevel *toplevel = GAIL_TOPLEVEL (data);
  AtkObject *atk_obj = ATK_OBJECT (toplevel);
  GObject *object;
  GtkWidget *widget;
  gint n_children;

  object = g_value_get_object (param_values + 0);

  if (!GTK_IS_WINDOW (object))
    return TRUE;

  widget = GTK_WIDGET (object);
  if (widget->parent)
    return TRUE;

  /* Add the window to the list & emit the signal */
  toplevel->window_list = g_list_append (toplevel->window_list, widget);

  n_children = g_list_length (toplevel->window_list);

 /*
  * Must subtract 1 from the n_children since the index is 0-based
  * but g_list_length is 1-based.
  */
  g_signal_emit_by_name (atk_obj, "children-changed::add",
                         n_children - 1, NULL, NULL);

  /* Connect destroy signal callback */
  g_signal_connect (G_OBJECT(object), 
                    "destroy",
                    G_CALLBACK (gail_toplevel_window_destroyed),
                    toplevel);

  return TRUE;
}

/*
 * Hide events on GtkWindow cause a child to be removed from the toplevel
 */
static gboolean
gail_toplevel_hide_event_watcher (GSignalInvocationHint *ihint,
                                  guint                  n_param_values,
                                  const GValue          *param_values,
                                  gpointer               data)
{
  GailToplevel *toplevel = GAIL_TOPLEVEL (data);
  GObject *object;

  object = g_value_get_object (param_values + 0);

  if (!GTK_IS_WINDOW (object))
    return FALSE;

  _gail_toplevel_remove_child (toplevel, GTK_WINDOW (object));
  return TRUE;
}

/*
 * Common code used by destroy and hide events on GtkWindow
 */
static void
_gail_toplevel_remove_child (GailToplevel *toplevel, 
                             GtkWindow    *window)
{
  AtkObject *atk_obj = ATK_OBJECT (toplevel);
  GList *l;
  guint window_count = 0;

  if (toplevel->window_list)
    {
        GtkWindow *tmp_window;

        /* Must loop through them all */
        for (l = toplevel->window_list; l; l = l->next)
        {
          tmp_window = GTK_WINDOW (l->data);

          if (window == tmp_window)
            {
              /* Remove the window from the window_list & emit the signal */
              g_signal_emit_by_name (atk_obj, "children-changed::remove",
                                     window_count, NULL, NULL);
              toplevel->window_list = g_list_remove (toplevel->window_list,
                                                     l->data);
              break;
            }

          window_count++;
        }
    }
}
