/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 *  fonts.c: test functions for the libgpa config database
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors:
 *    Tambet Ingo <tambet@ximian.com>
 *
 *  Copyright (C) 2002 Ximian Inc. and authors
 *
 */

#include <popt.h>
#include <stdio.h>
#include <glib.h>
#include <string.h>
#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-print-job.h>
#include <libgnomeprint/gnome-font.h>
#include <libgnomeprint/gnome-fontmap.h>

gboolean options_dump = FALSE;
gboolean options_dump_full = FALSE;
gint     options_generate = -1;
gboolean options_catalog = FALSE;
gboolean options_number = FALSE;
gchar*   options_output = NULL;
gboolean options_pdf = FALSE;

poptContext popt;

static struct poptOption options[] = {
	{ "dump",      'd', POPT_ARG_NONE,   &options_dump,   0,
	  "Dump the list of fonts", NULL},
 	{ "dump-full", 'f', POPT_ARG_NONE,   &options_dump_full,   0,
	  "Dump the list of fonts with their properties",  NULL},
	{ "generate",  'g', POPT_ARG_INT,    &options_generate,   0,
	  "Generate test output for a font",  NULL},
	{ "pdf",       'p', POPT_ARG_NONE,    &options_pdf,   0,
	  "Generate a pdf file instead of a Postscript one",  NULL},
	{ "number",    'n', POPT_ARG_NONE,   &options_number,   0,
	  "Return the number of fonts known to gnome-print",  NULL},
	{ "catalog",   'c', POPT_ARG_NONE,   &options_catalog,   0,
	  "Generate a font catalog",  NULL},
	{ "output",    'o', POPT_ARG_STRING, &options_output,   0,
	  "Specify the output file",  NULL},
	POPT_AUTOHELP
	{ NULL }
};


static void
print_font_info (GPFontEntry *entry, GnomePrintContext *gpc)
{
	GnomeFont *font;
	GPFontEntryTT *tt = (GPFontEntryTT *) entry;
	gdouble x, y, row_size;
	gchar *c;

	font = gnome_font_find (entry->name, 12);
	if (!font || (strcmp (gnome_font_get_name (font), entry->name) != 0)) {
		g_print ("Could not get font: %s\n", entry->name);
		exit (-2);
	}
	g_print ("Printing with \"%s\".\n", gnome_font_get_name (font));

	y = 780;
	x = 20;
	row_size = 20;
	
	gnome_print_beginpage (gpc, "1");
	gnome_print_setfont (gpc, font);

	gnome_print_moveto (gpc, x, y);
	gnome_print_show (gpc, "Name:");
	y -= row_size;
	gnome_print_moveto (gpc, x, y);
	gnome_print_show (gpc, "Version:");
	y -= row_size;
	gnome_print_moveto (gpc, x, y);
	gnome_print_show (gpc, "Family Name:");
	y -= row_size;
	gnome_print_moveto (gpc, x, y);
	gnome_print_show (gpc, "Speciesname:");
	y -= row_size;
	gnome_print_moveto (gpc, x, y);
	gnome_print_show (gpc, "PS Name:");
	y -= row_size;
	gnome_print_moveto (gpc, x, y);
	gnome_print_show (gpc, "Weight:");
	y -= row_size;
	gnome_print_moveto (gpc, x, y);
	gnome_print_show (gpc, "Italic Angle:");
	y -= row_size;
	gnome_print_moveto (gpc, x, y);
	gnome_print_show (gpc, "Type:");
	y -= row_size;
	gnome_print_moveto (gpc, x, y);
	gnome_print_show (gpc, "File:");
	y -= row_size;

	y = 780;
	x = x + 100;

	c = g_strdup_printf ("%g", entry->ItalicAngle);
	
	gnome_print_moveto (gpc, x, y);
	gnome_print_show (gpc, entry->name);
	y -= row_size;
	gnome_print_moveto (gpc, x, y);
	gnome_print_show (gpc, entry->version);
	y -= row_size;
	gnome_print_moveto (gpc, x, y);
	gnome_print_show (gpc, entry->familyname);
	y -= row_size;
	gnome_print_moveto (gpc, x, y);
	gnome_print_show (gpc, entry->speciesname);
	y -= row_size;
	gnome_print_moveto (gpc, x, y);
	gnome_print_show (gpc, entry->psname);
	y -= row_size;
	gnome_print_moveto (gpc, x, y);
	gnome_print_show (gpc, entry->weight);
	y -= row_size;
	gnome_print_moveto (gpc, x, y);
	gnome_print_show (gpc, c);
	y -= row_size;
	gnome_print_moveto (gpc, x, y);
	gnome_print_show (gpc, "TrueType");
	y -= row_size;
	gnome_print_moveto (gpc, x, y);
	gnome_print_show (gpc, tt->ttf.name);
	y -= row_size;

	g_free (c);

	gnome_print_showpage (gpc);
	g_object_unref (G_OBJECT (font));
	
	return;
}

static gint
number_of_fonts (void)
{
	GPFontMap *map;

	map = gp_fontmap_get ();
	return g_slist_length (map->fonts);
}
      

static void
generate_font_info (gint num)
{
	GnomePrintContext *gpc;
	GnomePrintConfig *config;
	GnomePrintJob *job;
	GPFontEntry *entry;
	GPFontMap *map;
	GSList *list;
	gint max;

	map = gp_fontmap_get ();
	list = g_slist_copy (map->fonts);
	max = g_slist_length (list);
      
	g_assert ((num > 0) && (num <= max));

	entry = (GPFontEntry *) ((GSList *)(g_slist_nth (map->fonts, num - 1)))->data;

	g_assert (entry);
	
	g_print ("Name: %s\n", entry->psname);
	
	if (entry->type != GP_FONT_ENTRY_TRUETYPE) {
		g_print ("Only TrueType fonts are supported for now\n");
		return;
	}

	job = gnome_print_job_new (NULL);
	gpc = gnome_print_job_get_context (job);
	config = gnome_print_job_get_config (job);

	if (options_pdf) {
		if (!gnome_print_config_set (config, "Printer", "PDF"))
			g_print ("Could not set the printer to PDF\n");
	} else {
		if (!gnome_print_config_set (config, "Printer", "GENERIC"))
			g_print ("Could not set the printer to GENERIC\n");
		if (!gnome_print_config_set (config, "Settings.Transport.Backend", "file"))
			g_print ("Could not set the backend to file\n");
	}
	if (!gnome_print_config_set (config, GNOME_PRINT_KEY_OUTPUT_FILENAME,
				     options_output ? options_output : (options_pdf ? "output.pdf" : "output.ps")))
		g_print ("Could not set the output filename\n");
	if (!gnome_print_config_set (config, GNOME_PRINT_KEY_OUTPUT_FILENAME,
				     options_output ? options_output : "output.ps"))
		g_print ("Could not set the output filename\n");
	if (!gnome_print_config_set (config, GNOME_PRINT_KEY_PAPER_SIZE, "USLetter"))
		g_print ("Could not set the Paper Size\n");
	
	print_font_info (entry, gpc);

	gnome_print_job_close (job);
	gnome_print_job_print (job);
}

static void
generate_catalog (void)
{
	g_print ("Implement me\n");
}

static void
dump_short_list (void)
{
	GList *list, *tmp;
	gchar *font;

	tmp = list = gnome_font_list ();

	while (tmp) {
		font = tmp->data;
		tmp = tmp->next;

		g_print ("%s\n", font);
	}

	gnome_font_list_free (list);
}

static void
dump_font_info (GPFontEntry *entry, gint num)
{
	GPFontEntryTT *tt;
	
	g_print ("\n\nEntry:\t\t%d\n", num);
	g_print ("Name:\t\t%s\n", entry->name);
	g_print ("Version:\t%s\n", entry->version);
	g_print ("Family Name:\t%s\n", entry->familyname);
	g_print ("Speciesname:\t%s\n", entry->speciesname);
	g_print ("PS name:\t%s\n", entry->psname);
	g_print ("Weight:\t\t%s\n", entry->weight);
	g_print ("Italic Angle:\t%g\n", entry->ItalicAngle);

	switch (entry->type) {
	case GP_FONT_ENTRY_UNKNOWN:
		g_print ("Type:\t\tUnknown\n");
		break;
	case GP_FONT_ENTRY_TYPE1:
		g_print ("Type:\t\tType 1\n");
		break;
	case GP_FONT_ENTRY_TRUETYPE:
		tt = (GPFontEntryTT *) entry;
		g_print ("Type:\t\tTrue Type\n");
		g_print ("File:\t\t%s\n", tt->ttf.name);
		g_print ("Face num:\t%d\n", tt->facenum);
		break;
	case GP_FONT_ENTRY_TYPE1_ALIAS:
		g_print ("Type:\t\tType1 Alias\n");
		break;
	case GP_FONT_ENTRY_ALIAS:
		g_print ("Type:\t\tAlias\n");
		break;
	case GP_FONT_ENTRY_SPECIAL:
		g_print ("Type:\t\tSpecial\n");
		break;
	}
}

static void
dump_long_list (void)
{
	GPFontMap *map;
	GSList *list, *l;
	gint i = 0;

	map = gp_fontmap_get ();
	list = g_slist_copy (map->fonts);
	l = list;

	while (l) {
		dump_font_info (l->data, ++i);
		l = l->next;
	}

	g_slist_free (list);
}

static void
usage (gint num)
{
	g_print ("\n");
	switch (num) {
	case 0:
		g_print ("Please specify an action\n");
		break;
	case 1:
		g_print ("Only one of --dump --dump-full --generate --catalog and --number can be specified\n");
		break;
	case 2:
		g_print ("Font number out of range\n");
		break;
	case 3:
		g_print ("Option --generate needs a font number to be specified\n");
		break;
	case 4:
		g_print ("Option --output can only be used with option --generate\n");
		break;
	case 5:
		g_print ("Option --pdf can only be used with option --generate\n");
		break;
	default:
		g_assert_not_reached ();
	}

	g_print ("\n");
	poptPrintHelp (popt, stdout, FALSE);
}

static void
check_options (int argc, const char *argv[])
{
	GPFontMap *map;
	gint max;
	gint num = 0;
	gint retval;

	popt = poptGetContext ("test", argc, argv, options, 0);
	retval = poptGetNextOpt (popt);
	/*
	g_print ("Ret:%d\n", retval);
	*/

	if (options_dump)
		num++;
	if (options_dump_full)
		num++;
	if ((options_generate != -1) || retval == POPT_ERROR_NOARG)
		num++;
	if (options_catalog)
		num++;
	if (options_number)
		num++;

	map = gp_fontmap_get ();
	max = g_slist_length (map->fonts);
	gp_fontmap_release (map);

	if (num > 1) {
		usage (1);
	} else if (retval == POPT_ERROR_NOARG ||
		   retval == POPT_ERROR_BADNUMBER) {
		usage (3);
	} else if (num == 0) {
		usage (0);
	} else if ((options_generate != -1) &&
		   ((options_generate < 1) ||
		    (options_generate > max))) {
		usage (2);
	} else if ((options_generate == -1) &&
		   (options_output != NULL)) {
		usage (4);
	} else if ((options_generate == -1) &&
		   (options_pdf == TRUE)) {
		usage (5);
	} else {
		poptFreeContext (popt);
		return;
	}

	poptFreeContext (popt);
	exit (1);
}

int
main (int argc, const char *argv[])
{
	check_options (argc, argv);

	g_type_init ();
	
	if (options_dump)
		dump_short_list ();
	if (options_dump_full)
		dump_long_list ();
	if (options_catalog)
		generate_catalog ();
	if (options_generate > 0)
		generate_font_info (options_generate);
	if (options_number)
		return number_of_fonts ();
	
	return 0;
}
