/* bug-buddy bug submitting program
 *
 * Copyright (C) 2001 Jacob Berkman
 * Copyright 2001 Ximian, Inc.
 *
 * Author:  jacob berkman  <jacob@bug-buddy.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include "bug-buddy.h"
#include "libglade-buddy.h"
#include "distribution.h"

#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <utime.h>
#include <errno.h>
#include <string.h>
#include <ctype.h>

#include <gnome.h>

#include <libgnome/gnome-desktop-item.h>
#include <gmenu-tree.h>

#include <bonobo/bonobo-exception.h>
#include <bonobo-activation/bonobo-activation.h>

#include <dirent.h>

#include <libxml/tree.h>
#include <libxml/parser.h>
#include <libxml/xmlmemory.h>

#include <libsoup/soup.h>
#include <libsoup/soup-xmlrpc-message.h>


#define APPLET_REQUIREMENTS \
	"has_all (repo_ids, ['IDL:Bonobo/Control:1.0'," \
	"		     'IDL:GNOME/Vertigo/PanelAppletShell:1.0']) && " \
	"defined (panel:icon)"

#define BUGZILLA_BUGZILLA               "X-GNOME-Bugzilla-Bugzilla"
#define BUGZILLA_PRODUCT                "X-GNOME-Bugzilla-Product"
#define BUGZILLA_COMPONENT              "X-GNOME-Bugzilla-Component"
#define BUGZILLA_EMAIL                  "X-GNOME-Bugzilla-Email"
#define BUGZILLA_VERSION                "X-GNOME-Bugzilla-Version"
#define BUGZILLA_OTHER_BINARIES         "X-Gnome-Bugzilla-OtherBinaries"

#define BA_BUGZILLA_BUGZILLA            "bugzilla:bugzilla"
#define BA_BUGZILLA_PRODUCT             "bugzilla:product"
#define BA_BUGZILLA_COMPONENT           "bugzilla:component"
#define BA_BUGZILLA_VERSION             "bugzilla:version"
#define BA_BUGZILLA_EMAIL               "bugzilla:email"
#define BA_BUGZILLA_OTHER_BINARIES      "bugzilla:other_binaries"

static void
add_bugzilla_application (GHashTable *hash, 
			  const char *name, 
			  const char *comment, 
			  const char *bugzilla, 
			  const char *product,
			  const char *component, 
			  const char *version, 
			  const char *email, 
			  const char *icon,
			  const char *program,
			  const char *other_programs)
{
	BugzillaApplication *app;
	char **programv;
	int i;

	app = g_new0 (BugzillaApplication, 1);
	
	app->name        = g_strdup (name);
	app->comment     = g_strdup (comment);
	app->bugzilla    = g_strdup (bugzilla);
	app->product     = g_strdup (product);
	app->component   = g_strdup (component);
	app->version     = g_strdup (version);
	app->email       = g_strdup (email);

	if (icon) {
		if (g_path_is_absolute (icon)) {
			app->pixbuf = gdk_pixbuf_new_from_file_at_size (icon, 48, 48, NULL);
		}
		if (app->pixbuf == NULL) {
			app->pixbuf = gtk_icon_theme_load_icon (gtk_icon_theme_get_default (),
								icon, 48, 0, NULL);
		}
		if (app->pixbuf == NULL && strrchr (icon, '.') != NULL) {
			char *name;
			name = g_strndup (icon, strlen (icon) - strlen (strrchr (icon, '.')));
			app->pixbuf = gtk_icon_theme_load_icon (gtk_icon_theme_get_default (),
								name, 48, 0, NULL);
			g_free (name);
		}
	
		if (app->pixbuf == NULL) {
			char *filename;
			filename = g_strdup_printf (BUDDY_ICONDIR"/pixmaps/%s", icon);
			app->pixbuf = gdk_pixbuf_new_from_file_at_size (filename, 48, 48, NULL);
			g_free (filename);
		}
	}

	if (app->pixbuf == NULL) {
		g_warning (_("Couldn't load icon for %s"), app->name);
	}
	
	if (program) {
		g_shell_parse_argv (program, &i, &programv, NULL);
		if (programv[0]) {
			char *s;
			s = strrchr (programv[0], G_DIR_SEPARATOR);
			s = s ? s+1 : programv[0];
			if (g_hash_table_lookup (hash, s) == NULL) {
				g_hash_table_insert (hash, g_strdup (s), app);
				g_print ("DEBUG: Added %s\n", s);
			} else {
				g_free (app);
				return;
			}
		} else {
			g_free (app);
			return;
		}
		if (programv)
			g_strfreev (programv);
	}
		
	if (other_programs) {
		programv = g_strsplit (other_programs, ";", -1);
		for (i=0; programv[i]; i++) {
			if (g_hash_table_lookup (hash, programv[i]) == NULL) {
				g_hash_table_insert (hash, g_strdup (programv[i]), app);
				g_print ("DEBUG: Added %s\n", programv[i]);
			}
		}
		g_strfreev (programv);
	}
}

static const GSList *
get_i18n_slist (void)
{
  GList *langs_glist;
  static GSList *langs_gslist;

  if (langs_gslist)
	  return langs_gslist;

  langs_glist = (GList *)gnome_i18n_get_language_list ("LC_MESSAGES");
  langs_gslist = NULL;
  while (langs_glist != NULL) {
    langs_gslist = g_slist_append (langs_gslist, langs_glist->data);
    langs_glist = langs_glist->next;
  }

  return langs_gslist;
}

static void
load_applets (GnomeIconTheme *git, GHashTable *hash)
{
	Bonobo_ServerInfoList *info_list;
	Bonobo_ServerInfo *info;
	CORBA_Environment ev;
	GSList *langs;
	int i;
        gchar *name;
	GSList *l;

	CORBA_exception_init (&ev);
	info_list = bonobo_activation_query (APPLET_REQUIREMENTS, NULL, &ev);
	if (BONOBO_EX (&ev)) {
		g_warning ("Applet list query failed: %s", BONOBO_EX_REPOID (&ev));
		CORBA_exception_free (&ev);
		return;
	}
	CORBA_exception_free (&ev);

	langs = (GSList *)get_i18n_slist ();

	for (i = 0; i < info_list->_length; i++) {
		info = info_list->_buffer + i;
		if (!bonobo_server_info_prop_lookup (info,
						     BA_BUGZILLA_BUGZILLA,
NULL)) {
			continue;
		}

		name = g_strdup (bonobo_server_info_prop_lookup (info, "name", langs));
		/*FIXME:
		for (l = applications; l; l = l->next) {
			BugzillaApplication *app = l->data;
		
			if (strcmp (app->name, name) == 0) {
				g_free (name);
				name = g_strdup_printf (_("%s (Panel Applet)"), bonobo_server_info_prop_lookup (info, "name", langs));
	
				break;
			}
		}*/

		add_bugzilla_application (hash,
			name,
			bonobo_server_info_prop_lookup (info, "description", langs),
			bonobo_server_info_prop_lookup (info, BA_BUGZILLA_BUGZILLA,  NULL),
			bonobo_server_info_prop_lookup (info, BA_BUGZILLA_PRODUCT,   NULL),
			bonobo_server_info_prop_lookup (info, BA_BUGZILLA_COMPONENT, NULL),
			bonobo_server_info_prop_lookup (info, BA_BUGZILLA_VERSION, NULL),
			bonobo_server_info_prop_lookup (info, BA_BUGZILLA_EMAIL,     NULL),
			bonobo_server_info_prop_lookup (info, "panel:icon", NULL),
			NULL,
			bonobo_server_info_prop_lookup (info, BA_BUGZILLA_OTHER_BINARIES, NULL));
		
		g_free (name);
	}

	CORBA_free (info_list);
}

static int
compare_applications (GMenuTreeEntry *a,
		      GMenuTreeEntry *b)
{
	return g_utf8_collate (gmenu_tree_entry_get_name (a),
			       gmenu_tree_entry_get_name (b));
}

static GSList *get_all_applications_from_dir (GMenuTreeDirectory *directory,
					      GSList             *list);

static GSList *
get_all_applications_from_alias (GMenuTreeAlias *alias,
				 GSList         *list)
{
	GMenuTreeItem *aliased_item;

	aliased_item = gmenu_tree_alias_get_item (alias);

	switch (gmenu_tree_item_get_type (aliased_item)) {
	case GMENU_TREE_ITEM_DIRECTORY:
		list = get_all_applications_from_dir (GMENU_TREE_DIRECTORY (aliased_item), list);
		break;

	case GMENU_TREE_ITEM_ENTRY:
		list = g_slist_append (list, gmenu_tree_item_ref (aliased_item));
		break;

	default:
		break;
	}

	gmenu_tree_item_unref (aliased_item);

	return list;
}

static GSList *
get_all_applications_from_dir (GMenuTreeDirectory *directory,
			       GSList             *list)
{
	GSList *items;
	GSList *l;

	items = gmenu_tree_directory_get_contents (directory);
	for (l = items; l; l = l->next) {
		GMenuTreeItem *item = l->data;

		switch (gmenu_tree_item_get_type (item)) {
		case GMENU_TREE_ITEM_DIRECTORY:
			list = get_all_applications_from_dir (GMENU_TREE_DIRECTORY (item), list);
			break;

		case GMENU_TREE_ITEM_ENTRY:
			list = g_slist_append (list, gmenu_tree_item_ref (item));
			break;

		case GMENU_TREE_ITEM_ALIAS:
			list = get_all_applications_from_alias (GMENU_TREE_ALIAS (item), list);
			break;

		default:
			break;
		}

		gmenu_tree_item_unref (item);
	}

	g_slist_free (items);

	return list;
}

static GSList *
get_all_applications (void)
{
	GMenuTree          *tree;
	GMenuTreeDirectory *root;
	GSList             *retval;

	tree = gmenu_tree_lookup ("applications.menu", GMENU_TREE_FLAGS_NONE);

	root = gmenu_tree_get_root_directory (tree);

	retval = get_all_applications_from_dir (root, NULL);

	gmenu_tree_item_unref (root);
	gmenu_tree_unref (tree);

	retval = g_slist_sort (retval, (GCompareFunc) compare_applications);

	return retval;
}

GQuark
bugzilla_error_quark (void)
{
	  return g_quark_from_static_string ("bugzilla_error");
}

GHashTable *
load_applications (void)
{
	GnomeIconTheme *git;
	GSList         *all_applications;
	GSList         *l;
	const char     *prev_name = NULL;
	GError	       *error = NULL;

	GHashTable *program_to_application = g_hash_table_new (g_str_hash, g_str_equal);

	git = gnome_icon_theme_new ();
	gnome_icon_theme_set_allow_svg (git, FALSE);

	all_applications = get_all_applications ();
	for (l = all_applications; l; l = l->next) {
		GnomeDesktopItem *ditem;
		char *icon;
		GMenuTreeEntry *entry = l->data;

		if (prev_name && strcmp (gmenu_tree_entry_get_name (entry), prev_name) == 0) {
			gmenu_tree_item_unref (entry);
			continue;
		}
		ditem = gnome_desktop_item_new_from_uri (gmenu_tree_entry_get_desktop_file_path (entry), 
							 0, &error);
		if (!ditem) {
			if (error) {
				g_warning ("Couldn't load %s: %s", gmenu_tree_entry_get_desktop_file_path (entry),
					   error->message);
				g_error_free (error);
				error = NULL;
			}
			gmenu_tree_item_unref (entry);
			continue;
		}
    
		if ((gnome_desktop_item_get_entry_type (ditem) != GNOME_DESKTOP_ITEM_TYPE_APPLICATION) || !gnome_desktop_item_get_string (ditem, BUGZILLA_BUGZILLA)) {
			gnome_desktop_item_unref (ditem);	
			gmenu_tree_item_unref (entry);
			continue;
		}

		icon = gnome_desktop_item_get_icon (ditem, git);
		add_bugzilla_application (program_to_application,
			gnome_desktop_item_get_localestring (ditem, GNOME_DESKTOP_ITEM_NAME),
			gnome_desktop_item_get_localestring (ditem, GNOME_DESKTOP_ITEM_COMMENT),
			gnome_desktop_item_get_string (ditem, BUGZILLA_BUGZILLA),
			gnome_desktop_item_get_string (ditem, BUGZILLA_PRODUCT),
			gnome_desktop_item_get_string (ditem, BUGZILLA_COMPONENT),
			gnome_desktop_item_get_string (ditem, BUGZILLA_VERSION),
			gnome_desktop_item_get_string (ditem, BUGZILLA_EMAIL),
			icon,
			gnome_desktop_item_get_string (ditem, GNOME_DESKTOP_ITEM_EXEC),
			gnome_desktop_item_get_string (ditem, BUGZILLA_OTHER_BINARIES));
		g_free (icon);
		prev_name = gmenu_tree_entry_get_name (entry);
		gnome_desktop_item_unref (ditem);	
		gmenu_tree_item_unref (entry);
	}
	g_slist_free (all_applications);

	load_applets (git, program_to_application);

	g_object_unref (git);

	return program_to_application;
}

int
bugzilla_parse_response (SoupMessage *msg, GError **err)
{
	SoupXmlrpcResponse *response;
        SoupXmlrpcValue *value;
	long bugid;
	int debug = 0;

	g_return_if_fail (err == NULL || *err == NULL);

	if (!SOUP_STATUS_IS_SUCCESSFUL (msg->status_code)) {
		g_set_error (err, BUGZILLA_ERROR, BUGZILLA_ERROR_RECV_BAD_STATUS, 
		             _("HTTP Response returned bad status code %d"), msg->status_code);
		return -1;
	}

	response = soup_xmlrpc_message_parse_response (SOUP_XMLRPC_MESSAGE (msg));
	if (!response) {
		g_set_error (err, BUGZILLA_ERROR, BUGZILLA_ERROR_RECV_PARSE_FAILED, 
		             _("Unable to parse XML-RPC Response")); 
		return -1;
	}

	g_print ("DEBUG %s\n", soup_xmlrpc_response_to_string (response));

	/* check to see if the XMLRPC response was a <fault> */
	if (soup_xmlrpc_response_is_fault (response)) {
		SoupXmlrpcValue *faultCode = NULL;
		SoupXmlrpcValue *faultString = NULL;
		GHashTable *fault = NULL;
		gchar *errormsg = NULL;
		long errorcode = -1;

		value = soup_xmlrpc_response_get_value (response);
		if (!value) {
			debug = 1;
			goto parse_error;
		}

		/* get the struct representing the fault */
		if (!soup_xmlrpc_value_get_struct (value, &fault)) {
			debug = 2;
			goto parse_error;
		}

		/* get the integer representing the fault code */
		faultCode = g_hash_table_lookup (fault, "faultCode");
		if (faultCode == NULL || !soup_xmlrpc_value_get_int (faultCode, &errorcode)) {
			debug = 3;
			goto parse_error;
		}
		
		/* get the string representing the fault string */
		faultString = g_hash_table_lookup (fault, "faultString");
		if (faultString == NULL || !soup_xmlrpc_value_get_string (faultString, &errormsg)) {
			debug = 4;
			goto parse_error;
		}
			
		/* send back a BUGZILLA_ERROR_FAULT, using the errorcode and errorstring to
		 * construct the GError message  */
		g_set_error (err, BUGZILLA_ERROR, BUGZILLA_ERROR_RECV_FAULT, 
		             "%ld:%s", errorcode, errormsg);
		return -1;

	}

	value = soup_xmlrpc_response_get_value (response);
	if (!value) {
		debug = 5;
		goto parse_error;
	}

	if (!soup_xmlrpc_value_get_int (value, &bugid)) {
		debug = 6;
		goto parse_error;
	}	

	/* whew, everything checked out, send back the bug id */
	g_object_unref (response);
	return bugid;

parse_error:
	g_set_error (err, BUGZILLA_ERROR, BUGZILLA_ERROR_RECV_PARSE_FAILED, 
	             _("Unable to parse XML-RPC Response\n\n%d\n\n%s"),
	             debug, soup_xmlrpc_response_to_string (response));
	g_object_unref (response);
	return -1;
}


SoupXmlrpcMessage*
bugzilla_create_report (BugzillaApplication *app, GnomeVersionInfo *gnome_version,
		        const char *username, const char *text,
		        GladeXML *xml, GError **err)
{
	SoupXmlrpcMessage *message;
	char *debug_string;
	char *title;
	char *distro;
	char *comment;
	char *user_agent;

	g_return_val_if_fail (app != NULL, NULL);
	g_return_val_if_fail (gnome_version != NULL, NULL);
	g_return_val_if_fail (username != NULL, NULL);
	g_return_val_if_fail (text != NULL, NULL);
	g_return_val_if_fail (xml != NULL, NULL);
	g_return_val_if_fail (err == NULL || *err == NULL, NULL);

	/* FIXME: Hardcoded right now */
	if (app->bugzilla == NULL || strcmp (app->bugzilla, "GNOME") != 0) {
		g_set_error (err, BUGZILLA_ERROR, BUGZILLA_ERROR_SEND_NOTSUPPORTED_APP,
		             _("Application does not track its bugs in the GNOME Bugzilla."));
		return NULL;
	}

	if (!app->product || !app->component) {
		g_set_error (err, BUGZILLA_ERROR, BUGZILLA_ERROR_SEND_NOTSUPPORTED_APP,
		             _("Product or component not specified."));
		return NULL;
	}

	message = soup_xmlrpc_message_new ("http://bugzilla.gnome.org/bugbuddy.cgi");
	if (message == NULL) {
		g_set_error (err, BUGZILLA_ERROR, BUGZILLA_ERROR_SEND_ERROR,
		             _("Unable to create XML-RPC message."));
		return NULL;
	}

	user_agent = g_strdup_printf ("Bug-Buddy: %s", VERSION);
	soup_message_add_header (SOUP_MESSAGE(message)->request_headers, "User-Agent", user_agent);
	g_free (user_agent);

	soup_xmlrpc_message_start_call (message, "BugBuddy.createBug");
	soup_xmlrpc_message_start_param (message);
	soup_xmlrpc_message_start_struct (message);

	soup_xmlrpc_message_start_member (message, "version");
	soup_xmlrpc_message_write_string (message, app->version ? app->version : "unspecified");
	soup_xmlrpc_message_end_member (message);

	soup_xmlrpc_message_start_member (message, "product");
	soup_xmlrpc_message_write_string (message, app->product);
	soup_xmlrpc_message_end_member (message);

	soup_xmlrpc_message_start_member (message, "component");
	soup_xmlrpc_message_write_string (message, app->component);
	soup_xmlrpc_message_end_member (message);

	soup_xmlrpc_message_start_member (message, "gnome_version");
	soup_xmlrpc_message_write_string (message, gnome_version->gnome_platform);
	soup_xmlrpc_message_end_member (message);

	soup_xmlrpc_message_start_member (message, "reporter");
	soup_xmlrpc_message_write_string (message, username);
	soup_xmlrpc_message_end_member (message);

	soup_xmlrpc_message_start_member (message, "priority");
	soup_xmlrpc_message_write_string (message, "High");
	soup_xmlrpc_message_end_member (message);

	soup_xmlrpc_message_start_member (message, "bug_severity");
	soup_xmlrpc_message_write_string (message, "critical");
	soup_xmlrpc_message_end_member (message);


	soup_xmlrpc_message_start_member (message, "short_desc");
	title = g_strdup_printf ("crash on %s", app->name);
	soup_xmlrpc_message_write_string (message, title); 
	soup_xmlrpc_message_end_member (message);

	soup_xmlrpc_message_start_member (message, "comment");
	soup_xmlrpc_message_write_string (message, text); 
	soup_xmlrpc_message_end_member (message);
	soup_xmlrpc_message_end_param (message);
	soup_xmlrpc_message_end_struct (message);

	soup_xmlrpc_message_end_call (message);
	debug_string = (char *)soup_xmlrpc_message_to_string (message);
	g_print ("Message xml-rpc: %s\n", debug_string);
	
	g_free (title);

	return message;

}
	
