/* gnome-stones - preferences.h
 *
 * Time-stamp: <2002/05/02 17:26:51 dave>
 *
 * Copyright (C) 1998 Carsten Schaar
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "preferences.h"
#include "status.h"
#include "main.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>


/*****************************************************************************/
/* Global Variables */

/* Enum for preferences dialog */
enum {
     GAME_STRING,
     CAVES_INT,
     FILENAME_STRING,
     INDEX_INT,
     N_COLS
};

/* The default game, that should be loaded, if no client state is to
   be restored.  If this variables value is NULL, than 'default.cave'
   will be used instead.  */

gchar *default_game= NULL;

/* This variable specifies the currently played game.  If 'game' is
   equal to 'NULL', than no game is loaded.  */

GStonesGame *game= NULL;

/* The currently played cave.  This cave must be a cave, that belongs
   to the game 'game'.  */

GStonesCave *cave= NULL;

/* The data about the player.  */

GStonesPlayer *player= NULL;

/* You may start a game in different cavs.  'start_cave' decides, in
   which cave the player will start.  */

guint start_cave= 0;

/* If you use a joystick as input device, this variable holds the
   device's id.  Setting it to GDK_CORE_POINTER disables the Joystick
   support.  */
#if 0
  guint32  joystick_deviceid    = GDK_CORE_POINTER;
#else
  guint32 joystick_deviceid = 0;
#endif
gfloat   joystick_switch_level= 0.5;


/* This is default scroll method for view function */

void (*view_scroll_method) (GStonesView *view, GStonesCave *cave)= atari_scroll;


/* The game can be in different states.  These state decides, how to
   react if some events occur.  */

GameState state= STATE_TITLE;


/* The preferences dialog.  */
GtkWidget *preferences_dialog= NULL;




/****************************************************************************/
/* Some extern stuff, that should be removed soon.  */

void joystick_set_properties (guint32 deviceid, gfloat switch_level);

void load_game (const char *filename, guint cave);


/****************************************************************************/
/* Stuff for managing the list of avaiable games.  */


typedef struct _GameFile GameFile;

struct _GameFile
{
  gchar *filename;
  gchar *gametitle;
  guint  caves;
};


GList *games= NULL;


static gint
compare_game_names (const GameFile *file, const char *name)
{
  return strcmp (file->filename, name);
}


static GameFile *
add_game (const char *filename)
{
  char     *prefix= NULL;
  GList    *tmp;
  GameFile *file;
  
  /* Maybe this game is already in the games list.  */
  tmp= g_list_find_custom (games, (gpointer) filename, 
			   (GCompareFunc) compare_game_names);
  if (tmp) return (GameFile *) tmp->data;
  
  /* FIXME: add some checks, if it's realy a Gnome-Stones game.  */
  file= g_malloc (sizeof (GameFile));
  
  if (file)
    {
      file->filename = g_strdup (filename);

      prefix= g_strconcat ("=", filename, "=/", NULL);
      gnome_config_push_prefix (prefix);
      
      file->gametitle= gnome_config_get_translated_string ("General/Title");
      file->caves    = gnome_config_get_int               ("Caves/Number");

      gnome_config_pop_prefix ();
      g_free (prefix);
      
      games= g_list_append (games, file);
    }

  return file;
}


gboolean caves_suffix (char *name)
{
 int l=strlen (name); 
 if (l<7) return FALSE;

 return strcmp (name+l-6, ".caves")==0;

}

void
game_directory_scan (const char *directory)
{
  DIR *dir;
  
  dir= opendir (directory);
  if (dir)
    {
      struct dirent *entry;
      
      while ((entry = readdir (dir)) != NULL)
	{
	  char *filename= g_strconcat (directory, "/", entry->d_name, NULL);
	  struct stat sbuf;
	  
	  if ( caves_suffix (filename) 
	       && (stat (filename, &sbuf)== 0) && S_ISREG (sbuf.st_mode))
	    add_game (filename);

	  g_free (filename);
	}
      
      closedir (dir);
    }
}


static void
load_game_by_number (guint n)
{
  GList *tmp= g_list_nth (games, n);

  g_return_if_fail (tmp != NULL);
  
  load_game (((GameFile *) tmp->data)->filename, 0);
}


gboolean
load_game_by_name (const char *filename, guint cave)
{
  GameFile *file= add_game (filename);
  
  if (file)
    {
      load_game (file->filename, cave);
      
      return TRUE;
    }
  
  return FALSE;
}



/*****************************************************************************/
/* string<-->scroll_method conversion  */


gchar *scroll_method_name (void)
{
  if (view_scroll_method==atari_scroll)
    return "atari_scroll";
  if (view_scroll_method==smooth_scroll)
    return "smooth_scroll";
  if (view_scroll_method==center_scroll)
    return "center_scroll";
 
  return "default";

}

void set_scroll_method (gchar *name)
{

  if (!strcmp (name,"atari_scroll"))  view_scroll_method=atari_scroll;
  if (!strcmp (name,"smooth_scroll")) view_scroll_method=smooth_scroll;
  if (!strcmp (name,"center_scroll")) view_scroll_method=center_scroll;

}




/*****************************************************************************/
/* Save preferences.  */


void 
preferences_save (gboolean global)
{
  gchar    *devicename= NULL;
  GList    *devices;

  gnome_config_clean_section ("Preferences");
#if 0
  for (devices= gdk_devices_list (); devices; devices= devices->next)
    {
      GdkDevice *info =  devices->data;
      
      if (joystick_deviceid == info->deviceid)
	{
	  devicename= info->name;
	  break;
	}
    }
#endif  
  if (devicename)
    gnome_config_set_string ("Preferences/Joystick device",  devicename);
  gnome_config_set_float ("Preferences/Joystick switch level", 
			  joystick_switch_level);

  gnome_config_set_string ("Preferences/Scroll method", scroll_method_name());

  if (game)
    {
      gnome_config_set_string ("Preferences/Game", game->filename);
      gnome_config_set_int ("Preferences/Start cave", start_cave);
    }
  
  gnome_config_clean_section ("Windows");

  if (!global)
    {
      /* Information about open windows is only stored, if a session
         manager issued a save.  */
      gnome_config_set_bool ("Windows/Preferences", 
			     preferences_dialog != NULL);
    }
  

  gnome_config_sync ();
}


void
preferences_save_global (void)
{
  gnome_config_push_prefix 
    (gnome_client_get_global_config_prefix (gnome_master_client ()));

  preferences_save (TRUE);

  gnome_config_pop_prefix ();
  gnome_config_sync ();
}


gint
preferences_save_local (GnomeClient        *client,
			gint                phase,
			GnomeSaveStyle      save_style,
			gint                shutdown,
			GnomeInteractStyle  interact_style,   
			gpointer            client_data)
{
  gchar *prefix= gnome_client_get_config_prefix (client);
  gchar *argv[3]= {"rm", "-r", NULL };
      
  gnome_config_push_prefix (prefix);
    
  preferences_save (FALSE);

  gnome_config_pop_prefix ();
  gnome_config_sync ();

  argv[2]= gnome_config_get_real_path (prefix);
  gnome_client_set_discard_command (client, 3, argv);

  return TRUE;
}



/*****************************************************************************/
/* Restoring the preferences from disc.  */


gboolean
preferences_restore (void)
{
  GnomeClient *client= gnome_master_client ();
  gchar       *devicename;
  char        *filename;
  gboolean     def;
  guint        cave;

  gnome_config_push_prefix (gnome_client_get_config_prefix (client));
#if 0
  devicename= gnome_config_get_string ("Preferences/Joystick device=");
  if (devicename)
    {
      GList *devices;
      
      for (devices= gdk_input_list_devices (); devices; devices= devices->next)
	{
	  GdkDeviceInfo *info = (GdkDeviceInfo *) devices->data;
	  
	  if (strcmp (info->name, devicename) == 0)
	    {
	      joystick_deviceid= info->deviceid;
	      break;
	    }
	}      
      g_free (devicename);
    }
  if (joystick_deviceid != GDK_CORE_POINTER)
    gdk_input_set_mode (joystick_deviceid, GDK_MODE_SCREEN);
#endif

  joystick_switch_level= 
    gnome_config_get_float ("Preferences/Joystick switch level=0.5");

  set_scroll_method ( 
    gnome_config_get_string ("Preferences/Scroll method=atari_scroll") );


  filename= gnome_config_get_string_with_default ("Preferences/Game", &def);

  cave= gnome_config_get_int ("Preferences/Start cave=0");
  if (!default_game || !load_game_by_name (default_game, 0))
    {
      if (def || !load_game_by_name (filename, cave))
	{
	  load_game_by_name (CAVESDIR"/default.caves", 0);
	}
    }

  g_free (filename);

  if (gnome_config_get_bool ("Windows/Preferences=0"))
    {
      preferences_dialog_show ();
    }

  gnome_config_pop_prefix ();

  return TRUE;
}



/****************************************************************************/
/* Preferences dialog stuff.  */

typedef struct _PreferencesData PreferencesData;

struct _PreferencesData
{
  /* Page one.  */
  GtkWidget        *game_list;
  gint              selected_game;

  /* Page two. */
  GtkWidget        *level_frame;
  
  guint32           joystick_deviceid;
  gfloat            joystick_switch_level;


  /* Page three */

  /* Page four */
  gchar            *scroll_method_name;

};


static void
preferences_response_cb (GtkWidget *w, gint response_id, gpointer data)
{
  PreferencesData *prdata= (PreferencesData *) data;

  g_return_if_fail (prdata != NULL);

  switch(response_id)
  {
  case GTK_RESPONSE_CLOSE:
  case GTK_RESPONSE_ACCEPT:
       /* FIXME: Add some checks and warnings here.  */
       if (prdata->selected_game > -1)
            load_game_by_number (prdata->selected_game);
       
       joystick_set_properties (prdata->joystick_deviceid,
                                prdata->joystick_switch_level);
       set_scroll_method (prdata->scroll_method_name);
       /* After setting all needed values, we can save the programs
          state to disc.  */
       preferences_save_global ();
       gtk_widget_destroy (preferences_dialog);
       preferences_dialog = NULL;
       break;

  case GTK_RESPONSE_HELP:
       break;
  }

}


static gint
preferences_destroy_cb (GtkWidget *w, gpointer data)
{
  g_free (data);

  preferences_dialog= NULL;

  return FALSE;
}


static void
preferences_changed_cb (GtkWidget *w, gpointer data)
{
  PreferencesData *prdata= (PreferencesData *) data;
  
  g_return_if_fail (prdata != NULL);

}


static void 
game_selector_select_row (GtkTreeSelection * selection,
                          gpointer data)
{
  PreferencesData *prdata= (PreferencesData *) data;
  GtkTreeView * view;
  GtkTreeIter iter;
  GtkTreeModel *model;

  g_return_if_fail (prdata != NULL);

  view = gtk_tree_selection_get_tree_view(selection);
  preferences_changed_cb (GTK_WIDGET(view), data);

  if(gtk_tree_selection_get_selected(selection, &model, &iter))
  {
       int index;
       gtk_tree_model_get(model, &iter, 
                          INDEX_INT, &index,
                          -1);
       prdata->selected_game= index;
  }

}


/* The joystick callbacks.  */

static void
preferences_set_joystick_device (GtkWidget *widget, gpointer data)
{
  guint32          deviceid= GPOINTER_TO_UINT(data);
  PreferencesData *prdata  = 
    (PreferencesData *) gtk_object_get_user_data (GTK_OBJECT (widget));

  prdata->joystick_deviceid= deviceid;
#if 0  
  if (deviceid == GDK_CORE_POINTER)
    {
      gtk_widget_set_sensitive (prdata->level_frame, FALSE);
    }
  else
    {
      gtk_widget_set_sensitive (prdata->level_frame, TRUE);
    }
#endif
}


static void
preferences_set_joystick_switch_level (GtkAdjustment *adjust, gpointer data)
{
  PreferencesData *prdata  = 
    (PreferencesData *) gtk_object_get_user_data (GTK_OBJECT (adjust));

  prdata->joystick_switch_level= adjust->value;

}



/* The scroll method callbacks.  */

static void
preferences_set_scroll_method (GtkWidget *widget, gpointer data)
{

  PreferencesData *prdata  = 
    (PreferencesData *) gtk_object_get_user_data (GTK_OBJECT (widget));

  prdata->scroll_method_name= (gchar *)data;

}




static GtkWidget *
preferences_dialog_new (void)
{
  GtkWidget *properties;
  GtkWidget *box;
  GtkWidget *notebook;
  GtkWidget *list_view;
  GtkWidget *label;
  GtkWidget *scrolled;
  GtkListStore *list;
  GtkTreeIter iter;
  GtkTreeSelection *selection;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;
  
  PreferencesData *prdata;
  
  prdata= g_malloc (sizeof (PreferencesData));
  
  properties = gtk_dialog_new_with_buttons(_("GNOME-Stones Preferences"),
                                           GTK_WINDOW(app),
                                           /*GTK_DIALOG_MODAL |*/ GTK_DIALOG_DESTROY_WITH_PARENT,
                                           GTK_STOCK_HELP, GTK_RESPONSE_HELP,
                                           GTK_STOCK_CLOSE, GTK_RESPONSE_ACCEPT,
                                           NULL);
  g_signal_connect (GTK_OBJECT(properties), "destroy",
                    GTK_SIGNAL_FUNC(gtk_widget_destroyed), &properties);
  
  notebook = gtk_notebook_new();
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(properties)->vbox), notebook, 
                      TRUE, TRUE, 0);
  gtk_widget_show (notebook);
  
  /* The first page of our preferences dialog. */
  box= gtk_vbox_new (FALSE, GNOME_PAD);
  gtk_container_set_border_width (GTK_CONTAINER (box), GNOME_PAD_SMALL);

  /* The list of game names.  */
  list= gtk_list_store_new (N_COLS, 
                            G_TYPE_STRING, /* Game title */ 
                            G_TYPE_INT, /* Number of caves */
                            G_TYPE_STRING, /* Filename */
                            G_TYPE_INT); /* Index - remains hidden */

  /* Populate list */
  {
    GList *tmp= games;
    int index = 0;

    while (tmp)
      {
	char buffer[10];
	GameFile *file= (GameFile *)tmp->data;
	gint      row;
	
        gtk_list_store_append(list, &iter);
        gtk_list_store_set (list, &iter,
                            GAME_STRING, file->gametitle,
                            CAVES_INT, file->caves,
                            FILENAME_STRING, file->filename,
                            INDEX_INT, index,
                            -1);
        index++;
	tmp= tmp->next;
      }
  }

  /* Create view */
  list_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(list));

  g_object_unref(list);

  prdata->game_list= list_view;

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("Game title"), 
                                                 renderer, 
                                                 "text", 0, 
                                                 NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(list_view), column);
  
  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("Caves"), renderer, 
                                                 "text", 1, 
                                                 NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(list_view), column);

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("Filename"), renderer, 
                                                 "text", 2, 
                                                 NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(list_view), column);

  scrolled = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
                                  GTK_POLICY_ALWAYS,
                                  GTK_POLICY_AUTOMATIC);

  gtk_container_add (GTK_CONTAINER (scrolled), list_view);
  gtk_box_pack_start (GTK_BOX (box), scrolled, TRUE, TRUE, GNOME_PAD_SMALL);

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(list_view));
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);

  {
       gboolean valid;
       GtkTreeModel *model;
       /* Get an iter associated with the view */
  
       model = gtk_tree_view_get_model(GTK_TREE_VIEW(list_view));
       valid = gtk_tree_model_get_iter_first( model, &iter);
  
       while(valid)
       {
            /* Walk through list, testing each row */
            gchar *filename;
            int index;
            gtk_tree_model_get(model, &iter,
                               FILENAME_STRING, &filename,
                               INDEX_INT, &index,
                               -1 );
            if (game && strcmp (filename, game->filename) == 0)
            {
                 gtk_tree_selection_select_iter (selection, &iter);
                 prdata->selected_game= index;
            }
            valid = gtk_tree_model_iter_next (model, &iter);
       }
  }

  g_signal_connect ( selection, 
                     "changed", 
                     GTK_SIGNAL_FUNC (game_selector_select_row),
                     prdata);

  gtk_widget_show (list_view);
  gtk_widget_show (scrolled);
  gtk_widget_show (box);

  label= gtk_label_new (_("Game"));
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), 
			    box, label);

  /* The second page of our preferences dialog. */

  prdata->joystick_deviceid    = joystick_deviceid;
  prdata->joystick_switch_level= joystick_switch_level;

  box= gtk_vbox_new (FALSE, GNOME_PAD);
  gtk_container_set_border_width (GTK_CONTAINER (box), GNOME_PAD_SMALL);
  
  {
    guint      i;
    GtkObject *adjust;
    GtkWidget *frame;
    GtkWidget *scale;
    GtkWidget *hbox;
    GtkWidget *vbox;
    GtkWidget *menuitem;
    GtkWidget *optionmenu;
    GtkWidget *device_menu;
    GList     *devices;

    frame= gtk_frame_new (_("Device"));
    gtk_box_pack_start (GTK_BOX (box), frame, FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_widget_show (frame);

    vbox= gtk_vbox_new (FALSE, GNOME_PAD);
    gtk_container_add (GTK_CONTAINER (frame), vbox);
    gtk_widget_show (vbox);

    hbox= gtk_hbox_new (FALSE, GNOME_PAD);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_widget_show (hbox);

    label= gtk_label_new (_("Joystick device:"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_widget_show (label);
    
    device_menu= gtk_menu_new ();

    /* We definatly have a "disable" entry.  */
    menuitem= gtk_menu_item_new_with_label (_("disabled"));
    gtk_object_set_user_data (GTK_OBJECT (menuitem), prdata);
#if 0
    g_signal_connect (GTK_OBJECT (menuitem), "activate",
			(GtkSignalFunc) preferences_set_joystick_device,
			GUINT_TO_POINTER (GDK_CORE_POINTER));
#endif
    gtk_menu_shell_append (GTK_MENU_SHELL (device_menu), menuitem);
    gtk_widget_show (menuitem);

#if 0    
    for (devices= gdk_input_list_devices (), i= 1; devices; 
	 devices= devices->next, i++)
      {
	GdkDeviceInfo *info = (GdkDeviceInfo *) devices->data;

	if (info->deviceid != GDK_CORE_POINTER)
	  {
	    menuitem= gtk_menu_item_new_with_label (info->name);

	    gtk_object_set_user_data (GTK_OBJECT (menuitem), prdata);
            g_signal_connect (GTK_OBJECT (menuitem), "activate",
                                (GtkSignalFunc) preferences_set_joystick_device,
                                GUINT_TO_POINTER (info->deviceid));

	    gtk_menu_shell_append (GTK_MENU_SHELL (device_menu), menuitem);
	    gtk_widget_show (menuitem);
	  }

	if (info->deviceid == prdata->joystick_deviceid)
	  gtk_menu_set_active (GTK_MENU (device_menu), i);
      }
#endif    
    optionmenu= gtk_option_menu_new ();
    gtk_option_menu_set_menu (GTK_OPTION_MENU (optionmenu), device_menu);
    gtk_box_pack_start (GTK_BOX (hbox), optionmenu, FALSE, FALSE, 2);
    gtk_widget_show (optionmenu);
    
    gtk_widget_show (label);
    gtk_widget_show (hbox);
    gtk_widget_show (optionmenu);

    frame= gtk_frame_new (_("Digital joystick emulation"));
    gtk_box_pack_start (GTK_BOX (box), frame, FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_widget_show (frame);

    vbox= gtk_vbox_new (FALSE, GNOME_PAD);
    gtk_container_add (GTK_CONTAINER (frame), vbox);
    gtk_widget_show (vbox);

    hbox= gtk_hbox_new (FALSE, GNOME_PAD);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_widget_show (hbox);

    label= gtk_label_new (_("Switch level:"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_widget_show (label);

    adjust= gtk_adjustment_new (prdata->joystick_switch_level,
				0.0, 1.0, 0.02, 0.1, 0.0);
    gtk_object_set_user_data (adjust, prdata);
    g_signal_connect (adjust, "value_changed",
			(GtkSignalFunc) preferences_set_joystick_switch_level,
			NULL);
    
    scale= gtk_hscale_new (GTK_ADJUSTMENT (adjust));
    gtk_scale_set_digits (GTK_SCALE (scale), 2);
    gtk_box_pack_start (GTK_BOX (hbox), scale, FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_widget_show (scale);
#if 0    
    if (prdata->joystick_deviceid == GDK_CORE_POINTER)
      {
	gtk_widget_set_sensitive (GTK_WIDGET (frame), FALSE);
      }
#endif
    prdata->level_frame= frame;
  }
  
  gtk_widget_show (box);

  label= gtk_label_new (_("Joystick"));
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), 
			    box, label);

  /* The third page of our preferences dialog. */

  box= gtk_vbox_new (FALSE, GNOME_PAD);
  gtk_container_set_border_width (GTK_CONTAINER (box), GNOME_PAD_SMALL);
  

  label= gtk_label_new (_("Sound enable/disable. Not yet implemented!"));
  gtk_box_pack_start (GTK_BOX (box), label, TRUE, FALSE, GNOME_PAD_SMALL);
  

  gtk_widget_show (label); 
  gtk_widget_show (box); 

  label= gtk_label_new (_("Sound"));
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook),
			    box, label);


  /* Fourth page is miscellaneous stuff */
  
  box= gtk_vbox_new (FALSE, GNOME_PAD);
  gtk_container_set_border_width (GTK_CONTAINER (box), GNOME_PAD_SMALL);
  

  
  {
    gint      i=0;
    GtkWidget *frame;
    GtkWidget *hbox;
    GtkWidget *vbox;
    GtkWidget *menuitem;
    GtkWidget *optionmenu;
    GtkWidget *scroll_method_menu;


    frame= gtk_frame_new (_("Scroll method"));
    gtk_box_pack_start (GTK_BOX (box), frame, FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_widget_show (frame);

    vbox= gtk_vbox_new (FALSE, GNOME_PAD);
    gtk_container_add (GTK_CONTAINER (frame), vbox);
    gtk_widget_show (vbox);

    hbox= gtk_hbox_new (FALSE, GNOME_PAD);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_widget_show (hbox);

    /*
    label= gtk_label_new (_("Scroll method:"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_widget_show (label);
    */
    
    scroll_method_menu= gtk_menu_new ();


    prdata->scroll_method_name = scroll_method_name();

    menuitem= gtk_menu_item_new_with_label (_("Atari like scrolling"));
    gtk_object_set_user_data (GTK_OBJECT (menuitem), prdata);
    g_signal_connect (GTK_OBJECT (menuitem), "activate",
			(GtkSignalFunc) preferences_set_scroll_method,
			"atari_scroll");
    gtk_menu_shell_append (GTK_MENU_SHELL (scroll_method_menu), menuitem);
    gtk_widget_show (menuitem);

    if (!strcmp(prdata->scroll_method_name,"atari_scroll"))
      gtk_menu_set_active (GTK_MENU (scroll_method_menu), i++);
    ++i;




    menuitem= gtk_menu_item_new_with_label (_("Smooth scrolling"));
    gtk_object_set_user_data (GTK_OBJECT (menuitem), prdata);
    g_signal_connect (GTK_OBJECT (menuitem), "activate",
			(GtkSignalFunc) preferences_set_scroll_method,
			"smooth_scroll");
    gtk_menu_shell_append (GTK_MENU_SHELL (scroll_method_menu), menuitem);
    gtk_widget_show (menuitem);

    if (!strcmp(prdata->scroll_method_name,"smooth_scroll"))
      gtk_menu_set_active (GTK_MENU (scroll_method_menu), i++);
    ++i;



    menuitem= gtk_menu_item_new_with_label (_("Always in the center"));
    gtk_object_set_user_data (GTK_OBJECT (menuitem), prdata);
    g_signal_connect (GTK_OBJECT (menuitem), "activate",
			(GtkSignalFunc) preferences_set_scroll_method,
			"center_scroll");
    gtk_menu_shell_append (GTK_MENU_SHELL (scroll_method_menu), menuitem);
    gtk_widget_show (menuitem);

    if (!strcmp(prdata->scroll_method_name,"center_scroll"))
      gtk_menu_set_active (GTK_MENU (scroll_method_menu), i);
    ++i;

    

   
    optionmenu= gtk_option_menu_new ();
    gtk_option_menu_set_menu (GTK_OPTION_MENU (optionmenu), scroll_method_menu);
    gtk_box_pack_start (GTK_BOX (hbox), optionmenu, FALSE, FALSE, 2);
    gtk_widget_show (optionmenu);
    
    gtk_widget_show (label);
    gtk_widget_show (hbox);
    gtk_widget_show (optionmenu);

  }

  gtk_widget_show (box); 

  label= gtk_label_new (_("Misc."));
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook),
			    box, label);


  g_signal_connect (GTK_OBJECT (properties), "response",
		      GTK_SIGNAL_FUNC (preferences_response_cb), prdata);
  g_signal_connect (GTK_OBJECT (properties), "close",
		      GTK_SIGNAL_FUNC (preferences_destroy_cb), prdata);

  return properties;
}


void
preferences_dialog_show (void)
{
  if (!preferences_dialog)
    {
      preferences_dialog = preferences_dialog_new ();
    }
  
  gtk_widget_show (preferences_dialog);
}



/****************************************************************************/
/* Initialize the session management stuff.  */



void
session_management_init (void)
{
  GnomeClient *client= gnome_master_client ();
  
  g_signal_connect (GTK_OBJECT (client), "save_yourself",
		      GTK_SIGNAL_FUNC (preferences_save_local), 
		      GINT_TO_POINTER (FALSE));
  g_signal_connect (GTK_OBJECT (client), "die",
		      GTK_SIGNAL_FUNC (gstones_exit), NULL);
}



/* Local Variables: */
/* mode:c */
/* eval:(load-library "time-stamp") */
/* eval:(make-local-variable 'write-file-hooks) */
/* eval:(add-hook 'write-file-hooks 'time-stamp) */
/* eval:(setq time-stamp-format '(time-stamp-yyyy/mm/dd time-stamp-hh:mm:ss user-login-name)) */
/* End: */





