/*
 *  Copyright (C) 2002  Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <libgnome/gnome-i18n.h>
#include "bookmarks-bonoboui-menu.h"
#include "gul-gobject-misc.h"
#include "galeon-marshal.h"
#include "gul-bonobo-extensions.h"
#include "gul-string.h"

#include "bookmarks-gtk-menu-item.h"

#undef DEBUG_BUIM

/**
 * Private data
 */
struct _GbBonoboUIMenuPrivate {
	GbFolder *root;
	gchar *path;
	gboolean context_only;
	BonoboUIComponent *uic;
	guint dirty_timeout;

	GbLocationSource *location_source;
	GSList *menuitems;
};

#define REBUILD_TIMEOUT 1000

/**
 * Private functions, only availble from this file
 */
static void		gb_bonobo_ui_menu_class_init		(GbBonoboUIMenuClass *klass);
static void		gb_bonobo_ui_menu_init			(GbBonoboUIMenu *e);
static void		gb_bonobo_ui_menu_finalize_impl		(GObject *o);
static void		gb_bonobo_ui_menu_build_submenu		(GbBonoboUIMenu *buim);
static void		gb_bonobo_ui_menu_build_root		(GbBonoboUIMenu *buim);
static void		gb_bonobo_ui_menu_build_bookmark 	(GbBonoboUIMenu *buim, GbBookmark *item, guint index);
static gboolean		gb_bonobo_ui_menu_rebuild_timeout_cb	(gpointer data);
static void		gb_bonobo_ui_menu_rebuild_submenu_with_timeout (GbBonoboUIMenu *buim);
static void		gb_bonobo_ui_menu_rebuild_submenu	(GbBonoboUIMenu *buim);
static void 		gb_bonobo_ui_menu_child_removed_cb	(GbFolder *f, GbBookmark *b, int pos,
								 GbBonoboUIMenu *buim);
static void 		gb_bonobo_ui_menu_child_added_cb	(GbFolder *f, GbBookmark *b, int pos,
								 GbBonoboUIMenu *buim);
static void		gb_bonobo_ui_menu_bookmark_activated_cb	(GObject *sender,
								 GbBookmarkEventActivated *ev,
								 GbBonoboUIMenu *buim);


static gpointer g_object_class;

enum GbBonoboUIMenuSignalsEnum {
	GB_BONOBO_UI_MENU_BOOKMARK_ACTIVATED,
	GB_BONOBO_UI_MENU_LAST_SIGNAL
};
static gint GbBonoboUIMenuSignals[GB_BONOBO_UI_MENU_LAST_SIGNAL];

/**
 * BonoboUIMenu object
 */

MAKE_GET_TYPE (gb_bonobo_ui_menu, "GbBonoboUIMenu", GbBonoboUIMenu, gb_bonobo_ui_menu_class_init,
	       gb_bonobo_ui_menu_init, G_TYPE_OBJECT);

static void
gb_bonobo_ui_menu_class_init (GbBonoboUIMenuClass *klass)
{
	G_OBJECT_CLASS (klass)->finalize = gb_bonobo_ui_menu_finalize_impl;
	g_object_class = g_type_class_peek_parent (klass);

	GbBonoboUIMenuSignals[GB_BONOBO_UI_MENU_BOOKMARK_ACTIVATED] = g_signal_new (
		"bookmark-activated", G_OBJECT_CLASS_TYPE (klass),  
		G_SIGNAL_RUN_FIRST | G_SIGNAL_RUN_LAST | G_SIGNAL_RUN_CLEANUP,
                G_STRUCT_OFFSET (GbBonoboUIMenuClass, gb_bonobo_ui_menu_bookmark_activated), 
		NULL, NULL, 
		g_cclosure_marshal_VOID__POINTER,
		G_TYPE_NONE, 1, G_TYPE_POINTER);
}

static void 
gb_bonobo_ui_menu_init (GbBonoboUIMenu *m)
{
	GbBonoboUIMenuPrivate *p = g_new0 (GbBonoboUIMenuPrivate, 1);
	m->priv = p;
}

static void
gb_bonobo_ui_menu_finalize_impl (GObject *o)
{
	GbBonoboUIMenu *buim = GB_BONOBO_UI_MENU (o);
	GbBonoboUIMenuPrivate *p = buim->priv;

	gb_bonobo_ui_menu_set_location_source (buim, NULL);

	if (p->dirty_timeout)
	{
		g_source_remove (p->dirty_timeout);
		p->dirty_timeout = 0;
	}

	g_slist_foreach (p->menuitems, (GFunc)g_object_unref, NULL);
	g_slist_free (p->menuitems);

	g_signal_handlers_disconnect_matched (p->root, G_SIGNAL_MATCH_DATA, 0, 0, NULL, NULL, buim);
	g_object_unref (G_OBJECT (p->root));
	g_object_unref (G_OBJECT (p->uic));

	g_free (p->path);
	g_free (p);

	G_OBJECT_CLASS (g_object_class)->finalize (o);
}

GbBonoboUIMenu *
gb_bonobo_ui_menu_new (GbFolder *root, BonoboUIComponent *uic, const gchar *path, gboolean context_only)
{
	GbBonoboUIMenu *ret = g_object_new (GB_TYPE_BONOBO_UI_MENU, NULL);
	GbBonoboUIMenuPrivate *p = ret->priv;

	p->root = g_object_ref (root);
	p->uic = g_object_ref (uic);
	p->path = g_strdup (path);
	p->dirty_timeout = 0;
	p->menuitems = NULL;
	p->context_only = context_only;

	gb_bonobo_ui_menu_build_root (ret);

	return ret;
}

static void
gb_bonobo_ui_menu_build_root (GbBonoboUIMenu *buim)
{
	GbBonoboUIMenuPrivate *p = buim->priv;

	gb_bonobo_ui_menu_rebuild_submenu (buim);

	g_signal_connect (p->root, "child-added", 
			  G_CALLBACK (gb_bonobo_ui_menu_child_added_cb), buim);
	g_signal_connect (p->root, "child-removed", 
			  G_CALLBACK (gb_bonobo_ui_menu_child_removed_cb), buim);
}

static void
gb_bonobo_ui_menu_build_submenu_context_only (GbBonoboUIMenu *buim, GbBookmark *b, guint *index)
{
	GbBonoboUIMenuPrivate *p = buim->priv;

	if (b->add_to_context_menu)
	{
		if  (GB_IS_FOLDER (b) || GB_IS_SITE (b))
		{
			gb_bonobo_ui_menu_build_bookmark (buim, b, *index);
		}
		else if (GB_IS_SEPARATOR (b))
		{
			gul_bonobo_add_menu_separator (buim->priv->uic, p->path);
		}
		else
		{
			g_warning ("something skipped when building bookmarks menu");
		}
		(*index)++;
	}

	if (GB_IS_FOLDER (b) && !gb_bookmark_is_alias (b))
	{
		GbBookmark *i;
		for (i = GB_FOLDER (b)->child; i != NULL; i = i->next)
		{
			gb_bonobo_ui_menu_build_submenu_context_only (buim, i, index);
		}
	}
}

static void
gb_bonobo_ui_menu_build_submenu (GbBonoboUIMenu *buim)
{
	GbBonoboUIMenuPrivate *p = buim->priv;
	guint index = 0;
	GbBookmark *i;

	if (p->context_only)
	{
		gb_bonobo_ui_menu_build_submenu_context_only (buim, GB_BOOKMARK (p->root), &index);
	}
	else
	{
		for (i = p->root->child; i != NULL; i = i->next)
		{
			if  (GB_IS_FOLDER (i) || GB_IS_SITE (i))
			{
				gb_bonobo_ui_menu_build_bookmark (buim, i, index);
			}
			else if (GB_IS_SEPARATOR (i))
			{
				gul_bonobo_add_menu_separator (buim->priv->uic, p->path);
			}
			else
			{
				g_warning ("something skipped when building bookmarks menu");
			}
			index++;
		}
	}
}

static void
gb_bonobo_ui_menu_build_bookmark (GbBonoboUIMenu *buim, GbBookmark *item,
				  guint index)
{
	GbBonoboUIMenuPrivate *p = buim->priv;
	GbGtkMenuItem *w;

	w = gb_gtk_menu_item_new (item);
	gb_gtk_menu_item_set_location_source (w, p->location_source);
	g_signal_connect (w, "bookmark-activated", 
			  G_CALLBACK (gb_bonobo_ui_menu_bookmark_activated_cb), buim);

	p->menuitems = g_slist_prepend (p->menuitems, g_object_ref (w));

	gb_gtk_menu_item_fill_submenu (w, FALSE);

	gtk_widget_show (GTK_WIDGET (w));
	gul_bonobo_add_numbered_widget (p->uic, GTK_WIDGET (w), index, p->path);
	
}


/* dealing with changes in the menu structure */

/* in an ideal world, this should not rebuild the whole submenu, but for now it's good
   enough to do so. The difference will not be noticed unless the menu has been torn off */

static void
gb_bonobo_ui_menu_child_removed_cb (GbFolder *f, GbBookmark *b, int pos,
				    GbBonoboUIMenu *buim)
{
	gb_bonobo_ui_menu_rebuild_submenu_with_timeout (buim);
}

static void
gb_bonobo_ui_menu_child_added_cb (GbFolder *f, GbBookmark *b, int pos,
				  GbBonoboUIMenu *buim)
{
	gb_bonobo_ui_menu_rebuild_submenu_with_timeout (buim);
}

static void
gb_bonobo_ui_menu_rebuild_submenu_with_timeout (GbBonoboUIMenu *buim)
{
	g_return_if_fail (GB_IS_BONOBO_UI_MENU (buim));

	if (!buim->priv->dirty_timeout)
	{
		buim->priv->dirty_timeout = g_timeout_add (REBUILD_TIMEOUT, 
							   gb_bonobo_ui_menu_rebuild_timeout_cb, buim);
	}
}

static gboolean
gb_bonobo_ui_menu_rebuild_timeout_cb (gpointer data)
{
	GbBonoboUIMenu *buim = data;
#ifdef DEBUG_BUIM
	g_print ("Rebuild timeout %x\n", (guint32) buim);
#endif
	buim->priv->dirty_timeout = 0;

	gb_bonobo_ui_menu_rebuild_submenu (buim);
	return FALSE;
}

void
gb_bonobo_ui_menu_rebuild (GbBonoboUIMenu *buim)
{
	gb_bonobo_ui_menu_rebuild_submenu (buim);
}

static void
gb_bonobo_ui_menu_rebuild_submenu (GbBonoboUIMenu *buim)
{
	GbBonoboUIMenuPrivate *p = buim->priv;

#ifdef DEBUG_BUIM
	g_print ("Rebuilding buim: %x\n", (guint32) buim);
#endif

	gul_bonobo_clear_path (p->uic, p->path);

	/* free the list of menuitems */
	g_slist_foreach (p->menuitems, (GFunc)g_object_unref, NULL);
	g_slist_free (p->menuitems);
	p->menuitems = NULL;

	gb_bonobo_ui_menu_build_submenu (buim);

}

void
gb_bonobo_ui_menu_set_location_source (GbBonoboUIMenu *buim, GbLocationSource *src)
{
	GbBonoboUIMenuPrivate *p = buim->priv;
	GSList *li;

	if (p->location_source)
	{
		g_object_remove_weak_pointer (G_OBJECT (p->location_source),
					      (gpointer *) &p->location_source);
	}

	p->location_source = src;

	if (p->location_source)
	{
		g_object_add_weak_pointer (G_OBJECT (p->location_source), 
					   (gpointer *) &p->location_source);
	}

	for (li = p->menuitems; li; li = li->next)
	{
		gb_gtk_menu_item_set_location_source (li->data, src);
	}
}

static void
gb_bonobo_ui_menu_bookmark_activated_cb (GObject *sender,
					 GbBookmarkEventActivated *ev,
					 GbBonoboUIMenu *buim)
{
	g_signal_emit (buim, GbBonoboUIMenuSignals[GB_BONOBO_UI_MENU_BOOKMARK_ACTIVATED], 0, ev);
}
