/*
 *  Copyright (C) 2000, 2001, 2002 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include "galeon-shell.h"
#include "galeon-config.h"
#include "window-commands.h"
#include "find-dialog.h"
#include "print-dialog.h"
#include "bookmarks-editor.h"
#include "eel-gconf-extensions.h"
#include "prefs-strings.h"
#include "galeon-embed-utils.h"
#include "gul-general.h"
#include "pdm-dialog.h"
#include "toolbar.h"
#include "galeon-window-bookmark-activated-proxy.h"
#include "gul-toolbar-editor.h"
#include "gul-bonobo-extensions.h"

#include <string.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <bonobo/bonobo-i18n.h>
#include <libgnomeui/gnome-about.h>
#include <libgnome/gnome-help.h>
#include <gtk/gtkmessagedialog.h>
#include <gtk/gtkeditable.h>

#define AVAILABLE_TOOLBAR_ITEMS \
	"new=new_button;" \
	"back=navigation_button(direction=back,arrow=FALSE);" \
	"back_menu=navigation_button(direction=back,arrow=TRUE);" \
	"forward=navigation_button(direction=forward,arrow=FALSE);" \
	"forward_menu=navigation_button(direction=forward,arrow=TRUE);" \
	"up=navigation_button(direction=up,arrow=FALSE);" \
	"up_menu=navigation_button(direction=up,arrow=TRUE);" \
	"stop=stop_button;" \
	"reload=reload_button;" \
	"home=home_button;" \
	"favicon=favicon;" \
	"location=location;" \
	"go=go_button;" \
	"zoom=zoom;" \
	"spinner=spinner;" \
	"separator;"



void
window_cmd_edit_find (BonoboUIComponent *uic, 
		      GaleonWindow *window, 
		      const char* verbname)
{
	GaleonDialog *dialog;
	dialog = galeon_window_get_find_dialog (window);

	g_object_ref (dialog);
	
	galeon_dialog_show (dialog);
}

static void 
print_dialog_preview_cb (PrintDialog *dialog, 
			 GaleonWindow *window)
{
	galeon_window_set_chrome (window, EMBED_CHROME_PPVIEWTOOLBARON);
}

void 
window_cmd_file_print (BonoboUIComponent *uic, 
		       GaleonWindow *window, 
		       const char* verbname)
{
	GaleonDialog *dialog;
	GaleonEmbed *embed;
	
	embed = galeon_window_get_active_embed (window);
	g_return_if_fail (embed != NULL);
	
	dialog = print_dialog_new_with_parent (GTK_WIDGET(window),
					       embed, NULL);
	g_signal_connect (G_OBJECT(dialog),
			  "preview",
			  G_CALLBACK (print_dialog_preview_cb),
			  window);
	galeon_dialog_set_modal (dialog, TRUE);
	galeon_dialog_show (dialog);
}

void
window_cmd_go_back (BonoboUIComponent *uic, 
		    GaleonWindow *window, 
		    const char* verbname)
{
	GaleonEmbed *embed;

	embed = galeon_window_get_active_embed (window);
	g_return_if_fail (embed != NULL);

	galeon_embed_go_back (embed);
}

void
window_cmd_go_up (BonoboUIComponent *uic, 
		    GaleonWindow *window, 
		    const char* verbname)
{
	GaleonEmbed *embed;

	embed = galeon_window_get_active_embed (window);
	g_return_if_fail (embed != NULL);

	galeon_embed_go_up (embed);
}

void 
window_cmd_file_send_to	(BonoboUIComponent *uic, 
		      	 GaleonWindow *window, 
		      	 const char* verbname)
{
	char *url;
	GaleonTab *tab;
	GaleonEmbed *embed;
	char *location;
	char *title;

	tab = galeon_window_get_active_tab (window);
	g_return_if_fail (tab);
	
	embed = galeon_window_get_active_embed (window);
	g_return_if_fail (tab);
	
	location = gnome_vfs_escape_string (galeon_tab_get_location (tab));
	if (galeon_embed_get_title (embed, &title) == G_OK)
	{
		char *tmp = gnome_vfs_escape_string (title);
		g_free (title);
		title = tmp;
	}
	else
	{
		title = gnome_vfs_escape_string (_("Check this out!"));
	}

	url = g_strconcat ("mailto:",
                           "?Subject=", title,
                           "&Body=", location, NULL);

	galeon_embed_load_url (embed, url);

	g_free (title);
	g_free (location);
	g_free (url);
}

void
window_cmd_go_forward (BonoboUIComponent *uic, 
		       GaleonWindow *window, 
		       const char* verbname)
{
	GaleonEmbed *embed;

	embed = galeon_window_get_active_embed (window);
	g_return_if_fail (embed != NULL);

	galeon_embed_go_forward (embed);
}

void
window_cmd_go_go (BonoboUIComponent *uic, 
		  GaleonWindow *window, 
		  const char* verbname)
{
	Toolbar *tb;
	
	g_return_if_fail (IS_GALEON_WINDOW (window));
	
	tb = galeon_window_get_toolbar (window);

	if (tb)
	{
		char *location = toolbar_get_location (tb);
		galeon_window_load_url (window, location);
		g_free (location);
	}
}

void
window_cmd_go_home (BonoboUIComponent *uic, 
		    GaleonWindow *window, 
		    const char* verbname)
{
	GaleonEmbed *embed;
	char *location;
	
	embed = galeon_window_get_active_embed (window);
	g_return_if_fail (embed != NULL);

	location = eel_gconf_get_string (CONF_GENERAL_HOMEPAGE);
	g_return_if_fail (location != NULL);
	
	galeon_embed_load_url (embed, location);

	g_free (location);
}

void
window_cmd_go_myportal (BonoboUIComponent *uic, 
		        GaleonWindow *window, 
		        const char* verbname)
{
	GaleonEmbed *embed;
	
	embed = galeon_window_get_active_embed (window);
	g_return_if_fail (embed != NULL);

	galeon_embed_load_url (embed, "myportal:");
}

void
window_cmd_go_location (BonoboUIComponent *uic, 
		        GaleonWindow *window, 
		        const char* verbname)
{
	galeon_window_edit_location (window);
}

void
window_cmd_go_stop (BonoboUIComponent *uic, 
		    GaleonWindow *window, 
		    const char* verbname)
{
	GaleonEmbed *embed;

	embed = galeon_window_get_active_embed (window);
	g_return_if_fail (embed != NULL);

	galeon_embed_stop_load (embed);
}

void
window_cmd_go_reload (BonoboUIComponent *uic, 
		      GaleonWindow *window, 
		      const char* verbname)
{
	GaleonEmbed *embed;

	embed = galeon_window_get_active_embed (window);
	g_return_if_fail (embed != NULL);

	galeon_embed_reload (embed, EMBED_RELOAD_NORMAL);
}

static void
open_new_tab	(GaleonWindow *window, GaleonNewTabFlags flags)
{
	GaleonTab *tab;
	Toolbar *tb;

	flags |= GALEON_NEW_TAB_HOMEPAGE | GALEON_NEW_TAB_JUMP;

	tab = galeon_window_get_active_tab (window);
	
	tab = galeon_shell_new_tab (galeon_shell, window, tab, NULL, flags);

	window = galeon_tab_get_window (tab);
	tb = galeon_window_get_toolbar (window);
	toolbar_set_location (tb, "");
	toolbar_edit_location (tb);
}

void
window_cmd_new (BonoboUIComponent *uic, 
		GaleonWindow *window, 
		const char* verbname)
{
	open_new_tab (window, 0);
}

void
window_cmd_new_window (BonoboUIComponent *uic, 
		       GaleonWindow *window, 
		       const char* verbname)
{
	open_new_tab (window, GALEON_NEW_TAB_IN_NEW_WINDOW);
}

void
window_cmd_new_tab (BonoboUIComponent *uic, 
		    GaleonWindow *window, 
		    const char* verbname)
{
	open_new_tab (window, GALEON_NEW_TAB_IN_EXISTING_WINDOW);
}

void
window_cmd_bookmarks_edit (BonoboUIComponent *uic, 
			   GaleonWindow *window, 
			   const char* verbname)
{
	GbBookmarkSet *set;
	GbEditor *e;
	GaleonWindowBookmarkActivatedProxy *wbap;

	set = galeon_shell_get_bookmark_set (galeon_shell);
	if (!set) return;

	e = gb_editor_new_for_set (set);
	gtk_widget_show (gb_editor_get_main_widget (e));

	wbap = galeon_window_bookmark_activated_proxy_new ();
	galeon_window_bookmark_activated_proxy_set_window (wbap, window);
	galeon_window_bookmark_activated_proxy_connect (wbap, G_OBJECT (e));

	g_signal_connect_swapped (gb_editor_get_main_widget (e), 
				  "delete_event", G_CALLBACK (g_object_unref), e);
	g_signal_connect_swapped (gb_editor_get_main_widget (e), 
				  "delete_event", G_CALLBACK (g_object_unref), wbap);
}

void
window_cmd_bookmarks_add_default (BonoboUIComponent *uic, 
				  GaleonWindow *window, 
				  const char* verbname)
{
	GaleonTab *tab;
	const char *title;
	const char *location;
	GbBookmarkSet *set;
	GbSite *b;

	tab = galeon_window_get_active_tab (window);
	g_return_if_fail (tab);

	title = galeon_tab_get_title (tab);
	location = galeon_tab_get_location (tab);

	set = galeon_shell_get_bookmark_set (galeon_shell);
	g_return_if_fail (set);

	b = gb_site_new (set, title, location);
	gb_bookmark_set_add_default (set, GB_BOOKMARK (b));
}

void
window_cmd_file_open (BonoboUIComponent *uic, 
		      GaleonWindow *window, 
		      const char* verbname)
{
	gchar *dir, *retDir;
        gchar *file;
        GnomeVFSURI *uri;
	GtkWidget *wmain;
	GaleonEmbedShell *embed_shell;
	gresult result;

	embed_shell = galeon_shell_get_embed_shell (galeon_shell);
	g_return_if_fail (embed_shell != NULL);
	
	wmain = GTK_WIDGET (window);
	g_return_if_fail (wmain != NULL);
	
        dir = eel_gconf_get_string (CONF_STATE_OPEN_DIR);

	result = galeon_embed_shell_show_file_picker 
		(embed_shell, wmain, 
		 _("Select the file to open"),
                 dir, NULL, modeOpen,
                 &file, NULL, NULL, NULL);

	if (result == G_OK)
	{
                uri = gnome_vfs_uri_new (file);
                if (uri)
                {
			
                        galeon_window_load_url(window, file);

                        retDir = gnome_vfs_uri_extract_dirname (uri);

                        /* set default open dir */
                        eel_gconf_set_string (CONF_STATE_OPEN_DIR, 
                                              retDir);

                        g_free (retDir);
                        gnome_vfs_uri_unref (uri);
                }

		g_free (file);
        }

        g_free (dir);   
}

void
window_cmd_file_save_as (BonoboUIComponent *uic, 
		      	 GaleonWindow *window, 
		         const char* verbname)
{
	GaleonEmbed *embed;
	GaleonEmbedPersist *persist;
	char *url;
	
	embed = galeon_window_get_active_embed (window);
	g_return_if_fail (embed != NULL);

	persist = galeon_embed_persist_new (embed);
	galeon_embed_persist_set_flags (persist,
					EMBED_PERSIST_MAINDOC);

	galeon_embed_get_location(embed, TRUE, TRUE, &url);
	galeon_embed_persist_set_source(persist, url);

	galeon_embed_utils_save (GTK_WIDGET(window), 
			    	 CONF_STATE_SAVE_DIR,
			    	 TRUE,
			    	 TRUE,
			    	 persist);

	g_object_unref (G_OBJECT(persist));
}

void window_cmd_file_quit(BonoboUIComponent *uic,
			  GaleonWindow *window,
			  const char* verbname)
{
	Session *session =
		galeon_shell_get_session(galeon_shell);

	session_close(session);
}

void
window_cmd_file_close_tab (BonoboUIComponent *uic, 
		           GaleonWindow *window, 
		           const char* verbname)
{
	GaleonTab *tab;
	
	tab = galeon_window_get_active_tab (window);
	g_return_if_fail (tab != NULL);

	galeon_window_remove_tab (window, tab);
}

void 
window_cmd_edit_cut (BonoboUIComponent *uic, 
		     GaleonWindow *window, 
		     const char* verbname)
{
	GtkWidget *widget = gtk_window_get_focus (GTK_WINDOW (window));

	if (GTK_IS_EDITABLE (widget))
	{
		gtk_editable_cut_clipboard (GTK_EDITABLE (widget));
	}
	else
	{
		GaleonEmbed *embed;
		embed = galeon_window_get_active_embed (window);
		g_return_if_fail (embed != NULL);
		
		galeon_embed_selection_cut (embed);
	}
}

void
window_cmd_edit_copy (BonoboUIComponent *uic, 
		      GaleonWindow *window, 
		      const char* verbname)
{
	GtkWidget *widget = gtk_window_get_focus (GTK_WINDOW (window));

	if (GTK_IS_EDITABLE (widget))
	{
		gtk_editable_copy_clipboard (GTK_EDITABLE (widget));
	}
	else
	{
		GaleonEmbed *embed;

		embed = galeon_window_get_active_embed (window);
		g_return_if_fail (embed != NULL);
		
		galeon_embed_selection_copy (embed);
	}
}

void
window_cmd_edit_paste (BonoboUIComponent *uic, 
		       GaleonWindow *window, 
		       const char* verbname)
{
	GtkWidget *widget = gtk_window_get_focus (GTK_WINDOW (window));

	if (GTK_IS_EDITABLE (widget))
	{
		gtk_editable_paste_clipboard (GTK_EDITABLE (widget));
	}
	else
	{
		GaleonEmbed *embed;

		embed = galeon_window_get_active_embed (window);
		g_return_if_fail (embed != NULL);
		
		galeon_embed_paste (embed);
	}
}

void 
window_cmd_edit_select_all (BonoboUIComponent *uic, 
		      	    GaleonWindow *window, 
		      	    const char* verbname)
{
	GtkWidget *widget = gtk_window_get_focus (GTK_WINDOW (window));

	if (GTK_IS_EDITABLE (widget))
	{
		gtk_editable_select_region (GTK_EDITABLE (widget), 0, -1);
	}
	else
	{
		GaleonEmbed *embed;

		embed = galeon_window_get_active_embed (window);
		g_return_if_fail (embed != NULL);
		
		galeon_embed_select_all (embed);
	}
}

void
window_cmd_edit_find_next (BonoboUIComponent *uic, 
			   GaleonWindow *window, 
			   const char* verbname)
{
	GaleonDialog *dialog;
	
	dialog = galeon_window_get_find_dialog (window);

	find_dialog_go_next (FIND_DIALOG(dialog), FALSE);

	galeon_window_update_control (window, FindControl);
}

void
window_cmd_edit_find_prev (BonoboUIComponent *uic, 
			   GaleonWindow *window, 
			   const char* verbname)
{
	GaleonDialog *dialog;
	
	dialog = galeon_window_get_find_dialog (window);

	find_dialog_go_prev (FIND_DIALOG(dialog), FALSE);

	galeon_window_update_control (window, FindControl);
}

void 
window_cmd_set_permission(BonoboUIComponent *uic, 
		      	  GaleonWindow *window, 
			  const char* verbname)
{
	gresult result;

	gchar *url;
	PermissionType type;
	gboolean value;

	GaleonEmbed *embed;
	GaleonEmbedShell *embed_shell;

	embed_shell = galeon_shell_get_embed_shell(galeon_shell);
	g_return_if_fail(embed_shell != NULL);

	embed = galeon_window_get_active_embed(window);
	result = galeon_embed_get_location(embed, TRUE, FALSE, &url);
	if(result != G_OK) return;

	if (!strcmp(verbname, "AllowCookieSite"))
	{
		type = COOKIES_PERMISSION;
		value = TRUE;
#if MOZILLA_SNAPSHOT <= 6
		gul_bonobo_set_sensitive(uic, "/commands/AllowCookieSite",
					 FALSE);
		gul_bonobo_set_sensitive(uic, "/commands/BlockCookieSite",
					 TRUE);
#endif
	}
	else if (!strcmp(verbname, "BlockCookieSite"))
	{
		type = COOKIES_PERMISSION;
		value = FALSE;
#if MOZILLA_SNAPSHOT <= 6
		gul_bonobo_set_sensitive(uic, "/commands/AllowCookieSite",
					 TRUE);
		gul_bonobo_set_sensitive(uic, "/commands/BlockCookieSite",
					 FALSE);
#endif
	}
	else if (!strcmp(verbname, "AllowImageSite"))
	{
		type = IMAGES_PERMISSION;
		value = TRUE;
#if MOZILLA_SNAPSHOT <= 6
		gul_bonobo_set_sensitive(uic, "/commands/AllowImageSite",
					 FALSE);
		gul_bonobo_set_sensitive(uic, "/commands/BlockImageSite",
					 TRUE);
#endif
	}
	else //if (!strcmp(verbname, "BlockImageSite"))
	{
		type = IMAGES_PERMISSION;
		value = FALSE;
#if MOZILLA_SNAPSHOT <= 6
		gul_bonobo_set_sensitive(uic, "/commands/AllowImageSite",
					 TRUE);
		gul_bonobo_set_sensitive(uic, "/commands/BlockImageSite",
					 FALSE);
#endif
	}

	galeon_embed_shell_set_permission(embed_shell, url, type, value);
}

void
window_cmd_show_java_console(BonoboUIComponent *uic,
			     GaleonWindow *window,
			     const char* verbname)
{
	GaleonEmbedShell *embed_shell;
	gresult result;

	embed_shell = galeon_shell_get_embed_shell(galeon_shell);
	g_return_if_fail(embed_shell != NULL);

	result = galeon_embed_shell_show_java_console(embed_shell);
}

void window_cmd_show_js_console(BonoboUIComponent *uic,
				GaleonWindow *window,
				const char* verbname)
{
	GaleonEmbedShell *embed_shell;
	gresult result;

	embed_shell = galeon_shell_get_embed_shell(galeon_shell);
	g_return_if_fail(embed_shell != NULL);

	result = galeon_embed_shell_show_js_console(embed_shell);
}

void 
window_cmd_view_zoom_in	(BonoboUIComponent *uic, 
		      	 GaleonWindow *window, 
		      	 const char *verbname)
{
	GaleonEmbed *embed;
	int zoom;
	
	embed = galeon_window_get_active_embed (window);
	g_return_if_fail (embed != NULL);

	galeon_embed_zoom_get (embed, &zoom);
	galeon_window_set_zoom (window, zoom + 10);
}

void 
window_cmd_view_zoom_out (BonoboUIComponent *uic, 
		      	  GaleonWindow *window, 
		          const char* verbname)
{
	GaleonEmbed *embed;
	int zoom;
	
	embed = galeon_window_get_active_embed (window);
	g_return_if_fail (embed != NULL);

	galeon_embed_zoom_get (embed, &zoom);
	if (zoom >= 10)
	{
		galeon_window_set_zoom (window, zoom - 10);
	}
}

void 
window_cmd_view_zoom_normal (BonoboUIComponent *uic, 
		      	     GaleonWindow *window, 
		      	     const char* verbname)
{
	galeon_window_set_zoom (window, 100);
}

void 
window_cmd_view_page_source (BonoboUIComponent *uic, 
		      	     GaleonWindow *window, 
		      	     const char* verbname)
{
	GaleonTab *tab;
	
	tab = galeon_window_get_active_tab (window);
	g_return_if_fail (tab != NULL);

	galeon_shell_new_tab (galeon_shell, window, tab, NULL, 
			      GALEON_NEW_TAB_VIEW_SOURCE);
}

void 
window_cmd_tools_history (BonoboUIComponent *uic, 
		      	  GaleonWindow *window, 
		      	  const char* verbname)
{
	galeon_window_show_history (window);
}

void 
window_cmd_tools_pdm (BonoboUIComponent *uic, 
		      GaleonWindow *window, 
		      const char* verbname)
{
	GaleonDialog *dialog;

	dialog = pdm_dialog_new (GTK_WIDGET(window));

	galeon_dialog_show (dialog);
}

void 
window_cmd_edit_prefs (BonoboUIComponent *uic, 
		       GaleonWindow *window, 
		       const char* verbname)
{
	GtkDialog *dialog;

	dialog = prefs_dialog_new ();
	prefs_dialog_show_page (PREFS_DIALOG(dialog),
				PREFS_PAGE_GENERAL);
	gtk_window_set_transient_for (GTK_WINDOW (dialog),
				      GTK_WINDOW (window));
	gtk_widget_show (GTK_WIDGET(dialog));
}

static void
window_cmd_settings_toolbar_editor_revert_clicked_cb (GtkButton *b, GulTbEditor *tbe)
{
	gchar *def;

	g_return_if_fail (GUL_IS_TB_EDITOR (tbe));
	
	eel_gconf_unset (CONF_TOOLBAR_SETUP);
	def = eel_gconf_get_string (CONF_TOOLBAR_SETUP);
	if (def)
	{
		GulToolbar *current;
		GulToolbar *avail;
		current = gul_tb_editor_get_toolbar (tbe);
		gul_toolbar_parse (current, def);
		g_free (def);

		avail = gul_tb_editor_get_available (tbe);
		g_object_ref (avail);
		gul_toolbar_parse (avail, AVAILABLE_TOOLBAR_ITEMS);
		gul_tb_editor_set_available (tbe, avail);
		g_object_unref (avail);
	}

}

static void
window_cmd_settings_toolbar_editor_current_changed_cb (GulToolbar *tb, gpointer data)
{
	gchar *current_str;

	g_return_if_fail (GUL_IS_TOOLBAR (tb));

	current_str = gul_toolbar_to_string (tb);
	eel_gconf_set_string (CONF_TOOLBAR_SETUP, current_str);
	g_free (current_str);
}

void 
window_cmd_settings_toolbar_editor (BonoboUIComponent *uic, 
				    GaleonWindow *window, 
				    const char* verbname)
{
	static GulTbEditor *tbe = NULL;
	GulToolbar *avail;
	GulToolbar *current;
	gchar *current_str;
	GtkButton *revert_button;

	avail = gul_toolbar_new ();
	gul_toolbar_parse (avail, AVAILABLE_TOOLBAR_ITEMS);

	current_str = eel_gconf_get_string (CONF_TOOLBAR_SETUP);
	current = gul_toolbar_new ();
	if (current_str)
	{
		gul_toolbar_parse (current, current_str);
		g_free (current_str);
	}

	if (!tbe)
	{
		tbe = gul_tb_editor_new ();
		g_object_add_weak_pointer (G_OBJECT (tbe),
                                           (void **)&tbe);
		gul_tb_editor_set_parent (tbe, 
					  GTK_WIDGET(window));
	}
	else
	{
		gul_tb_editor_show (tbe);
                return;
	}
	
	gul_tb_editor_set_toolbar (tbe, current);
	gul_tb_editor_set_available (tbe, avail);
	g_object_unref (avail);
	g_object_unref (current);

	g_signal_connect (current, "changed", 
			  G_CALLBACK (window_cmd_settings_toolbar_editor_current_changed_cb), NULL);

	revert_button = gul_tb_editor_get_revert_button (tbe);
	gtk_widget_show (GTK_WIDGET (revert_button));

	g_signal_connect (revert_button, "clicked", 
			  G_CALLBACK (window_cmd_settings_toolbar_editor_revert_clicked_cb), tbe);
	
	gul_tb_editor_show (tbe);
}

void
window_cmd_help_about (BonoboUIComponent *uic, 
		       GaleonWindow *window, 
		       const char* verbname)
{
	static GtkWidget *about = NULL;
	GdkPixbuf  	 *pixbuf;
	GError     	 *error = NULL;
	gchar      	 *file;

	const gchar *authors[] = {
		"Marco Pesenti Gritti <mpeseng@tin.it>",
		"Remco de Jong <rdj@rdj.cg.nu>",
		"Ricardo Fern\xc3\xa1ndez Pascual <ric@users.sourceforge.net>",
		"Nate Case <nd@kracked.com>",
		"Bob Phan <ex-freek@yahoo.com>",
		"Matt Aubury <matt@ookypooky.com>",
		"Joshua Horvath <jmhorvat@home.com>",
		/* this one is supposed to be in cyrillic */
		_("Yanko Kaneti <yaneti@declera.com>"),
		"Philip Langdale <philipl@mail.utexas.edu>",
		"Matthew Mueller <donut@azstarnet.com>",
		"Daniel Erat <dan_erat@pobox.com>",
		"Jorn Baayen <jorn@nl.linux.org>",
		/* don't delete the space */
		" ",
		_("Graphic arts:"),
		"Toussaint Fr\xc3\xa9" "d\xc3\xa9ric <fred@linuxgraphic.org>",
		NULL
	};

	gchar *documenters[] = {
		NULL
	};

	/* Translator credits */
	gchar *translator_credits = _("translator_credits");

	if (about != NULL)
	{
		gdk_window_show(about->window);
		gdk_window_raise(about->window);
		return;
	}

	file = gul_general_user_file("about_logo.png", FALSE);
	pixbuf = gdk_pixbuf_new_from_file (file, &error);
	
	if (error) {
		g_warning (G_STRLOC ": cannot open %s: %s", file, error->message);
		g_error_free (error);
	}
	g_free (file);
	
	about = gnome_about_new(
		       _("Galeon"), VERSION,
		       "Copyright \xc2\xa9 2000-2002 Marco Pesenti Gritti",
		       _("A GNOME browser based on Mozilla"),
		       (const char **)authors,
		       (const char **)documenters,
		       strcmp (translator_credits, "translator_credits") != 0 ? translator_credits : NULL,
		       pixbuf);
	if (pixbuf) {
		g_object_unref (G_OBJECT(pixbuf));
	}			
				
	gtk_window_set_transient_for (GTK_WINDOW (about),
				      GTK_WINDOW (window));
	g_object_add_weak_pointer (G_OBJECT (about), (gpointer *)&about);
	gtk_widget_show (about);
}

void 
window_cmd_set_charset (BonoboUIComponent *uic, 
			EncodingMenuData *data, 
			const char* verbname)
{
	GaleonWindow *window = data->data;
	GaleonEmbed *embed;
	
	embed = galeon_window_get_active_embed (window);
	g_return_if_fail (embed != NULL);

	g_print (data->encoding);
	galeon_embed_set_charset (embed, data->encoding);
}

void 
window_cmd_tabs_next (BonoboUIComponent *uic, 
		      GaleonWindow *window, 
		      const char* verbname)
{
	galeon_window_next_tab (window);
}

void 
window_cmd_tabs_previous (BonoboUIComponent *uic, 
			  GaleonWindow *window, 
			  const char* verbname)
{
	galeon_window_prev_tab (window);
}

void 
window_cmd_tabs_move_left  (BonoboUIComponent *uic, 
			    GaleonWindow *window, 
			    const char* verbname)
{
	GList *tabs;
	GaleonTab *tab;

	tab = galeon_window_get_active_tab (window);
	tabs = galeon_window_get_tabs (window);
	g_return_if_fail (tab != NULL);
	
	tabs = g_list_find (tabs, (gpointer)tab);
	tabs = tabs->prev;

	if (tabs)
	{
		GaleonTab *sibling;
		sibling = GALEON_TAB (tabs->data);
		galeon_window_move_tab (window, tab, sibling);
		g_list_free (tabs);
	}
}

void window_cmd_tabs_move_right (BonoboUIComponent *uic, 
				 GaleonWindow *window, 
				 const char* verbname)
{
	GList *tabs;
	GaleonTab *tab;

	tab = galeon_window_get_active_tab (window);
	tabs = galeon_window_get_tabs (window);
	g_return_if_fail (tab != NULL);
	
	tabs = g_list_find (tabs, (gpointer)tab);
	tabs = tabs->next;

	if (tabs)
	{
		GaleonTab *sibling;
		sibling = GALEON_TAB (tabs->data);
		galeon_window_move_tab (window, tab, sibling);
		g_list_free (tabs);
	}
}

void 
window_cmd_tabs_detach  (BonoboUIComponent *uic, 
			 GaleonWindow *window, 
			 const char* verbname)
{
	GaleonWindow *new_win;
	GaleonTab *tab;

	if (g_list_length (galeon_window_get_tabs (window)) <= 1) {
		return;
	}

	tab = galeon_window_get_active_tab (window);	
	new_win = galeon_window_new ();

	galeon_window_reparent_tab (new_win, window, tab);

	gtk_widget_show (GTK_WIDGET (new_win));
}

void 
window_cmd_help_manual (BonoboUIComponent *uic, 
			char *filename, 
			const char* verbname)
{
	GError *error;
	GtkWidget *dialog;

	error = NULL;
	gnome_help_display ("galeon.xml", NULL, &error);

	if (error) 
	{
		dialog = gtk_message_dialog_new (NULL,
						 GTK_DIALOG_MODAL,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_CLOSE,
						 _("There was an error displaying help: \n%s"),
						 error->message);
		g_signal_connect (G_OBJECT (dialog), "response",
				  G_CALLBACK (gtk_widget_destroy),
				  NULL);

		gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
		gtk_widget_show (dialog);
		g_error_free (error);
	}
}
