/* spui.c
 *
 * Copyright 2001, 2002 Sun Microsystems, Inc.,
 * Copyright 2001, 2002 BAUM Retec, A.G.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include "spui.h"
#include "SRMessages.h"
#include "libsrconf.h"
#include <glade/glade.h>
#include "srintl.h"
#include "defui.h"
#include "gnopiui.h"

#define INVALID_VOICE "test"

#define CHECK_RANGE_OF_VALUE(val,min,max)\
	(min > val ? min : (val > max ? max : val))

typedef enum
{
    PUNCTUATION_IGNORE,
    PUNCTUATION_SAME,
    PUNCTUATION_MOST,
    PUNCTUATION_ALL,
    PUNCTUATION_NUMBER
}PunctuationType;

typedef enum
{
    TEXT_ECHO_CHARACTER,
    TEXT_ECHO_WORD,
    TEXT_ECHO_NUMBER
}TextEchoType;

enum 
{
    VOICE_GNOPERNICUS_SPEAKER,
    VOICE_GNOME_SPEAK_SPEAKER,
    VOICE_VOLUME,
    VOICE_RATE,
    VOICE_PITCH,
    
    VOICE_NO_COLUMN
};

enum
{
    SPEECH_DICTIONARY_WORD,
    SPEECH_DICTIONARY_REPLACER,
    
    SPEECH_DICTIONARY_NO_COLUMN
};

struct {
    GtkWidget	*w_speech_settings;
    GtkWidget	*ck_count;
    GtkWidget	*ck_spaces;
    GtkWidget	*ck_cursors;
    GtkWidget	*ck_modifiers;
    GtkWidget	*ck_dictionary;
    GtkWidget	*bt_dictionary;
    GtkWidget	*rb_punctuation [PUNCTUATION_NUMBER];
    GtkWidget	*rb_text_echo [TEXT_ECHO_NUMBER];
} spui_speech_settings;

struct {
    GtkWidget	*w_dictionary;
    GtkWidget	*w_dict_add_modify;
    GtkWidget	*et_word;
    GtkWidget	*et_replace;
    GtkWidget	*tv_dictionary;
} spui_speech_dictionary;

struct {
    GtkWidget	*w_speech_voice_add_modify;
    GtkWidget	*w_speech_voice;
    GtkWidget	*tv_voice_table;
    GtkWidget	*cb_engine_voice;
    GtkWidget	*et_gnopernicus_voice;
    GtkWidget	*sp_voice_volume;
    GtkWidget	*sp_voice_rate;
    GtkWidget	*sp_voice_pitch;
    GtkWidget	*sp_voice_priority;
    GtkWidget	*hs_voice_volume;
    GtkWidget	*hs_voice_rate;
    GtkWidget	*hs_voice_pitch;
    GtkWidget	*hs_voice_priority;
/*    GtkWidget	*ck_apply_for_all;*/
} spui_speech_voice;


static GtkTreeIter			curr_iter;		
static SpeakerSettingsListType		*curr_speaker_settings;
static GnopernicusSpeakerListType	*curr_gnopernicus_speaker;
static SpeakerSettings			old_speaker_settings;
static gboolean				new_item;

extern Speech 				*speech_setting;
static gboolean				speech_changed;
static gboolean				changes_activable;

extern GnopernicusSpeakerListType 	*gnopernicus_speakers;
static GnopernicusSpeakerListType 	*gnopernicus_speakers_backup;
extern SpeakerSettingsListType  	*speakers_settings;
static SpeakerSettingsListType  	*speakers_settings_backup;
extern DictionaryListType		*dictionary_list;

gchar *punct_keys[]=
{
    "IGNORE",
    "SAME",
    "TRANSLATE",
    "BOTH"
};

gchar *common_keys[]=
{
    "NONE",
    "ALL"
};

gchar *dictionary_keys[]=
{
    "NO",
    "YES"
};

gchar *text_echo_keys[]=
{
    "CHARACTER",
    "WORD"
};

#define STR(X) (X == NULL? "" : X)
/******************************************************************************/

static void
spui_clean_old_speaker_setting (void)
{
    old_speaker_settings.volume = 0;
    old_speaker_settings.rate   = 0;
    old_speaker_settings.pitch  = 0;
    g_free (old_speaker_settings.gs_speaker);
    g_free (old_speaker_settings.gnopernicus_speaker);
    old_speaker_settings.gs_speaker = NULL;
    old_speaker_settings.gnopernicus_speaker = NULL;
}

static void
spui_set_speaker_parameter_in_list_store_at_iter (SpeakerSettings *item, 
					        GtkListStore *store,
					        GtkTreeIter  iter)
{
    sru_return_if_fail (store);
    sru_return_if_fail (item);
    
    gtk_list_store_set (GTK_LIST_STORE (store), &iter, 
			VOICE_GNOPERNICUS_SPEAKER,	(gchar*)item->gnopernicus_speaker,
			VOICE_GNOME_SPEAK_SPEAKER,  	(gchar*)item->gs_speaker,
			VOICE_VOLUME,			item->volume,
			VOICE_RATE,			item->rate,
			VOICE_PITCH,			item->pitch,
		    	-1);
}


static SpeakerSettings*
spui_add_modify_get_new_speaker_settings (void)
{
    SpeakerSettings *new_settings = NULL;
    
    if (!spui_speech_voice.w_speech_voice_add_modify)
	return NULL;
	    
    new_settings = spconf_speaker_settings_new ();
    new_settings->volume = CHECK_RANGE_OF_VALUE (gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spui_speech_voice.sp_voice_volume)), MIN_SPEECH_VOLUME, MAX_SPEECH_VOLUME);
    new_settings->rate   = CHECK_RANGE_OF_VALUE (gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spui_speech_voice.sp_voice_rate)), MIN_SPEECH_RATE, MAX_SPEECH_RATE);
    new_settings->pitch  = CHECK_RANGE_OF_VALUE (gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spui_speech_voice.sp_voice_pitch)), MIN_SPEECH_PITCH, MAX_SPEECH_PITCH);
    new_settings->gs_speaker = g_strdup (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (spui_speech_voice.cb_engine_voice)->entry)));
    new_settings->gnopernicus_speaker = g_strdup (gtk_entry_get_text (GTK_ENTRY (spui_speech_voice.et_gnopernicus_voice)));    
    
    return new_settings;
}

static gboolean
spui_add_modify_get_properties (void)
{
    SpeakerSettingsListType    *ss_found = NULL;
    GnopernicusSpeakerListType *gs_found = NULL;
    SpeakerSettings *new_setting = NULL;
    GtkTreeModel    *model = NULL;

    if (!spui_speech_voice.w_speech_voice_add_modify ||
	!curr_speaker_settings    ||
	!curr_gnopernicus_speaker) 
	return FALSE;

    model = gtk_tree_view_get_model (GTK_TREE_VIEW (spui_speech_voice.tv_voice_table));    
        
    new_setting = spui_add_modify_get_new_speaker_settings ();

    ss_found = spconf_speaker_settings_list_find (speakers_settings, new_setting->gnopernicus_speaker);
    gs_found = spconf_gnopernicus_speakers_find (gnopernicus_speakers, new_setting->gnopernicus_speaker);
    
    if (new_item)
    {
	if (!ss_found && !gs_found &&
	    strcmp (NONE_ELEMENT,  new_setting->gnopernicus_speaker) && 
	    strcmp (INVALID_VOICE, new_setting->gnopernicus_speaker)) 
	{
	    curr_speaker_settings->data = 
		spconf_speaker_settings_copy (curr_speaker_settings->data, 
					      new_setting);

	    g_free (curr_gnopernicus_speaker->data);
	    curr_gnopernicus_speaker->data 	= g_strdup (new_setting->gnopernicus_speaker);
	}
	else
	{
	    speakers_settings =
		spconf_speaker_settings_list_remove (speakers_settings,
						     ((SpeakerSettings*)(curr_speaker_settings->data))->gnopernicus_speaker);
	    gnopernicus_speakers =
		spconf_gnopernicus_speakers_remove (gnopernicus_speakers,
						    curr_gnopernicus_speaker->data);
	    gtk_list_store_remove (GTK_LIST_STORE (model), &curr_iter);

	    defui_send_msg (_("Invalide voice!"));
	    
	    return FALSE;
	}
    }
    else
    if (ss_found &&  gs_found)
    {
	    curr_speaker_settings->data = 
		spconf_speaker_settings_copy (curr_speaker_settings->data, 
						   new_setting);
    }
    else
	return FALSE;    
        

    spui_set_speaker_parameter_in_list_store_at_iter (new_setting, 
						    	GTK_LIST_STORE (model),
					    	        curr_iter);


    spconf_speaker_settings_free (new_setting);

    return TRUE;
}

static void
spui_remove_created_item (void)
{
    GtkTreeModel     *model;
    model = gtk_tree_view_get_model (GTK_TREE_VIEW (spui_speech_voice.tv_voice_table));
    
    if (new_item)
    {
	gtk_list_store_remove (GTK_LIST_STORE (model), &curr_iter);
	
	if (curr_speaker_settings && 
	    curr_speaker_settings->data)
	{
	    speakers_settings =
		spconf_speaker_settings_list_remove (speakers_settings,
		((SpeakerSettings*)curr_speaker_settings->data)->gnopernicus_speaker);
	}
    }
    else
    {
	spui_set_speaker_parameter_in_list_store_at_iter (&old_speaker_settings, 
							  GTK_LIST_STORE (model),
					    		  curr_iter);
	curr_speaker_settings->data =
		spconf_speaker_settings_copy (curr_speaker_settings->data, 
			    		      &old_speaker_settings);

	spconf_speaker_settings_save ((SpeakerSettings*)curr_speaker_settings->data);
    }
}

void
spui_engine_voice_list_set (GList *voice_list)
{
    if (!voice_list) 
	return;
    gtk_combo_set_popdown_strings (GTK_COMBO (spui_speech_voice.cb_engine_voice),
				    voice_list);
}

static void
spui_add_modify_select (const gchar *mode)
{	
    gboolean state = FALSE;
    if (!strcmp (mode, "ADD"))  state = TRUE;
		    else	state = FALSE;
    gtk_widget_set_sensitive (GTK_WIDGET (spui_speech_voice.et_gnopernicus_voice), state);
}

static gboolean
spui_modify_speaker_values_in_add_modify (SpeakerSettings *settings)
{
    if (!spui_speech_voice.w_speech_voice_add_modify) 
	return FALSE;
	
    if (strcmp (((SpeakerSettings*)curr_speaker_settings->data)->gnopernicus_speaker, 
		settings->gnopernicus_speaker))
	return FALSE;

    gtk_spin_button_set_value (GTK_SPIN_BUTTON (spui_speech_voice.sp_voice_volume), 	(gdouble)settings->volume);
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (spui_speech_voice.sp_voice_rate), 	(gdouble)settings->rate);
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (spui_speech_voice.sp_voice_pitch), 	(gdouble)settings->pitch);
    gtk_range_set_value (GTK_RANGE (spui_speech_voice.hs_voice_volume), settings->volume);
    gtk_range_set_value (GTK_RANGE (spui_speech_voice.hs_voice_rate), 	settings->rate);
    gtk_range_set_value (GTK_RANGE (spui_speech_voice.hs_voice_pitch), 	settings->pitch);
    gtk_entry_set_text  (GTK_ENTRY (spui_speech_voice.et_gnopernicus_voice), settings->gnopernicus_speaker);
    gtk_entry_set_text  (GTK_ENTRY (GTK_COMBO (spui_speech_voice.cb_engine_voice)->entry), settings->gs_speaker);
    
    return TRUE;
}

static gboolean
spui_add_modify_set_properties_settings (SpeakerSettings *settings)
{
    if (!spui_speech_voice.w_speech_voice_add_modify) 
	return FALSE;
	
    spui_clean_old_speaker_setting ();
    
    old_speaker_settings.volume 		= settings->volume;
    old_speaker_settings.rate   		= settings->rate;
    old_speaker_settings.pitch  		= settings->pitch;
    old_speaker_settings.gs_speaker 		= g_strdup (settings->gs_speaker);
    old_speaker_settings.gnopernicus_speaker 	= g_strdup (settings->gnopernicus_speaker);

/*    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (spui_speech_voice.ck_apply_for_all), FALSE);*/
    
    return 
	spui_modify_speaker_values_in_add_modify (settings);
}

static void 
spui_hs_voice_property_value_changed (GtkWidget	*widget,
			    	      gpointer	data)
{
    gint value1, value2;
    SpeakerSettings *curr_setting = NULL;
    
    if (!curr_speaker_settings) 
	return;

    value1 = (gint) gtk_range_get_value (GTK_RANGE (widget));
    value2 = (gint) gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON ((GtkWidget*)data));

    if (value1 == value2 || 
	!changes_activable)
	return;
    
    curr_setting = (SpeakerSettings*)curr_speaker_settings->data;
    
    curr_setting->volume = 
		    CHECK_RANGE_OF_VALUE ((gint)gtk_range_get_value (GTK_RANGE (spui_speech_voice.hs_voice_volume)), MIN_SPEECH_VOLUME, MAX_SPEECH_VOLUME);
    curr_setting->rate = 
		    CHECK_RANGE_OF_VALUE ((gint)gtk_range_get_value (GTK_RANGE (spui_speech_voice.hs_voice_rate)), MIN_SPEECH_RATE, MAX_SPEECH_RATE);
    curr_setting->pitch = 
		    CHECK_RANGE_OF_VALUE ((gint)gtk_range_get_value (GTK_RANGE (spui_speech_voice.hs_voice_pitch)), MIN_SPEECH_PITCH, MAX_SPEECH_PITCH);
    
    if (!new_item)
        spconf_speaker_settings_save (curr_setting);
    else
	gtk_spin_button_set_value (GTK_SPIN_BUTTON ((GtkWidget*)data), (gdouble)value1);
    	
}

static void
spui_sp_voice_property_value_changed (GtkWidget	*widget,
				      gpointer	data)
{

    gint value1, value2;
    SpeakerSettings *curr_setting = NULL;
    
    if (!curr_speaker_settings) 
	return;

    value1 = (gint) gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widget));
    value2 = (gint) gtk_range_get_value (GTK_RANGE ((GtkWidget*)data));

    if (value1 == value2 || 
	!changes_activable)
	return;

    curr_setting = (SpeakerSettings*)curr_speaker_settings->data;

    curr_setting->volume = 
		    CHECK_RANGE_OF_VALUE ((gint) gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spui_speech_voice.sp_voice_volume)), MIN_SPEECH_VOLUME, MAX_SPEECH_VOLUME);
    curr_setting->rate   = 
		    CHECK_RANGE_OF_VALUE ((gint) gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spui_speech_voice.sp_voice_rate)), MIN_SPEECH_RATE, MAX_SPEECH_RATE);
    curr_setting->pitch  = 
		    CHECK_RANGE_OF_VALUE ((gint) gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spui_speech_voice.sp_voice_pitch)), MIN_SPEECH_PITCH, MAX_SPEECH_PITCH);
    
    if (!new_item)
	spconf_speaker_settings_save (curr_setting);
    else
	gtk_range_set_value (GTK_RANGE ((GtkWidget*)data), (gdouble) value1);	
}

static void
spui_combo_changed (GtkWidget *widget,
		    gpointer  user_data)
{
    gchar *cvalue;
    SpeakerSettings *curr_setting = NULL;
    
    if (!curr_speaker_settings ||
	!changes_activable) 
	return;
    
    curr_setting = 
	(SpeakerSettings*)curr_speaker_settings->data;
    (const gchar*)cvalue = 
	gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (spui_speech_voice.cb_engine_voice)->entry));
    
    if (strcmp (curr_setting->gs_speaker, cvalue) && 
	strlen (cvalue) > 0 &&
	strcmp (cvalue, NONE_ELEMENT) &&
	strcmp (curr_setting->gnopernicus_speaker, NONE_ELEMENT))
    {
	g_free (curr_setting->gs_speaker);
	curr_setting->gs_speaker = g_strdup (cvalue);
	if (!new_item)
	    spconf_speaker_settings_save (curr_setting);
    }
}

static void
spui_voice_add_modify_settings_response (GtkDialog *dialog,
					gint       response_id,
					gpointer   user_data)
{
    switch (response_id)
    {
        case GTK_RESPONSE_OK: 
        {
	    if (spui_add_modify_get_properties ())
 	    {
    		spconf_speaker_settings_save ((SpeakerSettings*)curr_speaker_settings->data);
    		if (new_item)
		    spconf_gnopernicus_speakers_save (gnopernicus_speakers);
	    }
	    
	    changes_activable = FALSE;    
	    spui_clean_old_speaker_setting ();
	}
        break;
	case GTK_RESPONSE_CANCEL:
	    spui_remove_created_item ();
	    changes_activable = FALSE;
	    spui_clean_old_speaker_setting ();
        break;
	case GTK_RESPONSE_HELP: 
	    gn_load_help ("gnopernicus-speech-settings");
	    return;
	break;
        default:
        break;
    }
    gtk_widget_hide ((GtkWidget*)dialog);
}


static gint
spui_delete_emit_response_cancel (GtkDialog *dialog,
				  GdkEventAny *event,
				  gpointer data)
{
    gtk_dialog_response (GTK_DIALOG (dialog),
			 GTK_RESPONSE_CANCEL);
    return TRUE; /* Do not destroy */
}


static void
spui_set_voice_add_modify_handlers  (GladeXML *xml)
{
    spui_speech_voice.w_speech_voice_add_modify = glade_xml_get_widget (xml, "w_voice_add_modify");
    
    spui_speech_voice.cb_engine_voice 	   = glade_xml_get_widget (xml, "cb_engine_voice");
    spui_speech_voice.et_gnopernicus_voice = glade_xml_get_widget (xml, "et_gnopernicus_voice");
    
    spui_speech_voice.hs_voice_volume	= glade_xml_get_widget (xml, "hs_voice_volume");
    spui_speech_voice.hs_voice_rate	= glade_xml_get_widget (xml, "hs_voice_rate");
    spui_speech_voice.hs_voice_pitch	= glade_xml_get_widget (xml, "hs_voice_pitch");
    spui_speech_voice.hs_voice_priority	= glade_xml_get_widget (xml, "hs_voice_priority");
    spui_speech_voice.sp_voice_priority	= glade_xml_get_widget (xml, "sp_voice_priority");
    spui_speech_voice.sp_voice_volume	= glade_xml_get_widget (xml, "sp_voice_volume");
    spui_speech_voice.sp_voice_rate	= glade_xml_get_widget (xml, "sp_voice_rate");
    spui_speech_voice.sp_voice_pitch	= glade_xml_get_widget (xml, "sp_voice_pitch");

/*    spui_speech_voice.ck_apply_for_all	= glade_xml_get_widget (xml, "ck_apply_for_all");*/
    
    g_object_set_data (G_OBJECT (spui_speech_voice.hs_voice_pitch),"param", (gint *)VOICE_PITCH);
    
    g_signal_connect (spui_speech_voice.w_speech_voice_add_modify, "response",
		      G_CALLBACK (spui_voice_add_modify_settings_response), NULL);
    g_signal_connect (spui_speech_voice.w_speech_voice_add_modify, "delete_event",
                      G_CALLBACK (spui_delete_emit_response_cancel), NULL);

    glade_xml_signal_connect (xml,"on_combo-entry3_changed",		
			    GTK_SIGNAL_FUNC (spui_combo_changed));

    glade_xml_signal_connect_data (xml, "on_hs_voice_volume_value_changed",			
			    GTK_SIGNAL_FUNC (spui_hs_voice_property_value_changed), 
			    (gpointer)spui_speech_voice.sp_voice_volume);
    glade_xml_signal_connect_data (xml, "on_hs_voice_rate_value_changed",			
			    GTK_SIGNAL_FUNC (spui_hs_voice_property_value_changed), 
			    (gpointer)spui_speech_voice.sp_voice_rate);
    glade_xml_signal_connect_data (xml, "on_hs_voice_pitch_value_changed",			
			    GTK_SIGNAL_FUNC (spui_hs_voice_property_value_changed),
			    (gpointer)spui_speech_voice.sp_voice_pitch);
    glade_xml_signal_connect_data (xml, "on_hs_voice_priority_value_changed",			
			    GTK_SIGNAL_FUNC (spui_hs_voice_property_value_changed),
			    (gpointer)spui_speech_voice.sp_voice_priority);

    glade_xml_signal_connect_data (xml, "on_sp_voice_volume_changed",			
			    GTK_SIGNAL_FUNC (spui_sp_voice_property_value_changed), 
			    (gpointer)spui_speech_voice.hs_voice_volume);
    glade_xml_signal_connect_data (xml, "on_sp_voice_rate_changed",			
			    GTK_SIGNAL_FUNC (spui_sp_voice_property_value_changed), 
			    (gpointer)spui_speech_voice.hs_voice_rate);
    glade_xml_signal_connect_data (xml, "on_sp_voice_pitch_changed",			
			    GTK_SIGNAL_FUNC (spui_sp_voice_property_value_changed),
			    (gpointer)spui_speech_voice.hs_voice_pitch);
    glade_xml_signal_connect_data (xml, "on_sp_voice_priority_changed",			
			    GTK_SIGNAL_FUNC (spui_sp_voice_property_value_changed),
			    (gpointer)spui_speech_voice.hs_voice_priority);

}

static gboolean 
spui_load_speech_voice_add_modify (GtkWidget *parent_window)
{        
    if (!spui_speech_voice.w_speech_voice_add_modify)
    {
	GladeXML *xml;
	xml = gn_load_interface ("Speech_Settings/speech_settings.glade2", "w_voice_add_modify");
	sru_return_val_if_fail (xml, FALSE);
	spui_set_voice_add_modify_handlers  (xml);
	spconf_load_voices ();    
	g_object_unref (G_OBJECT (xml));
	gtk_window_set_transient_for ( GTK_WINDOW (spui_speech_voice.w_speech_voice_add_modify),
				           GTK_WINDOW (parent_window));
				    
	gtk_window_set_destroy_with_parent ( GTK_WINDOW (spui_speech_voice.w_speech_voice_add_modify), 
				         TRUE);
    }
    else
	gtk_widget_show (spui_speech_voice.w_speech_voice_add_modify);
        
    return TRUE;
}

/******************************************************************************/
static gboolean
spui_speaker_parameter_add (GtkWidget *widget,
	    		    gpointer  user_data)
{
    GtkTreeModel     *model;
    GtkTreeSelection *selection;

    model 	  = gtk_tree_view_get_model 	(GTK_TREE_VIEW (spui_speech_voice.tv_voice_table));
    selection     = gtk_tree_view_get_selection (GTK_TREE_VIEW (spui_speech_voice.tv_voice_table));

    if (!GTK_IS_LIST_STORE (model))
	return FALSE;

    spui_load_speech_voice_add_modify (spui_speech_voice.w_speech_voice);
    spui_add_modify_select ("ADD");

    gtk_list_store_append (GTK_LIST_STORE (model), &curr_iter);
	
    new_item = TRUE;
    gnopernicus_speakers = 
	spconf_gnopernicus_speakers_add (gnopernicus_speakers, NONE_ELEMENT);
    speakers_settings = 
	spconf_speaker_settings_list_add  (speakers_settings, NONE_ELEMENT,
							     DEFAULT_SPEECH_ENGINE_VOICE,
							     DEFAULT_SPEECH_VOLUME, 
							     DEFAULT_SPEECH_PITCH,
							     DEFAULT_SPEECH_RATE);
    curr_speaker_settings    = g_slist_last (speakers_settings);
    curr_gnopernicus_speaker = g_slist_last (gnopernicus_speakers);
    
    if (curr_speaker_settings)
	spui_add_modify_set_properties_settings ((SpeakerSettings*)curr_speaker_settings->data);
	    
    changes_activable = TRUE;
    
    return TRUE;
}

static gboolean
spui_speaker_parameter_modify (GtkWidget *widget,
			       gpointer  user_data)
{
    GtkTreeModel     *model;
    GtkTreeSelection *selection;
    gchar 	     *gnopernicus_speaker;

    model 	  = gtk_tree_view_get_model 	( GTK_TREE_VIEW (spui_speech_voice.tv_voice_table));
    selection     = gtk_tree_view_get_selection ( GTK_TREE_VIEW (spui_speech_voice.tv_voice_table));

    if (!GTK_IS_LIST_STORE (model))
	return FALSE;
    if (!gtk_tree_selection_get_selected (selection, NULL, &curr_iter))
	return FALSE;

    spui_load_speech_voice_add_modify (spui_speech_voice.w_speech_voice);
    spui_add_modify_select ("MODIFY");
    new_item = FALSE;

    gtk_tree_model_get (model, 	&curr_iter,
			VOICE_GNOPERNICUS_SPEAKER,	&gnopernicus_speaker,
                    	-1);
				    
    curr_speaker_settings = 
	    spconf_speaker_settings_list_find (speakers_settings,  
					      gnopernicus_speaker);
    curr_gnopernicus_speaker = 
	    spconf_gnopernicus_speakers_find (gnopernicus_speakers, 
					      gnopernicus_speaker);
    if (curr_speaker_settings)
	spui_add_modify_set_properties_settings ((SpeakerSettings*)curr_speaker_settings->data);
    
    g_free (gnopernicus_speaker);
    
    changes_activable = TRUE;

    return TRUE;
}

static void
spui_row_activated_cb (GtkTreeView       *tree_view,
            	       GtkTreePath       *path,
		       GtkTreeViewColumn *column)
{
    spui_speaker_parameter_modify (NULL, NULL);
}


static void
spui_voice_remove (GtkWidget *widget,
		    gpointer user_data)
{
    GtkTreeModel     *model;
    GtkTreeSelection *selection;
    GtkTreeIter	     iter;
    gchar 	     *gnoper_speaker;
    
    model 	  = gtk_tree_view_get_model 	( GTK_TREE_VIEW (spui_speech_voice.tv_voice_table));
    selection     = gtk_tree_view_get_selection ( GTK_TREE_VIEW (spui_speech_voice.tv_voice_table));

    if (!GTK_IS_LIST_STORE (model))
	return;
    if (!gtk_tree_selection_get_selected (selection, NULL, &iter))
	return;
    
    gtk_tree_model_get (model, 	&iter,
			VOICE_GNOPERNICUS_SPEAKER, 
			&gnoper_speaker,
                    	-1);

    if (!gnoper_speaker)
	return;
	
    gnopernicus_speakers =
	spconf_gnopernicus_speakers_remove (gnopernicus_speakers,
					    gnoper_speaker);
	    
    speakers_settings =
	spconf_speaker_settings_list_remove (speakers_settings,
					  gnoper_speaker);
					  
    gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
    g_free (gnoper_speaker);
}

static void
spui_free_lists (void)
{
    gnopernicus_speakers_backup =
	    spconf_gnopernicus_speakers_free (gnopernicus_speakers_backup);
    speakers_settings_backup  =
	    spconf_speaker_settings_list_free (speakers_settings_backup);
    gnopernicus_speakers =
	    spconf_gnopernicus_speakers_free (gnopernicus_speakers);
    speakers_settings  =
	    spconf_speaker_settings_list_free (speakers_settings);
}

static void
spui_voice_settings_response (GtkDialog *dialog,
			    gint       response_id,
			    gpointer   user_data)
{
    switch (response_id)
    {
	case GTK_RESPONSE_OK: 
        {
	    gtk_widget_hide ((GtkWidget*)dialog);
	    spconf_gnopernicus_speakers_save (gnopernicus_speakers);
	    spconf_speaker_settings_list_save (speakers_settings);
	    spui_free_lists ();
	}
        break;
	case GTK_RESPONSE_CANCEL:
        {
	    gtk_widget_hide ((GtkWidget*)dialog);
	    spconf_gnopernicus_speakers_save (gnopernicus_speakers_backup);
	    spconf_speaker_settings_list_save (speakers_settings_backup);
	    spui_free_lists ();
	}
        break;
	case GTK_RESPONSE_APPLY:
        {
	    spconf_gnopernicus_speakers_save (gnopernicus_speakers);
	    spconf_speaker_settings_list_save (speakers_settings);
	    gnopernicus_speakers_backup =
		    spconf_gnopernicus_speakers_free (gnopernicus_speakers_backup);
	    speakers_settings_backup  =
		    spconf_speaker_settings_list_free (speakers_settings_backup);
	    gnopernicus_speakers_backup = 
		    spconf_gnopernicus_speakers_clone (gnopernicus_speakers);			   
	    speakers_settings_backup  =
		    spconf_speaker_settings_list_clone (speakers_settings);
	}
        break;
	case GTK_RESPONSE_HELP: 
	    gn_load_help ("gnopernicus-speech-settings");
	break;
	default:
	    gtk_widget_hide ((GtkWidget*)dialog);
        break;
    }
    
}

static void
spui_modify_setting_for_all_voices  (gint volume,
				     gint rate,
				     gint pitch,
				     gboolean use_default)
{
    SpeakerSettingsListType *tmp = NULL;
    for (tmp = speakers_settings; tmp ; tmp = tmp->next)
    {
	if (!use_default)
	{
	    if (volume)
		((SpeakerSettings*)tmp->data)->volume = 
		    CHECK_RANGE_OF_VALUE (((SpeakerSettings*)tmp->data)->volume + volume, MIN_SPEECH_VOLUME, MAX_SPEECH_VOLUME);
	    if (rate)
		((SpeakerSettings*)tmp->data)->rate = 
		    CHECK_RANGE_OF_VALUE (((SpeakerSettings*)tmp->data)->rate + rate, MIN_SPEECH_RATE, MAX_SPEECH_RATE);
	    if (pitch)
		((SpeakerSettings*)tmp->data)->pitch = 
		    CHECK_RANGE_OF_VALUE (((SpeakerSettings*)tmp->data)->pitch + pitch, MIN_SPEECH_PITCH, MAX_SPEECH_PITCH);
	}
	else
	{
	    if (volume)
		((SpeakerSettings*)tmp->data)->volume = CHECK_RANGE_OF_VALUE (volume, MIN_SPEECH_VOLUME, MAX_SPEECH_VOLUME);
	    if (rate)
		((SpeakerSettings*)tmp->data)->rate   = CHECK_RANGE_OF_VALUE (rate, MIN_SPEECH_RATE, MAX_SPEECH_RATE);
	    if (pitch)
		((SpeakerSettings*)tmp->data)->pitch  = CHECK_RANGE_OF_VALUE (pitch, MIN_SPEECH_PITCH, MAX_SPEECH_PITCH);
	}
    }
    spconf_speaker_settings_list_save (speakers_settings);
}

static void
spui_volume_mode_clicked (GtkWidget *widget,
			 gpointer  user_data)
{
    const gchar *command = (const gchar*)user_data;
    gint     volume = 0;
    gboolean use_default = FALSE;
    if (!strcmp (command, "INCREMENT"))
    {
	volume = SPEECH_VOLUME_STEP;
    }
    else
    if (!strcmp (command, "DECREMENT"))
    {
	volume = -SPEECH_VOLUME_STEP;
    }
    else
    {
	volume = DEFAULT_SPEECH_VOLUME;
	use_default = TRUE;
    }
    spui_modify_setting_for_all_voices (volume, 0, 0, use_default);
}

static void
spui_rate_mode_clicked (GtkWidget *widget,
			 gpointer  user_data)
{
    const gchar *command = (const gchar*)user_data;
    gint     rate = 0;
    gboolean use_default = FALSE;
    if (!strcmp (command, "INCREMENT"))
    {
	rate = SPEECH_RATE_STEP;
    }
    else
    if (!strcmp (command, "DECREMENT"))
    {
	rate = -SPEECH_RATE_STEP;
    }
    else
    {
	rate = DEFAULT_SPEECH_RATE;
	use_default = TRUE;
    }
    spui_modify_setting_for_all_voices (0, rate, 0, use_default);
}

static void
spui_pitch_mode_clicked (GtkWidget *widget,
			 gpointer  user_data)
{
    const gchar *command = (const gchar*)user_data;
    gint     pitch = 0;
    gboolean use_default = FALSE;

    if (!strcmp (command, "INCREMENT"))
    {
	pitch = SPEECH_PITCH_STEP;
    }
    else
    if (!strcmp (command, "DECREMENT"))
    {
	pitch = -SPEECH_PITCH_STEP;
    }
    else
    {
    	pitch = DEFAULT_SPEECH_PITCH;
	use_default = TRUE;
    }
    spui_modify_setting_for_all_voices (0, 0, pitch, use_default);
}

static void
spui_voice_test_clicked (GtkWidget *widget,
			 gpointer  user_data)
{
    GtkTreeModel     *model;
    GtkTreeSelection *selection;
    GtkTreeIter	     iter;
    gchar 	     *gnop_speak;
    
    model 	  = gtk_tree_view_get_model 	( GTK_TREE_VIEW (spui_speech_voice.tv_voice_table));
    selection     = gtk_tree_view_get_selection ( GTK_TREE_VIEW (spui_speech_voice.tv_voice_table));

    if (!GTK_IS_LIST_STORE (model))
	return;
    if (!gtk_tree_selection_get_selected (selection, NULL, &iter))
	return;
    
    gtk_tree_model_get (model, 	&iter,
    			VOICE_GNOPERNICUS_SPEAKER, &gnop_speak,
                    	-1);
    if (!gnop_speak)
	return;
	    
    spconf_play_voice (gnop_speak);
    g_free (gnop_speak);    
}



static void
spui_modify_speaker_values_in_table (SpeakerSettings *value)
{
    gboolean valid;
    GtkTreeModel     *model;
    GtkTreeIter	     iter;
    
    model = gtk_tree_view_get_model (GTK_TREE_VIEW (spui_speech_voice.tv_voice_table));

    valid = gtk_tree_model_get_iter_first (model, &iter );
    
    if (!GTK_IS_LIST_STORE (model))
	return;
    
    while (valid)
    {
	gchar *ikey = NULL;
	gtk_tree_model_get (model, 			&iter,
			    VOICE_GNOPERNICUS_SPEAKER,  &ikey,
                    	    -1);
	if (!strcmp (ikey, value->gnopernicus_speaker))
	{
	    spui_set_speaker_parameter_in_list_store_at_iter (value, 
					GTK_LIST_STORE (model), iter);
	    g_free (ikey);
	    break;
	}
	g_free (ikey);
	
	valid = gtk_tree_model_iter_next (model, &iter);
    }    
}

void
spui_modify_speaker_values (SpeakerSettings *value)
{
    spui_modify_speaker_values_in_table 	(value);
    spui_modify_speaker_values_in_add_modify 	(value);
}


static void
spui_add_elem_in_table (GtkListStore *store)
{
    SpeakerSettingsListType *elem = NULL;
    GtkTreeIter iter;
    
    for (elem = speakers_settings; elem ; elem = elem->next)
    {
	SpeakerSettings *item = (SpeakerSettings*)elem->data;
	
	if (!item)
	    continue;
	    
	gtk_list_store_append ( GTK_LIST_STORE (store), &iter);
	
	spui_set_speaker_parameter_in_list_store_at_iter (item, store, iter);
    }    
}

static GtkTreeModel*
spui_create_model (void)
{
    GtkListStore *store;      
    store = gtk_list_store_new (VOICE_NO_COLUMN, 
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_INT,
				G_TYPE_INT,
				G_TYPE_INT);
    spui_add_elem_in_table (store);
    return GTK_TREE_MODEL (store) ;
}


static void
spui_set_voice_handlers  (GladeXML *xml)
{
    GtkTreeModel 	*model;
    GtkCellRenderer 	*cell_renderer;
    GtkTreeSelection 	*selection;
    GtkTreeViewColumn 	*column;

    spui_speech_voice.w_speech_voice = glade_xml_get_widget (xml, "w_sp_voice");
    spui_speech_voice.tv_voice_table = glade_xml_get_widget (xml, "tv_voice_table");
    
    g_signal_connect (spui_speech_voice.w_speech_voice, "response",
		      G_CALLBACK (spui_voice_settings_response), NULL);
    g_signal_connect (spui_speech_voice.w_speech_voice, "delete_event",
                      G_CALLBACK (spui_delete_emit_response_cancel), NULL);

    glade_xml_signal_connect_data (xml,"on_bt_voice_add_clicked",	
			    GTK_SIGNAL_FUNC (spui_speaker_parameter_add),
			    (gpointer)"ADD");

    glade_xml_signal_connect_data (xml,"on_bt_voice_modify_clicked",	
			    GTK_SIGNAL_FUNC (spui_speaker_parameter_modify),
			    (gpointer)"MODIFY");

    glade_xml_signal_connect (xml,"on_bt_voice_remove_clicked",	
			     GTK_SIGNAL_FUNC (spui_voice_remove));

    glade_xml_signal_connect_data (xml, "on_bt_inc_volume_clicked",			
			     GTK_SIGNAL_FUNC (spui_volume_mode_clicked),
			     (gpointer)"INCREMENT");

    glade_xml_signal_connect_data (xml, "on_bt_dec_volume_clicked",			
			     GTK_SIGNAL_FUNC (spui_volume_mode_clicked),
			     (gpointer)"DECREMENT");
    
    glade_xml_signal_connect_data (xml, "on_bt_def_volume_clicked",			
			     GTK_SIGNAL_FUNC (spui_volume_mode_clicked),
			     (gpointer)"DEFAULT");

    glade_xml_signal_connect_data (xml, "on_bt_inc_rate_clicked",			
			     GTK_SIGNAL_FUNC (spui_rate_mode_clicked),
			     (gpointer)"INCREMENT");

    glade_xml_signal_connect_data (xml, "on_bt_dec_rate_clicked",			
			     GTK_SIGNAL_FUNC (spui_rate_mode_clicked),
			     (gpointer)"DECREMENT");
    
    glade_xml_signal_connect_data (xml, "on_bt_def_rate_clicked",			
			     GTK_SIGNAL_FUNC (spui_rate_mode_clicked),
			     (gpointer)"DEFAULT");

    glade_xml_signal_connect_data (xml, "on_bt_inc_pitch_clicked",			
			     GTK_SIGNAL_FUNC (spui_pitch_mode_clicked),
			     (gpointer)"INCREMENT");

    glade_xml_signal_connect_data (xml, "on_bt_dec_pitch_clicked",			
			     GTK_SIGNAL_FUNC (spui_pitch_mode_clicked),
			     (gpointer)"DECREMENT");
    
    glade_xml_signal_connect_data (xml, "on_bt_def_pitch_clicked",			
			     GTK_SIGNAL_FUNC (spui_pitch_mode_clicked),
			     (gpointer)"DEFAULT");

    glade_xml_signal_connect (xml, "on_bt_add_mod_voice_test_clicked",			
			     GTK_SIGNAL_FUNC (spui_voice_test_clicked));

    model = spui_create_model ();
                
    gtk_tree_view_set_model (GTK_TREE_VIEW (spui_speech_voice.tv_voice_table), model);

    
    gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (model), 
					  VOICE_GNOPERNICUS_SPEAKER, 
					  GTK_SORT_ASCENDING);

    g_signal_connect (spui_speech_voice.tv_voice_table, "row_activated", 
		      G_CALLBACK (spui_row_activated_cb), model);					        

    cell_renderer = gtk_cell_renderer_text_new ();
    
    column = gtk_tree_view_column_new_with_attributes   (_("Gnopernicus"),
    							cell_renderer,
							"text", VOICE_GNOPERNICUS_SPEAKER,
							NULL);	
    gtk_tree_view_column_set_sort_column_id (column, VOICE_GNOPERNICUS_SPEAKER);
    gtk_tree_view_column_set_clickable (column, TRUE);
    gtk_tree_view_column_set_resizable (column, TRUE);
    gtk_tree_view_append_column (GTK_TREE_VIEW (spui_speech_voice.tv_voice_table), column);
    
    column = gtk_tree_view_column_new_with_attributes   (_("Speakers"),
    							cell_renderer,
							"text", VOICE_GNOME_SPEAK_SPEAKER,
							NULL);	
    gtk_tree_view_column_set_clickable (column, TRUE);
    gtk_tree_view_column_set_resizable (column, TRUE);
    gtk_tree_view_append_column (GTK_TREE_VIEW (spui_speech_voice.tv_voice_table), column);

    column = gtk_tree_view_column_new_with_attributes   (_("Volume"),
    							cell_renderer,
							"text", VOICE_VOLUME,
							NULL);	
    gtk_tree_view_column_set_clickable (column, TRUE);
    gtk_tree_view_column_set_resizable (column, TRUE);
    gtk_tree_view_append_column (GTK_TREE_VIEW (spui_speech_voice.tv_voice_table), column);

    column = gtk_tree_view_column_new_with_attributes   (_("Rate"),
    							cell_renderer,
							"text", VOICE_RATE,
							NULL);	
    gtk_tree_view_column_set_clickable (column, TRUE);
    gtk_tree_view_column_set_resizable (column, TRUE);
    gtk_tree_view_append_column (GTK_TREE_VIEW (spui_speech_voice.tv_voice_table), column);

    column = gtk_tree_view_column_new_with_attributes   (_("Pitch"),
    							cell_renderer,
							"text", VOICE_PITCH,
							NULL);	
    gtk_tree_view_column_set_clickable (column, TRUE);
    gtk_tree_view_column_set_resizable (column, TRUE);
    gtk_tree_view_append_column (GTK_TREE_VIEW (spui_speech_voice.tv_voice_table), column);
    
    g_object_unref (G_OBJECT (model));
    
    selection = gtk_tree_view_get_selection ( GTK_TREE_VIEW (spui_speech_voice.tv_voice_table));
    gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);    

}

static gboolean 
spui_load_speech_voice_settings (GtkWidget *parent_window)
{    
    gnopernicus_speakers = 
	    spconf_gnopernicus_speakers_load (gnopernicus_speakers);			   
    speakers_settings  =
	    spconf_speaker_settings_list_get ();
    gnopernicus_speakers_backup = 
	    spconf_gnopernicus_speakers_clone (gnopernicus_speakers);			   
    speakers_settings_backup  =
	    spconf_speaker_settings_list_clone (speakers_settings);
    
    if (!spui_speech_voice.w_speech_voice)
    {
	GladeXML *xml;
	xml = gn_load_interface ("Speech_Settings/speech_settings.glade2", "w_sp_voice");
	sru_return_val_if_fail (xml, FALSE);
	spui_set_voice_handlers  (xml);
	g_object_unref (G_OBJECT (xml));
	gtk_window_set_transient_for ( GTK_WINDOW (spui_speech_voice.w_speech_voice),
				           GTK_WINDOW (parent_window));
	gtk_window_set_destroy_with_parent ( GTK_WINDOW (spui_speech_voice.w_speech_voice), 
					         TRUE);
    }
    else
    {
	GtkTreeModel *model;
	model = gtk_tree_view_get_model ( GTK_TREE_VIEW (spui_speech_voice.tv_voice_table) );
	gtk_list_store_clear (GTK_LIST_STORE (model));
	spui_add_elem_in_table (GTK_LIST_STORE (model));
	gtk_widget_show (spui_speech_voice.w_speech_voice);
    }
                
    return TRUE;
}


/*******************************DICTIONARY***********************************/
static void
spui_dictionary_add_modify_set_value (void)
{
    GtkTreeModel     *model;
    gchar *word    = (gchar*) gtk_entry_get_text (GTK_ENTRY (spui_speech_dictionary.et_word));
    gchar *replace = (gchar*) gtk_entry_get_text (GTK_ENTRY (spui_speech_dictionary.et_replace));

    model = gtk_tree_view_get_model ( GTK_TREE_VIEW (spui_speech_dictionary.tv_dictionary));
    
    if (!word    || strlen (word)    == 0 ||
        !replace || strlen (replace) == 0)
	return;

    if (!GTK_IS_LIST_STORE (model))
	return;
	

    if (new_item)
    {
    	dictionary_list =	
    	    spconf_dictionary_add_word (dictionary_list,
					word, replace);
    }
    else
    {
	dictionary_list =	
	    spconf_dictionary_modify_word (dictionary_list,
			    		    word, replace);
    }			
    
    gtk_list_store_set (GTK_LIST_STORE (model), 	&curr_iter, 
    			SPEECH_DICTIONARY_WORD,	 	(gchar*)word,
			SPEECH_DICTIONARY_REPLACER,  	(gchar*)replace,
		    	-1);
}

static void
spui_dictionary_add_modify_restore_state (void)
{
    GtkTreeModel     *model;
    
    model = gtk_tree_view_get_model (GTK_TREE_VIEW (spui_speech_dictionary.tv_dictionary));
    
    if (!GTK_IS_LIST_STORE (model))
	return;

    if (new_item)
	gtk_list_store_remove (GTK_LIST_STORE (model), &curr_iter);
}

static void
spui_dictionary_add_modify_response (GtkDialog *dialog,
				    gint       response_id,
				    gpointer   user_data)
{
    switch (response_id)
    { 
	case GTK_RESPONSE_OK:
	    spui_dictionary_add_modify_set_value ();
	break;
	case GTK_RESPONSE_CANCEL:
	    spui_dictionary_add_modify_restore_state ();
	break;
	case GTK_RESPONSE_HELP:
    	    gn_load_help ("gnopernicus-speech-settings");
	    return;
	break;
	default:
	break;
    }
    gtk_widget_hide ((GtkWidget*)dialog);
}

static void
spui_dictionary_value_add_to_widgets (void)
{
    GtkTreeModel     *model;
    gchar *word = NULL;
    gchar *replacer = NULL;
    model = gtk_tree_view_get_model ( GTK_TREE_VIEW (spui_speech_dictionary.tv_dictionary));
    
    if (!GTK_IS_LIST_STORE (model))
	return;
    
    gtk_tree_model_get (model, 	&curr_iter,
			SPEECH_DICTIONARY_WORD, &word,
			SPEECH_DICTIONARY_REPLACER, &replacer,
                    	-1);

    gtk_entry_set_text (GTK_ENTRY (spui_speech_dictionary.et_word), STR(word));
    gtk_entry_set_text (GTK_ENTRY (spui_speech_dictionary.et_replace), STR(replacer));

    g_free (word);
    g_free (replacer);
}


static void
spui_set_dictionary_add_modify_handlers (GladeXML *xml)
{
    spui_speech_dictionary.w_dict_add_modify = glade_xml_get_widget (xml, "w_dict_add_modify");
    spui_speech_dictionary.et_word = glade_xml_get_widget (xml, "et_word"); 
    spui_speech_dictionary.et_replace = glade_xml_get_widget (xml, "et_replace"); 
        
    g_signal_connect (spui_speech_dictionary.w_dict_add_modify, "response",
		      G_CALLBACK (spui_dictionary_add_modify_response), NULL);
    g_signal_connect (spui_speech_dictionary.w_dict_add_modify, "delete_event",
                      G_CALLBACK (spui_delete_emit_response_cancel), NULL);

}

static gboolean 
spui_load_dictionary_add_modify (GtkWidget *parent_window)
{
    if (!spui_speech_dictionary.w_dict_add_modify)
    {
	GladeXML *xml;
	xml = gn_load_interface ("Speech_Settings/speech_settings.glade2", "w_dict_add_modify");
	sru_return_val_if_fail (xml, FALSE);
	spui_set_dictionary_add_modify_handlers (xml);
	g_object_unref (G_OBJECT (xml));
	gtk_window_set_transient_for ( GTK_WINDOW (spui_speech_dictionary.w_dict_add_modify),
				           GTK_WINDOW (parent_window));
				    
	gtk_window_set_destroy_with_parent ( GTK_WINDOW (spui_speech_dictionary.w_dict_add_modify), 
					         TRUE);
    }
    else
	gtk_widget_show (spui_speech_dictionary.w_dict_add_modify);
    
    spui_dictionary_value_add_to_widgets ();
    
    return TRUE;
}


static void
spui_dictionary_response (GtkDialog *dialog,
			  gint       response_id,
			  gpointer   user_data)
{
    switch (response_id)
    {
	case GTK_RESPONSE_OK:
    	    spconf_dictionary_save (dictionary_list);
	break;
	case GTK_RESPONSE_CANCEL:
	    dictionary_list =	
		spconf_dictionary_free (dictionary_list);
	break;
	case GTK_RESPONSE_HELP:
	    gn_load_help ("gnopernicus-speech-settings");
	    return;
	break;
	default:
	break;
    }
    
    gtk_widget_hide ((GtkWidget*)dialog);    
}

static void
spui_dict_add (GtkWidget *widget,
		gpointer data)
{
    GtkTreeModel     *model;
    GtkTreeSelection *selection;    
    
    selection = gtk_tree_view_get_selection ( GTK_TREE_VIEW (spui_speech_dictionary.tv_dictionary));
    model     = gtk_tree_view_get_model ( GTK_TREE_VIEW (spui_speech_dictionary.tv_dictionary));

    if (!GTK_IS_LIST_STORE (model))
	return;
        
    new_item  = TRUE;
    
    gtk_list_store_append (GTK_LIST_STORE (model), &curr_iter);
    gtk_tree_selection_select_iter (selection, &curr_iter);
    
    spui_load_dictionary_add_modify (spui_speech_dictionary.w_dictionary);
}

static void
spui_dict_modify (GtkWidget *widget,
		gpointer data)
{
    GtkTreeIter	     iter;
    GtkTreeSelection *selection;    
    selection = gtk_tree_view_get_selection ( GTK_TREE_VIEW (spui_speech_dictionary.tv_dictionary));
    
    if (!gtk_tree_selection_get_selected (selection, NULL, &iter))
	return;
	
    curr_iter = iter;
    new_item  = FALSE;
    
    spui_load_dictionary_add_modify (spui_speech_dictionary.w_dictionary);
}

static void
spui_dict_remove (GtkWidget *widget,
		gpointer data)
{
    GtkTreeModel     *model = NULL;
    GtkTreeIter	     iter;
    GtkTreeSelection *selection = NULL;    
    gchar 	     *word = NULL;
    
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (spui_speech_dictionary.tv_dictionary));
    model     = gtk_tree_view_get_model (GTK_TREE_VIEW (spui_speech_dictionary.tv_dictionary));
    
    if (!GTK_IS_LIST_STORE (model))
	return;
    
    if (!gtk_tree_selection_get_selected (selection, NULL, &iter))
	return;
	
    gtk_tree_model_get (model, &iter,
			SPEECH_DICTIONARY_WORD, &word,
                    	-1);
    dictionary_list =	
	    spconf_dictionary_remove_word (dictionary_list, word);

    gtk_list_store_remove (GTK_LIST_STORE (model), &iter);

    g_free (word);
}

static void
spui_dictionary_row_activated_cb (GtkTreeView       *tree_view,
            	    		  GtkTreePath       *path,
		    		  GtkTreeViewColumn *column)
{
    GtkTreeIter	     iter;
    GtkTreeSelection *selection;    
    selection = gtk_tree_view_get_selection ( GTK_TREE_VIEW (tree_view));
    
    if (!gtk_tree_selection_get_selected (selection, NULL, &iter))
	return;
	
    curr_iter = iter;
    new_item  = FALSE;
    
    spui_load_dictionary_add_modify (spui_speech_dictionary.w_dictionary);
}

static void
spui_fill_table (GtkListStore *store)
{
    DictionaryListType* elem = NULL;
    GtkTreeIter  iter;
    
    for (elem = dictionary_list; elem ; elem = elem->next)
    {
	gchar *word = NULL;
	gchar *replace = NULL;
	spconf_dictionary_split_entry ((gchar*)elem->data, &word, &replace);

	gtk_list_store_append ( GTK_LIST_STORE (store), &iter);
	gtk_list_store_set ( GTK_LIST_STORE (store), &iter, 
			    SPEECH_DICTIONARY_WORD,	(gchar*)word,
			    SPEECH_DICTIONARY_REPLACER,  (gchar*)replace,
		    	    -1);
	g_free (word);
	g_free (replace);
    }
}

static GtkTreeModel*
spui_create_dictionary_model (void)
{
    GtkListStore *store;
    store = gtk_list_store_new (SPEECH_DICTIONARY_NO_COLUMN, 
				G_TYPE_STRING,
				G_TYPE_STRING);

    spui_fill_table (store);
    return GTK_TREE_MODEL (store) ;
}


static void
spui_set_dictionary_handlers (GladeXML *xml)
{
    GtkTreeModel 	*model;
    GtkCellRenderer 	*cell_renderer;
    GtkTreeSelection 	*selection;
    GtkTreeViewColumn 	*column;

    spui_speech_dictionary.w_dictionary = glade_xml_get_widget (xml, "w_dictionary");
    spui_speech_dictionary.tv_dictionary = glade_xml_get_widget (xml, "tv_dictionary");
    
    g_signal_connect (spui_speech_dictionary.w_dictionary, "response",
		      G_CALLBACK (spui_dictionary_response), NULL);
    g_signal_connect (spui_speech_dictionary.w_dictionary, "delete_event",
                      G_CALLBACK (spui_delete_emit_response_cancel), NULL);

    glade_xml_signal_connect (xml,"on_bt_dict_add_clicked",			
			     GTK_SIGNAL_FUNC (spui_dict_add));

    glade_xml_signal_connect (xml,"on_bt_dict_modify_clicked",			
			     GTK_SIGNAL_FUNC (spui_dict_modify));

    glade_xml_signal_connect (xml,"on_bt_dict_remove_clicked",			
			     GTK_SIGNAL_FUNC (spui_dict_remove));

    glade_xml_signal_connect (xml,"on_et_word_selected_activate",			
			     GTK_SIGNAL_FUNC (spui_dict_modify));


    model = spui_create_dictionary_model ();
                
    gtk_tree_view_set_model (GTK_TREE_VIEW (spui_speech_dictionary.tv_dictionary), model);

    
    gtk_tree_sortable_set_sort_column_id ( GTK_TREE_SORTABLE (model), 
					    SPEECH_DICTIONARY_WORD, 
					    GTK_SORT_ASCENDING);

    g_signal_connect (spui_speech_dictionary.tv_dictionary, "row_activated", 
		      G_CALLBACK (spui_dictionary_row_activated_cb), model);					        

    cell_renderer = gtk_cell_renderer_text_new ();
    
    column = gtk_tree_view_column_new_with_attributes   (_("Word"),
    							cell_renderer,
							"text", SPEECH_DICTIONARY_WORD,
							NULL);	
    gtk_tree_view_column_set_sort_column_id (column, SPEECH_DICTIONARY_WORD);
    gtk_tree_view_column_set_clickable (column, TRUE);
    gtk_tree_view_column_set_resizable (column, TRUE);
    gtk_tree_view_append_column (GTK_TREE_VIEW (spui_speech_dictionary.tv_dictionary), column);
    
    column = gtk_tree_view_column_new_with_attributes   (_("Replacer"),
    							cell_renderer,
							"text", SPEECH_DICTIONARY_REPLACER,
							NULL);	
    gtk_tree_view_column_set_clickable (column, TRUE);
    gtk_tree_view_column_set_resizable (column, TRUE);
    gtk_tree_view_append_column (GTK_TREE_VIEW (spui_speech_dictionary.tv_dictionary), column);

    g_object_unref (G_OBJECT (model));
    
    selection = gtk_tree_view_get_selection ( GTK_TREE_VIEW (spui_speech_dictionary.tv_dictionary));
    gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);    

}

static gboolean 
spui_load_dictionary (GtkWidget *parent_window)
{
    if (!dictionary_list)
    {
	dictionary_list =
	    spconf_dictionary_load ();
    }
    
    if (!spui_speech_dictionary.w_dictionary)
    {
	GladeXML *xml;
	xml = gn_load_interface ("Speech_Settings/speech_settings.glade2", "w_dictionary");
	sru_return_val_if_fail (xml, FALSE);
	spui_set_dictionary_handlers (xml);
	g_object_unref (G_OBJECT (xml));
	gtk_window_set_transient_for ( GTK_WINDOW (spui_speech_dictionary.w_dictionary),
				           GTK_WINDOW (parent_window));
	gtk_window_set_destroy_with_parent ( GTK_WINDOW (spui_speech_dictionary.w_dictionary), 
					         TRUE);
    }
    else
    {
	GtkTreeModel *model;
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (spui_speech_dictionary.tv_dictionary));
	gtk_list_store_clear (GTK_LIST_STORE (model));
	spui_fill_table (GTK_LIST_STORE (model));
	gtk_widget_show (spui_speech_dictionary.w_dictionary);
    }
    
    return TRUE;
}

/*******************************SPEECH SETTINGS******************************/
static void
spui_apply_sensitivity (gboolean state)
{
    speech_changed = state;
}

static void 
spui_speech_setting_changing (void)
{
    gint iter;
    
    if (!speech_setting) 
	return;
	
    for (iter = PUNCTUATION_IGNORE ; iter < PUNCTUATION_NUMBER ; iter++)
    {
	if (gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON(spui_speech_settings.rb_punctuation[iter])) && 
	    strcmp (speech_setting->punctuation_type, punct_keys[iter]))
	{
	    g_free(speech_setting->punctuation_type);
	    speech_setting->punctuation_type = g_strdup (punct_keys[iter]);
	    spconf_punctuation_set (speech_setting->punctuation_type);
	    break;
	}
    }

    for (iter = TEXT_ECHO_CHARACTER ; iter < TEXT_ECHO_NUMBER ; iter++)
    {
	if (gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON(spui_speech_settings.rb_text_echo[iter])) && 
	    strcmp (speech_setting->text_echo_type, text_echo_keys[iter]))
	{
	    g_free(speech_setting->text_echo_type);
	    speech_setting->text_echo_type = g_strdup (text_echo_keys[iter]);
	    spconf_text_echo_set (speech_setting->text_echo_type);
	    break;
	}
    }

    g_free (speech_setting->modifiers_type);
    g_free (speech_setting->cursors_type);
    g_free (speech_setting->spaces_type);
    g_free (speech_setting->count_type);
    g_free (speech_setting->dictionary);
    
    speech_setting->modifiers_type = 
	g_strdup (common_keys [gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON (spui_speech_settings.ck_modifiers))]);
    spconf_modifiers_set (speech_setting->modifiers_type);		  

    speech_setting->cursors_type = 
	g_strdup (common_keys [gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON (spui_speech_settings.ck_cursors))]);
    spconf_cursors_set (speech_setting->cursors_type);		  

    speech_setting->spaces_type = 
	g_strdup (common_keys [gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON (spui_speech_settings.ck_spaces))]);
    spconf_spaces_set (speech_setting->spaces_type);		  

    speech_setting->dictionary = 
	g_strdup (dictionary_keys [gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON (spui_speech_settings.ck_dictionary))]);
    spconf_dictionary_set (speech_setting->dictionary);		  

    speech_setting->count_type = 
	g_strdup (common_keys [gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON (spui_speech_settings.ck_count))]);
    spconf_count_set (speech_setting->count_type);		  
}


static void
spui_voices_settings (GtkWidget *widget,
		      gpointer	user_data)
{
    spui_load_speech_voice_settings ((GtkWidget*)spui_speech_settings.w_speech_settings);    
}

static void
spui_dictionary (GtkWidget *widget,
		gpointer   user_data)
{
    spui_load_dictionary ((GtkWidget*)spui_speech_settings.w_speech_settings);    
}

static void
spui_state_changed (GtkWidget	*widget,
		    gpointer	data)
{
    spui_apply_sensitivity (TRUE);
}

static void
spui_dictionary_toggled (GtkWidget *widget,
			gpointer    data)
{
    spui_apply_sensitivity (TRUE);
    gtk_widget_set_sensitive (spui_speech_settings.bt_dictionary,
    				gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON (widget)));
}

static void
spui_settings_response (GtkDialog *dialog,
			gint       response_id,
			gpointer   user_data)
{
    switch (response_id)
    {
        case GTK_RESPONSE_OK: 
	    if (speech_changed)
    		spui_speech_setting_changing ();
	    gtk_widget_hide ((GtkWidget*)dialog);
        break;
        case GTK_RESPONSE_APPLY:
    	    spui_speech_setting_changing ();
	    spui_apply_sensitivity (FALSE);
        break;
        case GTK_RESPONSE_HELP:
	    gn_load_help ("gnopernicus-speech-settings");
	break;
	default:
	    gtk_widget_hide ((GtkWidget*)dialog);
        break;
    }
}

/**
 *
 * Set event handlers and get a widgets used in this interface.
 * xml - glade interface XML pointer
 *
**/
static void 
spui_set_handlers (GladeXML *xml)
{    
    spui_speech_settings.w_speech_settings 	= glade_xml_get_widget (xml, "w_sp_settings");

    spui_speech_settings.rb_punctuation [PUNCTUATION_IGNORE]	= glade_xml_get_widget (xml, "rb_punct_ignore"); 
    spui_speech_settings.rb_punctuation [PUNCTUATION_SAME]	= glade_xml_get_widget (xml, "rb_punct_same"); 
    spui_speech_settings.rb_punctuation [PUNCTUATION_MOST]	= glade_xml_get_widget (xml, "rb_punct_most"); 
    spui_speech_settings.rb_punctuation [PUNCTUATION_ALL]	= glade_xml_get_widget (xml, "rb_punct_all"); 

    spui_speech_settings.rb_text_echo [TEXT_ECHO_CHARACTER]	= glade_xml_get_widget (xml, "rb_techo_character"); 
    spui_speech_settings.rb_text_echo [TEXT_ECHO_WORD]	= glade_xml_get_widget (xml, "rb_techo_word"); 

    spui_speech_settings.ck_modifiers 	= glade_xml_get_widget (xml, "ck_modifiers"); 
    spui_speech_settings.ck_cursors 		= glade_xml_get_widget (xml, "ck_cursors"); 
    spui_speech_settings.ck_spaces 		= glade_xml_get_widget (xml, "ck_spaces"); 
    spui_speech_settings.ck_count 		= glade_xml_get_widget (xml, "ck_count"); 
    spui_speech_settings.ck_dictionary	= glade_xml_get_widget (xml, "ck_dictionary"); 
    spui_speech_settings.bt_dictionary	= glade_xml_get_widget (xml, "bt_dictionary"); 

    g_signal_connect (spui_speech_settings.w_speech_settings, "response",
		      G_CALLBACK (spui_settings_response), NULL);
    g_signal_connect (spui_speech_settings.w_speech_settings, "delete_event",
                      G_CALLBACK (spui_delete_emit_response_cancel), NULL);

    
    glade_xml_signal_connect_data (xml,"on_ck_count_toggled",			
			    GTK_SIGNAL_FUNC (spui_state_changed), 
			    (gpointer)FALSE);

    glade_xml_signal_connect_data (xml,"on_bt_voices_clicked",			
			    GTK_SIGNAL_FUNC (spui_voices_settings),
			    (gpointer)spui_speech_settings.w_speech_settings);

    glade_xml_signal_connect_data (xml,"on_bt_dictionary_clicked",			
			    GTK_SIGNAL_FUNC (spui_dictionary),
			    (gpointer)spui_speech_settings.w_speech_settings);
			    
    glade_xml_signal_connect (xml,"on_rb_punct_ignore_toggled",			
			    GTK_SIGNAL_FUNC (spui_state_changed));
    glade_xml_signal_connect (xml,"on_rb_punct_same_toggled",			
			    GTK_SIGNAL_FUNC (spui_state_changed));
    glade_xml_signal_connect (xml,"on_rb_punct_most_toggled",			
			    GTK_SIGNAL_FUNC (spui_state_changed));
    glade_xml_signal_connect (xml,"on_rb_punct_all_toggled",			
			    GTK_SIGNAL_FUNC (spui_state_changed));

    glade_xml_signal_connect (xml,"on_rb_techo_character_toggled",			
			    GTK_SIGNAL_FUNC (spui_state_changed));
    glade_xml_signal_connect (xml,"on_rb_techo_word_toggled",			
			    GTK_SIGNAL_FUNC (spui_state_changed));
    glade_xml_signal_connect (xml,"on_ck_modifiers_toggled",			
			    GTK_SIGNAL_FUNC (spui_state_changed));
    glade_xml_signal_connect (xml,"on_ck_dictionary_toggled",			
			    GTK_SIGNAL_FUNC (spui_dictionary_toggled));
    glade_xml_signal_connect (xml,"on_ck_cursors_toggled",			
			    GTK_SIGNAL_FUNC (spui_state_changed));
    glade_xml_signal_connect (xml,"on_ck_spaces_toggled",			
			    GTK_SIGNAL_FUNC (spui_state_changed));

}

/**
 *
 * Speech option user interface loader function
 *
**/
gboolean 
spui_load_speech_settings (GtkWidget *parent_window)
{
    if (!spui_speech_settings.w_speech_settings)
    {
	GladeXML *xml;
	xml = gn_load_interface ("Speech_Settings/speech_settings.glade2", "w_sp_settings");
	sru_return_val_if_fail (xml, FALSE);
	spui_set_handlers  (xml);
	g_object_unref (G_OBJECT (xml));
	gtk_window_set_transient_for ( GTK_WINDOW (spui_speech_settings.w_speech_settings),
				           GTK_WINDOW (parent_window));
				    
	gtk_window_set_destroy_with_parent ( GTK_WINDOW (spui_speech_settings.w_speech_settings), 
					         TRUE);
    }
    else
	gtk_widget_show (spui_speech_settings.w_speech_settings);
    
    spui_speech_setting_value_add_to_widgets (speech_setting);    
    
    spui_apply_sensitivity (FALSE);
                
    return TRUE;
}

/**
 *
 * Set the widgets with a current value.
 *
**/
gboolean
spui_speech_setting_value_add_to_widgets (Speech *speech_setting)
{    
    gint iter;
    if (!spui_speech_settings.w_speech_settings ||
	!speech_setting) 
	return FALSE;
            
    for (iter = PUNCTUATION_IGNORE ; iter < PUNCTUATION_NUMBER ; iter++)
    {
	if (!strcmp (punct_keys[iter], speech_setting->punctuation_type))
	{
	    gtk_toggle_button_set_active (  GTK_TOGGLE_BUTTON 
					    (spui_speech_settings.rb_punctuation [iter]), 
					    TRUE);
	    break;
	}
    }

    for (iter = TEXT_ECHO_CHARACTER ; iter < TEXT_ECHO_NUMBER ; iter++)
    {
	if (!strcmp (text_echo_keys[iter], speech_setting->text_echo_type))
	{
	    gtk_toggle_button_set_active (  GTK_TOGGLE_BUTTON 
					    (spui_speech_settings.rb_text_echo [iter]), 
					    TRUE);
	    break;
	}
    }

    gtk_toggle_button_set_active (  GTK_TOGGLE_BUTTON (spui_speech_settings.ck_count), 
				    !strcmp ("ALL", speech_setting->count_type)
				  );
    gtk_toggle_button_set_active (  GTK_TOGGLE_BUTTON (spui_speech_settings.ck_modifiers), 
				    !strcmp ("ALL", speech_setting->modifiers_type)
				  );
    gtk_toggle_button_set_active (  GTK_TOGGLE_BUTTON (spui_speech_settings.ck_cursors), 
				    !strcmp ("ALL", speech_setting->cursors_type)
				  );
    gtk_toggle_button_set_active (  GTK_TOGGLE_BUTTON (spui_speech_settings.ck_spaces), 
				    !strcmp ("ALL", speech_setting->spaces_type)
				  );

    gtk_toggle_button_set_active (  GTK_TOGGLE_BUTTON (spui_speech_settings.ck_dictionary), 
				    !strcmp ("YES", speech_setting->dictionary)
				  );

    gtk_widget_set_sensitive (spui_speech_settings.bt_dictionary,
    			    gtk_toggle_button_get_active ( 
			    GTK_TOGGLE_BUTTON (spui_speech_settings.ck_dictionary)));

    return TRUE;
}
