/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/**
 * gpdf bonobo persistor
 *
 * Authors:
 *   Michael Meeks (michael@ximian.com)
 *   Martin Kretzschmar <Martin.Kretzschmar@inf.tu-dresden.de>
 *
 * Copyright 1999, 2000 Ximian, Inc.
 * Copyright 2002 Martin Kretzschmar
 */

#include "gpdf-persist-stream.h"
#include "gpdf-util.h"
#include "gpdf-marshal.h"
#include "gpdf-g-switch.h"
#  include <libbonobo.h>
#include "gpdf-g-switch.h"
#include "Object.h"
#include "PDFDoc.h"
#include "BonoboStream.h"

#define noPDF_DEBUG
#ifdef PDF_DEBUG
#  define DBG(x) x
#else
#  define DBG(x)
#endif

BEGIN_EXTERN_C

struct _GPdfPersistStreamPrivate {
	PDFDoc        *pdf_doc;

	bonoboStream  *pdf_stream;
	Bonobo_Stream  bonobo_stream;
};

#define PARENT_TYPE BONOBO_TYPE_PERSIST_STREAM
static BonoboPersistStreamClass *parent_class;

enum {
	SET_PDF_SIGNAL,
	LAST_SIGNAL
};

static guint gpdf_persist_stream_signals [LAST_SIGNAL];

PDFDoc *
gpdf_persist_stream_get_pdf_doc (GPdfPersistStream *persist_stream)
{
	g_return_val_if_fail (persist_stream != NULL, NULL);
	g_return_val_if_fail (GPDF_IS_PERSIST_STREAM (persist_stream), NULL);

	return persist_stream->priv->pdf_doc;
}

glong
gpdf_persist_stream_get_length (GPdfPersistStream *persist_stream)
{
	g_return_val_if_fail (persist_stream != NULL, -1);
	g_return_val_if_fail (GPDF_IS_PERSIST_STREAM (persist_stream), -1);

	return bonobo_stream_client_get_length (
		persist_stream->priv->bonobo_stream, NULL);
}

static void
delete_pdf_doc_full (GPdfPersistStream *persist_stream)
{
	GPdfPersistStreamPrivate *priv;

	g_return_if_fail (persist_stream != NULL);
	g_return_if_fail (GPDF_IS_PERSIST_STREAM (persist_stream));

	priv = persist_stream->priv;

	if (priv->pdf_doc) {
		delete priv->pdf_doc;
		priv->pdf_doc = NULL;
		/* PDFDoc::~PDFDoc deletes its stream */
		priv->pdf_stream = NULL;
	}

	if (priv->bonobo_stream) {
		CORBA_Environment ev;
		CORBA_exception_init (&ev);

		Bonobo_Unknown_unref (priv->bonobo_stream, &ev);
		CORBA_Object_release (priv->bonobo_stream, &ev);
		priv->bonobo_stream = NULL;

		CORBA_exception_free (&ev);
	}
}
	

/*
 * Loads a PDF from a Bonobo_Stream
 */
static void
load_pdf_from_stream (BonoboPersistStream *ps, Bonobo_Stream stream,
		      Bonobo_Persist_ContentType type,
		      void *closure, CORBA_Environment *ev)
{
	GPdfPersistStream *gpdf_ps = GPDF_PERSIST_STREAM (ps);
	GPdfPersistStreamPrivate *priv;
	Object obj;

	bonobo_return_if_fail (ps != NULL, ev);
	bonobo_return_if_fail (GPDF_IS_PERSIST_STREAM (ps), ev);

	priv = gpdf_ps->priv;

	if (priv->pdf_doc || priv->pdf_stream || priv->bonobo_stream)
		delete_pdf_doc_full (gpdf_ps);

	/* FIXME: is this against Bonobo::PersistStream's contract ? */

	/* We need to keep the alive stream for later */

	CORBA_Object_duplicate (stream, ev);
	Bonobo_Unknown_ref (stream, ev);
	priv->bonobo_stream = stream;
	BONOBO_RET_EX (ev);

	DBG (g_message ("Loading PDF from PersistStream"));
	
	obj.initNull ();
	priv->pdf_stream = new bonoboStream (stream, 0, gFalse, 0, &obj);
	priv->pdf_doc = new PDFDoc (priv->pdf_stream);

	DBG (g_message ("Done load"));

	if (!priv->pdf_doc->isOk ()) {
		g_warning ("Duff pdf data");
		goto exit_bad_file;
	}
	if (priv->pdf_doc->getCatalog () == NULL) {
		g_warning ("Duff pdf catalog");
		goto exit_bad_file;
	}

	goto exit_clean;

exit_bad_file:
	delete_pdf_doc_full (gpdf_ps);

exit_clean:
	g_signal_emit (G_OBJECT (gpdf_ps),
		       gpdf_persist_stream_signals [SET_PDF_SIGNAL],
		       0);
}


static void
gpdf_persist_stream_destroy (BonoboObject *object)
{
	GPdfPersistStream *persist_stream = GPDF_PERSIST_STREAM (object);

	g_return_if_fail (object != NULL);
	g_return_if_fail (GPDF_IS_PERSIST_STREAM (object));
	
	DBG (g_message ("Destroying GPdfPersistStream"));

	delete_pdf_doc_full (persist_stream);

	BONOBO_CALL_PARENT (BONOBO_OBJECT_CLASS, destroy, (object));
}

static void
gpdf_persist_stream_finalize (GObject *object)
{
	g_return_if_fail (object != NULL);
	g_return_if_fail (GPDF_IS_PERSIST_STREAM (object));

	g_free ((GPDF_PERSIST_STREAM (object))->priv);

	BONOBO_CALL_PARENT (G_OBJECT_CLASS, finalize, (object));
}

static void
gpdf_persist_stream_class_init (GPdfPersistStreamClass *klass)
{
 	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	BonoboObjectClass *bonobo_object_class = BONOBO_OBJECT_CLASS (klass);

 	parent_class = 
	     BONOBO_PERSIST_STREAM_CLASS (gtk_type_class (PARENT_TYPE));

	gpdf_persist_stream_signals [SET_PDF_SIGNAL] =
		g_signal_new ("set_pdf",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GPdfPersistStreamClass,
					       set_pdf),
			      NULL, NULL,
			      gpdf_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);

	bonobo_object_class->destroy = gpdf_persist_stream_destroy;
 	object_class->finalize = gpdf_persist_stream_finalize;
}

static void
gpdf_persist_stream_init (GPdfPersistStream *persist_stream)
{
 	persist_stream->priv = g_new0 (GPdfPersistStreamPrivate, 1);
}

GPdfPersistStream *
gpdf_persist_stream_construct (GPdfPersistStream *persist_stream, 
			       const gchar *iid)
{
	return GPDF_PERSIST_STREAM (
	     bonobo_persist_stream_construct (
		  BONOBO_PERSIST_STREAM (persist_stream),
		  load_pdf_from_stream, NULL, NULL, iid,
		  NULL));
}

GPdfPersistStream *
gpdf_persist_stream_new (const gchar *iid)
{
	GPdfPersistStream *persist_stream;

	DBG (g_message ("Creating GPdfPersistStream"));

	persist_stream = 
	     GPDF_PERSIST_STREAM (g_object_new (GPDF_TYPE_PERSIST_STREAM, NULL));

	return gpdf_persist_stream_construct (persist_stream, iid);
}

BONOBO_TYPE_FUNC (GPdfPersistStream, PARENT_TYPE, gpdf_persist_stream)

END_EXTERN_C
