/* SRLow.c
 *
 * Copyright 2001, 2002 Sun Microsystems, Inc.,
 * Copyright 2001, 2002 BAUM Retec, A.G.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "SRLow.h"
#include <stdio.h>
#include "glib.h"
#include "cspi/spi.h"
#include <string.h>
#include <time.h>

typedef struct _SRLEvent
{
    gchar	*type;
    Accessible	*acc;
    long	detail1;
    long	detail2;
    AccessibleEvent *acc_ev;
}SRLEvent;

typedef struct _SRLLastInfo
{
    SRLong	text_char_count;
    SRLong	text_caret_offset;
    SRLong	text_selections_count;
    SRLong	text_crt_selection_length;
    gdouble	value_crt_value;
}SRLLastInfo;

/* maximum number of clients alowed */
#define	SRL_MAX_CLIENTS		1

/* array of all clients who need to be notified */
static SRLClient 	srl_clients[SRL_MAX_CLIENTS];

/* variable who retains if library was initialized */
static gboolean		srl_initialized		= FALSE;

/* last edit text object focused */
Accessible 		*srl_last_edit = NULL;

/* declaration for needded AccessibleEventListener */
static AccessibleEventListener 	*event_listener	= NULL;

/* functions used for accessible callbacks */
static void report_event (const AccessibleEvent *event, void *user_data);

static Accessible *srl_last_focus = NULL;
static Accessible *srl_last_create = NULL;
static Accessible *srl_last_restore = NULL;
static Accessible *srl_last_delete_acc = NULL;
static gchar	  *srl_last_deleted_text = NULL;

static SREvent    *srl_last_event = NULL;
static unsigned long srl_last_flags = 0;

#ifdef SRL_DEBUG
GList *srl_obj_list = NULL;
GList *srl_str_list = NULL;
#endif

#define SRL_LOG_AT_SPI		1
#define SRL_LOG_GNOPERNICUS	2
#define SRL_LOG_IMPORTANT	4
#define SRL_LOG_TERMINAL	8
#define SRL_LOG_REENTRANCY	16

static gint srl_log_flags = 0;

static gchar*
srl_get_toolkit (Accessible *acc)
{
    Accessible *parent;
    gchar *rv = NULL;
    
    srl_return_val_if_fail (acc, NULL);
    
    parent = acc;
    Accessible_ref (parent);
    
    while (parent && !Accessible_isApplication (parent))
    {
	acc = parent;
	parent = Accessible_getParent (parent);
	Accessible_unref (acc);
    }
    if (parent)
    {
	AccessibleApplication *app;
	app = Accessible_getApplication (parent);
	if (app)
	{
	    rv = AccessibleApplication_getToolkitName (app);
	    AccessibleApplication_unref (app);
	}
	Accessible_unref (parent);
    }
    return rv;
}


static void
srl_log_at_spi (const AccessibleEvent *event)
{
    gchar *name, *role, *toolkit;
    if (!(srl_log_flags & SRL_LOG_AT_SPI))
	return;
    if (!(srl_log_flags & SRL_LOG_TERMINAL) && 
	    Accessible_getRole (event->source) == SPI_ROLE_TERMINAL)
	return;

    name = Accessible_getName (event->source),
    role = Accessible_getRoleName (event->source);
    toolkit = srl_get_toolkit (event->source);
    fprintf (stderr, "\nAT:%xp----\"%s\" "
			"for %xp \"%s\" role \"%s\" "
			"from \"%s\" with details %ld and %ld", 
		(unsigned int)event, event->type,
		(unsigned int)event->source, name ? name : "", role ? role : "",
		toolkit ? toolkit : "", event->detail1, event->detail2);
    SPI_freeString (name);
    SPI_freeString (role);
    SPI_freeString (toolkit);
}

static void
srl_log_gnopernicus (SREvent *event)
{
    SREventType ev_type;
    
    if (!(srl_log_flags & SRL_LOG_GNOPERNICUS))
	return;

    sre_add_reference (event);
    sre_get_type (event, &ev_type);
    if (ev_type == SR_EVENT_MOUSE)
    {
	SRPoint *point;

	sre_get_event_data (event, (void**)&point);
	fprintf (stderr, "\nGN:%xp--\"mouse:move\" "
			"at %d, %d", (unsigned int)event, point->x, point->y);
    }
    else
    {
	SRObject *obj;
	gchar *name, *role, *type;
    
	sre_get_event_data (event, (void**)&obj);
	sro_get_name (obj, &name, SR_INDEX_CONTAINER);
	sro_get_role_name (obj, &role, SR_INDEX_CONTAINER);
	sro_get_reason (obj, &type);
	fprintf (stderr, "\nGN:%xp--\"%s\" "
			"for %xp \"%s\" role \"%s\" ", 
		    (unsigned int)event, type ? type : "",
		    (unsigned int)obj, name ? name : "", role ? role : "");
	SR_freeString (name);
	SR_freeString (role);
	SR_freeString (type);
    }
    sre_release_reference (event);
}

static void
srl_log_gnopernicus_real (SREvent *event)
{
    if (!(srl_log_flags & SRL_LOG_GNOPERNICUS))
	return;
}

static void
srl_log_important (SRLEvent *event, SREvent *ev)
{
    if (!(srl_log_flags & SRL_LOG_AT_SPI))
	return;
}

static void
srl_log_reentrancy (const AccessibleEvent *event)
{
    if (!(srl_log_flags & SRL_LOG_REENTRANCY))
	return;
    if (!(srl_log_flags & SRL_LOG_TERMINAL) && 
	    Accessible_getRole (event->source) == SPI_ROLE_TERMINAL)
	return;

    fprintf (stderr, "\nQU:%xp will be really added to gnopernicus queue",
    		(unsigned int)event);
}

/**
 * srl_init:
 *
 *   Make all the needed initialization and register necessary event listeners.
 *
 *   Returns: #TRUE if all initialization succed, #FALSE otherwise. 
**/
gboolean 
srl_init ()
{
    gboolean rv = TRUE;
    const gchar *log;
    int  i;
    
    srl_assert (srl_initialized == 0);

    for (i = 0; i < SRL_MAX_CLIENTS; i++)
    {
	srl_clients[i].event_proc = NULL;
    }
    
    event_listener = SPI_createAccessibleEventListener (report_event, NULL);

    SPI_registerGlobalEventListener (event_listener, "focus:");

    SPI_registerGlobalEventListener (event_listener,
					"object:link-selected");
    SPI_registerGlobalEventListener (event_listener,
					"object:text-caret-moved");
    SPI_registerGlobalEventListener (event_listener,
					"object:text-changed:insert");
    SPI_registerGlobalEventListener (event_listener,
    					"object:text-changed:delete");
    SPI_registerGlobalEventListener (event_listener,
					"object:text-selection-changed");					

    SPI_registerGlobalEventListener (event_listener,
					"object:property-change:accessible-value");

    SPI_registerGlobalEventListener (event_listener,
					"object:state-changed:checked");
    SPI_registerGlobalEventListener (event_listener, 
					"object:state-changed:selected");
    SPI_registerGlobalEventListener (event_listener,
					"object:state-changed:expanded");

    SPI_registerGlobalEventListener (event_listener,
					"object:state-changed:visible");

    SPI_registerGlobalEventListener (event_listener,
					"object:selection-changed");

    SPI_registerGlobalEventListener (event_listener,
					"window:create");
    SPI_registerGlobalEventListener (event_listener,
					"window:destroy");
    SPI_registerGlobalEventListener (event_listener,
					"window:minimize");
    SPI_registerGlobalEventListener (event_listener,
					"window:maximize");
    SPI_registerGlobalEventListener (event_listener,
					"window:restore");
    SPI_registerGlobalEventListener (event_listener,
					"window:activate");
    SPI_registerGlobalEventListener (event_listener,
					"window:deactivate");

    SPI_registerGlobalEventListener (event_listener,
					"mouse:rel");
    SPI_registerGlobalEventListener (event_listener,
					"mouse:abs");

    srl_last_delete_acc = NULL;
    srl_last_deleted_text = NULL;
    srl_last_edit = NULL;
    
    srl_last_focus = NULL;
    srl_last_create = NULL;
    srl_last_restore = NULL;
    
    srl_last_event = NULL;
    srl_last_flags = 0;
    
    srl_initialized = 1;
#ifdef SRL_DEBUG
    srl_obj_list = NULL;
    srl_str_list = NULL;
#endif
    log = g_getenv ("GNOPERNICUS_LOG");
    if (!log)
	log = "";
    srl_log_flags = 0;
    if (strstr (log, "at-spi"))
	srl_log_flags |= SRL_LOG_AT_SPI;
    if (strstr (log, "gnopernicus"))
	srl_log_flags |= SRL_LOG_GNOPERNICUS;
    if (strstr (log, "important"))
	srl_log_flags |= SRL_LOG_IMPORTANT;
    if (strstr (log, "terminal"))
	srl_log_flags |= SRL_LOG_TERMINAL;
    if (strstr (log, "reentrancy"))
	srl_log_flags |= SRL_LOG_REENTRANCY;
    return rv;
}



/**
 * srl_terminate:
 *
 *   Deregister all used event listeners and make other necessary clean stuff.
 *
 *   Returns: #TRUE if succeed, #FALSE if an error occurs. 
**/
gboolean 
srl_terminate ()
{
    gboolean rv = TRUE;
    
    srl_assert (srl_initialized == 1);

    SPI_deregisterGlobalEventListenerAll (event_listener);
    AccessibleEventListener_unref (event_listener);

    if (srl_last_delete_acc)
	Accessible_unref (srl_last_delete_acc);
    if (srl_last_deleted_text)
	/*FIXME:next line must be valid */
	/*SPI_freeString (srl_last_deleted_text);*/
	g_free (srl_last_deleted_text);
    if (srl_last_edit)
	Accessible_unref (srl_last_edit);

    if (srl_last_focus)
	Accessible_unref (srl_last_focus);
    if (srl_last_create)
	Accessible_unref (srl_last_create);
    if (srl_last_restore)
	Accessible_unref (srl_last_restore);
    if (srl_last_event)
	sre_release_reference (srl_last_event);

    srl_initialized = 0;
#ifdef SRL_DEBUG
    if (srl_obj_list != NULL)
    {
	GList *crt;
	fprintf (stderr, "\n Not all alocated objects are freed");
	for (crt = srl_obj_list; crt; crt = crt->next)
	{
	    gchar *name, *role;
	    sro_get_name ((SRObject*) crt->data, &name, SR_INDEX_CONTAINER);
	    sro_get_role_name ((SRObject*) crt->data, &role, SR_INDEX_CONTAINER);
	    fprintf (stderr, "\n Object %s role %s", name ? name : "", role ? role : "");
	    SR_freeString (name);
	    SR_freeString (role);	    
	}	
	fprintf (stderr, "\n");
    }
    if (srl_str_list != NULL)
    {
	fprintf (stderr, "\n Not all alocated strings are freed corectely\n");
/*
	{
	    GList *crt;
	    for (crt = srl_str_list; crt; crt = crt->next)
	        fprintf (stderr, "\n%xp---%s", (unsigned int)crt->data, (gchar*)crt->data);
	}
*/
    }
#endif

    return rv;
}


/* function check if a client is empty */
static inline gboolean 
is_client_empty	(int index)
{
    return (srl_clients[index].event_proc ? FALSE : TRUE);
}

/**
 * srl_add_client:
 *   @client: a #SRLClient client.
 *
 *   Add a new client who need to be notified when an event occurs.
 *
 *   Returns: a valid #SRLClientHandle if succed 
 *     or #SRL_CLIENT_HANDLE_INVALID if faild.
**/
SRLClientHandle 
srl_add_client (const SRLClient *client)
{
    int i;	
    SRLClientHandle client_handle = SRL_CLIENT_HANDLE_INVALID;
	
    srl_assert (srl_initialized == 1);	
    
    if (!client)
	return SRL_CLIENT_HANDLE_INVALID;

    for (i = 0; i < SRL_MAX_CLIENTS; i++)
    {
	if (is_client_empty (i))
	{
	    srl_clients[i] = *client;
	    client_handle = i;
	    break;
	};	
    }	
    return client_handle;
}


/**
 * srl_remove_client:
 *   @client : a #SRLClientHandle who doesn't need anymore 
 *     to be notified by SRLow.previously obtained with #srl_add_client.
 *
 *   Remove a client from list of clients who need notifications.
 *
 *   Returns: #TRUE if succed, #FALSE otherwise.
**/
gboolean 
srl_remove_client (SRLClientHandle client)
{	
    int i = client;
    
    srl_assert (srl_initialized == 1);	
    
    if (0 <= i && SRL_MAX_CLIENTS > i )
    {
	srl_clients[i].event_proc = NULL;
	return TRUE;     
    }    
	
    return FALSE;
}


static gboolean
notify_clients_ (gpointer data)
{
    static gboolean busy = FALSE;

    if (busy)
	return FALSE;
    busy = TRUE;

    while (srl_last_event)
    {
	SREvent *event;
	unsigned long flags;
	gint i;

	event = srl_last_event;
	flags = srl_last_flags;
	srl_last_event = NULL;

	srl_log_gnopernicus_real (event);

	for (i = 0; i < SRL_MAX_CLIENTS; i++)
	{
	    if (srl_clients[i].event_proc)
	    {    
		srl_clients[i].event_proc (event, flags);
	    }	 
	}
	sre_release_reference (event);
    }
    busy = FALSE;

    return FALSE;
}



#define SRL_TIMER	1
#define SRL_IDLE	2
#define SRL_FORCE	4
/* notify all clients with an event */
static void 
notify_clients (SREvent *event, 
		const unsigned long flags)
{
    static gint mode_ = 0;

    srl_return_if_fail (event);
    
    if (!mode_)
    {
	G_CONST_RETURN gchar *mode;
	mode = g_getenv ("GNOPERNICUS_TIMER");
        if (!mode || !mode[0])
	    mode = "idle";
	if (strstr (mode, "timer"))
	    mode_ |= SRL_TIMER;
	else
	    mode_ |= SRL_IDLE;
	if (strstr (mode, "force"))
	    mode_ |= SRL_FORCE;
	fprintf (stderr, "\ngnopernicus will use a \"%s\" callback and will %sforce sometimes it call",
			(mode_ & SRL_TIMER) ? "timer" : "idle", (mode_ & SRL_FORCE) ? "" : "NOT ");
    }
    if (srl_last_event)
	sre_release_reference (srl_last_event);
    else
    {
	if (mode_ & SRL_TIMER)
	    g_timeout_add (100, notify_clients_, NULL);
	else
	    g_idle_add (notify_clients_, NULL);
    }
    srl_log_gnopernicus (event);
    srl_last_event = event;
    sre_add_reference (srl_last_event);
    srl_last_flags = flags;

    if (mode_ & SRL_FORCE)
    {
	static GTimer *timer = NULL;

	if (!timer)
    	    timer = g_timer_new ();
	if (g_timer_elapsed (timer, NULL) > 0.5)
	{
	    notify_clients_ (NULL);
	    g_timer_reset (timer);
	}
    }
}


static void 
srl_set_text_info (SRLLastInfo *info,
		    Accessible *acc)
{
    AccessibleText *text;
    gint i;
    srl_return_if_fail (acc && info);
    
    text = Accessible_getText (acc);
    if (!text)
	return;
    
    info->text_caret_offset  	= AccessibleText_getCaretOffset (text);
    info->text_char_count    	= AccessibleText_getCharacterCount (text);
    info->text_selections_count = AccessibleText_getNSelections (text);
    for (i = 0; i < info->text_selections_count; i++)
    {
	long int start, end;
	AccessibleText_getSelection (text, i, &start, &end);
	if (start == info->text_caret_offset || end == info->text_caret_offset)
	{
	    info->text_crt_selection_length = end - start;
	    break;
	}
    }
    AccessibleText_unref (text);
}

static void 
srl_set_value_info (SRLLastInfo *info,
		    Accessible *acc)
{
    AccessibleValue *value;
    
    srl_return_if_fail (acc && info);
            
    value = Accessible_getValue (acc);
    if (!value)
	return;
    
    info->value_crt_value = AccessibleValue_getCurrentValue (value);
    AccessibleValue_unref (value);    
}

static gboolean
srl_set_info (SRLLastInfo *info,
		Accessible *acc)
{
    srl_return_val_if_fail (info && acc, FALSE);
    
    if (Accessible_isValue (acc))
    	srl_set_value_info (info, acc);
    if (Accessible_isText (acc))
    	srl_set_text_info (info, acc);
        	
    return TRUE;
}

extern gboolean sro_get_from_accessible_event (Accessible *acc, gchar *event, SRObject **obj);
extern gboolean sro_set_difference (SRObject *obj, gchar *difference);

static gboolean
srl_report_accessible_sro (Accessible *acc,
			   gchar *event,
			   gpointer data)
{
    SRObject *obj;
    
    srl_assert (acc && event);

    if (sro_get_from_accessible_event (acc, event, &obj))
    {
	SREvent *ev;

	if (data)
	    sro_set_difference (obj, (gchar *)data);

	ev = sre_new ();
	if (ev)
	{
	    ev->type = SR_EVENT_SRO;
	    ev->data = obj;
	    sro_add_reference (obj);
	    ev->data_destructor = (SREventDataDestructor) sro_release_reference;
	    srl_log_important (NULL, ev);
	    notify_clients (ev, 0);
	    sre_release_reference (ev);
	}
	sro_release_reference (obj);
    }
    return TRUE;
}

static gboolean
srl_report_delete_text (gpointer data)
{
    if (srl_last_delete_acc)
    {
	Accessible *acc;
	gchar *text;
	acc = srl_last_delete_acc;
	srl_last_delete_acc = NULL;
	text = srl_last_deleted_text;
	srl_last_deleted_text = NULL;
	srl_report_accessible_sro (acc, "object:text-changed:delete", text);
	Accessible_unref (acc);
	g_free (text);
    }
    return FALSE;    
}


static gboolean
srl_process_event_real_window (SRLEvent *event)
{
    SRObject *obj;
    
    srl_assert (event && event->acc);
    
    if (sro_get_from_accessible_event (event->acc, event->type, &obj))
    {
	SREvent *ev;
	
	ev = sre_new ();
	if (ev)
	{
	    ev->type = SR_EVENT_WINDOW;
	    ev->data = obj;
	    sro_add_reference (obj);
	    ev->data_destructor = (SREventDataDestructor) sro_release_reference;
	    srl_log_important (event, ev);
	    notify_clients (ev, 0);
	    sre_release_reference (ev);
	}
	sro_release_reference (obj);
    }
    return TRUE;
}


static gboolean
srl_process_event_real_mouse (SRLEvent *event)
{
    SREvent *ev;

    srl_assert (event);
    
    ev = sre_new ();
    if (ev)
    {
	SRPoint *point;
	point = (SRPoint *) g_malloc (sizeof (SRPoint));
	if (point)
	{
	    ev->type = SR_EVENT_MOUSE;
	    ev->data = point;
	    point->x = event->detail1;
	    point->y = event->detail2;
	    ev->data_destructor = (SREventDataDestructor) g_free;
	    srl_log_important (event, ev);
	    notify_clients (ev, strcmp (event->type, "mouse:rel") ? 0 : 1);
	}
	sre_release_reference (ev);
    }
    return TRUE;
}

static gboolean
srl_process_event_real_tooltip (SRLEvent *event)
{
    SRObject *obj;
    
    srl_assert (event && event->acc);
    
    if (sro_get_from_accessible_event (event->acc, event->type, &obj))
    {
	SREvent *ev;

	ev = sre_new ();
	if (ev)
	{
	    ev->type = SR_EVENT_TOOLTIP;
	    ev->data = obj;
	    sro_add_reference (obj);
	    ev->data_destructor = (SREventDataDestructor) sro_release_reference;
	    srl_log_important (event, ev);
	    notify_clients (ev, 0);
	    sre_release_reference (ev);
	}
	sro_release_reference (obj);
    }
    return TRUE;

}


static gboolean
srl_process_event_real_sro (SRLEvent *ev)
{
    srl_assert (ev);

    srl_log_important (ev, NULL);
    if (srl_last_delete_acc && 
	Accessible_getRole (srl_last_delete_acc) == SPI_ROLE_COMBO_BOX &&
	strcmp (ev->type, "object:property-changed:accessible-content") == 0)
    {
	Accessible *acc;
	acc = srl_last_delete_acc;
	srl_last_delete_acc = NULL;
	Accessible_unref (acc);
    }

    if (srl_last_delete_acc)
    {
	srl_report_delete_text (NULL);
    }

    if (strcmp (ev->type, "object:text-changed:delete") == 0)
    {
	srl_last_delete_acc = ev->acc;
	Accessible_ref (srl_last_delete_acc);
	/*FIXME: next line must be valid */
	/*srl_last_deleted_text = AccessibleTextChangedEvent_getChangeString (ev->acc_ev);*/
	srl_last_deleted_text = g_strdup (AccessibleTextChangedEvent_getChangeString (ev->acc_ev));
	g_timeout_add (100, srl_report_delete_text, NULL);
    }
    else
    {
	gchar *tmp;
	tmp = NULL;
        if (strcmp (ev->type, "object:text-changed:insert") == 0)
	{
	    tmp = AccessibleTextChangedEvent_getChangeString (ev->acc_ev);
	}
	srl_report_accessible_sro (ev->acc, ev->type, tmp);
	/*FIXME : next line must be valid */
	/*SPI_freeString (tmp);*/
    }
    
    return TRUE;
}



static gboolean
srl_check_toolkit (Accessible *acc, 
		   gchar *name)
{
    Accessible *parent;
    gboolean rv = FALSE;
    
    srl_return_val_if_fail (acc && name, FALSE);
    
    parent = acc;
    Accessible_ref (parent);
    
    while (parent && !Accessible_isApplication (parent))
    {
	acc = parent;
	parent = Accessible_getParent (parent);
	Accessible_unref (acc);
    }
    if (parent)
    {
	AccessibleApplication *app;
	app = Accessible_getApplication (parent);
	if (app)
	{
	    gchar *tmp;
	    tmp = AccessibleApplication_getToolkitName (app);
	    if (tmp && strcmp (tmp, name) == 0)
		rv = TRUE;
	    SPI_freeString (tmp);
	    AccessibleApplication_unref (app);
	}
	Accessible_unref (parent);
    }
    return rv;
}


static gboolean
srl_process_event (SRLEvent *ev)
{
    gboolean process = FALSE;
    static SRLLastInfo info;
    static long last_mozilla_index = -1;
    static gboolean was_mozilla_caret = FALSE;
    
    srl_assert (ev);

    if (strcmp (ev->type, "object:text-caret-moved") == 0 &&
		    srl_check_toolkit (ev->acc, "mozilla"))
    {
#if 0
	SRLLastInfo info2;
	srl_set_info (&info2, ev->acc);
	if (last_mozilla_index == info2.text_caret_offset)
	    return FALSE;
	last_mozilla_index = info2.text_caret_offset;
#endif
	/*FIXME: next lines should be removed*/
	if (!was_mozilla_caret)
	{
	    was_mozilla_caret = TRUE;
	    return FALSE;
	}
    }

    if (strcmp (ev->type, "object:link-selected") == 0)
    {
	if (srl_check_toolkit (ev->acc, "mozilla"))
	{
	    gboolean ret = TRUE;
    	    AccessibleText *text;
	    SRLLastInfo info2;
	    text = Accessible_getText (ev->acc);
	    if (text)
	    {
		AccessibleHypertext *hyper;
		hyper = Accessible_getHypertext (ev->acc);
		if (hyper)
		{
		    /*long int offset = AccessibleText_getCaretOffset (text);*/
		    AccessibleHyperlink *link;
		    link = AccessibleHypertext_getLink (hyper, 
						0);
/*FIXME:must be valid			    AccessibleHypertext_getLinkIndex (hyper, offset));*/
		    if (link)
		    {
			g_free (ev->type);
			ev->type = g_strdup ("focus:");
			Accessible_unref (ev->acc);
			ev->acc = AccessibleHyperlink_getObject (link, 0); 
			ret = FALSE;
			AccessibleHyperlink_unref (link);
		    }	    
		    AccessibleHypertext_unref (hyper);
		}
		AccessibleText_unref (text);
	    }
	    was_mozilla_caret = FALSE;
	    if (ret)
		return TRUE;
	    srl_set_info (&info2, ev->acc);
	    last_mozilla_index = info2.text_caret_offset;
	}
	else
	    return TRUE;
    }

    if (strncmp (ev->type, "window:", 7) == 0)
    {
	Accessible *child;
	child = Accessible_getChildAtIndex (ev->acc, 0);
	if (child)
	{
	    gboolean ret = FALSE;
	    if (Accessible_getRole (child) == SPI_ROLE_MENU)
		ret = TRUE;
	    Accessible_unref (child);
	    if (ret)
		return TRUE;
	}
    }

    if (strcmp (ev->type, "object:selection-changed") == 0)
    {
	gboolean process2 = FALSE;
	if (Accessible_getRole (ev->acc) == SPI_ROLE_COMBO_BOX)
	{
	    AccessibleSelection *selection;
	    selection = Accessible_getSelection (ev->acc);
	    if (!selection)
		return FALSE;
	    if (AccessibleSelection_getNSelectedChildren (selection) > 0)
		process2 = TRUE;
	    AccessibleSelection_unref (selection); 
	}
	
	if (!process2)
	{
	    return TRUE;	
	}
	if (process2)
	{
	    g_free (ev->type);
	    ev->type = g_strdup ("object:property-changed:accessible-content");
	}
    }

    if (Accessible_isText (ev->acc))
    {
	Accessible *parent;
	parent = Accessible_getParent (ev->acc);
	if (parent)
	{
	    if (Accessible_getRole (parent) == SPI_ROLE_COMBO_BOX)
	    {
		Accessible_unref (ev->acc);
		ev->acc = parent;
		Accessible_ref (ev->acc);
	    }
	    Accessible_unref (parent);
	}
    }
/*FIXME: folowing code seems to be no more needed for gtk */
#if 0
    if (Accessible_getRole (ev->acc) == SPI_ROLE_LIST_ITEM &&
	strcmp (ev->type, "focus:") == 0)
    {
	/* avoid this message in case of combo-box */
	Accessible *parent;
	gboolean process2 = TRUE;
	parent = Accessible_getParent (ev->acc);
	if (parent)
	{
	    Accessible *parent2;
	    parent2 = Accessible_getParent (parent);
	    
	    if (parent2)
	    {
	        if (Accessible_getRole (parent2) == SPI_ROLE_COMBO_BOX)
		{
		    process2 = FALSE;
		}
	    	Accessible_unref (parent2);
	    }
	    Accessible_unref (parent);
	}
	if (!process2)
	    return FALSE;
    }	    
#endif
/*FIXME: end*/
    if (strcmp (ev->type, "object:text-caret-moved") == 0)
    {
	if (srl_last_focus != ev->acc)
	{
	    if (srl_check_toolkit (ev->acc, "mozilla"))
	    {
		g_free (ev->type);
		ev->type = g_strdup ("focus:");
	    }
	}
    }
    
    if (strcmp (ev->type, "window:restore") == 0)
    {
	if (Accessible_getRole (ev->acc) == SPI_ROLE_TOOL_TIP)
	{
	    g_free (ev->type);
	    ev->type = g_strdup ("tooltip:show");
	}
    }
    
    if (strcmp (ev->type, "object:state-changed:visible") == 0)
    {
	if (!ev->detail1)
	{
	    if (Accessible_getRole (ev->acc) == SPI_ROLE_TOOL_TIP)
	    {
		g_free (ev->type);
		ev->type = g_strdup ("tooltip:hide");
	    }
	}
    }
    
    if (strcmp (ev->type, "focus:") == 0)
    {
	if (srl_last_focus)
	    Accessible_unref (srl_last_focus);
	srl_last_focus = ev->acc;
	if (srl_last_focus)
	    Accessible_ref (srl_last_focus);
	process = TRUE;
	srl_set_info (&info, ev->acc);
	if (srl_last_edit)
	    Accessible_unref (srl_last_edit);
	srl_last_edit = NULL;
	if (Accessible_isEditableText (ev->acc))
	{
	    srl_last_edit = ev->acc;
	    Accessible_ref (srl_last_edit);
	}
	if (process)
	    srl_process_event_real_sro (ev);
    }
    else if (srl_last_focus == ev->acc)
    {
	SRLLastInfo info2;
	
	srl_set_info (&info2, ev->acc);
    
	process = TRUE;
	if (strcmp (ev->type, "object:text-selection-changed") == 0)
	{
	    if (info2.text_selections_count != info.text_selections_count 
		|| (info2.text_selections_count != 0 && info2.text_crt_selection_length != info.text_crt_selection_length))
	    {}
	    else
	    {
		process = FALSE;	
	    }
	}
	else if (strcmp (ev->type, "object:text-caret-moved") == 0)
	{
/*
	    fprintf (stderr, "\nCC=%ld SC=%ld CO=%ld",
			    info2.text_char_count,
			    info2.text_selections_count,
			    info2.text_caret_offset);

	    fprintf (stderr, "\nCC=%ld SC=%ld CO=%ld",
			    info.text_char_count,
			    info.text_selections_count,
			    info.text_caret_offset);
*/
	    if (info2.text_selections_count == info.text_selections_count 
		&& info2.text_selections_count == 0
		&& info2.text_caret_offset != info.text_caret_offset
		&& info2.text_char_count == info.text_char_count)
	    {}
	    else
	    {
		process = FALSE;	
	    }
	}
	else if (strcmp (ev->type, "object:state-changed:expanded") == 0)
	{
	    AccessibleRole role;
	    role = Accessible_getRole (ev->acc);
	    if (role == SPI_ROLE_TABLE_CELL)
	    {}
	    else
	    {
		process = FALSE;
	    }
	}
	else if (strcmp (ev->type, "object:state-changed:checked") == 0)
	{
	    AccessibleRole role;
	    role = Accessible_getRole (ev->acc);
	    if (role == SPI_ROLE_CHECK_BOX ||
		role == SPI_ROLE_RADIO_BUTTON ||
		role == SPI_ROLE_TABLE_CELL ||
		role == SPI_ROLE_TOGGLE_BUTTON ||
		role == SPI_ROLE_RADIO_MENU_ITEM ||
		role == SPI_ROLE_CHECK_MENU_ITEM)
	    {}
	    else
	    {
		process = FALSE;
	    }
	}
	else if (strcmp (ev->type, "object:state-changed:selected") == 0)
	{
	    AccessibleRole role;
	    role = Accessible_getRole (ev->acc);
	    if (role == SPI_ROLE_PAGE_TAB)
	    {}
	    else
	    {
		process = FALSE;
	    }
	}
	
	if (strcmp (ev->type, "object:text-changed:delete") == 0 ||
	    strcmp (ev->type, "object:text-changed:insert") == 0 ||
	    strcmp (ev->type, "object:text-caret-moved") == 0)
	{
	    if (Accessible_getRole (ev->acc) == SPI_ROLE_SPIN_BUTTON)
	    {
		SRLLastInfo info2;
		srl_set_value_info (&info2, ev->acc);
		if (info2.value_crt_value != info.value_crt_value)
		{
		    process = FALSE;
		}
	    }
	}

	if (process)
	{
	    srl_set_info (&info, ev->acc);
	    srl_process_event_real_sro (ev);
	}
	
	if (strcmp (ev->type, "object:text-changed:delete") == 0)
	{
	    info.text_caret_offset	= ev->detail1;
	    info.text_selections_count	= 0;
	    info.text_char_count -= ev->detail2;
	}
    }
    else if (strncmp (ev->type, "window", 6) == 0)
    {
	if (Accessible_getRole (ev->acc) == SPI_ROLE_TOOL_TIP)
	{}
	else if (strcmp (ev->type, "window:activate") == 0)
	{
	    if (srl_last_restore == ev->acc)
	    {
		process = FALSE;
	    }
	    else
	    {
		g_free (ev->type);
		ev->type = g_strdup ("window:switch");
		process = TRUE;
	    }
	    if (srl_last_restore)
		Accessible_unref (srl_last_restore);
	    srl_last_restore = NULL;
	}
	else if (strcmp (ev->type, "window:deactivate") == 0)
	{
	    process = FALSE;
	}
	else if (strcmp (ev->type, "window:restore") == 0)
	{
	    if (srl_last_restore)
		Accessible_unref (srl_last_restore);
	    srl_last_restore = ev->acc;
	    if (srl_last_restore)
		Accessible_ref (srl_last_restore);
		
	    if (srl_last_create == ev->acc)
	    {
		Accessible_unref (srl_last_create);
		srl_last_create = NULL;
	    }
	    else
	    {
		process = TRUE;
	    }
	}
	else if (strcmp (ev->type, "window:minimize") == 0)
	{
	    AccessibleStateSet *state;
	    state = Accessible_getStateSet (ev->acc);
	    if (state)
	    {
		if (AccessibleStateSet_contains (state, SPI_STATE_ACTIVE))
		{
		    g_free (ev->type);
		    ev->type = g_strdup ("window:titlelize");
		}
		AccessibleStateSet_unref (state);
	    }
	    process = TRUE;
	}
	else
	{
	    process = TRUE;
	}
	
	if (srl_last_create)
	    Accessible_unref (srl_last_create);
	srl_last_create = NULL;
	if (strcmp (ev->type, "window:create") == 0)
	{
	    srl_last_create = ev->acc;
	    if (srl_last_create)
		Accessible_ref (srl_last_create);
	}
	if (process)
	    srl_process_event_real_window (ev);
    }
    else if (strncmp (ev->type, "mouse", 5) == 0)
    {
	process = TRUE;
	if (process)
	    srl_process_event_real_mouse (ev);
    }
    else if (strncmp (ev->type, "tooltip", 7) == 0)
    {
	process = TRUE;
	if (process)
	    srl_process_event_real_tooltip (ev);
    }

    return TRUE;
}


    
/* callback for events */
static void
report_event (const AccessibleEvent *event, 
	      void *user_data)
{
    static gboolean busy = FALSE;
    static GQueue *queue = NULL;
    SRLEvent *ev;

    srl_assert (event && event->source);
    
    ev = (SRLEvent *) g_malloc (sizeof (SRLEvent));
    if (!ev)
	return;

    Accessible_ref (event->source);
    
    ev->acc = event->source;
    ev->type = g_strdup (event->type);
    ev->detail1 = event->detail1;
    ev->detail2 = event->detail2;
    ev->acc_ev = (AccessibleEvent*)event;
    AccessibleEvent_ref (ev->acc_ev);    
    if (!queue)
	queue = g_queue_new ();

    g_queue_push_head (queue, ev);
    srl_log_at_spi (event);

    if (busy)
    {
	srl_log_reentrancy (event);
	return;
    }
    busy = TRUE;
    
    while (queue && !g_queue_is_empty (queue))
    {
	SRLEvent *ev1;
	ev1 = (SRLEvent *) g_queue_pop_tail (queue);
		
	srl_process_event (ev1);

	Accessible_unref (ev1->acc);
	g_free (ev1->type);
	AccessibleEvent_unref (ev1->acc_ev);
	g_free (ev1);
    }

    if (queue)
	g_queue_free (queue);
    queue = NULL;
    busy = FALSE;
}

gboolean
srl_mouse_move (gint x,
		gint y)
{
    return SPI_generateMouseEvent (x, y, "abs") ? TRUE : FALSE;
}		

gboolean
srl_mouse_click (gint button)
{
    gchar action[] = "b1c";
    
    if (button == SR_MOUSE_BUTTON_LEFT)
	action[1] = '1';
    else if (button == SR_MOUSE_BUTTON_RIGHT)
	action[1] = '2';
    else
	srl_assert_not_reached ();
    
    return SPI_generateMouseEvent (-1, -1, action) ? TRUE : FALSE;
}

gboolean
srl_mouse_button_down (gint button)
{
    gchar action[] = "b1p";
    
    if (button == SR_MOUSE_BUTTON_LEFT)
	action[1] = '1';
    else if (button == SR_MOUSE_BUTTON_RIGHT)
	action[1] = '2';
    else
	srl_assert_not_reached ();
    return SPI_generateMouseEvent (-1,-1 , action) ? TRUE : FALSE;
}		

gboolean
srl_mouse_button_up (gint button)
{
    gchar action[] = "b1r";
    
    if (button == SR_MOUSE_BUTTON_LEFT)
	action[1] = '1';
    else if (button == SR_MOUSE_BUTTON_RIGHT)
	action[1] = '2';
    else
	srl_assert_not_reached ();
    
    return SPI_generateMouseEvent (-1,-1 , action) ? TRUE : FALSE;
}		
