/*
 * Copyright (C) 2009 Jens Georg <mail@jensge.org>.
 *
 * Author: Jens Georg <mail@jensge.org>
 *
 * This file is part of Rygel.
 *
 * Rygel is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Rygel is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
using Gst;

/**
 * A Gst.Bin derivative that implements transcoding of any type of media (using
 * decodebin2) to mpeg transport stream containing mpeg 2 video and mp2 audio.
 */
internal class Rygel.WMVTranscoderBin : Gst.Bin {
    private const string DECODEBIN = "decodebin2";
    private const string MUXER = "ffmux_asf";

    private const string AUDIO_ENC_SINK = "audio-enc-sink-pad";
    private const string VIDEO_ENC_SINK = "sink";

    private dynamic Element audio_enc;
    private dynamic Element video_enc;
    private dynamic Element muxer;

    public WMVTranscoderBin (MediaItem     item,
                             Element       src,
                             WMVTranscoder transcoder)
                             throws Error {
        dynamic Element decodebin = GstUtils.create_element (DECODEBIN,
                                                             DECODEBIN);
        var wma_transcoder = new WMATranscoder ();
        this.audio_enc = wma_transcoder.create_encoder (item,
                                                        null,
                                                        AUDIO_ENC_SINK);
        this.video_enc = transcoder.create_encoder (item, null, VIDEO_ENC_SINK);
        this.muxer = GstUtils.create_element (MUXER, MUXER);

        this.add_many (src,
                       decodebin,
                       this.audio_enc,
                       this.video_enc,
                       this.muxer);
        src.link (decodebin);

        var src_pad = muxer.get_static_pad ("src");
        var ghost = new GhostPad (null, src_pad);
        this.add_pad (ghost);

        decodebin.pad_added += this.decodebin_pad_added;
        decodebin.autoplug_continue += this.autoplug_continue;
    }

    private bool autoplug_continue (Element decodebin,
                                    Pad     new_pad,
                                    Caps    caps) {
        var muxer_pad = this.muxer.get_compatible_pad (new_pad, null);

        if (muxer_pad == null) {
            return true;
        } else {
            return new_pad.link (muxer_pad) != PadLinkReturn.OK;
        }
    }

    private void decodebin_pad_added (Element decodebin, Pad new_pad) {
        Element encoder;
        Pad enc_pad;

        var audio_enc_pad = this.audio_enc.get_pad (AUDIO_ENC_SINK);
        var video_enc_pad = this.video_enc.get_pad (VIDEO_ENC_SINK);

        // Check which encoder to use
        if (new_pad.can_link (audio_enc_pad)) {
            encoder = this.audio_enc;
            enc_pad = audio_enc_pad;
        } else if (new_pad.can_link (video_enc_pad)) {
            encoder = this.video_enc;
            enc_pad = video_enc_pad;
        } else {
            return;
        }

        encoder.link (this.muxer);

        if (new_pad.link (enc_pad) != PadLinkReturn.OK) {
            var error = new GstError.LINK (_("Failed to link pad %s to %s"),
                                           new_pad.name,
                                           enc_pad.name);
            GstUtils.post_error (this, error);
        }
    }
}
