/* GStreamer
 * Copyright (C) <2001> Steve Baker <stevebaker_org@yahoo.co.uk>
 * Copyright (C) 2002, 2003 Andy Wingo <wingo at pobox dot com>
 *
 * int2float.c
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <gstint2float.h>
#include <gst/audio/audio.h>

/* elementfactory information */
static GstElementDetails int2float_details = {
  "Integer to Float effect",
  "Filter/Converter/Audio",
  "Convert from integer to floating point audio data",
  "Steve Baker <stevebaker_org@yahoo.co.uk>, Andy Wingo <wingo at pobox dot com>"
};

#define GST_INT2FLOAT_DEFAULT_BUFFER_FRAMES 256

#if 0
#define DEBUG(...) g_message (__VA_ARGS__)
#else
#define DEBUG(...)
#endif

GST_PAD_TEMPLATE_FACTORY (int2float_src_factory,
  "src%d",
  GST_PAD_SRC,
  GST_PAD_REQUEST,
  GST_CAPS_NEW (
    "float_src",
    "audio/x-raw-float",
    "rate",		GST_PROPS_INT_RANGE (4000, 96000),
    "buffer-frames",	GST_PROPS_INT_RANGE (1, G_MAXINT),
    "width",		GST_PROPS_INT (32),
    "channels",		GST_PROPS_INT (1),
    "endianness",	GST_PROPS_INT (G_BYTE_ORDER)
  )
);

GST_PAD_TEMPLATE_FACTORY (int2float_sink_factory,
  "sink",
  GST_PAD_SINK,
  GST_PAD_ALWAYS,
  GST_CAPS_NEW (
    "int_sink",
    "audio/x-raw-int",
    "channels",   GST_PROPS_INT_RANGE (1, G_MAXINT),
    "rate",       GST_PROPS_INT_RANGE (4000, 96000),
    "endianness", GST_PROPS_INT (G_BYTE_ORDER),
    "width",      GST_PROPS_INT (16),
    "depth",      GST_PROPS_INT (16),
    "signed",     GST_PROPS_BOOLEAN (TRUE)
  )
);

static void                 gst_int2float_class_init (GstInt2FloatClass *klass);
static void                 gst_int2float_base_init (GstInt2FloatClass *klass);
static void                 gst_int2float_init (GstInt2Float *this);

static GstPadLinkReturn	    gst_int2float_link (GstPad *pad, GstCaps *caps);

static GstPad*              gst_int2float_request_new_pad (GstElement *element, GstPadTemplate *temp, const gchar *unused);
static void                 gst_int2float_chain_gint16 (GstPad *pad, GstData *_data);
static GstElementStateReturn gst_int2float_change_state (GstElement *element);

static GstElementClass *parent_class = NULL;

GType
gst_int2float_get_type(void) {
  static GType int2float_type = 0;

  if (!int2float_type) {
    static const GTypeInfo int2float_info = {
      sizeof(GstInt2FloatClass),
      (GBaseInitFunc)gst_int2float_base_init,
      NULL,
      (GClassInitFunc)gst_int2float_class_init,
      NULL,
      NULL,
      sizeof(GstInt2Float),
      0,
      (GInstanceInitFunc)gst_int2float_init,
    };
    int2float_type = g_type_register_static(GST_TYPE_ELEMENT, "GstInt2Float", &int2float_info, 0);
  }
  return int2float_type;
}

static void
gst_int2float_base_init (GstInt2FloatClass *klass)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (klass);

  gst_element_class_set_details (gstelement_class, &int2float_details);
  gst_element_class_add_pad_template (gstelement_class,
	GST_PAD_TEMPLATE_GET (int2float_src_factory));
  gst_element_class_add_pad_template (gstelement_class,
	GST_PAD_TEMPLATE_GET (int2float_sink_factory));
}

static void
gst_int2float_class_init (GstInt2FloatClass *klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass*)klass;
  gstelement_class = (GstElementClass*)klass;

  parent_class = g_type_class_ref(GST_TYPE_ELEMENT);

  gstelement_class->request_new_pad = gst_int2float_request_new_pad;
  gstelement_class->change_state = gst_int2float_change_state;
}

static void
gst_int2float_init (GstInt2Float *this)
{
  this->sinkpad = gst_pad_new_from_template(
			GST_PAD_TEMPLATE_GET (int2float_sink_factory),"sink");

  gst_element_add_pad(GST_ELEMENT(this),this->sinkpad);
  gst_pad_set_chain_function(this->sinkpad,gst_int2float_chain_gint16);
  
  gst_pad_set_link_function(this->sinkpad, gst_int2float_link);

  this->numsrcpads = 0;
  this->srcpads = NULL;
  
  this->intcaps = NULL;
  this->floatcaps = NULL;
  this->channels = 0;
}

/* see notes in gstfloat2int.c */
static GstPadLinkReturn
gst_int2float_link (GstPad *pad, GstCaps *caps)
{
  GstInt2Float *this;
  GstCaps *intcaps, *floatcaps;
  GSList *l;
  
  this = GST_INT2FLOAT (GST_PAD_PARENT (pad));

  if (GST_CAPS_IS_FIXED (caps)) {
    if (pad == this->sinkpad) {
      /* it's the int pad */
      if (!this->intcaps) {
        floatcaps = gst_caps_copy (gst_pad_template_get_caps (int2float_src_factory ()));
        gst_caps_get_int (caps, "rate", &this->rate);
        gst_caps_set (floatcaps, "rate", GST_PROPS_INT (this->rate), NULL);

        if (!this->buffer_frames)
          this->buffer_frames = GST_INT2FLOAT_DEFAULT_BUFFER_FRAMES;
        gst_caps_set (floatcaps, "buffer-frames", GST_PROPS_INT (this->buffer_frames), NULL);

        this->in_capsnego = TRUE;

        this->floatcaps = gst_caps_ref (floatcaps);
        this->intcaps = gst_caps_ref (caps);
        gst_caps_get_int (caps, "channels", &this->channels);
    
        for (l=this->srcpads; l; l=l->next)
          if (gst_pad_try_set_caps ((GstPad*) l->data, floatcaps) <= 0)
            return GST_PAD_LINK_REFUSED;
        
        this->in_capsnego = FALSE;
        return GST_PAD_LINK_OK;
      } else { 
        gst_caps_set (this->intcaps, "channels", GST_PROPS_INT_RANGE (1, G_MAXINT));
        if (gst_caps_intersect (caps, this->intcaps)) {
          gst_caps_get_int (caps, "channels", &this->channels);
          gst_caps_set (this->intcaps, "channels", GST_PROPS_INT (this->channels));
          return GST_PAD_LINK_OK;
        } else {
          gst_caps_set (this->intcaps, "channels", GST_PROPS_INT (this->channels));
          return GST_PAD_LINK_REFUSED;
        }
      }
    } else {
      /* it's a float pad */
      gint rate, buffer_frames;

      gst_caps_get_int (caps, "rate", &rate);
      gst_caps_get_int (caps, "buffer-frames", &buffer_frames);

      if (this->in_capsnego) {
        /* caps were set on the sink pad, so i started setting on the src pads.
           capsnego reached the end and bounced back to float2int, that started
           setting all of its sink pads, which brought me back here. */
        if (rate == this->rate && buffer_frames == this->buffer_frames)
          return GST_PAD_LINK_OK;
        else
          return GST_PAD_LINK_REFUSED;
      }

      intcaps = gst_caps_copy (gst_pad_template_get_caps (int2float_sink_factory ()));
      gst_caps_set (intcaps, "rate", GST_PROPS_INT (rate), NULL);

      this->rate = rate;
      this->buffer_frames = buffer_frames;

      /* intcaps is now variable in channels */

      if (!this->intcaps) {
        if (GST_PAD_PEER (this->sinkpad)) {
          GstCaps *newcaps, *gottencaps;

          gottencaps = gst_pad_get_allowed_caps (GST_PAD_PEER (this->sinkpad));
          gst_caps_debug (intcaps, "internal int2float int caps");
          gst_caps_debug (gottencaps, "allowed caps on int2float's peer");
          newcaps = gst_caps_intersect (gottencaps, intcaps);
          if (!newcaps)
            return GST_PAD_LINK_REFUSED;
          
          if (GST_CAPS_IS_FIXED (newcaps)) {
            gst_caps_get_int (newcaps, "channels", &this->channels);
            if (gst_pad_try_set_caps (this->sinkpad, newcaps) <= 0)
              return GST_PAD_LINK_DELAYED;
          }

          this->intcaps = newcaps;
        } else {
          /* we assume channels is equal to the number of source pads */
          gst_caps_set (intcaps, "channels", GST_PROPS_INT (this->numsrcpads), NULL);
          this->channels = this->numsrcpads;
        
          if (gst_pad_try_set_caps (this->sinkpad, intcaps) <= 0)
            return GST_PAD_LINK_REFUSED;

          this->intcaps = intcaps;
        }
        
        for (l=this->srcpads; l; l=l->next)
          if (! ((GstPad*) l->data == pad || gst_pad_try_set_caps ((GstPad*) l->data, caps) > 0))
            return GST_PAD_LINK_REFUSED;
        
        this->floatcaps = gst_caps_ref (caps);
        return GST_PAD_LINK_OK;
      } else if (gst_caps_intersect (intcaps, this->intcaps)) {
        return GST_PAD_LINK_OK;
      } else {
        return GST_PAD_LINK_REFUSED;
      }
    }
  } else {
    return GST_PAD_LINK_DELAYED;
  }
}

static GstPad*
gst_int2float_request_new_pad (GstElement *element, GstPadTemplate *templ, const gchar *unused) 
{
  gchar *name;
  GstPad *srcpad;
  GstInt2Float *this;

  this = GST_INT2FLOAT (element);
  
  if (GST_STATE (element) == GST_STATE_PLAYING) {
    g_warning ("new pads can't be requested from gstint2float while it is PLAYING");
    return NULL;
  }

  name = g_strdup_printf ("src%d", this->numsrcpads);
  srcpad = gst_pad_new_from_template (templ, name);
  g_free (name);
  gst_element_add_pad (GST_ELEMENT (this), srcpad);
  gst_pad_set_link_function (srcpad, gst_int2float_link);
  
  this->srcpads = g_slist_append (this->srcpads, srcpad);
  this->numsrcpads++;
  
  return srcpad;
}

static inline void
fetch_buffers (GstInt2Float *this) 
{
  int i;

  g_return_if_fail (this->frames_to_write == 0);

  for (i=0; i<this->numsrcpads; i++) {
    this->buffers[i] = gst_buffer_new_from_pool (this->buffer_pool, 0, 0);
    this->data_out[i] = (gfloat*)GST_BUFFER_DATA (this->buffers[i]);
    GST_BUFFER_TIMESTAMP (this->buffers[i]) = this->offset * GST_SECOND / this->rate;
  }
  this->offset += this->buffer_frames;
  this->frames_to_write = this->buffer_frames;
}

static void
gst_int2float_chain_gint16 (GstPad *pad, GstData *_data)
{
  GstBuffer *buf_in = GST_BUFFER (_data);
  GstInt2Float *this;
  gint16 *data_in;
  gfloat **data_out;
  gint i, j;
  GSList *srcpads;
  GstBuffer **buffers;
  gint to_process, channels;
  
  this = GST_INT2FLOAT (GST_OBJECT_PARENT (pad));

  if (!this->channels) {
    gst_element_error (GST_ELEMENT (this), "capsnego was never performed, bailing...");
    return;
  }
  if (!this->buffer_pool) /* see the state change function */
    this->buffer_pool = gst_buffer_pool_get_default (this->buffer_frames * sizeof (float),
                                                     this->numsrcpads);

  buffers = this->buffers;
  data_out = this->data_out;
  channels = this->channels;

  DEBUG ("chaining a buffer with %d frames",
         GST_BUFFER_SIZE (buf_in) / channels / sizeof (gint16));

  if (!this->frames_to_write)
    fetch_buffers (this);

  to_process = this->leftover_frames;
  if (to_process) {
    DEBUG ("dealing with %d leftover frames", to_process);

    /* we are guaranteed that if some frames are left over, they can't fill a
       whole buffer */
    data_in = (gint16*)GST_BUFFER_DATA (this->in_buffer);
    data_in += GST_BUFFER_SIZE (this->in_buffer) / sizeof (gint16) - channels * to_process;
    for (i=0; i<this->numsrcpads; i++) {
      for (j=0; j<to_process; j++)
        data_out[i][j] = ((gfloat)data_in[(j*channels) + (i%channels)]) / 32767.0;
      data_out[i] += to_process;
    }
    this->frames_to_write -= to_process;
    gst_buffer_unref (this->in_buffer);
    this->in_buffer = NULL;
  }
  
  this->in_buffer = buf_in;
  this->leftover_frames = GST_BUFFER_SIZE (buf_in) / sizeof (gint16) / channels;
  data_in = (gint16*)GST_BUFFER_DATA (this->in_buffer);
  
  while (1) {
    to_process = MIN (this->leftover_frames, this->frames_to_write);
    DEBUG ("processing %d frames", to_process);

    for (i=0, srcpads=this->srcpads ; i<this->numsrcpads; i++) {
      for (j=0; j<to_process; j++){
        data_out[i][j] = ((gfloat)data_in[(j*channels) + (i%channels)]) / 32767.0;
      }
      data_out[i] += to_process;

      DEBUG ("if %d == %d then I'm pushing...", this->frames_to_write, to_process);
      if (this->frames_to_write == to_process)
        gst_pad_push (GST_INT2FLOAT_SRCPAD (srcpads), GST_DATA (buffers[i]));

      srcpads = srcpads->next;
    }

    if (this->leftover_frames != to_process) {
      this->leftover_frames -= to_process;
      data_in += channels * to_process;
      this->frames_to_write = 0;

      DEBUG ("%d frames remaining in input buffer", this->leftover_frames);

      if (this->leftover_frames < this->buffer_frames)
        return;
    } else {
      gst_buffer_unref (this->in_buffer);
      this->in_buffer = NULL;
      this->frames_to_write -= to_process;
      this->leftover_frames = 0;

      DEBUG ("finished with input buffer, %d frames needed to fill output. returning.",
             this->frames_to_write);
      return;
    }
    
    fetch_buffers (this);
  }
}

static GstElementStateReturn
gst_int2float_change_state (GstElement *element)
{
  GstInt2Float *this;

  this = GST_INT2FLOAT (element);
  
  switch (GST_STATE_TRANSITION (element)) {
  case GST_STATE_PAUSED_TO_PLAYING:
    this->buffers = g_new0 (GstBuffer*, this->numsrcpads);
    this->data_out = g_new0 (gfloat*, this->numsrcpads);
    /* maybe capsnego hasn't happened; i guess we can alloc the buffers in the
       chainfunc then... */
    if (this->buffer_frames)
      this->buffer_pool = gst_buffer_pool_get_default (this->buffer_frames * sizeof (float),
                                                       this->numsrcpads);
    break;
    
  case GST_STATE_PLAYING_TO_PAUSED:
    g_free (this->buffers);
    this->buffers = NULL;
    g_free (this->data_out);
    this->data_out = NULL;
    if (this->buffer_pool)
      gst_buffer_pool_unref (this->buffer_pool);
    this->buffer_pool = NULL;
    break;
    
  case GST_STATE_PAUSED_TO_READY:
    if (this->intcaps)
      gst_caps_unref (this->intcaps);
    this->intcaps = NULL;
    if (this->floatcaps)
      gst_caps_unref (this->floatcaps);
    this->floatcaps = NULL;
    break;

  default:
    break;
  }

  if (GST_ELEMENT_CLASS (parent_class)->change_state) 
    return GST_ELEMENT_CLASS (parent_class)->change_state (element);

  return GST_STATE_SUCCESS;

}
