/* GStreamer
 * Copyright (C) <2001> Steve Baker <stevebaker_org@yahoo.co.uk>
 * Copyright (C) 2002, 2003 Andy Wingo <wingo at pobox dot com>
 *
 * float2int.c
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/floatcast/floatcast.h>

#include "gstfloat2int.h"
#include <gst/audio/audio.h>

/* elementfactory information */
static GstElementDetails float2int_details = {
  "Float to Integer effect",
  "Filter/Converter/Audio",
  "Convert from floating point to integer audio data",
  "Steve Baker <stevebaker_org@yahoo.co.uk>, Andy Wingo <wingo at pobox dot com>"
};


#define GST_FLOAT2INT_DEFAULT_BUFFER_FRAMES 256
#if 0
#define F2I_DEBUG(...) g_message (__VA_ARGS__)
#else
#define F2I_DEBUG(...)
#endif


static GstStaticPadTemplate float2int_sink_factory =
GST_STATIC_PAD_TEMPLATE (
  "sink%d",
  GST_PAD_SINK,
  GST_PAD_REQUEST,
  GST_STATIC_CAPS (GST_AUDIO_FLOAT_STANDARD_PAD_TEMPLATE_CAPS)
);

static GstStaticPadTemplate float2int_src_factory =
GST_STATIC_PAD_TEMPLATE (
  "src",
  GST_PAD_SRC,
  GST_PAD_ALWAYS,
  GST_STATIC_CAPS ( "audio/x-raw-int, "
    "rate = (int) [ 1, MAX ], "
    "channels = (int) [ 1, MAX ], "
    "endianness = (int) BYTE_ORDER, "
    "width = (int) 16, "
    "depth = (int) 16, "
    "signed = (boolean) true, "
    "buffer-frames = (int) [ 1, MAX]"
  )
);

static void	gst_float2int_class_init		(GstFloat2IntClass *klass);
static void	gst_float2int_base_init			(GstFloat2IntClass *klass);
static void	gst_float2int_init			(GstFloat2Int *plugin);

static GstCaps * gst_float2int_sink_getcaps (GstPad *pad);
static GstCaps * gst_float2int_src_getcaps (GstPad *pad);
static GstPadLinkReturn gst_float2int_src_link (GstPad *pad, const GstCaps *caps);
static GstPadLinkReturn gst_float2int_sink_link (GstPad *pad, const GstCaps *caps);

static GstPad*  gst_float2int_request_new_pad		(GstElement *element, GstPadTemplate *temp,
                                                         const gchar *unused);
static void     gst_float2int_pad_removed               (GstElement *element, GstPad *pad);

static void	gst_float2int_loop			(GstElement *element);

static GstElementStateReturn gst_float2int_change_state (GstElement *element);

static GstElementClass *parent_class = NULL;

GType
gst_float2int_get_type(void) {
  static GType float2int_type = 0;

  if (!float2int_type) {
    static const GTypeInfo float2int_info = {
      sizeof(GstFloat2IntClass),
      (GBaseInitFunc)gst_float2int_base_init,
      NULL,
      (GClassInitFunc)gst_float2int_class_init,
      NULL,
      NULL,
      sizeof(GstFloat2Int),
      0,
      (GInstanceInitFunc)gst_float2int_init,
    };
    float2int_type = g_type_register_static(GST_TYPE_ELEMENT, "GstFloat2Int", &float2int_info, 0);
  }
  return float2int_type;
}

static void
gst_float2int_base_init (GstFloat2IntClass *klass)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (klass);

  gst_element_class_set_details (gstelement_class, &float2int_details);
  gst_element_class_add_pad_template (gstelement_class,
	gst_static_pad_template_get (&float2int_src_factory));
  gst_element_class_add_pad_template (gstelement_class,
	gst_static_pad_template_get (&float2int_sink_factory));
}

static void
gst_float2int_class_init (GstFloat2IntClass *klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass*)klass;
  gstelement_class = (GstElementClass*)klass;

  parent_class = g_type_class_ref(GST_TYPE_ELEMENT);

  gstelement_class->request_new_pad = gst_float2int_request_new_pad;
  gstelement_class->pad_removed = gst_float2int_pad_removed;
  gstelement_class->change_state = gst_float2int_change_state;
}

static void
gst_float2int_init (GstFloat2Int *plugin)
{
  gst_element_set_loop_function (GST_ELEMENT (plugin), gst_float2int_loop);

  plugin->srcpad = gst_pad_new_from_template (
			gst_static_pad_template_get (&float2int_src_factory), "src");
  gst_element_add_pad (GST_ELEMENT (plugin), plugin->srcpad);
  gst_pad_set_link_function (plugin->srcpad, gst_float2int_src_link);
  gst_pad_set_getcaps_function (plugin->srcpad, gst_float2int_src_getcaps);

  plugin->numchannels = 0;
  plugin->channelcount = 0;
  plugin->channels = NULL;
}


static GstCaps *
gst_float2int_get_allowed_sinkcaps (GstFloat2Int *this, GstPad *ignorepad)
{
  GSList *channels;
  GstCaps *caps;
  GstCaps *allowed_caps;
  GstCaps *icaps;
  GstPad *sinkpad;
  GstFloat2IntInputChannel *channel;

  GST_DEBUG ("here");

  channels = this->channels;
  caps = gst_caps_from_string (GST_AUDIO_FLOAT_STANDARD_PAD_TEMPLATE_CAPS);
  while (channels) {
    channel = GST_FLOAT2INT_CHANNEL (channels);
    sinkpad = channel->sinkpad;
    channels = g_slist_next (channels);
    if (sinkpad == ignorepad) continue;

    allowed_caps = gst_pad_get_allowed_caps(sinkpad);
    icaps = gst_caps_intersect (caps, allowed_caps);
    gst_caps_free (allowed_caps);
    gst_caps_free (caps);
    caps = icaps;

    channels = g_slist_next (channels);
  }

  GST_DEBUG ("allowed sinkcaps %s", gst_caps_to_string (caps));

  return caps;
}

static void
gst_float2int_caps_remove_format (GstCaps *caps, const char *name)
{
  int i;
  GstStructure *structure;

  GST_DEBUG ("here");

  for(i=0;i<gst_caps_get_size(caps);i++){
    structure = gst_caps_get_structure(caps, i);
    gst_structure_set_name (structure, name);
    gst_structure_remove_field (structure, "width");
    gst_structure_remove_field (structure, "channels");
    gst_structure_remove_field (structure, "endianness");
    gst_structure_remove_field (structure, "depth");
    gst_structure_remove_field (structure, "signed");
    //gst_structure_remove_field (structure, "buffer-frames");
  }
}

static GstCaps *
gst_float2int_src_getcaps (GstPad *pad)
{
  GstFloat2Int *this;
  GstCaps *caps;
  GstCaps *icaps;
  int i;
  GstStructure *structure;

  GST_DEBUG ("here");

  this = GST_FLOAT2INT (gst_pad_get_parent (pad));

  caps = gst_float2int_get_allowed_sinkcaps (this, NULL);
  gst_float2int_caps_remove_format (caps, "audio/x-raw-int");

  GST_DEBUG ("no format %s", gst_caps_to_string (caps));

  if (this->numchannels > 0) {
    for(i=0;i<gst_caps_get_size(caps);i++){
      structure = gst_caps_get_structure(caps, i);
      gst_structure_set (structure, "channels", G_TYPE_INT,
          this->numchannels, NULL);
    }
  }

  GST_DEBUG ("channels %s", gst_caps_to_string (caps));

  GST_DEBUG ("pad template %s", gst_caps_to_string (gst_pad_get_pad_template_caps(pad)));

  icaps = gst_caps_intersect (caps, gst_pad_get_pad_template_caps(pad));
  gst_caps_free (caps);

  return icaps;
}

static GstCaps *
gst_float2int_sink_getcaps (GstPad *pad)
{
  GstFloat2Int *this;
  GstCaps *sinkcaps;
  GstCaps *srccaps;
  GstCaps *icaps;
  GstCaps *caps;
  int i;
  GstStructure *structure;

  this = GST_FLOAT2INT (gst_pad_get_parent (pad));

  GST_DEBUG ("here");

  sinkcaps = gst_float2int_get_allowed_sinkcaps (this, pad);
  gst_float2int_caps_remove_format (sinkcaps, "audio/x-raw-float");

  srccaps = gst_pad_get_allowed_caps (this->srcpad);
  gst_float2int_caps_remove_format (srccaps, "audio/x-raw-float");

  icaps = gst_caps_intersect (srccaps, sinkcaps);
  gst_caps_free (sinkcaps);
  gst_caps_free (srccaps);

  for(i=0;i<gst_caps_get_size(icaps);i++){
    structure = gst_caps_get_structure(icaps, i);
    gst_structure_set (structure, "channels", G_TYPE_INT, 1, NULL);
  }

  caps = gst_caps_intersect (icaps, gst_pad_get_pad_template_caps(pad));
  gst_caps_free (icaps);

  return caps;
}

static GstPadLinkReturn
gst_float2int_src_link (GstPad *pad, const GstCaps *caps)
{
  GstFloat2Int *this;
  GstStructure *structure;
  GstCaps *fcaps;
  GstPad *sinkpad;
  int rate;
  int n_channels;
  int buffer_frames;
  GSList *channels;
  GstPadLinkReturn ret;
  GstFloat2IntInputChannel *channel;

  GST_DEBUG ("here");

  this = GST_FLOAT2INT (GST_PAD_PARENT (pad));

  structure = gst_caps_get_structure (caps, 0);

  gst_structure_get_int (structure, "rate", &rate);
  gst_structure_get_int (structure, "channels", &n_channels);
  gst_structure_get_int (structure, "buffer-frames", &buffer_frames);

  fcaps = gst_caps_from_string (GST_AUDIO_FLOAT_STANDARD_PAD_TEMPLATE_CAPS);
  gst_caps_set_simple (fcaps,
      "rate", G_TYPE_INT, rate,
      "buffer-frames", G_TYPE_INT, buffer_frames,
      NULL);

  channels = this->channels;
  while (channels) {
    channel = GST_FLOAT2INT_CHANNEL (channels);
    sinkpad = channel->sinkpad;
    channels = g_slist_next (channels);
    if (sinkpad == pad) continue;

    ret = gst_pad_try_set_caps (sinkpad, fcaps);
    if(GST_PAD_LINK_FAILED (ret)) {
      /* FIXME we should revert the other changes */
      return ret;
    }

  }

  /* ok, it worked */

  this->rate = rate;
  //this->n_channels = n_channels;
  this->buffer_frames = buffer_frames;

  return GST_PAD_LINK_OK;
}

static GstPadLinkReturn
gst_float2int_sink_link (GstPad *pad, const GstCaps *caps)
{
  GstFloat2Int *this;
  GstStructure *structure;
  GstCaps *icaps;
  GstPad *sinkpad;
  int rate;
  int n_channels;
  int buffer_frames;
  GSList *channels;
  GstPadLinkReturn ret;
  GstFloat2IntInputChannel *channel;

  GST_DEBUG ("here");

  this = GST_FLOAT2INT (GST_PAD_PARENT (pad));

  structure = gst_caps_get_structure (caps, 0);

  gst_structure_get_int (structure, "rate", &rate);
  gst_structure_get_int (structure, "channels", &n_channels);
  gst_structure_get_int (structure, "buffer-frames", &buffer_frames);

  icaps = gst_caps_copy (gst_pad_get_pad_template_caps (this->srcpad));
  gst_caps_set_simple (icaps,
      "rate", G_TYPE_INT, rate,
      "buffer-frames", G_TYPE_INT, buffer_frames,
      "channels", G_TYPE_INT, this->numchannels,
      NULL);
  ret = gst_pad_try_set_caps (this->srcpad, icaps);
  if(GST_PAD_LINK_FAILED (ret)) {
    return ret;
  }

  channels = this->channels;
  while (channels) {
    channel = GST_FLOAT2INT_CHANNEL (channels);
    sinkpad = channel->sinkpad;
    channels = g_slist_next (channels);
    if (sinkpad == pad) continue;

    ret = gst_pad_try_set_caps (sinkpad, caps);
    if(GST_PAD_LINK_FAILED (ret)) {
      /* FIXME we should revert the other changes */
      return ret;
    }
  }

  /* ok, it worked */

  this->rate = rate;
  this->buffer_frames = buffer_frames;

  return GST_PAD_LINK_OK;
}


static GstPad*
gst_float2int_request_new_pad (GstElement *element, GstPadTemplate *templ, const gchar *unused) 
{
  gchar *name;
  GstFloat2Int *plugin;
  GstFloat2IntInputChannel *channel;

  plugin = GST_FLOAT2INT(element);
  
  g_return_val_if_fail(plugin != NULL, NULL);
  g_return_val_if_fail(GST_IS_FLOAT2INT(plugin), NULL);

  if (templ->direction != GST_PAD_SINK) {
    g_warning ("float2int: request new pad that is not a SINK pad\n");
    return NULL;
  }

  channel = g_new0 (GstFloat2IntInputChannel, 1);
  name = g_strdup_printf ("sink%d", plugin->channelcount);
  channel->sinkpad = gst_pad_new_from_template (templ, name);
  gst_element_add_pad (GST_ELEMENT (plugin), channel->sinkpad);
  gst_pad_set_link_function (channel->sinkpad, gst_float2int_sink_link);
  gst_pad_set_getcaps_function (channel->sinkpad, gst_float2int_sink_getcaps);
  
  plugin->channels = g_slist_append (plugin->channels, channel);
  plugin->numchannels++;
  plugin->channelcount++;
  
  GST_DEBUG ("float2int added pad %s\n", name);

  g_free (name);
  return channel->sinkpad;
}

static void
gst_float2int_pad_removed (GstElement *element,
			   GstPad *pad)
{
  GstFloat2Int *plugin;
  GSList *p;
  
  GST_DEBUG ("float2int removed pad %s\n", GST_OBJECT_NAME (pad));
  
  plugin = GST_FLOAT2INT (element);

  /* Find our channel for this pad */
  for (p = plugin->channels; p;) {
    GstFloat2IntInputChannel *channel = p->data;
    GSList *p_copy;

    if (channel->sinkpad == pad) {
      p_copy = p;

      p = p->next;
      plugin->channels = g_slist_remove_link (plugin->channels, p_copy);
      plugin->numchannels--;
      
      g_slist_free (p_copy);

      if (channel->buffer)
        gst_buffer_unref (channel->buffer);
      g_free (channel);
    } else {
      p = p->next;
    }
  }
}

static void
gst_float2int_loop (GstElement *element)
{
  GstFloat2Int *plugin = (GstFloat2Int*)element;
  GstBuffer *buf_out;
  gfloat *data_in;
  gint16 *data_out;
  gint num_frames, i, j;
  GSList *l;
  GstFloat2IntInputChannel *channel;
  gint frames_to_write;
  gint numchannels = plugin->numchannels;
    
  if (!plugin->channels) {
    GST_ELEMENT_ERROR (element, CORE, PAD, (NULL), ("float2int: at least one sink pad needs to be connected"));
    return;
  }

  /* get new output buffer */
  /* FIXME the 1024 is arbitrary */
  buf_out = gst_buffer_new_and_alloc (1024);
  num_frames = GST_BUFFER_SIZE (buf_out) / numchannels / sizeof (gint16);
  g_return_if_fail (GST_BUFFER_SIZE (buf_out) == num_frames * numchannels * sizeof (gint16));
  
  data_out = (gint16*)GST_BUFFER_DATA(buf_out);
  
  i = 0;
  frames_to_write = num_frames;

  while (frames_to_write) {
    if (!plugin->frames_remaining) {
      /* the buffer-frames GstPropsEntry ensures us that all buffers coming in
         will be the same size. if they are not, that pad will send EOS on the
         next pull. */
      F2I_DEBUG ("pulling buffers...");
      plugin->frames_remaining = G_MAXINT;

      for (l = plugin->channels; l; l=l->next) {
        channel = GST_FLOAT2INT_CHANNEL (l);

        if (channel->buffer)
          gst_buffer_unref (channel->buffer);
        channel->buffer = NULL;
      get_buffer:
        channel->buffer = GST_BUFFER (gst_pad_pull (channel->sinkpad));
        g_return_if_fail (channel->buffer != NULL);

        if (GST_IS_EVENT (channel->buffer)) {
          F2I_DEBUG ("got an event");
          if (GST_EVENT_TYPE (channel->buffer) == GST_EVENT_EOS) {
            plugin->frames_remaining = 0;
            GST_INFO ("float2int got eos from pad %s%s", GST_DEBUG_PAD_NAME (channel->sinkpad));
          } else {
            gst_pad_event_default (plugin->srcpad, GST_EVENT (channel->buffer));
            goto get_buffer; /* goto, hahaha */
          }
	} else {
          F2I_DEBUG ("got a buffer %d frames big", GST_BUFFER_SIZE (channel->buffer)/4);
          plugin->frames_remaining = MIN (plugin->frames_remaining,
                                          GST_BUFFER_SIZE (channel->buffer) / sizeof (gfloat));
        }
      }
    }

    if (plugin->frames_remaining) {
      gint to_process;

      F2I_DEBUG ("processing %d%s", plugin->frames_remaining,
                 frames_to_write >= plugin->frames_remaining ? " (last time)" : "");
      to_process = MIN (frames_to_write, plugin->frames_remaining);
      for (l=plugin->channels, i=0; l; l=l->next, i++) {
        F2I_DEBUG ("processing for %d frames on channel %d", to_process, i);
        channel = GST_FLOAT2INT_CHANNEL (l);
        g_return_if_fail (channel->buffer != NULL);
        data_in = (gfloat*) GST_BUFFER_DATA (channel->buffer);
        data_in += GST_BUFFER_SIZE (channel->buffer) / sizeof (float) - plugin->frames_remaining;
        
        for (j = 0; j < to_process; j++)
          data_out[(j*numchannels) + i] = (gint16)(gst_cast_float(data_in[j] * 32767.0F));
      }
      data_out += to_process * numchannels;
      
      if (to_process < plugin->frames_remaining) {
        /* we're finished with this output buffer */
        plugin->frames_remaining -= to_process;
        frames_to_write = 0;
      } else {
        /* note there's still the possibility that one of the buffers had less
           data, so we're possibly discarding some data from other float pads;
           that's ok though, because the pad that didn't give us buffer_frames
           worth of data will send EOS on the next pull */
        frames_to_write -= plugin->frames_remaining;
        plugin->frames_remaining = 0;
      }
    } else {
      /* a pad has run out of data */
      GST_BUFFER_SIZE (buf_out) = GST_BUFFER_SIZE (buf_out)
        - frames_to_write * numchannels * sizeof (gint16);

      if (GST_BUFFER_SIZE (buf_out)) {
        GST_BUFFER_TIMESTAMP (buf_out) = plugin->offset * GST_SECOND / plugin->rate;
        gst_pad_push (plugin->srcpad, GST_DATA (buf_out));
      } else {
        /* we got the eos event on one of the pads */
        gst_buffer_unref (buf_out);
        gst_pad_push (plugin->srcpad, GST_DATA ((GstBuffer*)gst_event_new (GST_EVENT_EOS)));
        gst_element_set_eos (element);
      }
      
      return;
    }
  }
  
  GST_BUFFER_TIMESTAMP (buf_out) = plugin->offset * GST_SECOND / plugin->rate;
  
  plugin->offset += num_frames;
  gst_pad_push (plugin->srcpad, GST_DATA (buf_out));
}

static GstElementStateReturn
gst_float2int_change_state (GstElement *element)
{
  GstFloat2Int *plugin = (GstFloat2Int *) element;
  GSList *p;

  switch (GST_STATE_TRANSITION (element)) {
    case GST_STATE_READY_TO_PAUSED:
      plugin->offset = 0;
      break;

    case GST_STATE_PAUSED_TO_PLAYING:
      for (p = plugin->channels; p; p = p->next) {
	GstFloat2IntInputChannel *c = p->data;

	c->eos = FALSE;
      }
      break;
      
    case GST_STATE_PAUSED_TO_READY:
      for (p = plugin->channels; p; p = p->next) {
	GstFloat2IntInputChannel *c = p->data;

        if (c->buffer)
          gst_buffer_unref (c->buffer);
        c->buffer = NULL;
	c->eos = FALSE;
      }
      plugin->frames_remaining = 0;
      break;

    default:
      break;
  }

  if (GST_ELEMENT_CLASS (parent_class)->change_state)
    return GST_ELEMENT_CLASS (parent_class)->change_state (element);

  return GST_STATE_SUCCESS;
}
