/* $Id: gdict-applet.c,v 1.82 2004/12/17 17:15:17 vnoel Exp $ */

/*
 *  Papadimitriou Spiros <spapadim+@cs.cmu.edu>
 *
 *  This code released under the GNU GPL.
 *  Read the file COPYING for more information.
 *
 *  GDict panel applet
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <panel-applet.h>

#include "gdict-app.h"
#include "gdict-about.h"
#include "gdict-applet.h"
#include "gdict-pref.h"
#include "gdict-pref-dialog.h"

static const gchar *tooltip_str = N_("Lookup words in an online dictionary");

static void
entry_activate_cb (GtkEntry *entry, gpointer data)
{
	GDictApplet *applet = data;
	GDictWindow *gdict = applet->gdict;

	gchar *text = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
    
    	if (!gdict->context)
		gdict_init_context (applet->gdict);

	gtk_window_set_screen (GTK_WINDOW (gdict),
			       gtk_widget_get_screen (applet->applet_widget));	
	gtk_window_move (GTK_WINDOW (gdict), gdict->x, gdict->y);
    	gtk_window_present (GTK_WINDOW (gdict));
    
    	if (text==NULL || *text == 0)
        	return;

	gtk_editable_delete_text (GTK_EDITABLE (entry), 0, -1);
	if (gdict->search_type == SEARCH_TYPE_REGULAR)
		gdict_app_do_lookup (gdict, text);
	else
		gdict_launch_web_search (gdict, text);
    	g_free (text);
}

static gboolean
image_clicked (GtkWidget *w, GdkEventButton *event, gpointer data)
{
	GDictApplet *applet = data;
	BonoboUIComponent *component;
	gchar *text;

	if (event->button == 3 || event->button ==2) {
		gtk_propagate_event (GTK_WIDGET(applet->applet_widget), (GdkEvent *) event);
		return TRUE;
	}

	if (GTK_WIDGET_VISIBLE (GTK_WIDGET(applet->gdict))) {	       
		gchar *text = gtk_editable_get_chars (GTK_EDITABLE (applet->entry_widget), 0, -1);
		if (*text == 0) {
			gdict_window_store_geometry (applet->gdict);
			gtk_widget_hide (GTK_WIDGET(applet->gdict));
		}
		else
			entry_activate_cb (GTK_ENTRY(applet->entry_widget), applet);
	} else
		entry_activate_cb (GTK_ENTRY(applet->entry_widget), applet);

	return FALSE;
}

static void
lookup_button_drag_cb (GtkWidget *widget, GdkDragContext *context, gint x, gint y,
		       GtkSelectionData *sd, guint info, guint t, gpointer data)
{
	GDictApplet *applet = data;
	gchar *text;
	
	text = gtk_selection_data_get_text (sd);
	
	if (text && *text) {
    		gtk_entry_set_text (GTK_ENTRY (applet->entry_widget), text);
		gtk_window_set_focus_on_map (GTK_WINDOW (applet->gdict), TRUE);
		entry_activate_cb (GTK_ENTRY (applet->entry_widget), applet);
    		g_free (text);
	}
	
}

static void
setup_applet_widgets (GDictApplet *applet)
{
 	static GtkTargetEntry drop_targets [] = {
    		{ "UTF8_STRING", 0, 0 },
  		{ "COMPOUND_TEXT", 0, 0 },
  		{ "TEXT", 0, 0 },
  		{ "text/plain", 0, 0 },
  		{ "STRING",     0, 0 }
    	};
	GtkWidget *eventbox;

	applet->about_dialog = NULL;

	if (applet->box)
		gtk_widget_destroy (applet->box);

	if (applet->orient == PANEL_APPLET_ORIENT_LEFT || applet->orient == PANEL_APPLET_ORIENT_RIGHT)
		applet->box = gtk_vbox_new (FALSE, 2);
	else
		applet->box = gtk_hbox_new (FALSE, 2);
	
	gtk_container_set_border_width (GTK_CONTAINER (applet->box), 0);

	eventbox = gtk_event_box_new ();
	gtk_event_box_set_visible_window (GTK_EVENT_BOX(eventbox), FALSE);
	applet->image_widget = gtk_image_new_from_icon_name ("gdict", applet->panel_size);
	gtk_container_add (GTK_CONTAINER (eventbox), applet->image_widget);
	gtk_tooltips_set_tip (applet->tooltips, eventbox, _(tooltip_str), NULL);
	gtk_drag_dest_set (eventbox, GTK_DEST_DEFAULT_ALL, drop_targets, 
    		           G_N_ELEMENTS (drop_targets), GDK_ACTION_COPY | GDK_ACTION_MOVE);

	gtk_box_pack_start (GTK_BOX (applet->box), eventbox, TRUE, TRUE, 0);
	
        if (GTK_IS_ACCESSIBLE (gtk_widget_get_accessible (applet->image_widget)))          
        gdict_add_atk_namedesc (GTK_WIDGET (applet->image_widget),
				_("Dictionary Lookup"), NULL);

	applet->entry_widget = gtk_entry_new ();
	if (GTK_IS_ACCESSIBLE (gtk_widget_get_accessible (applet->entry_widget)))
		gdict_add_atk_namedesc (GTK_WIDGET (applet->entry_widget),
					_("Dictionary word entry"), NULL);
        gtk_entry_set_editable (GTK_ENTRY(applet->entry_widget), TRUE);

	gtk_entry_set_width_chars (GTK_ENTRY (applet->entry_widget), 12);
	gtk_tooltips_set_tip (applet->tooltips, applet->entry_widget, _(tooltip_str), NULL);
	gtk_box_pack_end (GTK_BOX (applet->box), applet->entry_widget, TRUE, TRUE, 0);

 	gtk_widget_show_all (applet->box);

 	gtk_container_add (GTK_CONTAINER(applet->applet_widget), applet->box);                              

 	gtk_widget_show_all (applet->applet_widget);

 	g_signal_connect (G_OBJECT (applet->entry_widget), "activate",
 			  G_CALLBACK (entry_activate_cb), applet);
	g_signal_connect (G_OBJECT (eventbox), "button_release_event",
 			  G_CALLBACK (image_clicked), applet);
 	g_signal_connect (G_OBJECT (eventbox), "drag_data_received",
    		          G_CALLBACK (lookup_button_drag_cb), applet);
}


static void
change_background (PanelApplet *applet,
		   PanelAppletBackgroundType type,
		   GdkColor *colour,
		   GdkPixmap *pixmap)
{
	GtkRcStyle *rc_style;
	GtkStyle *style;

	/* reset style */
	gtk_widget_set_style (GTK_WIDGET (applet), NULL);
	rc_style = gtk_rc_style_new ();
	gtk_widget_modify_style (GTK_WIDGET (applet), rc_style);
	g_object_unref (rc_style);

	switch (type) {
	case PANEL_NO_BACKGROUND:
		break;
	case PANEL_COLOR_BACKGROUND:
		gtk_widget_modify_bg (GTK_WIDGET (applet),
				      GTK_STATE_NORMAL, colour);
		break;
	case PANEL_PIXMAP_BACKGROUND:
		style = gtk_style_copy (GTK_WIDGET (applet)->style);
		if (style->bg_pixmap[GTK_STATE_NORMAL])
			g_object_unref (style->bg_pixmap[GTK_STATE_NORMAL]);
		style->bg_pixmap[GTK_STATE_NORMAL] = g_object_ref (pixmap);
		gtk_widget_set_style (GTK_WIDGET (applet), style);
		break;
	}
}

static void
size_allocate_cb (PanelApplet *widget, GtkAllocation *allocation, GDictApplet *applet)
{
	if (applet->orient == PANEL_APPLET_ORIENT_UP || applet->orient == PANEL_APPLET_ORIENT_DOWN) {
		if (applet->panel_size == allocation->height)
			return;
		applet->panel_size = allocation->height;
	} else {
		if (applet->panel_size == allocation->width)
			return;
		applet->panel_size = allocation->width;
	}
	setup_applet_widgets (applet);
}

static void
change_orient_cb (PanelApplet *widget, PanelAppletOrient orient, GDictApplet *applet)
{
	applet->orient = orient;
	setup_applet_widgets (applet);
}

static gboolean
destroy_applet (GtkWidget *widget, gpointer data)
{
	GDictApplet *applet = data;

	if (applet->about_dialog)
		gtk_widget_destroy (applet->about_dialog);

	gtk_widget_destroy (applet->image_widget);
	gtk_widget_destroy (applet->entry_widget);
	gtk_widget_destroy (applet->box);
	gtk_widget_destroy (applet->applet_widget);
}

static void
text_received (GtkClipboard *clipboard, const gchar *text, gpointer data)
{
	GDictApplet *applet = data;
	gchar *lookup_text;
	
	if (!text)
		return;
	
	lookup_text = g_strdup_printf ("%s", text);
	g_strstrip (lookup_text);
	
	if (*lookup_text == 0)
        	return;

    	gtk_entry_set_text (GTK_ENTRY (applet->entry_widget), text);
    	entry_activate_cb (GTK_ENTRY (applet->entry_widget), applet);
    	g_free (lookup_text);
}

static void
lookup_selected_text (BonoboUIComponent *uic, gpointer data, const gchar *verbname)
{
	GDictApplet *applet = data;
	GtkClipboard *clipboard;

	clipboard = gtk_clipboard_get (GDK_SELECTION_PRIMARY);
 	gtk_clipboard_request_text (clipboard, text_received, applet);
 	
}

static void
about_cb (BonoboUIComponent *uic,
          gpointer           user_data,
          const gchar       *verbname)
{
	GDictApplet *applet = user_data;

	if (applet->about_dialog) {
		gtk_window_set_screen (GTK_WINDOW (applet->about_dialog),
				       gtk_widget_get_screen (applet->applet_widget));

		gtk_window_present (GTK_WINDOW (applet->about_dialog));
		return;
	}

	applet->about_dialog = gdict_about_new ();

	gtk_window_set_screen (GTK_WINDOW (applet->about_dialog),
			       gtk_widget_get_screen (applet->applet_widget));

	g_signal_connect (G_OBJECT (applet->about_dialog), "response",
			  G_CALLBACK (gtk_widget_destroy), NULL);
	g_signal_connect (G_OBJECT (applet->about_dialog), "destroy",
			  G_CALLBACK (gtk_widget_destroyed), &applet->about_dialog);

	gtk_widget_show (applet->about_dialog);
}

static void
help_cb (BonoboUIComponent *uic,
          gpointer           user_data,
          const gchar       *verbname)
{
	GError *error = NULL;
	gnome_help_display("gnome-dictionary",NULL,&error);

	if (error) {
		GtkWidget *dialog;

		dialog = gtk_message_dialog_new_with_markup (NULL, GTK_DIALOG_MODAL, 
							     GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
							     "<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
							     _("Could not display help"),
							     error->message);

		g_signal_connect (G_OBJECT (dialog), "response",
				  G_CALLBACK (gtk_widget_destroy),
				  NULL);

		gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
		gtk_widget_show (dialog);
		g_error_free (error);
	}
}

static void
prefs_cb (BonoboUIComponent *uic,
          gpointer           user_data,
          const gchar       *verbname)
{
	GDictApplet *applet = user_data;
	if (!applet->gdict->context)
		gdict_init_context (applet->gdict);

	if (!applet->pref_dialog)
		applet->pref_dialog = gdict_pref_dialog_new (applet->gdict, 
							     applet->gdict->pref);

	gtk_window_present (GTK_WINDOW (applet->pref_dialog));
}

static const BonoboUIVerb gdict_applet_menu_verbs [] = {
	BONOBO_UI_VERB ("sel", lookup_selected_text),
        BONOBO_UI_VERB ("prefs", prefs_cb),
        BONOBO_UI_VERB ("help", help_cb),
        BONOBO_UI_VERB ("about", about_cb),
        BONOBO_UI_VERB_END
};

static gboolean
gdict_applet_new (PanelApplet *parent_applet)
{
	GDictApplet *applet;
	GDictWindow *gdict;
	GtkWidget *window;
	BonoboUIComponent *component;
	GtkIconInfo *icon_info;

 	applet = g_new0 (GDictApplet, 1);

        applet->parent_applet = parent_applet;
	applet->applet_widget = GTK_WIDGET(parent_applet);
	applet->panel_size = panel_applet_get_size (PANEL_APPLET (applet->applet_widget));
 	applet->orient = panel_applet_get_orient (PANEL_APPLET (applet->applet_widget));
	
	gtk_window_set_default_icon_name ("gdict");
 	
	panel_applet_set_flags (parent_applet, PANEL_APPLET_EXPAND_MINOR);

	applet->tooltips = gtk_tooltips_new ();
	g_object_ref (applet->tooltips);
	gtk_object_sink (GTK_OBJECT (applet->tooltips));

	setup_applet_widgets (applet);

	window = gdict_window_new ();
	gdict = GDICT_WINDOW (window);

	gdict->applet = applet;
	applet->gdict = gdict;

 	panel_applet_setup_menu_from_file (PANEL_APPLET (applet->applet_widget),
					   NULL,
                            		   "GNOME_GDictApplet.xml",
					   NULL,
                            		   gdict_applet_menu_verbs,
                            		   applet);                               

	if (panel_applet_get_locked_down (PANEL_APPLET (applet->applet_widget))) {
		BonoboUIComponent *popup_component;

		popup_component = panel_applet_get_popup_component (PANEL_APPLET (applet->applet_widget));
		bonobo_ui_component_set_prop (popup_component,
					      "/commands/prefs",
					      "hidden", "1",
					      NULL);
	}

 	/* server will be contacted when an action is performed */
 	gdict->context = NULL;
	
	component = panel_applet_get_popup_component (PANEL_APPLET(applet->applet_widget));
	
 	g_signal_connect (G_OBJECT (applet->applet_widget), "size_allocate",
 			  G_CALLBACK (size_allocate_cb), applet);
 	g_signal_connect (G_OBJECT (applet->applet_widget), "change_orient",
 			  G_CALLBACK (change_orient_cb), applet);
	g_signal_connect (G_OBJECT (applet->applet_widget), "destroy",
			  G_CALLBACK (destroy_applet), applet);
	g_signal_connect (G_OBJECT (applet->applet_widget), "change_background",
			  G_CALLBACK (change_background), applet);

	return TRUE;
}

static gboolean
gdict_applet_factory (PanelApplet *applet,
                     const gchar          *iid,
                     gpointer              data)
{
        gboolean retval = FALSE;

        if (!strcmp (iid, "OAFIID:GNOME_GDictApplet"))
                retval = gdict_applet_new (applet);

        return retval;
}

PANEL_APPLET_BONOBO_FACTORY ("OAFIID:GNOME_GDictApplet_Factory",
			     PANEL_TYPE_APPLET,
                             "gnome-dictionary",
                             "0",
                             gdict_applet_factory,
                             NULL)
 	
