/* ATK -  Accessibility Toolkit
 * Copyright 2001 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "atkdocument.h"

GType
atk_document_get_type (void)
{
  static GType type = 0;

  if (!type) {
    static const GTypeInfo tinfo =
    {
      sizeof (AtkDocumentIface),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,

    };

    type = g_type_register_static (G_TYPE_INTERFACE, "AtkDocument", &tinfo, 0);
  }

  return type;
}

/**
 * atk_document_get_document_type:
 * @document: a #GObject instance that implements AtkDocumentIface
 *
 * Gets a string indicating the document type.
 *
 * Returns: a string indicating the document type
 **/
G_CONST_RETURN gchar*
atk_document_get_document_type (AtkDocument *document)
{
  AtkDocumentIface *iface;

  g_return_val_if_fail (ATK_IS_DOCUMENT (document), NULL);

  iface = ATK_DOCUMENT_GET_IFACE (document);

  if (iface->get_document_type)
    {
      return (iface->get_document_type) (document);
    }
  else
    {
      return NULL;
    }
}

/**
 * atk_document_get_document:
 * @document: a #GObject instance that implements AtkDocumentIface
 *
 * Gets a %gpointer that points to an instance of the DOM.  It is
 * up to the caller to check atk_document_get_type to determine
 * how to cast this pointer.
 *
 * Returns: a %gpointer that points to an instance of the DOM.
 **/
gpointer 
atk_document_get_document (AtkDocument *document)
{
  AtkDocumentIface *iface;

  g_return_val_if_fail (ATK_IS_DOCUMENT (document), NULL);

  iface = ATK_DOCUMENT_GET_IFACE (document);

  if (iface->get_document)
    {
      return (iface->get_document) (document);
    }
  else
    {
      return NULL;
    }
}

/**
 * atk_document_get_locale:
 * @document: a #GObject instance that implements AtkDocumentIface
 *
 * Gets a UTF-8 string indicating the POSIX-style LC_MESSAGES locale
 *          of the content of this document instance.  Individual
 *          text substrings or images within this document may have
 *          a different locale, see atk_text_get_attributes and
 *          atk_image_get_image_locale.
 *
 * Returns: a UTF-8 string indicating the POSIX-style LC_MESSAGES
 *          locale of the document content as a whole, or NULL if
 *          the document content does not specify a locale.
 **/
G_CONST_RETURN gchar *
atk_document_get_locale (AtkDocument *document)
{
  AtkDocumentIface *iface;

  g_return_val_if_fail (ATK_IS_DOCUMENT (document), NULL);

  iface = ATK_DOCUMENT_GET_IFACE (document);

  if (iface->get_document_locale)
    {
      return (iface->get_document_locale) (document);
    }
  else
    {
      return NULL;
    }
}


/**
 * atk_document_get_attributes:
 * @document: a #GObject instance that implements AtkDocumentIface
 *
 * Gets an AtkAttributeSet which describes document-wide
 *          attributes as name-value pairs.
 *
 * Returns: An AtkAttributeSet containing the explicitly
 *          set name-value-pair attributes associated with this document
 *          as a whole.
 * @Since: ATK 1.12
 **/
AtkAttributeSet *
atk_document_get_attributes (AtkDocument *document)
{
  AtkDocumentIface *iface;

  g_return_val_if_fail (ATK_IS_DOCUMENT (document), NULL);

  iface = ATK_DOCUMENT_GET_IFACE (document);

  if (iface->get_document_attributes)
    {
      return (iface->get_document_attributes) (document);
    }
  else
    {
      return NULL;
    }
}

/**
 * atk_document_get_attribute_value:
 * @document: a #GObject instance that implements AtkDocumentIface
 * @attribute_name: a character string representing the name of the attribute
 *            whose value is being queried.
 *
 * Returns: a string value associated with the named attribute for this
 *    document, or NULL if a value for #attribute_name has not been specified
 *    for this document.
 * @Since: ATK 1.12
 */
G_CONST_RETURN gchar *
atk_document_get_attribute_value (AtkDocument *document, 
				  const gchar *attribute_name)
{
  AtkDocumentIface *iface;

  g_return_val_if_fail (ATK_IS_DOCUMENT (document), NULL);

  iface = ATK_DOCUMENT_GET_IFACE (document);

  if (iface->get_document_attribute_value)
    {
      return (iface->get_document_attribute_value) (document, attribute_name);
    }
  else
    {
      return NULL;
    }
}

/**
 * atk_document_set_attribute_value:
 * @document: a #GObject instance that implements AtkDocumentIface
 * @attribute_name: a character string representing the name of the attribute
 *            whose value is being set.
 * @value: a string value to be associated with #attribute_name.
 *
 * Returns: TRUE if #value is successfully associated with #attribute_name
 *          for this document, FALSE otherwise (e.g. if the document does not
 *          allow the attribute to be modified).
 * @Since ATK 1.12
 */
gboolean
atk_document_set_attribute_value (AtkDocument *document, 
				  const gchar *attribute_name,
				  const gchar *value)
{
  AtkDocumentIface *iface;

  g_return_val_if_fail (ATK_IS_DOCUMENT (document), FALSE);

  iface = ATK_DOCUMENT_GET_IFACE (document);

  if (iface->set_document_attribute)
    {
      return (iface->set_document_attribute) (document, attribute_name, value);
    }
  else
    {
      return FALSE;
    }
}
