
/* GConf
 * Copyright (C) 1999 Red Hat Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gconf-value.h"
#include "gconf-error.h"
#include "gconf-schema.h"
#include <errno.h>
#include <string.h>
#include <stdlib.h>

/* Quick hack so I can mark strings */

#ifdef _ 
#warning "_ already defined"
#else
#define _(x) x
#endif

#ifdef N_ 
#warning "N_ already defined"
#else
#define N_(x) x
#endif

static void
set_string(gchar** dest, const gchar* src)
{
  if (*dest != NULL)
    g_free(*dest);

  *dest = src ? g_strdup(src) : NULL;
}

/*
 * Values
 */

GConfValue* 
g_conf_value_new(GConfValueType type)
{
  GConfValue* value;

  /* Probably want to use mem chunks here eventually. */
  value = g_new0(GConfValue, 1);

  value->type = type;

  /* the g_new0() is important: sets list type to invalid, NULLs all
     pointers */
  
  return value;
}

GConfValue* 
g_conf_value_new_from_string(GConfValueType type, const gchar* value_str,
                             GConfError** err)
{
  GConfValue* value;

  value = g_conf_value_new(type);

  switch (type)
    {
    case G_CONF_VALUE_INT:
      {
        char* endptr = NULL;
        glong result;

        errno = 0;
        result = strtol(value_str, &endptr, 10);

        if (endptr == value_str)
          {
            if (err)
              *err = g_conf_error_new(G_CONF_PARSE_ERROR,
                                      _("Didn't understand `%s' (expected integer)"),
                                      value_str);
            
            g_conf_value_destroy(value);
            value = NULL;
          }
        else if (errno == ERANGE)
          {
            if (err)
              *err = g_conf_error_new(G_CONF_PARSE_ERROR,
                                      _("Integer `%s' is too large or small"),
                                      value_str);
            g_conf_value_destroy(value);
            value = NULL;
          }
        else
          g_conf_value_set_int(value, result);
      }
      break;
    case G_CONF_VALUE_FLOAT:
      {
        gchar* endptr = 0;
        double num;
        num = g_strtod(value_str, &endptr);
        if (value_str != endptr)
          {
            g_conf_value_set_float(value, num);
          }
        else
          {
            if (err)
              *err = g_conf_error_new(G_CONF_PARSE_ERROR,
                                      _("Didn't understand `%s' (expected real number)"),
                                      value_str);
            
            g_conf_value_destroy(value);
            value = NULL;
          }
      }
      break;
    case G_CONF_VALUE_STRING:
      g_conf_value_set_string(value, value_str);
      break;
    case G_CONF_VALUE_BOOL:
      if (*value_str == 't' || *value_str == 'T' || *value_str == '1')
        g_conf_value_set_bool(value, TRUE);
      else if (*value_str == 'f' || *value_str == 'F' || *value_str == '0')
        g_conf_value_set_bool(value, FALSE);
      else
        {
          if (err)
            *err = g_conf_error_new(G_CONF_PARSE_ERROR,
                                    _("Didn't understand `%s' (expected true or false)"),
                                    value_str);

          g_conf_value_destroy(value);
          value = NULL;
        }
      break;
    case G_CONF_VALUE_LIST:
    case G_CONF_VALUE_PAIR:
    case G_CONF_VALUE_IGNORE_SUBSEQUENT:
    default:
      g_assert_not_reached();
      break;
    }

  return value;
}

GConfValue*
g_conf_value_new_list_from_string(GConfValueType list_type,
                                  const gchar* str)
{
  g_return_val_if_fail(list_type != G_CONF_VALUE_LIST, NULL);
  g_return_val_if_fail(list_type != G_CONF_VALUE_PAIR, NULL);

  /* FIXME (parse the string generated by g_conf_value_to_string();
     probably need to add string escaping to that function to make
     lists/pairs of string work */
  
  return NULL;
}

GConfValue*
g_conf_value_new_pair_from_string(GConfValueType car_type,
                                  GConfValueType cdr_type,
                                  const gchar* str)
{
  g_return_val_if_fail(car_type != G_CONF_VALUE_LIST, NULL);
  g_return_val_if_fail(car_type != G_CONF_VALUE_PAIR, NULL);

  /* FIXME (parse the string generated by g_conf_value_to_string();
     probably need to add string escaping to that function to make
     lists/pairs of string work */

  return NULL;
}

gchar*
g_conf_value_to_string(GConfValue* value)
{
  /* These strings shouldn't be translated; they're primarily 
     intended for machines to read, not humans, though I do
     use them in some debug spew
  */
  gchar* retval = NULL;

  switch (value->type)
    {
    case G_CONF_VALUE_INT:
      retval = g_malloc(64);
      g_snprintf(retval, 64, "%d", g_conf_value_int(value));
      break;
    case G_CONF_VALUE_FLOAT:
      retval = g_malloc(64);
      g_snprintf(retval, 64, "%g", g_conf_value_float(value));
      break;
    case G_CONF_VALUE_STRING:
      retval = g_strdup(g_conf_value_string(value));
      break;
    case G_CONF_VALUE_BOOL:
      retval = g_conf_value_bool(value) ? g_strdup("true") : g_strdup("false");
      break;
    case G_CONF_VALUE_LIST:
      {
        GSList* list;

        list = g_conf_value_list(value);

        if (list == NULL)
          retval = g_strdup("[]");
        else
          {
            gchar* buf;
            guint bufsize = 128;
            guint cur = 0;

            g_assert(list != NULL);
            
            buf = g_malloc(bufsize);
            
            buf[0] = '[';
            ++cur;

            while (list != NULL)
              {
                gchar* elem;
                guint len;
                
                elem = g_conf_value_to_string((GConfValue*)list->data);

                g_assert(elem != NULL);

                len = strlen(elem);

                if ((cur + len) >= bufsize)
                  {
                    bufsize *= 2;
                    buf = g_realloc(buf, bufsize);
                    buf[bufsize-1] = '\0'; /* paranoia */
                  }
                
                strcpy(&buf[cur], elem);
                cur += len;

                g_free(elem);

                buf[cur] = ',';
                ++cur;
                
                list = g_slist_next(list);
              }

            buf[cur-1] = ']'; /* overwrites last comma */
            buf[cur] = '\0';

            retval = buf;
          }
      }
      break;
    case G_CONF_VALUE_PAIR:
      {
        gchar* car;
        gchar* cdr;

        car = g_conf_value_to_string(g_conf_value_car(value));
        cdr = g_conf_value_to_string(g_conf_value_cdr(value));
        retval = g_strdup_printf("(%s, %s)", car, cdr);
        g_free(car);
        g_free(cdr);
      }
      break;
      /* These remaining shouldn't really be used outside of debug spew... */
    case G_CONF_VALUE_INVALID:
      retval = g_strdup("Invalid");
      break;
    case G_CONF_VALUE_SCHEMA:
      retval = g_strdup("Schema");
      break;
    case G_CONF_VALUE_IGNORE_SUBSEQUENT:
      retval = g_strdup("Ignore Subsequent");
      break;
    default:
      g_assert_not_reached();
      break;
    }

  return retval;
}

GConfValue* 
g_conf_value_copy(GConfValue* src)
{
  GConfValue* dest;

  g_return_val_if_fail(src != NULL, NULL);

  dest = g_conf_value_new(src->type);

  switch (src->type)
    {
    case G_CONF_VALUE_INT:
    case G_CONF_VALUE_FLOAT:
    case G_CONF_VALUE_BOOL:
    case G_CONF_VALUE_INVALID:
      dest->d = src->d;
      break;
    case G_CONF_VALUE_STRING:
      set_string(&dest->d.string_data, src->d.string_data);
      break;
    case G_CONF_VALUE_SCHEMA:
      if (src->d.schema_data)
        dest->d.schema_data = g_conf_schema_copy(src->d.schema_data);
      else
        dest->d.schema_data = NULL;
      break;
      
    case G_CONF_VALUE_LIST:
      {
        GSList* copy = NULL;
        GSList* tmp = src->d.list_data.list;

        while (tmp != NULL)
          {
            copy = g_slist_prepend(copy, g_conf_value_copy(tmp->data));

            tmp = g_slist_next(tmp);
          }
        
        copy = g_slist_reverse(copy);

        dest->d.list_data.list = copy;
        dest->d.list_data.type = src->d.list_data.type;
      }
      break;
      
    case G_CONF_VALUE_PAIR:

      if (src->d.pair_data.car)
        dest->d.pair_data.car = g_conf_value_copy(src->d.pair_data.car);
      else
        dest->d.pair_data.car = NULL;

      if (src->d.pair_data.cdr)
        dest->d.pair_data.cdr = g_conf_value_copy(src->d.pair_data.cdr);
      else
        dest->d.pair_data.cdr = NULL;

      break;

    case G_CONF_VALUE_IGNORE_SUBSEQUENT:
      break;
      
    default:
      g_assert_not_reached();
    }
  
  return dest;
}

static void
g_conf_value_free_list(GConfValue* value)
{
  GSList* tmp;
  g_return_if_fail(value != NULL);
  g_return_if_fail(value->type != G_CONF_VALUE_LIST);
  
  tmp = value->d.list_data.list;

  while (tmp != NULL)
    {
      g_conf_value_destroy(tmp->data);
      
      tmp = g_slist_next(tmp);
    }
  g_slist_free(value->d.list_data.list);

  value->d.list_data.list = NULL;
}

void 
g_conf_value_destroy(GConfValue* value)
{
  g_return_if_fail(value != NULL);

  switch (value->type)
    {
    case G_CONF_VALUE_STRING:
      if (value->d.string_data != NULL)
        g_free(value->d.string_data);
      break;
    case G_CONF_VALUE_SCHEMA:
      if (value->d.schema_data != NULL)
        g_conf_schema_destroy(value->d.schema_data);
      break;
    case G_CONF_VALUE_LIST:
      g_conf_value_free_list(value);
      break;
    case G_CONF_VALUE_PAIR:
      if (value->d.pair_data.car != NULL)
        g_conf_value_destroy(value->d.pair_data.car);

      if (value->d.pair_data.cdr != NULL)
        g_conf_value_destroy(value->d.pair_data.cdr);
      break;
    default:
      break;
    }
  
  g_free(value);
}

void        
g_conf_value_set_int(GConfValue* value, gint the_int)
{
  g_return_if_fail(value != NULL);
  g_return_if_fail(value->type == G_CONF_VALUE_INT);

  value->d.int_data = the_int;
}

void        
g_conf_value_set_string(GConfValue* value, const gchar* the_str)
{
  g_return_if_fail(value != NULL);
  g_return_if_fail(value->type == G_CONF_VALUE_STRING);

  set_string(&value->d.string_data, the_str);
}

void        
g_conf_value_set_float(GConfValue* value, gdouble the_float)
{
  g_return_if_fail(value != NULL);
  g_return_if_fail(value->type == G_CONF_VALUE_FLOAT);

  value->d.float_data = the_float;
}

void        
g_conf_value_set_bool(GConfValue* value, gboolean the_bool)
{
  g_return_if_fail(value != NULL);
  g_return_if_fail(value->type == G_CONF_VALUE_BOOL);

  value->d.bool_data = the_bool;
}

void       
g_conf_value_set_schema(GConfValue* value, GConfSchema* sc)
{
  g_return_if_fail(value != NULL);
  g_return_if_fail(value->type == G_CONF_VALUE_SCHEMA);
  
  if (value->d.schema_data != NULL)
    g_conf_schema_destroy(value->d.schema_data);

  value->d.schema_data = g_conf_schema_copy(sc);
}

void        
g_conf_value_set_schema_nocopy(GConfValue* value, GConfSchema* sc)
{
  g_return_if_fail(value != NULL);
  g_return_if_fail(value->type == G_CONF_VALUE_SCHEMA);
  g_return_if_fail(sc != NULL);
  
  if (value->d.schema_data != NULL)
    g_conf_schema_destroy(value->d.schema_data);

  value->d.schema_data = sc;
}

void
g_conf_value_set_car(GConfValue* value, GConfValue* car)
{
  g_return_if_fail(car != NULL);

  g_conf_value_set_car_nocopy(value, g_conf_value_copy(car));
}

void
g_conf_value_set_car_nocopy(GConfValue* value, GConfValue* car)
{
  g_return_if_fail(value != NULL);
  g_return_if_fail(value->type == G_CONF_VALUE_PAIR);
  g_return_if_fail(car != NULL);
  
  if (value->d.pair_data.car != NULL)
    g_conf_value_destroy(value->d.pair_data.car);

  value->d.pair_data.car = car;
}

void
g_conf_value_set_cdr(GConfValue* value, GConfValue* cdr)
{
  g_return_if_fail(cdr != NULL);

  g_conf_value_set_cdr_nocopy(value, g_conf_value_copy(cdr));
}

void
g_conf_value_set_cdr_nocopy(GConfValue* value, GConfValue* cdr)
{
  g_return_if_fail(value != NULL);
  g_return_if_fail(value->type == G_CONF_VALUE_PAIR);
  g_return_if_fail(cdr != NULL);
  
  if (value->d.pair_data.cdr != NULL)
    g_conf_value_destroy(value->d.pair_data.cdr);

  value->d.pair_data.cdr = cdr;
}

void
g_conf_value_set_list_type(GConfValue* value,
                           GConfValueType type)
{
  g_return_if_fail(value != NULL);
  g_return_if_fail(value->type == G_CONF_VALUE_LIST);
  g_return_if_fail(type != G_CONF_VALUE_LIST);
  g_return_if_fail(type != G_CONF_VALUE_PAIR);
  /* If the list is non-NULL either we already have the right
     type, or we shouldn't be changing it without deleting
     the list first. */
  g_return_if_fail(value->d.list_data.list == NULL);

  value->d.list_data.type = type;
}

void
g_conf_value_set_list_nocopy(GConfValue* value,
                             GSList* list)
{
  g_return_if_fail(value != NULL);
  g_return_if_fail(value->type == G_CONF_VALUE_LIST);
  g_return_if_fail(value->d.list_data.type != G_CONF_VALUE_INVALID);
  
  if (value->d.list_data.list)
    g_conf_value_free_list(value);

  value->d.list_data.list = list;
}

/*
 * GConfMetaInfo
 */

GConfMetaInfo*
g_conf_meta_info_new(void)
{
  GConfMetaInfo* gcmi;

  gcmi = g_new0(GConfMetaInfo, 1);

  /* pointers and time are NULL/0 */
  
  return gcmi;
}

void
g_conf_meta_info_destroy(GConfMetaInfo* gcmi)
{
  g_free(gcmi);
}

void
g_conf_meta_info_set_schema  (GConfMetaInfo* gcmi,
                              const gchar* schema_name)
{
  set_string(&gcmi->schema, schema_name);
}

void
g_conf_meta_info_set_mod_user(GConfMetaInfo* gcmi,
                              const gchar* mod_user)
{
  set_string(&gcmi->mod_user, mod_user);
}

void
g_conf_meta_info_set_mod_time(GConfMetaInfo* gcmi,
                              GTime mod_time)
{
  gcmi->mod_time = mod_time;
}

/*
 * GConfEntry
 */

GConfEntry* 
g_conf_entry_new(gchar* key, GConfValue* val)
{
  GConfEntry* pair;

  pair = g_new(GConfEntry, 1);

  pair->key   = key;
  pair->value = val;

  return pair;
}

void
g_conf_entry_destroy(GConfEntry* pair)
{
  g_free(pair->key);
  g_conf_value_destroy(pair->value);
  g_free(pair);
}
