/* -*- Mode: C; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* NetworkManager -- Network link manager
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Copyright (C) 2008 Red Hat, Inc.
 */

#include <glib.h>
#include <string.h>

#include "NetworkManager.h"
#include "nm-dbus-manager.h"
#include "nm-dhcp4-config.h"
#include "nm-dhcp4-config-glue.h"
#include "nm-dbus-glib-types.h"
#include "nm-properties-changed-signal.h"
#include "nm-utils.h"


G_DEFINE_TYPE (NMDHCP4Config, nm_dhcp4_config, G_TYPE_OBJECT)

#define NM_DHCP4_CONFIG_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), NM_TYPE_DHCP4_CONFIG, NMDHCP4ConfigPrivate))

typedef struct {
	char *dbus_path;
	GHashTable *options;
} NMDHCP4ConfigPrivate;


enum {
	PROP_0,
	PROP_OPTIONS,

	LAST_PROP
};

enum {
	PROPERTIES_CHANGED,

	LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };


NMDHCP4Config *
nm_dhcp4_config_new (void)
{
	return NM_DHCP4_CONFIG (g_object_new (NM_TYPE_DHCP4_CONFIG, NULL));
}

void
nm_dhcp4_config_add_option (NMDHCP4Config *self,
                            const char *key,
                            const char *option)
{
	GValue *svalue;

	g_return_if_fail (NM_IS_DHCP4_CONFIG (self));
	g_return_if_fail (key != NULL);
	g_return_if_fail (option != NULL);

	svalue = g_slice_new0 (GValue);
	g_value_init (svalue, G_TYPE_STRING);
	g_value_set_string (svalue, option);
	g_hash_table_insert (NM_DHCP4_CONFIG_GET_PRIVATE (self)->options, g_strdup (key), svalue);
	g_object_notify (G_OBJECT (self), NM_DHCP4_CONFIG_OPTIONS);
}

void
nm_dhcp4_config_reset (NMDHCP4Config *self)
{
	g_return_if_fail (NM_IS_DHCP4_CONFIG (self));

	g_hash_table_remove_all (NM_DHCP4_CONFIG_GET_PRIVATE (self)->options);
	g_object_notify (G_OBJECT (self), NM_DHCP4_CONFIG_OPTIONS);
}

const char *
nm_dhcp4_config_get_option (NMDHCP4Config *self, const char *key)
{
	GValue *value;

	g_return_val_if_fail (NM_IS_DHCP4_CONFIG (self), NULL);
	g_return_val_if_fail (key != NULL, NULL);

	value = g_hash_table_lookup (NM_DHCP4_CONFIG_GET_PRIVATE (self)->options, key);
	return value ? g_value_get_string (value) : NULL;
}

const char *
nm_dhcp4_config_get_dbus_path (NMDHCP4Config *self)
{
	g_return_val_if_fail (NM_IS_DHCP4_CONFIG (self), NULL);

	return NM_DHCP4_CONFIG_GET_PRIVATE (self)->dbus_path;
}

static void
nm_gvalue_destroy (gpointer data)
{
	GValue *value = (GValue *) data;

	g_value_unset (value);
	g_slice_free (GValue, value);
}

static void
nm_dhcp4_config_init (NMDHCP4Config *self)
{
	NMDHCP4ConfigPrivate *priv = NM_DHCP4_CONFIG_GET_PRIVATE (self);
	static guint32 counter = 0;
	DBusGConnection *connection;
	NMDBusManager *dbus_mgr;

	dbus_mgr = nm_dbus_manager_get ();
	connection = nm_dbus_manager_get_connection (dbus_mgr);
	priv->dbus_path = g_strdup_printf (NM_DBUS_PATH "/DHCP4Config/%d", counter++);
	dbus_g_connection_register_g_object (connection, priv->dbus_path, G_OBJECT (self));
	g_object_unref (dbus_mgr);

	priv->options = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, nm_gvalue_destroy);
}

static void
finalize (GObject *object)
{
	NMDHCP4ConfigPrivate *priv = NM_DHCP4_CONFIG_GET_PRIVATE (object);

	g_free (priv->dbus_path);
	g_hash_table_destroy (priv->options);
}

static void
get_property (GObject *object, guint prop_id,
			  GValue *value, GParamSpec *pspec)
{
	NMDHCP4ConfigPrivate *priv = NM_DHCP4_CONFIG_GET_PRIVATE (object);

	switch (prop_id) {
	case PROP_OPTIONS:
		g_value_set_boxed (value, priv->options);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

static void
nm_dhcp4_config_class_init (NMDHCP4ConfigClass *config_class)
{
	GObjectClass *object_class = G_OBJECT_CLASS (config_class);

	g_type_class_add_private (config_class, sizeof (NMDHCP4ConfigPrivate));

	/* virtual methods */
	object_class->get_property = get_property;
	object_class->finalize = finalize;

	/* properties */
	g_object_class_install_property
		(object_class, PROP_OPTIONS,
		 g_param_spec_boxed (NM_DHCP4_CONFIG_OPTIONS,
		                     "Options",
		                     "DHCP configuration options returned by the server",
		                     DBUS_TYPE_G_MAP_OF_VARIANT,
		                     G_PARAM_READABLE));

	/* Signals */
	signals[PROPERTIES_CHANGED] = 
		nm_properties_changed_signal_new (object_class,
								    G_STRUCT_OFFSET (NMDHCP4ConfigClass, properties_changed));

	dbus_g_object_type_install_info (G_TYPE_FROM_CLASS (config_class),
									 &dbus_glib_nm_dhcp4_config_object_info);
}
