/* SPDX-License-Identifier: GPL-2.0-or-later */
/*
 * Copyright (C) 2015 - 2018 Red Hat, Inc.
 */

#include "src/core/nm-default-daemon.h"

#include <linux/rtnetlink.h>

#include "libnm-platform/nm-platform-utils.h"
#include "libnm-platform/nm-linux-platform.h"

#include "nm-test-utils-core.h"

/*****************************************************************************/

G_STATIC_ASSERT(G_STRUCT_OFFSET(NMPlatformIPAddress, address_ptr)
                == G_STRUCT_OFFSET(NMPlatformIP4Address, address));
G_STATIC_ASSERT(G_STRUCT_OFFSET(NMPlatformIPAddress, address_ptr)
                == G_STRUCT_OFFSET(NMPlatformIP6Address, address));

G_STATIC_ASSERT(G_STRUCT_OFFSET(NMPlatformIPRoute, network_ptr)
                == G_STRUCT_OFFSET(NMPlatformIP4Route, network));
G_STATIC_ASSERT(G_STRUCT_OFFSET(NMPlatformIPRoute, network_ptr)
                == G_STRUCT_OFFSET(NMPlatformIP6Route, network));

/*****************************************************************************/

static void
test_init_linux_platform(void)
{
    gs_unref_object NMPlatform *platform = NULL;

    platform = nm_linux_platform_new(TRUE, NM_PLATFORM_NETNS_SUPPORT_DEFAULT, TRUE);
}

/*****************************************************************************/

static void
test_link_get_all(void)
{
    gs_unref_object NMPlatform  *platform = NULL;
    gs_unref_ptrarray GPtrArray *links    = NULL;

    platform = nm_linux_platform_new(TRUE, NM_PLATFORM_NETNS_SUPPORT_DEFAULT, TRUE);

    links = nm_platform_link_get_all(platform, TRUE);
}

/*****************************************************************************/

static void
test_nm_platform_link_flags2str(void)
{
    int i;

    for (i = 0; i < 100; i++) {
        char        buf[NM_PLATFORM_LINK_FLAGS2STR_MAX_LEN + 100];
        const char *s;
        const guint flags = ((i == 0) ? ~0u : nmtst_get_rand_uint());
        gsize       l;

        s = nm_platform_link_flags2str(flags, buf, sizeof(buf));
        g_assert(s);

        l = strlen(s);
        if (l > NM_PLATFORM_LINK_FLAGS2STR_MAX_LEN)
            g_error(
                "nm_platform_link_flags2str(%x) produced a longer output than %zu chars: \"%s\"",
                flags,
                NM_PLATFORM_LINK_FLAGS2STR_MAX_LEN,
                s);
        if (flags == ~0u && l != NM_PLATFORM_LINK_FLAGS2STR_MAX_LEN)
            g_error("nm_platform_link_flags2str(%x) is expected to produce %zu chars, but produced "
                    "%zu: \"%s\"",
                    flags,
                    NM_PLATFORM_LINK_FLAGS2STR_MAX_LEN,
                    l,
                    s);
    }
}

/*****************************************************************************/

static int
_address_pretty_sort_cmp(gconstpointer a, gconstpointer b, gpointer test_data)
{
    const int                   TEST_DATA_I      = GPOINTER_TO_INT(test_data);
    const int                   addr_family      = (TEST_DATA_I == 0 ? AF_INET : AF_INET6);
    const int                   IPV6_PREFER_TEMP = (TEST_DATA_I == 2);
    const NMPlatformIPXAddress *a_a              = a;
    const NMPlatformIPXAddress *a_b              = b;

    if (addr_family == AF_INET)
        return nm_platform_ip4_address_pretty_sort_cmp(&a_a->a4, &a_b->a4);

    return nm_platform_ip6_address_pretty_sort_cmp(&a_a->a6, &a_b->a6, IPV6_PREFER_TEMP);
}

static void
test_platform_ip_address_pretty_sort_cmp(gconstpointer test_data)
{
    static const char *const EXPECTED_BUFFER[3] = {
        [0] = ("8d1e69c83dc21056b973972c17a47631c169f11ff9119c40b407b6630000007a034f43340d01683c"
               "0045097aea4a849f060ddf57b24a5be9b10123e6e4f15ecd9c88b02a06d0dc5504b0595daf37deb4"
               "99996bfb667f072dec025d9cdc8a11f4409bcfb3c167090e00d90a8543961b2ccd47724a3c460ba8"
               "5183f4c5daae7d63ecc8fb1103000000fcebbaeb0c56535923f14874042a8aff1d018e5e000000cd"
               "36bbe3c9bb0ec36f00e007bb64a2827b7cdd38d0314c178e925b568ae56bc4e92931f2d303000000"
               "50b8d10ab700cc3a7cf51be0403b654ba2026808092069affaa100fdfdd68d3400184524a6bdc878"
               "9cf851782d895bcc8da98489c55b8c1230d2c50503000000210aeb0934cb2a247000b96d06992b6a"
               "5803d399000000d2df48f333ef467092008aa77d6732b678cf844ef5a943825fcd743f59bd14c89b"
               "df6f628a03000000f25d5895fb57fa60d9279fb2b8fcbac6ab0274f70000007bf6989d2d6a909e98"
               "00f7add5df2c93e05459507b5d8c6a0f466554d2ae4d52a89427c2260300000086ed27184b443181"
               "a5edc511c1507a141e03515600000033f57609453df7803300ae79989870ccec3def0ad92749e016"
               "663fe6ee0228c1da5f5b481b0300000068cb3bc873b04d937a6ed8f7bc51e5406603d0980000008b"
               "92d3adc69a84eb47002400207799416af1f0a086fbd7e2f7dea0077ae9c0c0d9f264af6703000000"
               "c38b6b8ee2614344bea73223e59bf0193402e9fa0000002d8ec7e4b89bf5a5d00076e66a9d80ab13"
               "2e1ac921eb76adbb229df32e561fa80aabf85f0c0300000008d33a2ea67776d7f88d69c8c7e3b2d3"
               "c902f05400000020abb42316c533d9c90088189c471a1ff2f0ce3ff66e782110125ae2ed74993ade"
               "507e308d030000008dab2dade2ab2fc9c56a7c86be5f96228e02f2f4000000b32b9c161c36b745ce"
               "00d06352cb2c354434819f4b248eb2b8e3ff1a5ffc386da337c27c15030000007e5c53e755c90fa0"
               "9e9653ddcb1b14d2750d8a130000001734b49e955cbcf46d006a1818053da43ec5085d1df74d1e28"
               "7271092897e26251de485b1303000000957c22721583c453d15ade4386ffd00b3600858900000013"
               "f09e601d88d4cb0c004d15c69e89f94ef7b30486591030639eb0e99f7f53702f3c84c4eb03000000"
               "87780281d47a32f2235376239daeaacb3c02864b000000ae91921681c2162b9bfa40546cdb544e44"
               "df2d7c2790d3119a051bb2ee8192ac0cfa3abc1e0300000075a2f42b50c6a36340132378b95c5931"
               "3b03ba640000002069e66142e22753344fda8a3e0b841cf76391dd68269b53eccb44a831714acfb5"
               "c442139803000000cee64b38b56e03e0f121014a2f70249eef027bce0000006f3973dc42a1003be9"
               "4f89d8eeb1f14d985d5bbe6c10085f3d6176f78a19bb8df1449a774903000000ec852d59f3d17232"
               "edd86ad6c3103a6884029aef00000082594f24a17dba1ee7e143fd52599364e13468f80fd514573f"
               "b572671cf6932f6909f4ecf103000000a1c748288ae21e371051852c972ea7954012884a00000045"
               "fd13a6826a4c619f9037d7a754655354c47cf0f551e96770a754ac19ef820fe0eafb1ac403000000"
               "3aa9986700b2a2b3ed257b8b0489f48f0500c8cac72105b327335a25a364d1015ffac03089f45539"
               "e1b0178462156cd6b83579b8da27345a4d9ca9d20300000074e786047d8683facd7db68b9ae586da"
               "cd03070c00000027c4dd3d031e33f22c636360d696be085aad7c499dd538d34574c038404923e75d"
               "36506ebd03000000ef369bb8ebfdcec3f8c8fa6867a982be8a054f85000000d82bc0ec7303f99f8f"
               "e8d07c84965a316af2abb982b001ec16901f55f960c55c22c896c9a7030000000a5f6abd850b06e3"
               "d9662795ce2cc1cee00002ab000000a318dc091e9aa1477f6fdc9830d7fbc95829a8838314dff34d"
               "24c332219a1b163a8483a2d0030000008ed3481e0bccc460e89f7abdb682380a6202e3740000004b"
               "a2b980b6aba92da6409969aabae87d60e00473755d31e5b2de252167c1c91b3a891715df03000000"
               "edae579e2e10c7f7a22ba0340436242cbe11248c000000174632d4f28aa28c08f80dd50c6712dfb4"
               "d184e820a28fa772a19cdf6dd1c173f6bc7ef08203000000ef60a8b540b26f66761d1f13badfad0f"
               "e8038f3c1aad2a82530b7447d9c2f401012f2f4c329f997184c9d2578e4cf9c48147507e1cd59ea1"
               "6e8e99cd03000000644f0e8ab3121cd5ef51e696c816290dba010e77000000e1530ff5397c9125f5"
               "9577d71c4b258a005116d11354edff62ceaa458fc75a91c45cfd5ce603000000152f2b70eaef7443"
               "e0f79ab6902dde53361dec71000000254ac9588319558041fc62ed564153e17478f8c3344d89c0bb"
               "5408c1b7d17fb084b16c7376030000005219061ab4c5c79489b2cc6a883c1469721dcb8b5adac8f5"
               "e6d66df46ea139107754dee62c36d2fc70ccc567df7a49b8a4c73cf5ffa4ffc8acdca10f03000000"
               "a36b178aabc78a4a0d80684ee5dbf45bba0210fb9733cc24904f288262e6ea77a22fa5f863eaeefa"
               "701d120523bd98f29b098b43cd68be6e37686b5e030000006215de4bb6f6f3d031a790287e45e9c1"
               "8b0236c0a25ea71f5daaed4d99eb2ebebe3c432fec7d3abc4ccf30d3aaaf02a4f05d7308b67f91bf"
               "0408423e03000000886dc91e2e5b0f0b65843fc961f7ad18010a3c67d9714f19f1e41c073164d93b"
               "1171719c7f81773302d007c983566139c695e0f4b661ee6b3cf09e8b02000000e79c10acfce165e3"
               "32a62384ec04e5bab40285ad00000071a36dd51323b2c54200154d12f86c260a9edfa7a74c1c83c1"
               "8e47935e8354846e27c219630200000054830c68e44ea74b4f2b83f9b3bee14d86034c9e000000f6"
               "abba4a32d635edea00fe423d01883918039249f398f9b37ea0914651b2a16e642c19a09202000000"
               "92defa453a3efc240b9977e166bee737fe10670c0000004c323b59b4cd20eb7d00d38785c88ffad4"
               "5e0dfe491d1ba96742c7b5e02b227122fcd32a7f0200000023a5223a2fb2c0764e08db62a3db07d0"
               "900260b922a7b4a1565e63d3b610c9df00e32703b0ed3a7d0f8b13cd736aaa72021a381fb50e2c42"
               "54270e46020000006a894e387625da376feecb80ac245409c005cc271d9c2f67179bff0644399ae7"
               "0064ed0432d599eaa660ac824b7fae389861419c50da49bf8c3b371502000000710c030690f5f18e"
               "a125dbf7d7e93bd65c01a56dfdcfc1155f236c8b9c79a62000660bbf024b03ff0a8e27c405e64244"
               "57393c6fd7ac5f99544e4b2702000000a7f810d249adc17647176b7c07d856e3b806efebc09dd9d6"
               "1966b7ae7412041d00fe7d361f2d5dfd66a3ae3939762df33a2d55060c78d551fcb6a03502000000"
               "9726d80d9c2d89df765b4d9a130257dc5f19f03a0000002bc9a89192c1c8be55281a59d1fd338f35"
               "7e75f8ce6e8cc34573654ea6624f138e8c45eb5002000000562dcd24a27c8212127da516b1091d1d"
               "4f0008170000000d32d823b0617718eb0f3195e2fe917423d314ae73572ff808b373f5b193d661ed"
               "e45aaf610200000040125e6baf4648cbaaa37b861e6d46dafe1f6ec400000053010911915ddb05f2"
               "a84e27c18a06e2c9cb29efdf672d2fa57fc85ebbe276c566804c122f0100000059747cf23a1349c8"
               "ab2dadc5a39411fd4f03116d000000316a553fc28cff0bcd00ebe434a7ae8ff040483e310819e3b1"
               "0db116431ec6f76949ba9be3010000000c67c3d4afa9c8649da90bd2fff2f2568101870b0000004b"
               "70c6d18d6242b356005e1a13d1111b067acbce3e3bcdf3824f1b134847ff26ba32d1d5f501000000"
               "bf7a70ba5193a162bc92ec1b11d9172cf8020347000000d8883b0f3fd6ed84d60059e553b887edcd"
               "61c2f5d2f381f843b809d259e499db7d2b092583010000009f9f0251becc987907879cca68fec7bb"
               "bb008edf000000973d136caca5054b010002a4c2167721673f2b02b3256ead1f2fea9bc12f0cfca1"
               "62fb81e701000000ca7feefe1933631e8e69f6d9d6962d2c6e0304f50000009e47150b9b14010469"
               "003acb72bb89182f931121966e91b9153b36bc60be5b534e1dff04520100000036674cdefb57b9ae"
               "54cfe8f42bf0e9381b00c4396de60f032bc7f34de295987100d4c0d4eb720c4ae3c9472e5f2da5d9"
               "c5cfd9f2343b21363f81268101000000585287dc6ca972a2cfb18b2b2cb749e2e002544d0eb4f73e"
               "75039fcf251b32fc00685b05ddd3aa9ba511d2e40edb4d758fb554158ae5c7c0fde69eec01000000"
               "bd68ab722af14548b29572e504265f6c72003e2200000079e208ed2e2ebb07710046a54cadb245ea"
               "fde6b02567f814de39dd8f88152a5845a4bab14801000000e6aae816430607ca0675e09818e9bbec"
               "5900c5900000006ab08940f6fc3bbdc700090f756aae660b3b02239efc1fb23d24438387def0f4c6"
               "cbb110fd01000000b9aca07b6e4925dd27943a272c171ed15a02be1c0000007b56e8f3fededea203"
               "00a8042602a1f8e6f5657f3f9e3eb807cbad76451055edb81539ce410100000018238487a417f3da"
               "01d99dae5f190096cc0e2cab0000005b2363d13edc5aa115165eb914d8fbe9a4fdb3d117d76b0de5"
               "416f95e7bb75d72fc8f1853b010000000e2b00b667b7b16dae02dfa0c26e7c3add003bd3000000f1"
               "2f2f3313d8469d09a92409c0b7f0318a575a4f8e061db3dd7fde25654a4059d5d8a4c5cf01000000"
               "fcc2056cc1ff8e51bbce426cfa4b861cc70192be7b14e7ba9c15acb881ae55f0f512dc1c0aa20624"
               "fbab244735d67c61283031667b2d74a103e0b1b10100000090a2312e17f1a35900459582f4ef43c7"
               "80038872000000efa9a89f87189b1d0afcdf6d9be94030b34774d1d7dddedd9899f0f627be16965a"
               "8704133101000000d9b4e3ce848518c4e3f6cbf25e5e1b990f026767000000bba23eb66e720da9e0"
               "7ac998b5690807d52617369ee1af4ca5a6a95581af5fd7ce732d803e010000001a0dd8c052ed6b13"
               "e2815a1a37a52bddd106f101000000ed72bb58f0e9201f17bd99fcf72909ac9e7a55299e9bb4fd53"
               "bc7417940fcffe3fb60291ea01000000def85b75827b26e8a0bef8675b29a197b71ecceaf5f1bb12"
               "c535256ab0916e5d58099ffc4a503a712350acbd48411f0dc15d2f0f49dad34565dbc8a900000000"
               "b550c5db56189572f32f4af595d785d5c117aad4000000f08bd7a09bdd214443008cd9bf66ec305a"
               "225a0c71b2ce78bb16104c8eaf18c56587e6f7bb00000000e612098610e52bb2a16a4008aefd545b"
               "7a019db50000008b362956cd46646a0e002160e5f769bb295766ed37a2123c5b6a813387e99bb834"
               "b7a350f900000000fa7db4532545666937c22a3b6e9686dcc403cbee000000529ed8901d7a0165fe"
               "00744edb6ba04f146e4c699341b95f41d9c1e67743efb54e4fc812bd000000006d064126666befbb"
               "98c39e7cc282208fef007ff40000009751adcb22fd50793f007a3c7dadd1c86e759d0c90bc512e13"
               "7dcef5e4a27985bd348352d7000000001ea012e5307294b6e346dac428ab20075e05e38d000000e6"
               "ccd2f50c30706cae00dcc2f0ee3d5e2612ea3b95e1d2a5d0fb95e8332fdfff650a6527f600000000"
               "aca0c7ee4197338581cafd63e66e3a23d61e257800000020aefd4503189f7630008e501c9052c22b"
               "9f3c39f24b0572b1de745cb25851429bf32099a8000000004d4647f53b1921af0f88e3759a08e7a5"
               "661f861eea1bf42ccd3c72f2ad0638570082ac5180101be4f85cef468ea086ea9aaafdc388811ec7"
               "3bbfb50100000000df6b3542f4bb5f574c5cf8107ba282f4f9038219000000772d08f0550b55726f"
               "2292362e5d68265d9f98c82d9b7a559be3acf4fc36fa6b513722d9b7000000001fb7c8307f06b4b2"
               "9088f20d9ac676d55b00ea34000000fec7c3523f76cf1ce34ca17a9a32f3f0f218424e48836dd1cb"
               "504e03d53e1124c5d9e1825f00000000284a024b7f9e85ff73e050a2cd89fd276e00cb8800000037"
               "4ec3127c750fabdd69121aec129cdc690fcc00ac2daa755e5292cd32dc096cd5438a72e100000000"
               "2495a609675344f7e2e3a5ebaec3c85f0a1242f7000000c70345132a61eda239d9d083c3bf085387"
               "046ef8a36f0e9e6982d15956ec2e2924ae6da48b00000000bcfe97e1761e12457ad742d3d7320a4f"
               "8800f47f5dd0cf69cb22840f09b3fe7749509cb6752b2cee30b7e7736a0afc9879ea40e69710fc9f"
               "57d6249b000000007dc7f17f666969e45c3baf7a119a65ee0b1ffa9e000000f2de6dc8dfcf2a3ad3"
               "bbcc0fa3b314f695bd82e9ea8891ed0a3d1b91caae8cb60bab90135f0000000064d210a256be75e0"
               "509c4477d19ac64b2e03220d6c0fc09c25d6534c809829af4a9df58dbfef186d416f3a1ef170d10f"
               "2c932e655203d82a87bb24e500000000e9cf9f6df9615189f78ee815b2781ea55c1755876b6a3f13"
               "922b6cb8360feaac1d5c302df59dd32a677e5a3af4fe297f608df061d45d864db442548600000000"
               "b0216050ed7afd741024e8365ea1aa11f90397c0c0ec468e9de5a9caf493472201e48d163106cc39"
               "7446a33fdc335c8b152c507a0a3c8e7cc67b2e48b47003c81785d3ffdbd9a617857ae8126b028c0e"
               "3e0f5354a9feb690c628d048eacce74500213cb1a84035b632ef3be6070dafa202187470e9da5570"
               "40e55b8b03000000c0cfb2b4386bec092fa5757ecde9348bda012ebda9fea6aa667224dae775e5cc"
               "0041aca2ff0f576767d3648102b61886b62f22ca33478e1ff9531cd903000000d241989477a363d5"
               "5ad47b8a11090df8da00d88da9fed75bc404156225b7eedefc2b44574b15e2ae496ad01bf007eacb"
               "0a28aec8682825108c54dda903000000d43e6505621b9a7f0422ae3fc8979ee0c400d794a9fe895e"
               "5bfe47c6f53a54e01b0c1d89414d94d0e02d2ec587fc330003781b0f294a2b7387bc45db03000000"
               "36bd39a44e8e8bc17c01ac63eca0c1cfbb004132a9fe13e4862573c2303dd1d65c7b2cffeca6d1ad"
               "accae11f8045a80f855ebf3b772eb2b1182c028703000000ec4501a1e27acfee7a8588ffd5a06cae"
               "4201e595a9fed65d67c624f285d7174177a47579dda0b6ead1e1b34901eaedd5bcd7be07d78e6222"
               "d611d7b103000000f9f23b5f1570a4648a0ff80dd9a37a82e10120aaa9fe258532449deb7afb217d"
               "58c0bb20de6f36d664873c5be82c5361358730bcde43782d87b8c3b9020000008982c42f53700a34"
               "37c5156e072b2f2fb300c9cfa9fe34ddb3980deb14ab7b5a001de913df1424ce3ee1d76fc1ea76e9"
               "8c9dfc78997ab53ec386e9c60200000010ee58bdc75183609985d627b6a4b1cbe402cbbba9fe3fd6"
               "a77eb3c5bf64b67100f18dbf463f9b8306c83bc0269779114bdade862c224f6fdd65aeb202000000"
               "902383028e3b47f2111d891d9f3c2c88cd007619a9fe44e644d7cf1d06b2b115d882549e62c12fba"
               "5b1cdec48d10bbb13b8313d8cd2a24d3be46bae5020000006665a268abfe8a41d149f074a378b881"
               "1500ce68a9fe5cec25a59273e7519f34c4458d70911714f1819aa46fbe143934c933ae6dec0fb124"
               "59970695020000000d3c723b91adb0da7c4aa7e7eb5a15bcde035fb9a9fe1fd84cc43c510385b9a4"
               "c2aac1d67a909b297c703915312e9c3c5a06c40b20ea99d5078a806d020000008bbfc9723107f3f5"
               "b2cd3dbddc74e26de905e48da9feb0313e7f8933e315130fa9a84c8288f83f6e70b1646d8026e9f1"
               "704f1b16286ba2da72437f9b01000000ec9e71ee4b951507beb42b3473cef00e89015f24a9fe2333"
               "02ad3b82d639272e0046eadfbd2146bf8cfdb205f90e54b58a6ee136a779f37c0d58fbcc01000000"
               "d0d44ff1e51c35157666c05348e6f507ff01e898a9fe9510b900b2061d0c334b009f2dd1cdf64b0f"
               "9a60e0e289f08db3e4bafc4e095a3230b4d9e89b010000006855676c277cf1bd017c9148da5892bb"
               "4903c3a7a9fe689d3391ce7d3fd48469005f0c233dbdd2a97835df1f1782dee88487fd0db5971b46"
               "9d82c85601000000050f63f8282f3464443cacf59c6379a44b0392f4a9fea98cb121c19c69d9942f"
               "000bcea75481f25e307d0de9a7afe283bd2f22ead05ff72045aaf95e01000000b9ef1b2110613fc9"
               "f90adac616a31dc2e4002345a9fe7151b38fc9c909cf8d71e64b0bd761fdf274fc5dca8ce1bd1871"
               "ae17bb451585609236ba0c700100000040318db913a353edc66efc068829c0e66a17143aa9feefb4"
               "6b382ab0404cdf029a6ce5d906837dbca6a5ee19d6f63fe9cbdd01f7246f13b2050424a200000000"
               "a029c896b4132bd895072cfeffe9d6812200bb02a9fe259b173c333cd03bc90500b899afeb760e3f"
               "004a2efc8ffc96b89b1bc8f415bd4e77c7a4c65600000000cd625009a40aeb62a42b6a62548e3c38"
               "b9022002a9fec5fbfaf1809becd2506300a6da29b2e94d3ab51fe0e9af98b180a36035fb6297dfa7"
               "0e09e2fc00000000cc807cdce682f80f94315c45ef817264c802736ea9fed637b077ddad3108be69"
               "f9b97d05c917ad6b10e693bb6e26f2ba90c8e909e85e20e591bf821c00000000c37ff4283a631480"
               "91acc42cac4ef3ad3702fec7a9fe62475588456e4ead7cbda620a3abae816e34cba1d74f18d1c4ed"
               "b99c0e49bcb946f71a64192d000000005eb888e9eb37bb046686943b101e1f553203abf8a9fe17c3"
               "641e447514ca565fb9b68c08a5e4351ea349e1ccf662e058b879a45176740744a19da86c00000000"
               "548759d313a5b92de36f90d4bc8b496afd03fe47a9fe370b6ec6584d64496089570968ead4d1ae91"
               "c819bb068196d599efde3246e43f5e7934f0386a000000004064f6b021281e235f94ae20ac26c7d4"
               "1f0350bfa9fe055bcfe3756ccdb3455c60ca7b11c66af76e80f062ea1a7e0b7be218d067530e1b04"),
        [1] = ("4d9ca9d20300000074e786047d8683facd7db68b9ae586dacd0b070c50320427c4dd3d031e33f22c"
               "636360d696be085aad7c499dd538d34574c038404923e75d30d2c50503000000210aeb0934cb2a24"
               "7000b96d06992b6a5875d3995b9663d2df48f333ef467092818aa77d6732b678cf844ef5a943825f"
               "cd743f59bd14c89bdf6f628a02000000f25d5895fb57fa60d9279fb2b8fcbac6ab6674f7a6df7979"
               "f6989d2d6a909e986ff7add5df2c93e05459507b5d8c6a0f466554d2ae4d52a8de485b1301000000"
               "957c22721583c453d15ade4386ffd00b36518589be395312f09e601d88d4cb0c2f4d15c69e89f94e"
               "f7b30486591030639eb0e99f7f53702fa4bab14801000000e6aae816430607ca0675e09818e9bbec"
               "5966c59068baf76ab08940f6fc3bbdc7f6090f756aae660b3b02239efc1fb23d24438387def0f4c6"
               "36ba0c700000000040318db913a353edc66efc068829c0e66a32143a0554efb46b382ab0404cdf02"
               "9a6ce5d906837dbca6a5ee19d6f63fe9cbdd01f7246f13b254270e46000000006a894e387625da37"
               "6feecb80ac245409c06bcc271d9c2f67179bff0644399ae79c64ed0432d599eaa660ac824b7fae38"
               "9861419c50da49bf5cfd5ce600000000152f2b70eaef7443e0f79ab6902dde53362fec7127e19f25"
               "4ac9588319558041fc62ed564153e17478f8c3344d89c0bb5408c1b7d17fb084b16c737600000000"
               "5219061ab4c5c79489b2cc6a883c14697280cb8b5adac8f5e6d66df46ea139107754dee62c36d2fc"
               "70ccc567df7a49b8a4c73cf5ffa4ffc83722d9b7000000001fb7c8307f06b4b29088f20d9ac676d5"
               "5b2dea34648cabfec7c3523f76cf1ce34ca17a9a32f3f0f218424e48836dd1cb504e03d53e1124c5"
               "87bb24e500000000e9cf9f6df9615189f78ee815b2781ea55c5f55876b6a3f13922b6cb8360feaac"
               "1d5c302df59dd32a677e5a3af4fe297f608df061d45d864d348352d7000000001ea012e5307294b6"
               "e346dac428ab20075e11e38dd16f8ee4ccd2f50c30706cae3fdcc2f0ee3d5e2612ea3b95e1d2a5d0"
               "fb95e8332fdfff65fa3abc1e0200000075a2f42b50c6a36340132378b95c59313b58ba64dbe99620"
               "69e66142e22753344fda8a3e0b841cf76391dd68269b53eccb44a831714acfb5f32099a800000000"
               "4d4647f53b1921af0f88e3759a08e7a56655861efedbf42ccd3c72f2ad063857ca82ac5180101be4"
               "f85cef468ea086ea9aaafdc388811ec7c442139803000000cee64b38b56e03e0f121014a2f70249e"
               "ef217bcefeec856e3973dc42a1003be94f89d8eeb1f14d985d5bbe6c10085f3d6176f78a19bb8df1"
               "09f4ecf100000000a1c748288ae21e371051852c972ea7954066884afed7b044fd13a6826a4c619f"
               "9037d7a754655354c47cf0f551e96770a754ac19ef820fe0732d803e000000001a0dd8c052ed6b13"
               "e2815a1a37a52bddd109f101fed87eed72bb58f0e9201f17bd99fcf72909ac9e7a55299e9bb4fd53"
               "bc7417940fcffe3feafb1ac4020000003aa9986700b2a2b3ed257b8b0489f48f055ac8cafea105b3"
               "27335a25a364d1015ffac03089f45539e1b0178462156cd6b83579b8da27345a544e4b2700000000"
               "a7f810d249adc17647176b7c07d856e3b87befebfe9dd9d61966b7ae7412041de5fe7d361f2d5dfd"
               "66a3ae3939762df33a2d55060c78d551dd65aeb202000000902383028e3b47f2111d891d9f3c2c88"
               "cd1676190000000000000000000000000000000162c12fba5b1cdec48d10bbb13b8313d8cd2a24d3"
               "27c219630100000054830c68e44ea74b4f2b83f9b3bee14d866e4c9e000000000000000000000000"
               "0000000101883918039249f398f9b37ea0914651b2a16e64a19da86c64fefab2548759d313a5b92d"
               "e36f90d4bc8b496afd57fe4700000000000000000000000000000000d4d1ae91c819bb068196d599"
               "efde3246e43f5e79507e308d030000008dab2dade2ab2fc9c56a7c86be5f96228e05f2f400000000"
               "000000000000000000000000cb2c354434819f4b248eb2b8e3ff1a5ffc386da38483a2d003000000"
               "8ed3481e0bccc460e89f7abdb682380a6208e37400000000000000000000000000000000bae87d60"
               "e00473755d31e5b2de252167c1c91b3a87e6f7bb03000000e612098610e52bb2a16a4008aefd545b"
               "7a219db500000000000000000000000000000000f769bb295766ed37a2123c5b6a813387e99bb834"
               "bc7ef08203000000ef60a8b540b26f66761d1f13badfad0fe8178f3c000000000000000000000000"
               "00000000329f997184c9d2578e4cf9c48147507e1cd59ea13f81268103000000585287dc6ca972a2"
               "cfb18b2b2cb749e2e022544d00000000000000000000000000000000ddd3aa9ba511d2e40edb4d75"
               "8fb554158ae5c7c09d82c85602000000050f63f8282f3464443cacf59c6379a44b8092f400000000"
               "0000000000000000000000005481f25e307d0de9a7afe283bd2f22ead05ff72045aaf95e02000000"
               "b9ef1b2110613fc9f90adac616a31dc2e44923450000000000000000000000000000000061fdf274"
               "fc5dca8ce1bd1871ae17bb4515856092cbb110fd02000000b9aca07b6e4925dd27943a272c171ed1"
               "5a66be1c0000000000000000000000000000000002a1f8e6f5657f3f9e3eb807cbad76451055edb8"
               "5f5b481b0100000068cb3bc873b04d937a6ed8f7bc51e5406674d098000000000000000000000000"
               "000000007799416af1f0a086fbd7e2f7dea0077ae9c0c0d9ecc8fb1101000000fcebbaeb0c565359"
               "23f14874042a8aff1d008e5e0000000000000000000000000000000064a2827b7cdd38d0314c178e"
               "925b568ae56bc4e9438a72e1000000002495a609675344f7e2e3a5ebaec3c85f0a5742f700000000"
               "000000000000000000000000bf085387046ef8a36f0e9e6982d15956ec2e2924891715df00000000"
               "edae579e2e10c7f7a22ba0340436242cbe53248c000000000000000000000000000000006712dfb4"
               "d184e820a28fa772a19cdf6dd1c173f6e45aaf610000000040125e6baf4648cbaaa37b861e6d46da"
               "fe556ec4000000000000000000000000000000008a06e2c9cb29efdf672d2fa57fc85ebbe276c566"
               "8c45eb5003000000562dcd24a27c8212127da516b1091d1d4f5f0817000000000000000000000000"
               "00000000fe917423d314ae73572ff808b373f5b193d661edbe46bae5030000006665a268abfe8a41"
               "d149f074a378b8811526ce6800000000000000000000000000000000911714f1819aa46fbe143934"
               "c933ae6dec0fb124fcd32a7f0300000023a5223a2fb2c0764e08db62a3db07d0907360b900000000"
               "000000000000000000000000b0ed3a7d0f8b13cd736aaa72021a381fb50e2c421dff045203000000"
               "36674cdefb57b9ae54cfe8f42bf0e9381b04c43900000000000000000000000000000000eb720c4a"
               "e3c9472e5f2da5d9c5cfd9f2343b21363c84c4eb0200000087780281d47a32f2235376239daeaacb"
               "3c54864b00000000000000000000000000000000db544e44df2d7c2790d3119a051bb2ee8192ac0c"
               "804c122f0100000059747cf23a1349c8ab2dadc5a39411fd4f60116d000000000000000000000000"
               "00000000a7ae8ff040483e310819e3b10db116431ec6f76987b8c3b9010000008982c42f53700a34"
               "37c5156e072b2f2fb372c9cf00000000000000000000000000000000df1424ce3ee1d76fc1ea76e9"
               "8c9dfc78997ab53ef9531cd901000000d241989477a363d55ad47b8a11090df8da18d88d00000000"
               "0000000000000000000000004b15e2ae496ad01bf007eacb0a28aec86828251065dbc8a900000000"
               "b550c5db56189572f32f4af595d785d5c10eaad40000000000000000000000000000000066ec305a"
               "225a0c71b2ce78bb16104c8eaf18c56557d6249b000000007dc7f17f666969e45c3baf7a119a65ee"
               "0b34fa9e00000000000000000000000000000000b314f695bd82e9ea8891ed0a3d1b91caae8cb60b"
               "fcb6a035000000009726d80d9c2d89df765b4d9a130257dc5f2bf03a000000000000000000000000"
               "00000000fd338f357e75f8ce6e8cc34573654ea6624f138e91bf821c03000000c37ff4283a631480"
               "91acc42cac4ef3ad373bfec700000000000000000000000000000002ae816e34cba1d74f18d1c4ed"
               "b99c0e49bcb946f734f0386a010000004064f6b021281e235f94ae20ac26c7d41f2a50bf00000000"
               "000000000000000000000002c66af76e80f062ea1a7e0b7be218d067530e1b04c386e9c601000000"
               "10ee58bdc75183609985d627b6a4b1cbe413cbbb00000000000000000000000000000002463f9b83"
               "06c83bc0269779114bdade862c224f6f1a64192d010000005eb888e9eb37bb046686943b101e1f55"
               "3273abf800000000000000000000000000000002a5e4351ea349e1ccf662e058b879a45176740744"
               "32d1d5f503000000bf7a70ba5193a162bc92ec1b11d9172cf83a0347000000000000000000000000"
               "00000002b887edcd61c2f5d2f381f843b809d259e499db7dfde69eec01000000bd68ab722af14548"
               "b29572e504265f6c726d3e2200000000000000000000ffffdb46a54cadb245eafde6b02567f814de"
               "39dd8f88152a584562fb81e702000000ca7feefe1933631e8e69f6d9d6962d2c6e5804f57239c09c"
               "47150b9b14010469823acb72bb89182f931121966e91b9153b36bc60be5b534e6e8e99cd01000000"
               "644f0e8ab3121cd5ef51e696c816290dba040e7726d082e1530ff5397c9125f59577d71c4b258a00"
               "5116d11354edff62ceaa458fc75a91c4b4d9e89b010000006855676c277cf1bd017c9148da5892bb"
               "4945c3a7ed55689d3391ce7d3fd48469845f0c233dbdd2a97835df1f1782dee88487fd0db5971b46"
               "2c19a0920000000092defa453a3efc240b9977e166bee737fe1c670c439a644c323b59b4cd20eb7d"
               "d3d38785c88ffad45e0dfe491d1ba96742c7b5e02b227122ab90135f0300000064d210a256be75e0"
               "509c4477d19ac64b2e64220d6c0fc09c25d6534c809829af4a9df58dbfef186d416f3a1ef170d10f"
               "2c932e655203d82a59970695030000000d3c723b91adb0da7c4aa7e7eb5a15bcde525fb98b841fd8"
               "4cc43c510385b9a4c2aac1d67a909b297c703915312e9c3c5a06c40b20ea99d58c3b371502000000"
               "710c030690f5f18ea125dbf7d7e93bd65c03a56dfdcfc1155f236c8b9c79a6205f660bbf024b03ff"
               "0a8e27c405e6424457393c6fd7ac5f9940e55b8b01000000c0cfb2b4386bec092fa5757ecde9348b"
               "da4c2ebd4f78a6aa667224dae775e5ccd141aca2ff0f576767d3648102b61886b62f22ca33478e1f"
               "abf85f0c0100000008d33a2ea67776d7f88d69c8c7e3b2d3c954f054e93f8120abb42316c533d9c9"
               "4d88189c471a1ff2f0ce3ff66e782110125ae2ed74993adeae6da48b00000000bcfe97e1761e1245"
               "7ad742d3d7320a4f8815f47f5dd0cf69cb22840f09b3fe7749509cb6752b2cee30b7e7736a0afc98"
               "79ea40e69710fc9f0408423e00000000886dc91e2e5b0f0b65843fc961f7ad18014c3c67d9714f19"
               "f1e41c073164d93b1171719c7f81773302d007c983566139c695e0f4b661ee6bc896c9a703000000"
               "0a5f6abd850b06e3d9662795ce2cc1cee00802abfec2aaa218dc091e9aa1477f6fdc9830d7fbc958"
               "29a8838314dff34d24c332219a1b163a3bbfb50103000000df6b3542f4bb5f574c5cf8107ba282f4"
               "f9188219fef93e742d08f0550b55726f2292362e5d68265d9f98c82d9b7a559be3acf4fc36fa6b51"
               "c8f1853b030000000e2b00b667b7b16dae02dfa0c26e7c3add113bd3febfa0f02f2f3313d8469d09"
               "a92409c0b7f0318a575a4f8e061db3dd7fde25654a4059d5b442548601000000b0216050ed7afd74"
               "1024e8365ea1aa11f92797c0feac468e9de5a9caf493472201e48d163106cc397446a33fdc335c8b"
               "152c507a0a3c8e7cacdca10f01000000a36b178aabc78a4a0d80684ee5dbf45bba3010fbfeb3cc24"
               "904f288262e6ea77a22fa5f863eaeefa701d120523bd98f29b098b43cd68be6e37c27c1500000000"
               "7e5c53e755c90fa09e9653ddcb1b14d2757f8a13feac3c1734b49e955cbcf46d256a1818053da43e"
               "c5085d1df74d1e287271092897e2625103e0b1b10100000090a2312e17f1a35900459582f4ef43c7"
               "807b887200000000000000000000000000000001e94030b34774d1d7dddedd9899f0f627be16965a"
               "87bc45db0100000036bd39a44e8e8bc17c01ac63eca0c1cfbb4f4132000000000000000000000000"
               "00000001eca6d1adaccae11f8045a80f855ebf3b772eb2b1449a774903000000ec852d59f3d17232"
               "edd86ad6c3103a68844c9aef00000000000000000000000000000001599364e13468f80fd514573f"
               "b572671cf6932f69d9e1825f03000000284a024b7f9e85ff73e050a2cd89fd276e0acb8800000000"
               "000000000000000000000001129cdc690fcc00ac2daa755e5292cd32dc096cd572437f9b02000000"
               "ec9e71ee4b951507beb42b3473cef00e896b5f2400000000000000000000000000000001bd2146bf"
               "8cfdb205f90e54b58a6ee136a779f37c2931f2d30300000050b8d10ab700cc3a7cf51be0403b654b"
               "a242680800000000000000000000000000000000a6bdc8789cf851782d895bcc8da98489c55b8c12"
               "37686b5e030000006215de4bb6f6f3d031a790287e45e9c18b7136c0000000000000000000000000"
               "00000000ec7d3abc4ccf30d3aaaf02a4f05d7308b67f91bfc7a4c65603000000cd625009a40aeb62"
               "a42b6a62548e3c38b96a200200000000000000000000000000000000b2e94d3ab51fe0e9af98b180"
               "a36035fb6297dfa78704133102000000d9b4e3ce848518c4e3f6cbf25e5e1b990f13676700000000"
               "000000000000000000000000690807d52617369ee1af4ca5a6a95581af5fd7ceb7a350f902000000"
               "fa7db4532545666937c22a3b6e9686dcc43ccbee000000000000000000000000000000006ba04f14"
               "6e4c699341b95f41d9c1e67743efb54e3cf09e8b01000000e79c10acfce165e332a62384ec04e5ba"
               "b46785ad00000000000000000000000000000000f86c260a9edfa7a74c1c83c18e47935e8354846e"
               "4fc812bd010000006d064126666befbb98c39e7cc282208fef4a7ff4000000000000000000000000"
               "00000000add1c86e759d0c90bc512e137dcef5e4a27985bdd8a4c5cf01000000fcc2056cc1ff8e51"
               "bbce426cfa4b861cc73992be000000000000000000000000000000000aa20624fbab244735d67c61"
               "283031667b2d74a10d58fbcc01000000d0d44ff1e51c35157666c05348e6f507ff72e89800000000"
               "000000000000000000000000cdf64b0f9a60e0e289f08db3e4bafc4e095a32302b09258301000000"
               "9f9f0251becc987907879cca68fec7bbbb4d8edf0000000000000000000000000000000016772167"
               "3f2b02b3256ead1f2fea9bc12f0cfca1078a806d000000008bbfc9723107f3f5b2cd3dbddc74e26d"
               "e956e48d0000000000000000000000000000000088f83f6e70b1646d8026e9f1704f1b16286ba2da"
               "0a6527f600000000aca0c7ee4197338581cafd63e66e3a23d6572578000000000000000000000000"
               "000000009052c22b9f3c39f24b0572b1de745cb25851429b8c54dda900000000d43e6505621b9a7f"
               "0422ae3fc8979ee0c400d79400000000000000000000000000000000414d94d0e02d2ec587fc3300"
               "03781b0f294a2b73050424a203000000a029c896b4132bd895072cfeffe9d6812232bb0200000000"
               "000000000000000000000000eb760e3f004a2efc8ffc96b89b1bc8f415bd4e77d611d7b103000000"
               "f9f23b5f1570a4648a0ff80dd9a37a82e10c20aa00000000000000000000000000000000de6f36d6"
               "64873c5be82c5361358730bcde43782df264af6703000000c38b6b8ee2614344bea73223e59bf019"
               "345ae9fa000000000000000000000000000000009d80ab132e1ac921eb76adbb229df32e561fa80a"
               "49ba9be3020000000c67c3d4afa9c8649da90bd2fff2f2568108870b000000000000000000000000"
               "00000000d1111b067acbce3e3bcdf3824f1b134847ff26ba0e09e2fc01000000cc807cdce682f80f"
               "94315c45ef817264c83c736e00000000000000000000000000000000c917ad6b10e693bb6e26f2ba"
               "90c8e909e85e20e5182c028701000000ec4501a1e27acfee7a8588ffd5a06cae4228e59500000000"
               "000000000000000000000000dda0b6ead1e1b34901eaedd5bcd7be07d78e62229427c22601000000"
               "86ed27184b443181a5edc511c1507a141e4a5156000000000000000000000000000000009870ccec"
               "3def0ad92749e016663fe6ee0228c1da1539ce410000000018238487a417f3da01d99dae5f190096"
               "cc292cab00000000000000000000000000000000d8fbe9a4fdb3d117d76b0de5416f95e7bb75d72f"
               "b60291ea00000000def85b75827b26e8a0bef8675b29a197b71bccea000000000000000000000000"
               "000000004a503a712350acbd48411f0dc15d2f0f49dad3458d1e69c801000000b973972c17a47631"
               "c169f11ff9119c40b47cb66300000000000000000000000000000002ea4a849f060ddf57b24a5be9"
               "b10123e6e4f15ecdc67b2e48000000001785d3ffdbd9a617857ae8126b028c0e3e4b535400000000"
               "000000000000000000000002a84035b632ef3be6070dafa202187470e9da55709c88b02a02000000"
               "04b0595daf37deb499996bfb667f072dec2c5d9c00000000000000000000ffff99d90a8543961b2c"
               "cd47724a3c460ba85183f4c5daae7d6336506ebd00000000ef369bb8ebfdcec3f8c8fa6867a982be"
               "8a534f8500000000000000000000ffffe8d07c84965a316af2abb982b001ec16901f55f960c55c22"),
        [2] = ("df6f628a02000000f25d5895fb57fa60d9279fb2b8fcbac6ab6674f7a6df7979f6989d2d6a909e98"
               "6ff7add5df2c93e05459507b5d8c6a0f466554d2ae4d52a8fa3abc1e0200000075a2f42b50c6a363"
               "40132378b95c59313b58ba64dbe9962069e66142e22753344fda8a3e0b841cf76391dd68269b53ec"
               "cb44a831714acfb554270e46000000006a894e387625da376feecb80ac245409c06bcc271d9c2f67"
               "179bff0644399ae79c64ed0432d599eaa660ac824b7fae389861419c50da49bfb16c737600000000"
               "5219061ab4c5c79489b2cc6a883c14697280cb8b5adac8f5e6d66df46ea139107754dee62c36d2fc"
               "70ccc567df7a49b8a4c73cf5ffa4ffc83722d9b7000000001fb7c8307f06b4b29088f20d9ac676d5"
               "5b2dea34648cabfec7c3523f76cf1ce34ca17a9a32f3f0f218424e48836dd1cb504e03d53e1124c5"
               "87bb24e500000000e9cf9f6df9615189f78ee815b2781ea55c5f55876b6a3f13922b6cb8360feaac"
               "1d5c302df59dd32a677e5a3af4fe297f608df061d45d864d4d9ca9d20300000074e786047d8683fa"
               "cd7db68b9ae586dacd0b070c50320427c4dd3d031e33f22c636360d696be085aad7c499dd538d345"
               "74c038404923e75d30d2c50503000000210aeb0934cb2a247000b96d06992b6a5875d3995b9663d2"
               "df48f333ef467092818aa77d6732b678cf844ef5a943825fcd743f59bd14c89bde485b1301000000"
               "957c22721583c453d15ade4386ffd00b36518589be395312f09e601d88d4cb0c2f4d15c69e89f94e"
               "f7b30486591030639eb0e99f7f53702fa4bab14801000000e6aae816430607ca0675e09818e9bbec"
               "5966c59068baf76ab08940f6fc3bbdc7f6090f756aae660b3b02239efc1fb23d24438387def0f4c6"
               "36ba0c700000000040318db913a353edc66efc068829c0e66a32143a0554efb46b382ab0404cdf02"
               "9a6ce5d906837dbca6a5ee19d6f63fe9cbdd01f7246f13b25cfd5ce600000000152f2b70eaef7443"
               "e0f79ab6902dde53362fec7127e19f254ac9588319558041fc62ed564153e17478f8c3344d89c0bb"
               "5408c1b7d17fb084348352d7000000001ea012e5307294b6e346dac428ab20075e11e38dd16f8ee4"
               "ccd2f50c30706cae3fdcc2f0ee3d5e2612ea3b95e1d2a5d0fb95e8332fdfff65c442139803000000"
               "cee64b38b56e03e0f121014a2f70249eef217bcefeec856e3973dc42a1003be94f89d8eeb1f14d98"
               "5d5bbe6c10085f3d6176f78a19bb8df109f4ecf100000000a1c748288ae21e371051852c972ea795"
               "4066884afed7b044fd13a6826a4c619f9037d7a754655354c47cf0f551e96770a754ac19ef820fe0"
               "f32099a8000000004d4647f53b1921af0f88e3759a08e7a56655861efedbf42ccd3c72f2ad063857"
               "ca82ac5180101be4f85cef468ea086ea9aaafdc388811ec7732d803e000000001a0dd8c052ed6b13"
               "e2815a1a37a52bddd109f101fed87eed72bb58f0e9201f17bd99fcf72909ac9e7a55299e9bb4fd53"
               "bc7417940fcffe3f544e4b2700000000a7f810d249adc17647176b7c07d856e3b87befebfe9dd9d6"
               "1966b7ae7412041de5fe7d361f2d5dfd66a3ae3939762df33a2d55060c78d551eafb1ac402000000"
               "3aa9986700b2a2b3ed257b8b0489f48f055ac8cafea105b327335a25a364d1015ffac03089f45539"
               "e1b0178462156cd6b83579b8da27345add65aeb202000000902383028e3b47f2111d891d9f3c2c88"
               "cd1676190000000000000000000000000000000162c12fba5b1cdec48d10bbb13b8313d8cd2a24d3"
               "27c219630100000054830c68e44ea74b4f2b83f9b3bee14d866e4c9e000000000000000000000000"
               "0000000101883918039249f398f9b37ea0914651b2a16e648483a2d0030000008ed3481e0bccc460"
               "e89f7abdb682380a6208e37400000000000000000000000000000000bae87d60e00473755d31e5b2"
               "de252167c1c91b3a8c45eb5003000000562dcd24a27c8212127da516b1091d1d4f5f081700000000"
               "000000000000000000000000fe917423d314ae73572ff808b373f5b193d661edbe46bae503000000"
               "6665a268abfe8a41d149f074a378b8811526ce6800000000000000000000000000000000911714f1"
               "819aa46fbe143934c933ae6dec0fb124fcd32a7f0300000023a5223a2fb2c0764e08db62a3db07d0"
               "907360b900000000000000000000000000000000b0ed3a7d0f8b13cd736aaa72021a381fb50e2c42"
               "1dff04520300000036674cdefb57b9ae54cfe8f42bf0e9381b04c439000000000000000000000000"
               "00000000eb720c4ae3c9472e5f2da5d9c5cfd9f2343b21363c84c4eb0200000087780281d47a32f2"
               "235376239daeaacb3c54864b00000000000000000000000000000000db544e44df2d7c2790d3119a"
               "051bb2ee8192ac0c804c122f0100000059747cf23a1349c8ab2dadc5a39411fd4f60116d00000000"
               "000000000000000000000000a7ae8ff040483e310819e3b10db116431ec6f76987b8c3b901000000"
               "8982c42f53700a3437c5156e072b2f2fb372c9cf00000000000000000000000000000000df1424ce"
               "3ee1d76fc1ea76e98c9dfc78997ab53eecc8fb1101000000fcebbaeb0c56535923f14874042a8aff"
               "1d008e5e0000000000000000000000000000000064a2827b7cdd38d0314c178e925b568ae56bc4e9"
               "f9531cd901000000d241989477a363d55ad47b8a11090df8da18d88d000000000000000000000000"
               "000000004b15e2ae496ad01bf007eacb0a28aec86828251065dbc8a900000000b550c5db56189572"
               "f32f4af595d785d5c10eaad40000000000000000000000000000000066ec305a225a0c71b2ce78bb"
               "16104c8eaf18c56557d6249b000000007dc7f17f666969e45c3baf7a119a65ee0b34fa9e00000000"
               "000000000000000000000000b314f695bd82e9ea8891ed0a3d1b91caae8cb60b891715df00000000"
               "edae579e2e10c7f7a22ba0340436242cbe53248c000000000000000000000000000000006712dfb4"
               "d184e820a28fa772a19cdf6dd1c173f6fcb6a035000000009726d80d9c2d89df765b4d9a130257dc"
               "5f2bf03a00000000000000000000000000000000fd338f357e75f8ce6e8cc34573654ea6624f138e"
               "a19da86c64fefab2548759d313a5b92de36f90d4bc8b496afd57fe47000000000000000000000000"
               "00000000d4d1ae91c819bb068196d599efde3246e43f5e79507e308d030000008dab2dade2ab2fc9"
               "c56a7c86be5f96228e05f2f400000000000000000000000000000000cb2c354434819f4b248eb2b8"
               "e3ff1a5ffc386da387e6f7bb03000000e612098610e52bb2a16a4008aefd545b7a219db500000000"
               "000000000000000000000000f769bb295766ed37a2123c5b6a813387e99bb834bc7ef08203000000"
               "ef60a8b540b26f66761d1f13badfad0fe8178f3c00000000000000000000000000000000329f9971"
               "84c9d2578e4cf9c48147507e1cd59ea13f81268103000000585287dc6ca972a2cfb18b2b2cb749e2"
               "e022544d00000000000000000000000000000000ddd3aa9ba511d2e40edb4d758fb554158ae5c7c0"
               "9d82c85602000000050f63f8282f3464443cacf59c6379a44b8092f4000000000000000000000000"
               "000000005481f25e307d0de9a7afe283bd2f22ead05ff72045aaf95e02000000b9ef1b2110613fc9"
               "f90adac616a31dc2e44923450000000000000000000000000000000061fdf274fc5dca8ce1bd1871"
               "ae17bb4515856092cbb110fd02000000b9aca07b6e4925dd27943a272c171ed15a66be1c00000000"
               "00000000000000000000000002a1f8e6f5657f3f9e3eb807cbad76451055edb85f5b481b01000000"
               "68cb3bc873b04d937a6ed8f7bc51e5406674d098000000000000000000000000000000007799416a"
               "f1f0a086fbd7e2f7dea0077ae9c0c0d9438a72e1000000002495a609675344f7e2e3a5ebaec3c85f"
               "0a5742f700000000000000000000000000000000bf085387046ef8a36f0e9e6982d15956ec2e2924"
               "e45aaf610000000040125e6baf4648cbaaa37b861e6d46dafe556ec4000000000000000000000000"
               "000000008a06e2c9cb29efdf672d2fa57fc85ebbe276c56691bf821c03000000c37ff4283a631480"
               "91acc42cac4ef3ad373bfec700000000000000000000000000000002ae816e34cba1d74f18d1c4ed"
               "b99c0e49bcb946f732d1d5f503000000bf7a70ba5193a162bc92ec1b11d9172cf83a034700000000"
               "000000000000000000000002b887edcd61c2f5d2f381f843b809d259e499db7d34f0386a01000000"
               "4064f6b021281e235f94ae20ac26c7d41f2a50bf00000000000000000000000000000002c66af76e"
               "80f062ea1a7e0b7be218d067530e1b04c386e9c60100000010ee58bdc75183609985d627b6a4b1cb"
               "e413cbbb00000000000000000000000000000002463f9b8306c83bc0269779114bdade862c224f6f"
               "1a64192d010000005eb888e9eb37bb046686943b101e1f553273abf8000000000000000000000000"
               "00000002a5e4351ea349e1ccf662e058b879a45176740744fde69eec01000000bd68ab722af14548"
               "b29572e504265f6c726d3e2200000000000000000000ffffdb46a54cadb245eafde6b02567f814de"
               "39dd8f88152a5845ab90135f0300000064d210a256be75e0509c4477d19ac64b2e64220d6c0fc09c"
               "25d6534c809829af4a9df58dbfef186d416f3a1ef170d10f2c932e655203d82a5997069503000000"
               "0d3c723b91adb0da7c4aa7e7eb5a15bcde525fb98b841fd84cc43c510385b9a4c2aac1d67a909b29"
               "7c703915312e9c3c5a06c40b20ea99d562fb81e702000000ca7feefe1933631e8e69f6d9d6962d2c"
               "6e5804f57239c09c47150b9b14010469823acb72bb89182f931121966e91b9153b36bc60be5b534e"
               "8c3b371502000000710c030690f5f18ea125dbf7d7e93bd65c03a56dfdcfc1155f236c8b9c79a620"
               "5f660bbf024b03ff0a8e27c405e6424457393c6fd7ac5f996e8e99cd01000000644f0e8ab3121cd5"
               "ef51e696c816290dba040e7726d082e1530ff5397c9125f59577d71c4b258a005116d11354edff62"
               "ceaa458fc75a91c440e55b8b01000000c0cfb2b4386bec092fa5757ecde9348bda4c2ebd4f78a6aa"
               "667224dae775e5ccd141aca2ff0f576767d3648102b61886b62f22ca33478e1fabf85f0c01000000"
               "08d33a2ea67776d7f88d69c8c7e3b2d3c954f054e93f8120abb42316c533d9c94d88189c471a1ff2"
               "f0ce3ff66e782110125ae2ed74993adeb4d9e89b010000006855676c277cf1bd017c9148da5892bb"
               "4945c3a7ed55689d3391ce7d3fd48469845f0c233dbdd2a97835df1f1782dee88487fd0db5971b46"
               "ae6da48b00000000bcfe97e1761e12457ad742d3d7320a4f8815f47f5dd0cf69cb22840f09b3fe77"
               "49509cb6752b2cee30b7e7736a0afc9879ea40e69710fc9f2c19a0920000000092defa453a3efc24"
               "0b9977e166bee737fe1c670c439a644c323b59b4cd20eb7dd3d38785c88ffad45e0dfe491d1ba967"
               "42c7b5e02b2271220408423e00000000886dc91e2e5b0f0b65843fc961f7ad18014c3c67d9714f19"
               "f1e41c073164d93b1171719c7f81773302d007c983566139c695e0f4b661ee6bc896c9a703000000"
               "0a5f6abd850b06e3d9662795ce2cc1cee00802abfec2aaa218dc091e9aa1477f6fdc9830d7fbc958"
               "29a8838314dff34d24c332219a1b163a3bbfb50103000000df6b3542f4bb5f574c5cf8107ba282f4"
               "f9188219fef93e742d08f0550b55726f2292362e5d68265d9f98c82d9b7a559be3acf4fc36fa6b51"
               "c8f1853b030000000e2b00b667b7b16dae02dfa0c26e7c3add113bd3febfa0f02f2f3313d8469d09"
               "a92409c0b7f0318a575a4f8e061db3dd7fde25654a4059d5acdca10f01000000a36b178aabc78a4a"
               "0d80684ee5dbf45bba3010fbfeb3cc24904f288262e6ea77a22fa5f863eaeefa701d120523bd98f2"
               "9b098b43cd68be6e37c27c15000000007e5c53e755c90fa09e9653ddcb1b14d2757f8a13feac3c17"
               "34b49e955cbcf46d256a1818053da43ec5085d1df74d1e287271092897e26251b442548601000000"
               "b0216050ed7afd741024e8365ea1aa11f92797c0feac468e9de5a9caf493472201e48d163106cc39"
               "7446a33fdc335c8b152c507a0a3c8e7c449a774903000000ec852d59f3d17232edd86ad6c3103a68"
               "844c9aef00000000000000000000000000000001599364e13468f80fd514573fb572671cf6932f69"
               "d9e1825f03000000284a024b7f9e85ff73e050a2cd89fd276e0acb88000000000000000000000000"
               "00000001129cdc690fcc00ac2daa755e5292cd32dc096cd572437f9b02000000ec9e71ee4b951507"
               "beb42b3473cef00e896b5f2400000000000000000000000000000001bd2146bf8cfdb205f90e54b5"
               "8a6ee136a779f37c87bc45db0100000036bd39a44e8e8bc17c01ac63eca0c1cfbb4f413200000000"
               "000000000000000000000001eca6d1adaccae11f8045a80f855ebf3b772eb2b103e0b1b101000000"
               "90a2312e17f1a35900459582f4ef43c7807b887200000000000000000000000000000001e94030b3"
               "4774d1d7dddedd9899f0f627be16965a050424a203000000a029c896b4132bd895072cfeffe9d681"
               "2232bb0200000000000000000000000000000000eb760e3f004a2efc8ffc96b89b1bc8f415bd4e77"
               "d611d7b103000000f9f23b5f1570a4648a0ff80dd9a37a82e10c20aa000000000000000000000000"
               "00000000de6f36d664873c5be82c5361358730bcde43782df264af6703000000c38b6b8ee2614344"
               "bea73223e59bf019345ae9fa000000000000000000000000000000009d80ab132e1ac921eb76adbb"
               "229df32e561fa80ac7a4c65603000000cd625009a40aeb62a42b6a62548e3c38b96a200200000000"
               "000000000000000000000000b2e94d3ab51fe0e9af98b180a36035fb6297dfa749ba9be302000000"
               "0c67c3d4afa9c8649da90bd2fff2f2568108870b00000000000000000000000000000000d1111b06"
               "7acbce3e3bcdf3824f1b134847ff26ba8704133102000000d9b4e3ce848518c4e3f6cbf25e5e1b99"
               "0f13676700000000000000000000000000000000690807d52617369ee1af4ca5a6a95581af5fd7ce"
               "b7a350f902000000fa7db4532545666937c22a3b6e9686dcc43ccbee000000000000000000000000"
               "000000006ba04f146e4c699341b95f41d9c1e67743efb54e0e09e2fc01000000cc807cdce682f80f"
               "94315c45ef817264c83c736e00000000000000000000000000000000c917ad6b10e693bb6e26f2ba"
               "90c8e909e85e20e5182c028701000000ec4501a1e27acfee7a8588ffd5a06cae4228e59500000000"
               "000000000000000000000000dda0b6ead1e1b34901eaedd5bcd7be07d78e62223cf09e8b01000000"
               "e79c10acfce165e332a62384ec04e5bab46785ad00000000000000000000000000000000f86c260a"
               "9edfa7a74c1c83c18e47935e8354846e4fc812bd010000006d064126666befbb98c39e7cc282208f"
               "ef4a7ff400000000000000000000000000000000add1c86e759d0c90bc512e137dcef5e4a27985bd"
               "9427c2260100000086ed27184b443181a5edc511c1507a141e4a5156000000000000000000000000"
               "000000009870ccec3def0ad92749e016663fe6ee0228c1da0d58fbcc01000000d0d44ff1e51c3515"
               "7666c05348e6f507ff72e89800000000000000000000000000000000cdf64b0f9a60e0e289f08db3"
               "e4bafc4e095a32302b092583010000009f9f0251becc987907879cca68fec7bbbb4d8edf00000000"
               "000000000000000000000000167721673f2b02b3256ead1f2fea9bc12f0cfca1078a806d00000000"
               "8bbfc9723107f3f5b2cd3dbddc74e26de956e48d0000000000000000000000000000000088f83f6e"
               "70b1646d8026e9f1704f1b16286ba2da0a6527f600000000aca0c7ee4197338581cafd63e66e3a23"
               "d6572578000000000000000000000000000000009052c22b9f3c39f24b0572b1de745cb25851429b"
               "1539ce410000000018238487a417f3da01d99dae5f190096cc292cab000000000000000000000000"
               "00000000d8fbe9a4fdb3d117d76b0de5416f95e7bb75d72fb60291ea00000000def85b75827b26e8"
               "a0bef8675b29a197b71bccea000000000000000000000000000000004a503a712350acbd48411f0d"
               "c15d2f0f49dad3452931f2d30300000050b8d10ab700cc3a7cf51be0403b654ba242680800000000"
               "000000000000000000000000a6bdc8789cf851782d895bcc8da98489c55b8c1237686b5e03000000"
               "6215de4bb6f6f3d031a790287e45e9c18b7136c000000000000000000000000000000000ec7d3abc"
               "4ccf30d3aaaf02a4f05d7308b67f91bfd8a4c5cf01000000fcc2056cc1ff8e51bbce426cfa4b861c"
               "c73992be000000000000000000000000000000000aa20624fbab244735d67c61283031667b2d74a1"
               "8c54dda900000000d43e6505621b9a7f0422ae3fc8979ee0c400d794000000000000000000000000"
               "00000000414d94d0e02d2ec587fc330003781b0f294a2b738d1e69c801000000b973972c17a47631"
               "c169f11ff9119c40b47cb66300000000000000000000000000000002ea4a849f060ddf57b24a5be9"
               "b10123e6e4f15ecdc67b2e48000000001785d3ffdbd9a617857ae8126b028c0e3e4b535400000000"
               "000000000000000000000002a84035b632ef3be6070dafa202187470e9da557036506ebd00000000"
               "ef369bb8ebfdcec3f8c8fa6867a982be8a534f8500000000000000000000ffffe8d07c84965a316a"
               "f2abb982b001ec16901f55f960c55c229c88b02a0200000004b0595daf37deb499996bfb667f072d"
               "ec2c5d9c00000000000000000000ffff99d90a8543961b2ccd47724a3c460ba85183f4c5daae7d63"),
    };
    const int   TEST_DATA_I      = GPOINTER_TO_INT(test_data);
    const int   addr_family      = (TEST_DATA_I == 0 ? AF_INET : AF_INET6);
    const int   IPV6_PREFER_TEMP = (TEST_DATA_I == 2);
    const guint N_ADDRESSES      = 100;
    const gsize ELM_SIZE =
        addr_family == AF_INET ? sizeof(NMPlatformIP4Address) : sizeof(NMPlatformIP6Address);
    const gboolean   DO_REGENERATE = FALSE;
    const gboolean   PRINT_RESULT  = DO_REGENERATE;
    const gboolean   CHECK_RESULT  = !DO_REGENERATE;
    gs_free guint8  *addresses     = NULL;
    gs_free guint64 *rand_map      = NULL;
    gsize            i, j;

#if !defined(__amd64__)
    /* The test generates a random array of NMPlatformIPXAddress (by crudely randomizing the memory,
     * not the structures themself) and then compares the sorted result with the expected output.
     * The sole purpose is to ensure that the sorting order stays stable.
     *
     * This only works on an architecture for which the test was made, otherwise
     * the expected data does not match (due to different layout of the structures
     * in memory).
     *
     * That's fine. Skip the test. */
    g_test_skip("skip test on non-amd64 architecture");
    return;
#endif

    /*
     * First we create a list of addresses filled with (stable) random bytes.
     * We tweak some fields explicitly (stable randomly), so that we cover all
     * relevant cases.
     *
     * Then we sort the list of addresses, and compare that the result is
     * as our EXPECTED_BUFFER.
     */

    addresses = g_malloc(ELM_SIZE * N_ADDRESSES);
    rand_map  = g_malloc(sizeof(rand_map[0]) * N_ADDRESSES);
    nmtst_stable_rand(258829693, addresses, ELM_SIZE * N_ADDRESSES);
    nmtst_stable_rand(710086081, rand_map, sizeof(rand_map[0]) * N_ADDRESSES);

    for (i = 0; i < N_ADDRESSES; i++) {
        NMPlatformIPXAddress *a  = (gpointer) (&addresses[i * ELM_SIZE]);
        guint64               r  = rand_map[i];
        guint64               r0 = r;
        struct in6_addr      *a6;

#define CONSUME_BITS(r, nbits)                        \
    ({                                                \
        guint64   _r     = (r);                       \
        const int _nbits = (nbits);                   \
                                                      \
        r >>= (_nbits);                               \
        (_r & (((((guint64) 1u) << (_nbits)) - 1u))); \
    })

        if (addr_family == AF_INET) {
            if (CONSUME_BITS(r, 1)) {
                /* randomly create a link-local address or not. */
                g_assert((NM_IPV4LL_NETWORK & ~NM_IPV4LL_NETMASK) == 0);
                a->a4.address = (a->a4.address & ~NM_IPV4LL_NETMASK) | NM_IPV4LL_NETWORK;
            }
        } else {
            a6 = &a->a6.address;
            switch (CONSUME_BITS(r, 4)) {
            case 0:
                a6->s6_addr32[0] = 0;
                a6->s6_addr32[1] = 0;
                a6->s6_addr32[2] = htonl(0xffff);
                g_assert(IN6_IS_ADDR_V4MAPPED(a6));
                break;
            case 1:
                a6->s6_addr32[0] = 0;
                a6->s6_addr32[1] = 0;
                a6->s6_addr32[2] = 0;
                a6->s6_addr32[3] = htonl(2);
                g_assert(IN6_IS_ADDR_V4COMPAT(a6));
                break;
            case 2:
                a6->s6_addr32[0] = 0;
                a6->s6_addr32[1] = 0;
                a6->s6_addr32[2] = 0;
                a6->s6_addr32[3] = 0;
                g_assert(IN6_IS_ADDR_UNSPECIFIED(a6));
                break;
            case 3:
                a6->s6_addr32[0] = 0;
                a6->s6_addr32[1] = 0;
                a6->s6_addr32[2] = 0;
                a6->s6_addr32[3] = htonl(1);
                g_assert(IN6_IS_ADDR_LOOPBACK(a6));
                break;
            case 4:
                a6->s6_addr32[0] = (a6->s6_addr32[0] & ~htonl(0xffc00000)) | htonl(0xfe800000);
                g_assert(IN6_IS_ADDR_LINKLOCAL(a6));
                break;
            case 6:
                a6->s6_addr32[0] = (a6->s6_addr32[0] & ~htonl(0xffc00000)) | htonl(0xfec00000);
                g_assert(IN6_IS_ADDR_SITELOCAL(a6));
                break;
            case 7:
            case 8:
            case 9:
            case 10:
                break;
            default:
                memset(a6, 0, sizeof(*a6));
                break;
            }
        }

        if (CONSUME_BITS(r, 5) != 0) {
            /* randomly make addr-source the same (so that several addresses compare
             * equal). */
            a->a4.addr_source = CONSUME_BITS(r, 2);
        }

        if (addr_family == AF_INET) {
            if (CONSUME_BITS(r, 1)) {
                /* randomly make the label empty or not. */
                a->a4.label[0] = '\0';
            }
        }
        if (a->ax.plen > 0)
            a->ax.plen = (r0 + 209284393u) % ((NM_IS_IPv4(addr_family) ? 32 : 128) + 1);
        if (addr_family == AF_INET) {
            if (CONSUME_BITS(r, 2) != 0) {
                a->a4.plen = CONSUME_BITS(r, 2);
            }
        }
        if (addr_family == AF_INET) {
            if (CONSUME_BITS(r, 1)) {
                a->a4.address = (a->a4.address & ~0xFFFFFFu) | (CONSUME_BITS(r, 2) << 24);
            }
        }
    }

    g_qsort_with_data(addresses,
                      N_ADDRESSES,
                      ELM_SIZE,
                      _address_pretty_sort_cmp,
                      (gpointer) test_data);

    for (i = 0; i < N_ADDRESSES; i++) {
        const NMPlatformIPXAddress *a = (gconstpointer) (&addresses[i * ELM_SIZE]);

        for (j = i + 1; j < N_ADDRESSES; j++) {
            const NMPlatformIPXAddress *b = (gconstpointer) (&addresses[j * ELM_SIZE]);
            int                         c1;
            int                         c2;

            if (addr_family == AF_INET) {
                c1 = nm_platform_ip4_address_pretty_sort_cmp(&a->a4, &b->a4);
                c2 = nm_platform_ip4_address_pretty_sort_cmp(&b->a4, &a->a4);
            } else {
                c1 = nm_platform_ip6_address_pretty_sort_cmp(&a->a6, &b->a6, IPV6_PREFER_TEMP);
                c2 = nm_platform_ip6_address_pretty_sort_cmp(&b->a6, &a->a6, IPV6_PREFER_TEMP);
            }

            g_assert_cmpint(c1, >=, -1);
            g_assert_cmpint(c1, <=, 1);
            g_assert_cmpint(c1, ==, -c2);
        }
    }

    if (PRINT_RESULT) {
        g_print("\n        [%d] = (", TEST_DATA_I);
        for (i = 0; i < ELM_SIZE * N_ADDRESSES;) {
            if (i > 0)
                g_print("\n               ");
            g_print("\"");
            for (j = 0; j < 40 && i < ELM_SIZE * N_ADDRESSES; j++, i++)
                g_print("%02x", addresses[i]);
            g_print("\"");
        }
        g_print("),\n");
        return;
    }

    if (CHECK_RESULT) {
        gs_free guint8 *bin_arr = NULL;
        gsize           bin_len;

        bin_arr = nm_utils_hexstr2bin_alloc(EXPECTED_BUFFER[TEST_DATA_I],
                                            FALSE,
                                            FALSE,
                                            NULL,
                                            0,
                                            &bin_len);

        if (bin_len != ELM_SIZE * N_ADDRESSES || memcmp(addresses, bin_arr, bin_len) != 0) {
            gs_free char *addresses_str = NULL;

            NM_PRAGMA_WARNING_DISABLE_DANGLING_POINTER
            g_error(">>> test_platform_ip_address_pretty_sort_cmp() will fail:\n"
                    ">>> addresses[%zu]: %s\n"
                    ">>> expected [%zu]: %s\n",
                    ELM_SIZE * N_ADDRESSES,
                    (addresses_str = nm_utils_bin2hexstr(addresses, ELM_SIZE * N_ADDRESSES, -1)),
                    bin_len,
                    EXPECTED_BUFFER[TEST_DATA_I]);
            NM_PRAGMA_WARNING_REENABLE
        }

        g_assert_cmpmem(addresses, ELM_SIZE * N_ADDRESSES, bin_arr, bin_len);
    }
}

/*****************************************************************************/

static void
test_route_type_is_nodev(void)
{
    int i;

    for (i = -1; i <= 257; i++) {
        gboolean is_nodev;

        switch ((guint8) i) {
        case RTN_BLACKHOLE:
        case RTN_UNREACHABLE:
        case RTN_PROHIBIT:
        case RTN_THROW:
            is_nodev = TRUE;
            break;
        default:
            is_nodev = FALSE;
            break;
        }

        g_assert_cmpint(is_nodev, ==, nm_platform_route_type_is_nodev(i));
    }
}

/*****************************************************************************/

static void
test_nmp_genl_family_type_from_name(void)
{
    int n_run;
    int i;

    for (i = 0; i < (int) _NMP_GENL_FAMILY_TYPE_NUM; i++) {
        const char *name = nmp_genl_family_infos[i].name;

        g_assert(name);
        if (i > 0)
            g_assert_cmpint(strcmp(nmp_genl_family_infos[i - 1].name, name), <, 0);
    }

    g_assert_cmpint(nmp_genl_family_type_from_name("ethtool"), ==, NMP_GENL_FAMILY_TYPE_ETHTOOL);
    g_assert_cmpint(nmp_genl_family_type_from_name("mptcp_pm"), ==, NMP_GENL_FAMILY_TYPE_MPTCP_PM);
    g_assert_cmpint(nmp_genl_family_type_from_name("nl80211"), ==, NMP_GENL_FAMILY_TYPE_NL80211);
    g_assert_cmpint(nmp_genl_family_type_from_name("nl802154"), ==, NMP_GENL_FAMILY_TYPE_NL802154);
    g_assert_cmpint(nmp_genl_family_type_from_name("wireguard"),
                    ==,
                    NMP_GENL_FAMILY_TYPE_WIREGUARD);

    g_assert_cmpint(nmp_genl_family_type_from_name(NULL), ==, _NMP_GENL_FAMILY_TYPE_NONE);
    g_assert_cmpint(nmp_genl_family_type_from_name("a"), ==, _NMP_GENL_FAMILY_TYPE_NONE);
    g_assert_cmpint(nmp_genl_family_type_from_name("wireguara"), ==, _NMP_GENL_FAMILY_TYPE_NONE);
    g_assert_cmpint(nmp_genl_family_type_from_name("wireguarb"), ==, _NMP_GENL_FAMILY_TYPE_NONE);
    g_assert_cmpint(nmp_genl_family_type_from_name(""), ==, _NMP_GENL_FAMILY_TYPE_NONE);
    g_assert_cmpint(nmp_genl_family_type_from_name("z"), ==, _NMP_GENL_FAMILY_TYPE_NONE);

    for (n_run = 0; n_run < 20; n_run++) {
        for (i = 0; i < (int) _NMP_GENL_FAMILY_TYPE_NUM; i++) {
            const char *cname  = nmp_genl_family_infos[i].name;
            const int   ch_idx = nmtst_get_rand_uint() % strlen(cname);
            char        name[200];
            char        ch;
            gsize       l;

            l = g_strlcpy(name, cname, sizeof(name));
            g_assert_cmpint(l, <, sizeof(name));

            if (n_run == 0) {
                g_assert_cmpint(nmp_genl_family_type_from_name(cname), ==, i);
                g_assert_cmpint(nmp_genl_family_type_from_name(name), ==, i);
            }

            /* randomly change one character in the name. Such a name becomes invalid.
             * There are no two valid names which only differ by one characters. */
            do {
                ch = nmtst_get_rand_uint() % 256;
            } while (cname[ch_idx] == ch);
            name[ch_idx] = ch;
            g_assert_cmpint(nmp_genl_family_type_from_name(name), ==, _NMP_GENL_FAMILY_TYPE_NONE);
        }
    }
}

/*****************************************************************************/

NMTST_DEFINE();

int
main(int argc, char **argv)
{
    nmtst_init_assert_logging(&argc, &argv, "WARN", "DEFAULT");

    g_test_add_func("/general/init_linux_platform", test_init_linux_platform);
    g_test_add_func("/general/link_get_all", test_link_get_all);
    g_test_add_func("/general/nm_platform_link_flags2str", test_nm_platform_link_flags2str);
    g_test_add_data_func("/general/platform_ip_address_pretty_sort_cmp/4",
                         GINT_TO_POINTER(0),
                         test_platform_ip_address_pretty_sort_cmp);
    g_test_add_data_func("/general/platform_ip_address_pretty_sort_cmp/6/1",
                         GINT_TO_POINTER(1),
                         test_platform_ip_address_pretty_sort_cmp);
    g_test_add_data_func("/general/platform_ip_address_pretty_sort_cmp/6/2",
                         GINT_TO_POINTER(2),
                         test_platform_ip_address_pretty_sort_cmp);
    g_test_add_func("/general/test_route_type_is_nodev", test_route_type_is_nodev);
    g_test_add_func("/nm-platform/test_nmp_genl_family_type_from_name",
                    test_nmp_genl_family_type_from_name);

    return g_test_run();
}
