#include "orbit-c-backend.h"

static void cbe_output_demarshal_type_integer(CBEDemarshalInfo mi);
static void cbe_output_demarshal_type_float(CBEDemarshalInfo mi);
static void cbe_output_demarshal_type_char(CBEDemarshalInfo mi);
static void cbe_output_demarshal_type_wide_char(CBEDemarshalInfo mi);
static void cbe_output_demarshal_type_boolean(CBEDemarshalInfo mi);
static void cbe_output_demarshal_type_octet(CBEDemarshalInfo mi);
static void cbe_output_demarshal_type_enum(CBEDemarshalInfo mi);
static void cbe_output_demarshal_type_fixed(CBEDemarshalInfo mi);
static void cbe_output_demarshal_type_string(CBEDemarshalInfo mi);
static void cbe_output_demarshal_type_wide_string(CBEDemarshalInfo mi);
static void cbe_output_demarshal_type_any(CBEDemarshalInfo mi);
static void cbe_output_demarshal_type_object(CBEDemarshalInfo mi);
static void cbe_output_demarshal_type_sequence(CBEDemarshalInfo mi);
static void cbe_output_demarshal_type_array(CBEDemarshalInfo mi);
static void cbe_output_demarshal_type_struct(CBEDemarshalInfo mi);
static void cbe_output_demarshal_type_union(CBEDemarshalInfo mi);
static void cbe_output_demarshal_case_stmt(CBEDemarshalInfo mi);

void
cbe_output_demarshaller(CBEDemarshalInfo dmi)
{
  IDL_tree ts;

  switch(IDL_NODE_TYPE(dmi.param)) {
  case IDLN_CASE_STMT:
    dmi.typespec = ts = dmi.param;
    break;
  default:
    dmi.typespec = ts = cbe_get_typespec(dmi.param);
  }

  switch(IDL_NODE_TYPE(ts)) {
  case IDLN_TYPE_INTEGER:
    cbe_output_demarshal_type_integer(dmi);
    break;

  case IDLN_TYPE_FLOAT:
    cbe_output_demarshal_type_float(dmi);
    break;

  case IDLN_TYPE_CHAR:
    cbe_output_demarshal_type_char(dmi);
    break;

  case IDLN_TYPE_WIDE_CHAR:
    cbe_output_demarshal_type_boolean(dmi);
    break;

  case IDLN_TYPE_BOOLEAN:
    cbe_output_demarshal_type_boolean(dmi);
    break;

  case IDLN_TYPE_ENUM:
    cbe_output_demarshal_type_enum(dmi);
    break;

  case IDLN_TYPE_OCTET:
    cbe_output_demarshal_type_octet(dmi);
    break;

  case IDLN_TYPE_FIXED:
    cbe_output_demarshal_type_fixed(dmi);
    break;

  case IDLN_TYPE_STRING:
    cbe_output_demarshal_type_string(dmi);
    break;

  case IDLN_TYPE_WIDE_STRING:
    cbe_output_demarshal_type_wide_string(dmi);
    break;

  case IDLN_TYPE_ANY:
    cbe_output_demarshal_type_any(dmi);
    break;

  case IDLN_TYPE_OBJECT:
    cbe_output_demarshal_type_object(dmi);
    break;

  case IDLN_TYPE_SEQUENCE:
    cbe_output_demarshal_type_sequence(dmi);
    break;

  case IDLN_TYPE_ARRAY:
    cbe_output_demarshal_type_array(dmi);
    break;

  case IDLN_TYPE_STRUCT:
  case IDLN_EXCEPT_DCL:
    cbe_output_demarshal_type_struct(dmi);
    break;

  case IDLN_TYPE_UNION:
    cbe_output_demarshal_type_union(dmi);
    break;

  case IDLN_CASE_STMT:
    cbe_output_demarshal_case_stmt(dmi); /* Part of unions */
    break;

  default:
    g_warning("NOT producing demarshaller for %s\n",
	      IDL_tree_type_names[IDL_NODE_TYPE(ts)]);
    break;

  }
}

static void
cbe_output_demarshal_type_atom(CBEDemarshalInfo mi)
{
  int boundary;
  /* XXX fixme: We could use some more smarts here.
     For example, if we've marshalled exactly two shorts as parameters,
     we don't need to do alignment for an integer parameter.
     ALL fixed-length types can be handled this way... */

  boundary = cbe_get_type_alignment(cbe_get_typespec(mi.param));

  if(mi.previous_param && (cbe_get_type_alignment(cbe_get_typespec(mi.previous_param)) < boundary))
    fprintf(mi.of, "  GIOP_RECV_BUFFER(_ORBIT_recv_buffer)->cur = ALIGN_ADDRESS(GIOP_RECV_BUFFER(_ORBIT_recv_buffer)->cur, %d);\n", boundary);

  if(mi.byteswap_version)
    fprintf(mi.of, "  GET_ATOM(%s);\n", mi.param_name);
  else {
    fprintf(mi.of, "  %s = *((", mi.param_name);
    orbit_cbe_write_typespec(mi.of, mi.typespec);
    fprintf(mi.of, " *)GIOP_RECV_BUFFER(_ORBIT_recv_buffer)->cur);\n");
    fprintf(mi.of, "  GIOP_RECV_BUFFER(_ORBIT_recv_buffer)->cur += sizeof(");
    orbit_cbe_write_typespec(mi.of, mi.typespec);
    fprintf(mi.of, ");\n");
  }
}

static void cbe_output_demarshal_type_integer(CBEDemarshalInfo mi)
{
  cbe_output_demarshal_type_atom(mi);
}

static void cbe_output_demarshal_type_float(CBEDemarshalInfo mi)
{
  cbe_output_demarshal_type_atom(mi);
}

static void cbe_output_demarshal_type_fixed(CBEDemarshalInfo mi)
{
  g_assert(!"Not yet implemented\n");
}

static void cbe_output_demarshal_type_char(CBEDemarshalInfo mi)
{
  cbe_output_demarshal_type_atom(mi);
}

static void cbe_output_demarshal_type_wide_char(CBEDemarshalInfo mi)
{
  cbe_output_demarshal_type_atom(mi);
}

static void cbe_output_demarshal_type_string(CBEDemarshalInfo mi)
{
  CBEDemarshalInfo submi = mi;

  submi.param_name = "len";
  submi.typespec = IDL_type_integer_new(0, IDL_INTEGER_TYPE_LONG);
  submi.param = submi.typespec;
  fprintf(mi.of, "  {\n");
  fprintf(mi.of, "    GIOP_unsigned_long len;\n");
  cbe_output_demarshaller(submi);

  if(mi.allocate_memory) {

    fprintf(mi.of, "    %s = CORBA_string_alloc(len);\n", mi.param_name);
    fprintf(mi.of, "    if(len)\n");
    fprintf(mi.of,
	    "    strncpy(%s, GIOP_RECV_BUFFER(_ORBIT_recv_buffer)->cur, len);\n",
	    mi.param_name);
    fprintf(mi.of, "    GIOP_RECV_BUFFER(_ORBIT_recv_buffer)->cur += len;\n");

  } else {
    fprintf(mi.of, "    %s = len?(GIOP_RECV_BUFFER(_ORBIT_recv_buffer)->cur):NULL;", mi.param_name);
    fprintf(mi.of, "    GIOP_RECV_BUFFER(_ORBIT_recv_buffer)->cur += len;\n");
  }

  fprintf(mi.of, "  }\n");

  IDL_tree_free(submi.typespec);
}

static void cbe_output_demarshal_type_wide_string(CBEDemarshalInfo mi)
{
  g_assert(!"Not yet implemented\n");
}

static void cbe_output_demarshal_type_boolean(CBEDemarshalInfo mi)
{
  cbe_output_demarshal_type_atom(mi);
}

static void cbe_output_demarshal_type_octet(CBEDemarshalInfo mi)
{
  cbe_output_demarshal_type_atom(mi);
}

static void cbe_output_demarshal_type_any(CBEDemarshalInfo mi)
{
  g_assert(!"Not yet implemented\n");
}

static void cbe_output_demarshal_type_object(CBEDemarshalInfo mi)
{
  g_assert(!"Not yet implemented\n");
}

static void cbe_output_demarshal_type_enum(CBEDemarshalInfo mi)
{
  cbe_output_demarshal_type_atom(mi);
}

static void cbe_output_demarshal_type_sequence(CBEDemarshalInfo mi)
{
  GString *tmpstr = g_string_new(NULL);
  CBEDemarshalInfo subdmi = mi;

  g_string_sprintf(tmpstr, "(%s)._length", mi.param_name);
  subdmi.param_name = tmpstr->str;
  subdmi.typespec = IDL_type_integer_new(0, IDL_INTEGER_TYPE_LONG);
  subdmi.param = subdmi.typespec;

  cbe_output_demarshaller(subdmi);

  IDL_tree_free(subdmi.param);

  fprintf(mi.of, "  (%s)._buffer = CORBA_sequence_", mi.param_name);
  orbit_cbe_write_typename(mi.of, IDL_TYPE_SEQUENCE(mi.typespec).simple_type_spec);
  fprintf(mi.of, "_allocbuf((%s)._length);\n", mi.param_name);

  subdmi.param = subdmi.typespec = IDL_TYPE_SEQUENCE(mi.typespec).simple_type_spec;
  g_string_sprintf(tmpstr, "(%s)._buffer[i]", mi.param_name);
  subdmi.param_name = tmpstr->str;
  fprintf(mi.of, "  {\n    int i;\n    for(i = 0; i < (%s)._length; i++) {\n",
	  mi.param_name);
  cbe_output_demarshaller(subdmi);
  fprintf(mi.of, "  }\n");

  g_string_free(tmpstr, TRUE);
}

static void cbe_output_demarshal_type_array(CBEDemarshalInfo mi)
{
  GString *tmpstr = g_string_new(NULL);
  int dimn;
  IDL_tree curitem, dcl;
  CBEDemarshalInfo subdmi = mi;
  
  dcl = IDL_get_parent_node(mi.typespec, IDLN_TYPE_DCL, NULL);

  g_assert(dcl);

  if(!mi.byteswap_version && cbe_type_is_fixed_length(IDL_TYPE_DCL(dcl).type_spec)) {
    /* need to test this... :-) */
    fprintf(mi.of,
	    "  memcpy(&%s, GIOP_RECV_BUFFER(_ORBIT_recv_buffer)->cur, sizeof(%s));\n",
	    mi.param_name, mi.param_name);
    fprintf(mi.of,
	    "  GIOP_RECV_BUFFER(_ORBIT_recv_buffer)->cur += sizeof(%s);\n",
	    mi.param_name);
    return;
  }

  fprintf(mi.of, "  {\n");

  fprintf(mi.of, "  int n0");

  for(dimn = 1, curitem = IDL_LIST(IDL_TYPE_ARRAY(mi.typespec).size_list).next;
      curitem; dimn++, curitem = IDL_LIST(curitem).next) {
    fprintf(mi.of, ", n%d", dimn);
  }
  fprintf(mi.of, ";\n");

  for(dimn = 0, curitem = IDL_TYPE_ARRAY(mi.typespec).size_list;
      curitem; dimn++, curitem = IDL_LIST(curitem).next) {
    fprintf(mi.of, "  for(n%d = 0; n%d < %lld; n%d++) {\n",
	    dimn, dimn, IDL_INTEGER(IDL_LIST(curitem).data).value, dimn);
  }

  g_string_sprintf(tmpstr, "(%s)", mi.param_name);
  for(dimn = 0, curitem = IDL_TYPE_ARRAY(mi.typespec).size_list;
      curitem; dimn++, curitem = IDL_LIST(curitem).next) {
    g_string_sprintfa(tmpstr, "[n%d]", dimn);
  }

  subdmi.param = subdmi.typespec = IDL_TYPE_DCL(dcl).type_spec;
  subdmi.param_name = tmpstr->str;
  cbe_output_demarshaller(subdmi);

  for(dimn--, curitem = IDL_TYPE_ARRAY(mi.typespec).size_list;
      curitem; dimn--, curitem = IDL_LIST(curitem).next) {
    fprintf(mi.of, "  } /* end loop for n%d */\n", dimn);
  }

  fprintf(mi.of, "  }\n");

  g_string_free(tmpstr, TRUE);
}

static void cbe_output_demarshal_type_struct(CBEDemarshalInfo mi)
{
  GString *tmpstr = g_string_new(NULL);
  CBEDemarshalInfo subdmi = mi;
  IDL_tree curitem, curdcl;

  if(!mi.byteswap_version && cbe_type_is_fixed_length(mi.typespec)) {

    /* Someone needs to verify that this shortcut */

    fprintf(mi.of, "  memcpy(&%s, GIOP_RECV_BUFFER(_ORBIT_recv_buffer)->cur, sizeof(%s));\n", mi.param_name, mi.param_name);
    fprintf(mi.of,
	    "  GIOP_RECV_BUFFER(_ORBIT_recv_buffer)->cur += sizeof(%s);\n",
	    mi.param_name);
    return;
  }

  for(curitem = IDL_TYPE_STRUCT(mi.typespec).member_list;
      curitem; curitem = IDL_LIST(curitem).next) {

    subdmi.param = subdmi.typespec =
      IDL_MEMBER(IDL_LIST(curitem).data).type_spec;

    for(curdcl = IDL_MEMBER(IDL_LIST(curitem).data).dcls; curdcl;
	curdcl = IDL_LIST(curdcl).next) {

      g_string_sprintf(tmpstr, "(%s).%s", mi.param_name,
		       IDL_IDENT(IDL_LIST(curdcl).data).str);
      subdmi.param_name = tmpstr->str;
      cbe_output_demarshaller(subdmi);
    }
  }
  g_string_free(tmpstr, TRUE);
}

static void cbe_output_demarshal_type_union(CBEDemarshalInfo mi)
{
  GString *tmpstr = g_string_new(NULL);
  CBEDemarshalInfo subdmi = mi;
  IDL_tree curitem;

  g_string_sprintf(tmpstr, "(%s)._d", mi.param_name);
  subdmi.param_name = tmpstr->str;

  subdmi.param = subdmi.typespec = IDL_TYPE_UNION(mi.typespec).switch_type_spec;

  cbe_output_demarshaller(subdmi);

  subdmi.previous_param = subdmi.param;

  fprintf(mi.of, "  switch((%s)._d) {\n", mi.param_name);

  g_string_sprintf(tmpstr, "(%s)._u", mi.param_name);
  subdmi.param_name = tmpstr->str;

  for(curitem = IDL_TYPE_UNION(mi.typespec).switch_body;
      curitem; curitem = IDL_LIST(curitem).next) {

    subdmi.param = subdmi.typespec = IDL_LIST(curitem).data;

    cbe_output_demarshaller(subdmi);
  }
  fprintf(mi.of, "  }\n\n");

  g_string_free(tmpstr, TRUE);
}

static void
cbe_output_demarshal_case_stmt(CBEDemarshalInfo mi)
{
  GString *tmpstr = g_string_new(NULL);
  IDL_tree curitem;
  CBEDemarshalInfo subdmi = mi;

  for(curitem = IDL_CASE_STMT(mi.typespec).labels;
      curitem;
      curitem = IDL_LIST(curitem).next) {
    if(IDL_LIST(curitem).data) {
      fprintf(mi.of, "  case ");
      orbit_cbe_write_const(mi.of, IDL_LIST(curitem).data);
      fprintf(mi.of, ":\n");
    } else
      fprintf(mi.of, "  default:\n");
  }  

  g_string_sprintf(tmpstr, "(%s).%s", mi.param_name,
		   IDL_IDENT(IDL_LIST(IDL_MEMBER(IDL_CASE_STMT(mi.typespec).element_spec).dcls).data).str);
  subdmi.param_name = tmpstr->str;
  subdmi.param = subdmi.typespec = IDL_MEMBER(IDL_CASE_STMT(mi.typespec).element_spec).type_spec;

  cbe_output_demarshaller(subdmi);

  g_string_free(tmpstr, TRUE);
}
