#include "orbit-c-backend.h"

static void cbe_output_marshal_type_atom(CBEMarshalInfo mi);
static void cbe_output_marshal_type_integer(CBEMarshalInfo mi);
static void cbe_output_marshal_type_float(CBEMarshalInfo mi);
static void cbe_output_marshal_type_fixed(CBEMarshalInfo mi);
static void cbe_output_marshal_type_char(CBEMarshalInfo mi);
static void cbe_output_marshal_type_wide_char(CBEMarshalInfo mi);
static void cbe_output_marshal_type_string(CBEMarshalInfo mi);
static void cbe_output_marshal_type_wide_string(CBEMarshalInfo mi);
static void cbe_output_marshal_type_boolean(CBEMarshalInfo mi);
static void cbe_output_marshal_type_octet(CBEMarshalInfo mi);
static void cbe_output_marshal_type_any(CBEMarshalInfo mi);
static void cbe_output_marshal_type_object(CBEMarshalInfo mi);
static void cbe_output_marshal_type_enum(CBEMarshalInfo mi);
static void cbe_output_marshal_type_sequence(CBEMarshalInfo mi);
static void cbe_output_marshal_type_array(CBEMarshalInfo mi);
static void cbe_output_marshal_type_struct(CBEMarshalInfo mi);
static void cbe_output_marshal_type_union(CBEMarshalInfo mi);

void cbe_output_marshaller(CBEMarshalInfo mi)
{
  IDL_tree ts;

  mi.typespec = ts = cbe_get_typespec(mi.param);

  switch(IDL_NODE_TYPE(ts)) {
  case IDLN_TYPE_INTEGER:
  case IDLN_TYPE_FLOAT:
  case IDLN_TYPE_CHAR:
  case IDLN_TYPE_WIDE_CHAR:
  case IDLN_TYPE_BOOLEAN:
  case IDLN_TYPE_OCTET:
  case IDLN_TYPE_ENUM:
    cbe_output_marshal_type_atom(mi);
    break;

  case IDLN_TYPE_FIXED:
    cbe_output_marshal_type_fixed(mi);
    break;

  case IDLN_TYPE_STRING:
    cbe_output_marshal_type_string(mi);
    break;

  case IDLN_TYPE_WIDE_STRING:
    cbe_output_marshal_type_wide_string(mi);
    break;

  case IDLN_TYPE_ANY:
    cbe_output_marshal_type_any(mi);
    break;

  case IDLN_TYPE_OBJECT:
    cbe_output_marshal_type_object(mi);
    break;

  case IDLN_TYPE_SEQUENCE:
    cbe_output_marshal_type_sequence(mi);
    break;

  case IDLN_TYPE_ARRAY:
    cbe_output_marshal_type_array(mi);
    break;

  case IDLN_TYPE_STRUCT:
    cbe_output_marshal_type_struct(mi);
    break;

  case IDLN_TYPE_UNION:
    cbe_output_marshal_type_union(mi);
    break;

  default:
    g_warning("NOT producing marshaller for %s\n", IDL_tree_type_names[IDL_NODE_TYPE(ts)]);
    break;
  }
}

static void cbe_output_marshal_type_atom(CBEMarshalInfo mi)
{
  int boundary;

  boundary = cbe_get_type_alignment(cbe_get_typespec(mi.param));
  /* If we don't need to align-up for a parameter, why bother doing the work? */

  if(mi.previous_param && (cbe_get_type_alignment(cbe_get_typespec(mi.previous_param)) >= boundary)) {
    fprintf(mi.of,
	    "  giop_%s(GIOP_%s_BUFFER(_ORBIT_send_buffer), &%s, sizeof(%s));\n",
	    mi.require_indirect?"send_buffer_append_mem_indirect":"message_buffer_append_mem",
	    mi.require_indirect?"SEND":"MESSAGE",
	    mi.param_name, mi.param_name);
  } else {
    fprintf(mi.of,
	    "  giop_%s_a(GIOP_%s_BUFFER(_ORBIT_send_buffer), &%s, sizeof(%s));\n",
	    mi.require_indirect?"send_buffer_append_mem_indirect":"message_buffer_append_mem",
	    mi.require_indirect?"SEND":"MESSAGE",
	    mi.param_name, mi.param_name);
  }
}

static void cbe_output_marshal_type_fixed(CBEMarshalInfo mi)
{
}

static void cbe_output_marshal_type_string(CBEMarshalInfo mi)
{
  CBEMarshalInfo submi = mi;

  fprintf(mi.of, "  {\n");
  fprintf(mi.of, "    GIOP_unsigned_long len = %s?(strlen(%s)+1):0;\n",
	  mi.param_name,
	  mi.param_name);

  submi.param_name = "len";
  submi.typespec = IDL_type_integer_new(0, IDL_INTEGER_TYPE_LONG);
  submi.param = submi.typespec;
  submi.require_indirect = TRUE;
  cbe_output_marshaller(submi);
  IDL_tree_free(submi.typespec);

  fprintf(mi.of, "    if(%s)\n", mi.param_name);
  fprintf(mi.of, "      giop_message_buffer_append_mem(GIOP_MESSAGE_BUFFER(_ORBIT_send_buffer), %s, len);\n", mi.param_name);
  fprintf(mi.of, "  }\n");
}

static void cbe_output_marshal_type_wide_string(CBEMarshalInfo mi)
{
  /* XXX fixme: Not sure what this is supposed to look like - do
     "something" for now */
  cbe_output_marshal_type_string(mi);
}

static void cbe_output_marshal_type_any(CBEMarshalInfo mi)
{
  /* marshal typecode, marshal type.
     we should hook into the DII marshalling functions here... */
  g_error("Can't marshal type ANY yet\n");
}

static void cbe_output_marshal_type_object(CBEMarshalInfo mi)
{
  g_error("Don't know how to put object refs on the wire... Look in object_to_string and find a way to hook in so we can get at the marshal buffer");
}

static void cbe_output_marshal_type_sequence(CBEMarshalInfo mi)
{
  fprintf(mi.of, "  {\n");
  if(!cbe_type_is_fixed_length(mi.typespec)) {
    fprintf(mi.of, "    GIOP_unsigned_long i;\n");
  }
  fprintf(mi.of, "    giop_message_buffer_append_mem_a(GIOP_MESSAGE_BUFFER(_ORBIT_send_buffer), (%s)._length, sizeof(GIOP_unsigned_long));\n", mi.param_name);

  if(cbe_type_is_fixed_length(mi.param)) {
    fprintf(mi.of, "    giop_message_buffer_append_mem(GIOP_MESSAGE_BUFFER(_ORBIT_send_buffer), (%s)._buffer, sizeof((%s)._buffer[0])*((%s)._length));\n",
	    mi.param_name, mi.param_name, mi.param_name);
  } else {
    GString *subname = g_string_new(NULL);
    CBEMarshalInfo submi = mi;
    submi.param = cbe_get_typespec(IDL_TYPE_SEQUENCE(mi.param).simple_type_spec);
    submi.previous_param = NULL;
    fprintf(mi.of, "    for(i = 0; i < (%s)._length; i++) {\n", mi.param_name);

    g_string_sprintf(subname, "(%s)._buffer[i]", mi.param_name);

    submi.param_name = subname->str;
    cbe_output_marshaller(submi);

    fprintf(mi.of, "    }\n");

    g_string_free(subname, TRUE);
  }
  fprintf(mi.of, "  }\n");
}

static void cbe_output_marshal_type_array(CBEMarshalInfo mi)
{
  fprintf(mi.of, "  {\n");
  if(!cbe_type_is_fixed_length(mi.typespec)) {
    fprintf(mi.of, "    GIOP_unsigned_long i;\n");
  }
  fprintf(mi.of, "\
    giop_message_buffer_append_mem_a(GIOP_MESSAGE_BUFFER(_ORBIT_send_buffer),\n\
				     &((%s)._length),\n\
				     sizeof(GIOP_unsigned_long));\n",
	  mi.param_name);

  if(cbe_type_is_fixed_length(mi.param)) {
    fprintf(mi.of, "\
    giop_message_buffer_append_mem(GIOP_MESSAGE_BUFFER(_ORBIT_send_buffer),\n\
				   (%s)._buffer,\n\
				   sizeof((%s)._buffer[0])*((%s)._length));\n",
	    mi.param_name, mi.param_name, mi.param_name);
  } else {
    GString *subname = g_string_new(NULL);
    CBEMarshalInfo submi = mi;

    submi.param = cbe_get_typespec(IDL_TYPE_SEQUENCE(mi.param).simple_type_spec);
    submi.previous_param = NULL;

    fprintf(mi.of, "    for(i = 0; i < (%s)._length; i++) {\n", mi.param_name);

    g_string_sprintf(subname, "(%s)._buffer[i]", mi.param_name);
    submi.param_name = subname->str;
    cbe_output_marshaller(submi);
    g_string_free(subname, TRUE);

    fprintf(mi.of, "    }\n");
  }

  fprintf(mi.of, "  }\n");
}

static void cbe_output_marshal_type_struct(CBEMarshalInfo mi)
{
  IDL_tree curitem, curmember;
  GString *subname = g_string_new(NULL);
  CBEMarshalInfo submi = mi;

  submi.param = NULL;
  for(curitem = IDL_TYPE_STRUCT(mi.typespec).member_list; curitem;
      curitem = IDL_LIST(curitem).next) {

    for(curmember = IDL_MEMBER(IDL_LIST(curitem).data).dcls; curmember;
	curmember = IDL_LIST(curmember).next) {
      submi.previous_param = submi.param;
      submi.param = cbe_get_typespec(IDL_LIST(curmember).data);
      g_string_sprintf(subname, "(%s).%s", mi.param_name,
		       IDL_IDENT(IDL_LIST(curmember).data).str);
      submi.param_name = subname->str;
      cbe_output_marshaller(submi);
    }

  }

  g_string_free(subname, TRUE);
}

static void cbe_output_marshal_type_union(CBEMarshalInfo mi)
{
  GString *subname  = g_string_new(NULL);
  IDL_tree curitem, curcase, ts;
  CBEMarshalInfo submi = mi;

  fprintf(mi.of, "  switch((%s)._d) {\n", mi.param_name);
  for(curitem = IDL_TYPE_UNION(mi.typespec).switch_body;
      curitem; curitem = IDL_LIST(curitem).next) {

    for(curcase = IDL_CASE_STMT(IDL_LIST(curitem).data).labels;
	curcase; curcase = IDL_LIST(curcase).next) {
      if(IDL_LIST(curcase).data) {
	fprintf(mi.of, "   case ");
	orbit_cbe_write_const(mi.of, IDL_LIST(curcase).data);
	fprintf(mi.of, ":\n");
      } else
	fprintf(mi.of, "   default:\n");
    }

    ts = IDL_CASE_STMT(IDL_LIST(curitem).data).element_spec; /* the IDL_MEMBER */

    /* shades of LISP */
    g_string_sprintf(subname, "(%s)._u.%s", mi.param_name,
		     IDL_IDENT(IDL_LIST(IDL_MEMBER(ts).dcls).data).str);
    submi.param_name = subname->str;

    submi.param = submi.typespec =
      IDL_MEMBER(ts).type_spec;

    cbe_output_marshaller(submi);

    fprintf(mi.of, "    break;");
  }

  g_string_free(subname, TRUE);
}
