#include "orbit-c-backend.h"
#include <glib.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

typedef struct {
  FILE *of;
  IDL_ns ns;
  IDL_tree tree;
} CBESkelInfo;

static void orbit_cbe_skel_process_piece(CBESkelInfo *ski);
static void cbe_skel_do_attr_dcl(CBESkelInfo *ski);
static void cbe_skel_do_binop(CBESkelInfo *ski);
static void cbe_skel_do_boolean(CBESkelInfo *ski);
static void cbe_skel_do_case_stmt(CBESkelInfo *ski);
static void cbe_skel_do_char(CBESkelInfo *ski);
static void cbe_skel_do_const_dcl(CBESkelInfo *ski);
static void cbe_skel_do_except_dcl(CBESkelInfo *ski);
static void cbe_skel_do_fixed(CBESkelInfo *ski);
static void cbe_skel_do_float(CBESkelInfo *ski);
static void cbe_skel_do_forward_dcl(CBESkelInfo *ski);
static void cbe_skel_do_gentree(CBESkelInfo *ski);
static void cbe_skel_do_ident(CBESkelInfo *ski);
static void cbe_skel_do_integer(CBESkelInfo *ski);
static void cbe_skel_do_interface(CBESkelInfo *ski);
static void cbe_skel_do_list(CBESkelInfo *ski);
static void cbe_skel_do_member(CBESkelInfo *ski);
static void cbe_skel_do_module(CBESkelInfo *ski);
static void cbe_skel_do_none(CBESkelInfo *ski);
static void cbe_skel_do_op_dcl(CBESkelInfo *ski);
static void cbe_skel_do_param_dcl(CBESkelInfo *ski);
static void cbe_skel_do_string(CBESkelInfo *ski);
static void cbe_skel_do_type_any(CBESkelInfo *ski);
static void cbe_skel_do_type_array(CBESkelInfo *ski);
static void cbe_skel_do_type_boolean(CBESkelInfo *ski);
static void cbe_skel_do_type_char(CBESkelInfo *ski);
static void cbe_skel_do_type_dcl(CBESkelInfo *ski);
static void cbe_skel_do_type_enum(CBESkelInfo *ski);
static void cbe_skel_do_type_fixed(CBESkelInfo *ski);
static void cbe_skel_do_type_float(CBESkelInfo *ski);
static void cbe_skel_do_type_integer(CBESkelInfo *ski);
static void cbe_skel_do_type_object(CBESkelInfo *ski);
static void cbe_skel_do_type_octet(CBESkelInfo *ski);
static void cbe_skel_do_type_sequence(CBESkelInfo *ski);
static void cbe_skel_do_type_string(CBESkelInfo *ski);
static void cbe_skel_do_type_struct(CBESkelInfo *ski);
static void cbe_skel_do_type_union(CBESkelInfo *ski);
static void cbe_skel_do_type_wide_char(CBESkelInfo *ski);
static void cbe_skel_do_type_wide_string(CBESkelInfo *ski);
static void cbe_skel_do_unaryop(CBESkelInfo *ski);
static void cbe_skel_do_wide_char(CBESkelInfo *ski);
static void cbe_skel_do_wide_string(CBESkelInfo *ski);

void
orbit_cbe_write_skeletons(FILE *outfile, IDL_ns ns, IDL_tree tree,
			  const char *header_filename)
{
  CBESkelInfo ski = {outfile, ns, tree};

  fprintf(outfile, "#include <string.h>\n");
  fprintf(outfile, "#include \"%s\"\n", header_filename);
  fprintf(outfile, "#define GET_ATOM(x) ({ GIOP_RECV_BUFFER(_ORBIT_recv_buffer)->decoder(&x, (GIOP_RECV_BUFFER(_ORBIT_recv_buffer)->cur), sizeof(x)); GIOP_RECV_BUFFER(_ORBIT_recv_buffer)->cur += sizeof(x); })\n");

  orbit_cbe_skel_process_piece(&ski);
}

static void
orbit_cbe_skel_process_piece(CBESkelInfo *ski)
{
  switch(IDL_NODE_TYPE(ski->tree)) {
  case IDLN_ATTR_DCL:
    cbe_skel_do_attr_dcl(ski);
    break;
  case IDLN_BINOP:
    cbe_skel_do_binop(ski);
    break;
  case IDLN_BOOLEAN:
    cbe_skel_do_boolean(ski);
    break;
  case IDLN_CASE_STMT:
    cbe_skel_do_case_stmt(ski);
    break;
  case IDLN_CHAR:
    cbe_skel_do_char(ski);
    break;
  case IDLN_CONST_DCL:
    cbe_skel_do_const_dcl(ski);
    break;
  case IDLN_EXCEPT_DCL:
    cbe_skel_do_except_dcl(ski);
    break;
  case IDLN_FIXED:
    cbe_skel_do_fixed(ski);
    break;
  case IDLN_FLOAT:
    cbe_skel_do_float(ski);
    break;
  case IDLN_FORWARD_DCL:
    cbe_skel_do_forward_dcl(ski);
    break;
  case IDLN_GENTREE:
    cbe_skel_do_gentree(ski);
    break;
  case IDLN_IDENT:
    cbe_skel_do_ident(ski);
    break;
  case IDLN_INTEGER:
    cbe_skel_do_integer(ski);
    break;
  case IDLN_INTERFACE:
    cbe_skel_do_interface(ski);
    break;
  case IDLN_LIST:
    cbe_skel_do_list(ski);
    break;
  case IDLN_MEMBER:
    cbe_skel_do_member(ski);
    break;
  case IDLN_MODULE:
    cbe_skel_do_module(ski);
    break;
  case IDLN_NONE:
    cbe_skel_do_none(ski);
    break;
  case IDLN_OP_DCL:
    cbe_skel_do_op_dcl(ski);
    break;
  case IDLN_PARAM_DCL:
    cbe_skel_do_param_dcl(ski);
    break;
  case IDLN_STRING:
    cbe_skel_do_string(ski);
    break;
  case IDLN_TYPE_ANY:
    cbe_skel_do_type_any(ski);
    break;
  case IDLN_TYPE_ARRAY:
    cbe_skel_do_type_array(ski);
    break;
  case IDLN_TYPE_BOOLEAN:
    cbe_skel_do_type_boolean(ski);
    break;
  case IDLN_TYPE_CHAR:
    cbe_skel_do_type_char(ski);
    break;
  case IDLN_TYPE_DCL:
    cbe_skel_do_type_dcl(ski);
    break;
  case IDLN_TYPE_ENUM:
    cbe_skel_do_type_enum(ski);
    break;
  case IDLN_TYPE_FIXED:
    cbe_skel_do_type_fixed(ski);
    break;
  case IDLN_TYPE_FLOAT:
    cbe_skel_do_type_float(ski);
    break;
  case IDLN_TYPE_INTEGER:
    cbe_skel_do_type_integer(ski);
    break;
  case IDLN_TYPE_OBJECT:
    cbe_skel_do_type_object(ski);
    break;
  case IDLN_TYPE_OCTET:
    cbe_skel_do_type_octet(ski);
    break;
  case IDLN_TYPE_SEQUENCE:
    cbe_skel_do_type_sequence(ski);
    break;
  case IDLN_TYPE_STRING:
    cbe_skel_do_type_string(ski);
    break;
  case IDLN_TYPE_STRUCT:
    cbe_skel_do_type_struct(ski);
    break;
  case IDLN_TYPE_UNION:
    cbe_skel_do_type_union(ski);
    break;
  case IDLN_TYPE_WIDE_CHAR:
    cbe_skel_do_type_wide_char(ski);
    break;
  case IDLN_TYPE_WIDE_STRING:
    cbe_skel_do_type_wide_string(ski);
    break;
  case IDLN_UNARYOP:
    cbe_skel_do_unaryop(ski);
    break;
  case IDLN_WIDE_CHAR:
    cbe_skel_do_wide_char(ski);
    break;
  case IDLN_WIDE_STRING:
    cbe_skel_do_wide_string(ski);
    break;
  case IDLN_ANY:
    g_error("IDLN_ANY not handled\n");
    break;
  }
}

static void
cbe_skel_do_attr_dcl(CBESkelInfo *ski)
{
  IDL_tree curop, curitem;
  GString *attrname = g_string_new(NULL);
  CBESkelInfo subski = *ski;

  for(curitem = IDL_ATTR_DCL(ski->tree).simple_declarations;
      curitem; curitem = IDL_LIST(curitem).next) {

    /* Fake the attribute get/set methods as operation declarations */
    IDL_tree ident, ns_data_save;
    int i;

    for (i = 0; i < 2; ++i) {

	    if (i && IDL_ATTR_DCL(ski->tree).f_readonly)
		    break;
	    /* Output the operation on this attribute */
	    g_string_sprintf(attrname, i ? "_set_%s" : "_get_%s",
			     IDL_IDENT(IDL_LIST(curitem).data).str);
	    ident = IDL_ident_new(strdup(attrname->str));
	    
	    /* Tell the ident where our namespace node is, and request a return value
	       if this is the _get operation */
	    IDL_IDENT_TO_NS(ident) = IDL_IDENT_TO_NS(IDL_LIST(curitem).data);
	    curop = IDL_op_dcl_new(0, i == 0 ?
				   IDL_ATTR_DCL(ski->tree).param_type_spec : NULL,
				   ident, NULL, NULL, NULL);
	    
	    curop->up = ski->tree->up;
	    subski.tree = curop;
	    
	    /* Save the namespace ident (IDL_GENTREE data) reference, assign
	       back to the temporary tree, output the operation, then restore
	       the namespace ident link */
	    ns_data_save = IDL_GENTREE(IDL_IDENT_TO_NS(IDL_LIST(curitem).data)).data;
	    IDL_GENTREE(IDL_IDENT_TO_NS(IDL_LIST(curitem).data)).data = ident;

	    if (i) {
		    /* The set routine also needs the value, so we
		       temporarily add that to the operation
		       declaration */
		    IDL_OP_DCL(curop).parameter_dcls = IDL_list_new(
			    IDL_param_dcl_new(IDL_PARAM_IN,
					      IDL_ATTR_DCL(ski->tree).param_type_spec,
					      IDL_ident_new(strdup("value"))));
	    }
	    
	    orbit_cbe_skel_process_piece(&subski);

	    /* Restore the fake link to the original in the namespace */
	    IDL_GENTREE(IDL_IDENT_TO_NS(IDL_LIST(curitem).data)).data = ns_data_save;

	    if (i) {
		    /* Free only what we've created for the fake node, so remove 
		       the attribute node element and then free the rest */
		    IDL_PARAM_DCL(IDL_LIST(
			    IDL_OP_DCL(curop).parameter_dcls).data).param_type_spec = NULL;
	    }
	    
	    /* Remove what we've "borrowed" from ATTR_DCL from the
	       fake curop node then free the rest */
	    IDL_OP_DCL(curop).op_type_spec = NULL;
	    IDL_tree_free(curop);
    }
  }

  g_string_free(attrname, TRUE);
}

static void
cbe_skel_do_binop(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_boolean(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_case_stmt(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_char(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_const_dcl(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_except_dcl(CBESkelInfo *ski)
{
  /* Nothing to do in skels */
  cbe_skel_do_type_struct(ski);
  g_warning("Finish up exception handling in skeletons\n");
}

static void
cbe_skel_do_fixed(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_float(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_forward_dcl(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_gentree(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_ident(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_integer(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

void
cbe_skel_print_skelptr(FILE *of, IDL_tree tree)
{
  char *id = NULL;
  IDL_tree curitem;

  switch(IDL_NODE_TYPE(tree)) {
  case IDLN_OP_DCL:
    id = IDL_ns_ident_to_qstring(IDL_IDENT_TO_NS(IDL_OP_DCL(tree).ident),
				 "_", 0);
    fprintf(of, "  skel_%s,\n", id);
    break;
  case IDLN_ATTR_DCL:
    id = IDL_ns_ident_to_qstring(IDL_IDENT_TO_NS(IDL_INTERFACE(IDL_get_parent_node(tree, IDLN_INTERFACE, NULL)).ident),
				 "_", 0);
    for(curitem = IDL_ATTR_DCL(tree).simple_declarations;
	curitem; curitem = IDL_LIST(curitem).data) {
      fprintf(of, "  skel_%s__get_%s,\n", id,
	      IDL_IDENT(IDL_LIST(curitem).data).str);
      if(!IDL_ATTR_DCL(tree).f_readonly)
	fprintf(of, "  skel_%s__set_%s,\n", id,
		IDL_IDENT(IDL_LIST(curitem).data).str);
    }
    break;
  default:
  }
  free(id);
}

void
cbe_skel_interface_print_relayer(CBESkelInfo *ski)
{
  /* XXX Prime candidate for optimization here. Is using gperf reasonable?
     In any case, there IS a better way ;-)
     -- Elliot */
  /* XXX this won't work with inheritance. Fix. */
  char *id;
  IDL_tree curitem, curdcl;

  id = IDL_ns_ident_to_qstring(IDL_IDENT_TO_NS(IDL_INTERFACE(ski->tree).ident), "_", 0);
  fprintf(ski->of, "static ORBitSkeleton get_skel_%s(POA_%s *servant, GIOPRecvBuffer *_ORBIT_recv_buffer, gpointer *epv)\n", id, id);
  fprintf(ski->of, "{\n");

  for(curitem = IDL_INTERFACE(ski->tree).body;
      curitem; curitem = IDL_LIST(curitem).next) {

    switch(IDL_NODE_TYPE(IDL_LIST(curitem).data)) {

    case IDLN_OP_DCL:
      fprintf(ski->of, "  if(!strcmp(_ORBIT_recv_buffer->message.u.request.operation, \"%s\")) {\n",
	      IDL_IDENT(IDL_OP_DCL(IDL_LIST(curitem).data).ident).str);
      fprintf(ski->of, "    *epv = servant->vepv->%s_epv->%s;\n", id,
	      IDL_IDENT(IDL_OP_DCL(IDL_LIST(curitem).data).ident).str);
      fprintf(ski->of,
	      "    return (gpointer)skel_%s_%s;\n",
	      id, IDL_IDENT(IDL_OP_DCL(IDL_LIST(curitem).data).ident).str);
      fprintf(ski->of, "  } else");
      break;

    case IDLN_ATTR_DCL:
      for(curdcl = IDL_ATTR_DCL(IDL_LIST(curitem).data).simple_declarations;
	  curdcl; curdcl = IDL_LIST(curdcl).next) {

	fprintf(ski->of, "  if(!strcmp(_ORBIT_recv_buffer->message.u.request.operation, \"_get_%s\")) {\n",
		IDL_IDENT(IDL_LIST(curdcl).data).str);
	fprintf(ski->of, "    *epv = servant->vepv->%s_epv->_get_%s;\n", id,
		IDL_IDENT(IDL_LIST(curdcl).data).str);
	fprintf(ski->of,
		"    return (gpointer)skel_%s__get_%s;\n",
		id,
		IDL_IDENT(IDL_LIST(curdcl).data).str);
	fprintf(ski->of, "  } else");

	if(!IDL_ATTR_DCL(IDL_LIST(curitem).data).f_readonly) {
	  fprintf(ski->of,
		  "  if(!strcmp(_ORBIT_recv_buffer->message.u.request.operation, \"_set_%s\")) {\n",
		  IDL_IDENT(IDL_LIST(curdcl).data).str);
	  fprintf(ski->of, "    *epv = servant->vepv->%s_epv->_set_%s;\n", id,
		IDL_IDENT(IDL_LIST(curdcl).data).str);
	  fprintf(ski->of,
		  "    return (gpointer)skel_%s__set_%s;\n",
		  id, IDL_IDENT(IDL_LIST(curdcl).data).str);
	  fprintf(ski->of, "  } else");
	}
      }
      break;
    default:
    }
  }

  fprintf(ski->of, "  {\n");
  fprintf(ski->of, "    g_error(\"Unknown operation\");\n");
  fprintf(ski->of, "  }\n");
  fprintf(ski->of, "  return NULL;\n");


  fprintf(ski->of, "}\n");
}

static void
cbe_skel_do_interface(CBESkelInfo *ski)
{
  CBESkelInfo subski = *ski;
  char *id, *id2;
  IDL_tree curitem;
  int i;

  subski.tree = IDL_INTERFACE(ski->tree).body;
  if(subski.tree)
    orbit_cbe_skel_process_piece(&subski);

  id = IDL_ns_ident_to_qstring(IDL_IDENT_TO_NS(IDL_INTERFACE(ski->tree).ident), "_", 0);

  cbe_skel_interface_print_relayer(ski);

  fprintf(ski->of,
	  "void POA_%s__init(POA_%s *servant, CORBA_Environment *env)\n",
	  id, id);
  fprintf(ski->of, "{\n");
  fprintf(ski->of, "  static PortableServer_ClassInfo class_info = {(gpointer)&get_skel_%s, \"IDL:%s:1.0\"};\n",
	  id, id);

  fprintf(ski->of,
	  "  PortableServer_ServantBase__init(((PortableServer_ServantBase *)servant), env);\n");

  for(curitem = IDL_INTERFACE(ski->tree).inheritance_spec; curitem;
      curitem = IDL_LIST(curitem).next) {
    id2 = IDL_ns_ident_to_qstring(IDL_IDENT_TO_NS(IDL_LIST(curitem).data),
				  "_", 0);
    fprintf(ski->of, "  POA_%s__init(((POA_%s *)servant), env);\n", id2, id2);
    free(id2);
  }

  fprintf(ski->of, "  ORBIT_OBJECT_KEY(servant->_private)->class_info = &class_info;\n");
  
  fprintf(ski->of, "}\n\n");

  fprintf(ski->of,
	  "void POA_%s__fini(POA_%s *servant, CORBA_Environment *env)\n",
	  id, id);
  fprintf(ski->of, "{\n");
  if(IDL_INTERFACE(ski->tree).inheritance_spec)
    {
      for(i = IDL_list_length(IDL_INTERFACE(ski->tree).inheritance_spec) - 1;
	  i >= 0; i--) {
	curitem = IDL_list_nth(IDL_INTERFACE(ski->tree).inheritance_spec, i);
	id2 = IDL_ns_ident_to_qstring(IDL_IDENT_TO_NS(IDL_LIST(curitem).data),
				      "_", 0);
	fprintf(ski->of, "  POA_%s__fini(((POA_%s *)servant), env);\n",
		id2, id2);
	free(id2);
      }
    }
  fprintf(ski->of, "  PortableServer_ServantBase__fini(servant, env);\n");
  fprintf(ski->of, "}\n\n");
}

static void
cbe_skel_do_list(CBESkelInfo *ski)
{
  IDL_tree curitem;
  CBESkelInfo subski = *ski;

  for(curitem = ski->tree; curitem; curitem = IDL_LIST(curitem).next) {
    subski.tree = IDL_LIST(curitem).data;
    orbit_cbe_skel_process_piece(&subski);
  }
}

static void
cbe_skel_do_member(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_module(CBESkelInfo *ski)
{
  char *id;

  id = IDL_ns_ident_to_qstring(IDL_IDENT_TO_NS(IDL_MODULE(ski->tree).ident), "_", 1);
  fprintf(ski->of, "/***************** Begin module %s ***************/\n", id);

  ski->tree = IDL_MODULE(ski->tree).definition_list;
  orbit_cbe_skel_process_piece(ski);
  fprintf(ski->of, "/***************** End module %s ***************/\n", id);
  free(id);
}

static void
cbe_skel_do_none(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_op_dcl(CBESkelInfo *ski)
{
  /* XXX we don't handle contexts here yet - fix that */
  char *id, *id2;
  IDL_tree curitem;
  int level, i, n;
  CBEMarshalInfo mi = {ski->of, NULL, NULL, NULL, NULL};
  CBEDemarshalInfo dmi = {ski->of, NULL, NULL, NULL, NULL, TRUE, FALSE};
  GString *tmpstr = g_string_new(NULL);

  id = IDL_ns_ident_to_qstring(IDL_IDENT_TO_NS(IDL_OP_DCL(ski->tree).ident),
			       "_", 0);

  if(IDL_OP_DCL(ski->tree).op_type_spec && IDL_OP_DCL(ski->tree).f_oneway) {
    g_error("[%s] You cannot have a return value from a oneway operation!\n", id);
  }
  curitem = IDL_get_parent_node(ski->tree, IDLN_INTERFACE, &level);

  g_assert(curitem);
  id2 = IDL_ns_ident_to_qstring(IDL_IDENT_TO_NS(IDL_INTERFACE(curitem).ident), "_", 0);

  fprintf(ski->of, "\nvoid\nskel_%s(POA_%s *_ORBIT_servant, GIOPRecvBuffer *_ORBIT_recv_buffer, CORBA_Environment *ev,\n", id, id2);

  orbit_cbe_write_typespec(ski->of, IDL_OP_DCL(ski->tree).op_type_spec);
  orbit_cbe_param_printptrs(ski->of, IDL_OP_DCL(ski->tree).op_type_spec,
			    DATA_RETURN);

  fprintf(ski->of, " (*%s)(PortableServer_Servant servant, ",
	  IDL_IDENT(IDL_OP_DCL(ski->tree).ident).str);

  for(curitem = IDL_OP_DCL(ski->tree).parameter_dcls; curitem; curitem = IDL_LIST(curitem).next) {
    cbe_print_param_dcl(ski->of, IDL_LIST(curitem).data);
    fprintf(ski->of, ", ");
  }
  fprintf(ski->of, "CORBA_Environment *ev))\n");

  fprintf(ski->of, "{\n");

  /* Print the variables into with we will be demarshalling
     & marshalling parameters */
  for(curitem = IDL_OP_DCL(ski->tree).parameter_dcls; curitem;
      curitem = IDL_LIST(curitem).next) {
    fprintf(ski->of, "  ");
    cbe_print_var_dcl(ski->of, IDL_LIST(curitem).data);
    fprintf(ski->of, ";\n");
  }

  if(!IDL_OP_DCL(ski->tree).f_oneway)
    fprintf(ski->of, "  GIOPSendBuffer *_ORBIT_send_buffer;\n");

  if(IDL_OP_DCL(ski->tree).op_type_spec) {
    fprintf(ski->of, "  ");
    cbe_print_param_dcl(ski->of, IDL_OP_DCL(ski->tree).op_type_spec);
    fprintf(ski->of, ";\n");
  }
  fprintf(ski->of, "\n");

  if(cbe_op_dcl_nparams(ski->tree, DATA_INOUT|DATA_IN) > 0) {

    fprintf(ski->of, "  if(giop_msg_conversion_needed(GIOP_MESSAGE_BUFFER(_ORBIT_recv_buffer))) {\n");

    do {
      for(curitem = IDL_OP_DCL(ski->tree).parameter_dcls;
	  curitem; curitem = IDL_LIST(curitem).next) {

	if(IDL_PARAM_DCL(IDL_LIST(curitem).data).attr == IDL_PARAM_IN
	   || IDL_PARAM_DCL(IDL_LIST(curitem).data).attr == IDL_PARAM_INOUT) {
	
	  if(IDL_PARAM_DCL(IDL_LIST(curitem).data).attr == IDL_PARAM_IN)
	    n = orbit_cbe_param_numptrs(mi.param, DATA_IN);
	  else
	    n = orbit_cbe_param_numptrs(mi.param, DATA_INOUT) /* only getting, not setting */ - 1;
	
	  dmi.previous_param = dmi.param;
	  dmi.param = IDL_LIST(curitem).data;
	
	  g_string_assign(tmpstr, "");
	  for(i = 0; i < n; i++)
	    g_string_append_c(tmpstr, '*');
	  g_string_sprintfa(tmpstr, "%s",
			    IDL_IDENT(IDL_PARAM_DCL(dmi.param).simple_declarator).str);
	  dmi.param_name = tmpstr->str;
	  fprintf(ski->of, "\n\n  /* demarshal parameter %s */\n", dmi.param_name);
	  cbe_output_demarshaller(dmi);
	}
      }
      if(dmi.byteswap_version)
	fprintf(ski->of, "  } else {\n");

    } while(dmi.byteswap_version-- > 0);

    fprintf(ski->of, "  }\n\n");

  }
  
  if(IDL_OP_DCL(ski->tree).op_type_spec)
    fprintf(ski->of, "  _ORBIT_retval = ");
  fprintf(ski->of, "%s(_ORBIT_servant, ",
	  IDL_IDENT(IDL_OP_DCL(ski->tree).ident).str);

  for(curitem = IDL_OP_DCL(ski->tree).parameter_dcls; curitem;
      curitem = IDL_LIST(curitem).next) {
    if(IDL_PARAM_DCL(IDL_LIST(curitem).data).attr != IDL_PARAM_IN)
      fprintf(ski->of, "&");
    fprintf(ski->of, "%s,",
	    IDL_IDENT(IDL_PARAM_DCL(IDL_LIST(curitem).data).simple_declarator).str);
  }
  fprintf(ski->of, " ev);\n");

  if(!IDL_OP_DCL(ski->tree).f_oneway) {
    fprintf(ski->of,
	    "  _ORBIT_send_buffer = \n");
    fprintf(ski->of,
	    "    giop_send_reply_buffer_use(GIOP_MESSAGE_BUFFER(_ORBIT_recv_buffer)->connection, NULL,\n");
    fprintf(ski->of,
	    "	 _ORBIT_recv_buffer->message.u.request.request_id,\n");
    fprintf(ski->of,
	    "	 GIOP_NO_EXCEPTION);\n\n");


    if(cbe_op_dcl_nparams(ski->tree, DATA_INOUT|DATA_OUT|DATA_RETURN) > 0) {

      mi.param = NULL;

      if(IDL_OP_DCL(ski->tree).op_type_spec) {
	fprintf(ski->of, "  /* marshal return value */\n");
	mi.param = IDL_OP_DCL(ski->tree).op_type_spec;
	mi.param_name = "_ORBIT_retval";
	cbe_output_marshaller(mi);
      }

      for(curitem = IDL_OP_DCL(ski->tree).parameter_dcls;
	  curitem; curitem = IDL_LIST(curitem).next) {
	
	if(IDL_PARAM_DCL(IDL_LIST(curitem).data).attr == IDL_PARAM_OUT
	   || IDL_PARAM_DCL(IDL_LIST(curitem).data).attr == IDL_PARAM_INOUT) {

	  mi.previous_param = mi.param;
	  mi.param = IDL_LIST(curitem).data;

	  if(IDL_PARAM_DCL(IDL_LIST(curitem).data).attr == IDL_PARAM_OUT)
	    n = orbit_cbe_param_numptrs(mi.param, DATA_OUT);
	  else
	    n = orbit_cbe_param_numptrs(mi.param, DATA_INOUT); /* Clarify */

	  g_string_assign(tmpstr, "");
#if 0
	  for(i = 0; i < n; i++)
	    g_string_append_c(tmpstr, '*');
#endif
	  g_string_sprintfa(tmpstr, "%s",
			    IDL_IDENT(IDL_PARAM_DCL(mi.param).simple_declarator).str);
	  mi.param_name = tmpstr->str;

	  fprintf(ski->of, "  /* marshal parameter %s */\n", mi.param_name);
	  cbe_output_marshaller(mi);
	}
      }
    }

    fprintf(ski->of, "  giop_send_buffer_write(_ORBIT_send_buffer);\n");
    fprintf(ski->of, "  giop_send_buffer_unuse(_ORBIT_send_buffer);\n");

    if(IDL_OP_DCL(ski->tree).op_type_spec
       && IDL_NODE_TYPE(IDL_OP_DCL(ski->tree).op_type_spec) == IDLN_TYPE_STRING) {
      fprintf(ski->of, "  CORBA_free(_ORBIT_retval);\n");
    }

    for(curitem = IDL_OP_DCL(ski->tree).parameter_dcls;
	curitem; curitem = IDL_LIST(curitem).next) {
      if(IDL_PARAM_DCL(IDL_LIST(curitem).data).attr == IDL_PARAM_OUT
	 || IDL_PARAM_DCL(IDL_LIST(curitem).data).attr == IDL_PARAM_INOUT)
	switch(IDL_NODE_TYPE(IDL_PARAM_DCL(IDL_LIST(curitem).data).param_type_spec)) {
	case IDLN_TYPE_STRING:
	  fprintf(ski->of, "  CORBA_free(%s);\n",
		  IDL_IDENT(IDL_PARAM_DCL(IDL_LIST(curitem).data).simple_declarator).str);
	  break;
	default:
	}
    }
  }

  fprintf(ski->of, "}\n\n");
  g_string_free(tmpstr, TRUE);
  free(id); free(id2);
}

static void
cbe_skel_do_param_dcl(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_string(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_type_any(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_type_array(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_type_boolean(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_type_char(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_type_dcl(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_type_enum(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_type_fixed(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_type_float(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_type_integer(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_type_object(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_type_octet(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_type_sequence(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_type_string(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_type_struct(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_type_union(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_type_wide_char(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_type_wide_string(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_unaryop(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_wide_char(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

static void
cbe_skel_do_wide_string(CBESkelInfo *ski)
{
  g_assert(!"Not yet implemented");
}

/* Dead code repo */
#if 0 /* not sure if we need this at all, unless we think up
	 a generic version of the relayer */
  fprintf(ski->of, "static ORBitSkeleton %s_skeletons[] = {\n", id);
  for(curitem = IDL_INTERFACE(ski->tree).body; curitem; curitem = IDL_LIST(curitem).next) {
    cbe_skel_print_skelptr(ski->of, IDL_LIST(curitem).data);
  }
  fprintf(ski->of, "};\n\n");
#endif

