/**************************************************************************

    main.c (Driver program for the IDL parser & backend)

    Copyright (C) 1998 Elliot Lee

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: main.c,v 1.22 1998/12/08 21:03:39 andrewtv Exp $

***************************************************************************/

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <glib.h>
#include <libIDL/IDL.h>

#include "orbit-idl.h"

/************ Function Prototypes *********/
static void orbit_idl_usage(void);
static int orbit_idl_to_backend(const char *filename, const char *cpp_args);

/************ Multi-function (NOT global) variables ********/

/************ Global variables *********/
gboolean enable_typecodes = FALSE,
  enable_inherited_ops = TRUE,
  enable_skeleton_impl = FALSE,
  enable_dcall = FALSE;
int debuglevel = IDL_WARNING1;


/*--------------------------------
  int main(int argc, char *argv[])

  Input: argument count, argument strings (including program name)
  Output: error code

  Description:
      Alters state based on options (argument strings that begin with a -)
      Iterates through non-option arguments.
	  Calls orbit_idl_to_c() on these non-option arguments
	  with the assumption that they are idl filenames.
  --------------------------------
*/
int main(int argc, char *argv[])
{
  GString *cpp_args;
  int i, nprocessed;

  cpp_args = g_string_new(NULL);

  for(i = 1; i < argc; i++) {
    if(argv[i][0] == '-') {
      if (!strcmp(argv[i]+1, "-libIDL-version")) {
	g_print("libIDL %s ", IDL_get_libver_string());
	g_print("(CORBA %s)\n", IDL_get_IDLver_string());
	return 0;
      }
      switch(argv[i][1]) {
      case 'd':
	debuglevel = atoi(argv[i] + 2);
	break;
      case 'D':
      case 'I':
	g_string_append(cpp_args, argv[i]);
	g_string_append_c(cpp_args, ' ');
	break;
      case 'E':
	if(!strcmp(argv[i]+2, "typecodes"))
	  enable_typecodes ^= TRUE;
	else if(!strcmp(argv[i]+2, "inherited_ops"))
	  enable_inherited_ops ^= TRUE;
	else if(!strcmp(argv[i]+2, "skeleton_impl"))
	  enable_skeleton_impl ^= TRUE;
	else if(!strcmp(argv[i]+2, "dcall"))
	  enable_dcall ^= TRUE;
	break;
      default:
	orbit_idl_usage();
	return 1;
      }
    }
  }

  for(i = 1, nprocessed = 0; i < argc; i++) {
    if(argv[i][0] && argv[i][0] != '-')
      nprocessed += orbit_idl_to_backend(argv[i], cpp_args->str);
  }

  if(nprocessed == 0) {
    orbit_idl_usage();
    return 1;
  }

  return 0;
}

/* no input, no output. Displays a help message. */
static void
orbit_idl_usage(void)
{
  g_print("\n"
	  "Usage: orbit-idl [options] file.idl ...\n"
	  "\n"
	  "Generic options\n"
	  "  -dX                Debug level, from 0 to %d.  Default is %d.\n"
	  "  -Efeature          Enable feature, where feature can be one of:\n"
	  "      skeleton_impl  Generate a template implementation (destructive)\n"
	  "      inherited_ops  Disable output of inherited operations\n"
	  "\n"
	  "Preprocessor options\n"
	  "  -Ddefine[=value]   Define value in preprocessor\n"
          "  -Ipath             Add search path for include files\n"
	  "\n",
	  IDL_WARNINGMAX, debuglevel);
}


/*---------------
  orbit_idl_to_backend:
     Input: IDL filename, and any arguments to be passed to CPP
     Output: Number of IDL files (1 or 0) that were successfully processed
     Does: Reads in 'filename' & parses it into a tree, using libIDL.
	   Calls the backend producer.
 ---------------
*/
static int
orbit_idl_to_backend(const char *filename, const char *cpp_args)
{
  IDL_ns namespace;
  IDL_tree tree;
  int x;
  char *basename, *ctmp;

  /* XXX todo: allow user to say "orbit-idl t" and have to read
     t.idl
  */
  x = IDL_parse_filename(filename, cpp_args, NULL, &tree, &namespace,
			 IDLF_TYPECODES, debuglevel);

  if(IDL_SUCCESS != x) {
    if (x == -1)
      g_warning("Parse of %s failed: %s", filename, g_strerror(errno));
    else
      g_warning("Parse of %s failed", filename);
    return 0;
  }

  basename = g_strdup(filename);
  ctmp = strrchr(basename, '.');
  if(ctmp) *ctmp = '\0';

  /* For now we only support the C backend :) */
  orbit_IDL_tree_to_c(basename, namespace, tree);

  g_free(basename);

  return 1;
}
