/* Atomixed -- the level-/themeeditor for atomix.
 * Copyright (C) 1999-2000 Jens Finke
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "theme_ctrl.h"
#include "main.h"
#include "support.h"
#include <stdlib.h>

/*=================================================================
 
  Declaration of global variables

  ---------------------------------------------------------------*/
Theme *selected_theme;     /* actual selected theme */
GList *loaded_themes;      /* the list of all loaded themes */


/*=================================================================
 
  Declaration of internal functions
  ---------------------------------------------------------------*/

void theme_ctrl_set_theme_modified(Theme *theme);

void update_theme_item(Theme *theme, GtkCList *clist);

void theme_ctrl_update_view(void);

void theme_ctrl_select_list_row(gint pos);

void destroy_theme_item(Theme *theme, gpointer user_data);

gint find_theme_equal_name(Theme *theme, const gchar *theme_name);

gint find_theme_equal_dir(Theme *theme, const gchar *dir_name);

gboolean exec_synchronous(gchar *dir, gchar *command);

void add_theme_name(Theme *theme, GList **list);

void clear_update_flag(Theme *theme, gpointer user_data);

gint theme_equal_name(Theme *theme_a, Theme *theme_b);

/* test suite */
void test_existance(const gchar *theme_name, const gchar *dir_name);
void check_locale(const gchar *theme_name);

/*=================================================================
 
  Theme Control initialisation and clean up

  ---------------------------------------------------------------*/
void theme_ctrl_init(void)
{
	GList *avail_themes;
	gint list_size;
	
	// create list of loaded levels
	avail_themes = theme_get_available_themes();
	list_size = g_list_length(avail_themes);

	if(list_size > 0)
	{
		int l;

		for(l = 0; l < list_size; l++)
		{
			Theme *theme;
			gchar *theme_name;

			theme_name = (gchar*)g_list_nth_data(avail_themes, l);
			theme = theme_load_xml(theme_name);
		       
			loaded_themes = g_list_append(loaded_themes, theme);
		}			
		selected_theme = (Theme*)g_list_nth_data(loaded_themes,0);
		
		// update graphical representation
		theme_ctrl_update_view();

		// select first level
		theme_ctrl_select_list_row(0);
	}
}

void theme_ctrl_destroy(void)
{
	g_list_foreach(loaded_themes, (GFunc)destroy_theme_item, NULL);
	g_list_free(loaded_themes);
}

void destroy_theme_item(Theme *theme, gpointer user_data)
{
	theme_destroy(theme);
}

/*=================================================================
 
  Functions of theme control.

  ---------------------------------------------------------------*/

void theme_ctrl_select_list_row(gint pos)
{
	GtkCList *clist;
	clist = GTK_CLIST(lookup_widget(get_app(), "theme_list"));

	gtk_clist_select_row(clist, pos, 0);
}


void theme_ctrl_update_view()
{
	GtkCList *clist;

	clist = GTK_CLIST(lookup_widget(get_app(), "theme_list"));
	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);

	g_list_foreach(loaded_themes, (GFunc)update_theme_item,
		       clist);

	gtk_clist_thaw(clist);
}

Theme* theme_ctrl_get_theme(gint row)
{
	GtkCList *clist;

	clist = GTK_CLIST(lookup_widget(get_app(), "theme_list"));
	return (Theme*)gtk_clist_get_row_data(clist, row);
}

Theme* theme_ctrl_get_actual_theme(void)
{
	return selected_theme;
}

void theme_ctrl_select_row(gint row)
{
	selected_theme = theme_ctrl_get_theme(row);
}

void theme_ctrl_set_actual_theme_modified(void)
{
	theme_ctrl_set_theme_modified(selected_theme);
}

void theme_ctrl_actual_theme_renamed(void)
{
	theme_ctrl_set_actual_theme_modified();      
	theme_ctrl_update_view();
	theme_ctrl_select_list_row(g_list_index(loaded_themes, selected_theme));
}

void theme_ctrl_new_theme(const gchar *name, const gchar *dir_name)
{	
	gchar *path;
	gchar *command;
	gchar *home_dir;
	gchar *tmp;

	/* FIXME: no absolute paths here */
	home_dir = g_get_home_dir();
	tmp = g_concat_dir_and_file(".atomix/themes", dir_name);
	path = g_concat_dir_and_file(home_dir, tmp);
	g_free(tmp);

	/* create directory */
	command = g_strconcat("mkdir ", dir_name, NULL);
	if(exec_synchronous("~/.atomix/themes", command))
	{
		GtkWidget *clist;
		Theme *theme;
		gint row;

		/* initialise new theme */
		theme = theme_new();	
		theme->name = g_strdup(name);
		theme->path = g_strdup(path);
		theme->modified = TRUE;
		
		/* add it to the list */
		loaded_themes = g_list_insert_sorted(loaded_themes, theme,
						     (GCompareFunc) theme_equal_name);	
		
		/* update view */
		theme_ctrl_update_view();
		
		clist = lookup_widget(get_app(), "theme_list");
		row = gtk_clist_find_row_from_data(GTK_CLIST(clist), (gpointer) theme);
		theme_ctrl_select_list_row(row);
	}
	else
	{
		g_free(path);
	}

	g_free(command);
}

gint theme_equal_name(Theme *theme_a, Theme *theme_b)
{
	g_return_val_if_fail(theme_a != NULL, 0);
	g_return_val_if_fail(theme_b != NULL, 0);

	g_return_val_if_fail(theme_a->name != NULL, 0);
	g_return_val_if_fail(theme_b->name != NULL, 0);

	return g_strcasecmp(theme_a->name, theme_b->name);
}

Theme* theme_ctrl_get_theme_by_name(const gchar *name)
{
	GList *result;
	
	result = g_list_find_custom(loaded_themes, (gchar*)name, 
				    (GCompareFunc)find_theme_equal_name);
	if(result == NULL)
	{
		return NULL;
	}
	else
	{
		return (Theme*) result->data;
	}
}

GList* 
theme_ctrl_get_theme_name_list(void)
{
	GList *list = NULL;

	g_list_foreach(loaded_themes, (GFunc) add_theme_name, &list);

	return list;
}

void 
add_theme_name(Theme *theme, GList **list)
{
	(*list) = g_list_append((*list), theme->name); 
}

void theme_ctrl_does_theme_exist(gchar *theme_name, gchar *dir_name, 
				 gboolean *name_exists, gboolean *dir_exists)
{
	GList *result;
	
	if(theme_name != NULL)
	{
		result = g_list_find_custom(loaded_themes, theme_name, 
					    (GCompareFunc) find_theme_equal_name);
		*name_exists = (result != NULL);
	}
	else
	{
		*name_exists = FALSE;
	}

	if(dir_name != NULL)
	{
		result = g_list_find_custom(loaded_themes, dir_name,
					    (GCompareFunc) find_theme_equal_dir);
		*dir_exists = (result != NULL);
	}
	else
	{
		*dir_exists = FALSE;
	}
}

gboolean theme_ctrl_check_for_modifications(void)
{
	gboolean result = FALSE;
	gint i = 0;
	gint length = g_list_length(loaded_themes);

	for(i = 0; i < length; i++)
	{
		Theme *theme = (Theme*) g_list_nth_data(loaded_themes, i);
		result = (result || theme->modified);
	}

	return result;
}

void theme_ctr_clear_update_flag(void)
{
	g_list_foreach(loaded_themes, (GFunc) clear_update_flag, NULL);
}

void clear_update_flag(Theme *theme, gpointer user_data)
{
	if(theme != NULL)
	{
		theme->need_update = FALSE;
	}
}

void theme_ctrl_save_actual_theme(void)
{
	if(selected_theme->modified)
	{
		theme_ctrl_save_theme(selected_theme, NULL);
	}
}

void theme_ctrl_save_theme(Theme *theme, gpointer data)
{
	gchar *file;

	g_return_if_fail(theme != NULL);

	file = g_concat_dir_and_file(theme->path, "theme");
	theme_save_xml(theme, file);
	g_free(file);

	theme->modified = FALSE;
}

void theme_ctrl_save_all_themes(void)
{
	gint length = g_list_length(loaded_themes);
	gint index;
	
	for(index = 0; index < length; index++)
	{
		Theme *theme = (Theme*)g_list_nth_data(loaded_themes, index);
		if(theme->modified)
		{
			theme_ctrl_save_theme(theme, NULL);
		}
	}
	
	theme_ctrl_update_view();		
	index = g_list_index(loaded_themes, selected_theme);
	theme_ctrl_select_list_row(index);
}

/* functions which work on arbitrary themes */
void theme_ctrl_set_theme_modified(Theme *theme)
{
	gint row;

	g_return_if_fail(theme != NULL);

	if(!theme->modified)
	{
		theme->modified = TRUE;

		theme_ctrl_update_view();
		row = g_list_index(loaded_themes, selected_theme);
		theme_ctrl_select_list_row(row);
	}
	theme->need_update = TRUE;
}

void theme_ctrl_remove_image(Theme *theme, ThemeElement *element)
{
	g_return_if_fail(theme != NULL);
	g_return_if_fail(element != NULL);

	if(element->file != NULL)
	{
		gchar *path;
		gchar *command;
		path = g_concat_dir_and_file(theme->path, element->file);
		command = g_strconcat("rm ", path, NULL);
		
		exec_synchronous(NULL, command);
		
		g_free(command);
		g_free(path);
	}
}

void theme_ctrl_new_selector_image(gchar *path)
{
	gchar *command;
	
	g_return_if_fail(selected_theme!=NULL);
	
	/* copy new image file into theme dir */
	command = g_strconcat("cp ", path, " .", NULL);
	exec_synchronous(selected_theme->path, command);
	g_free(command);

	theme_set_selector_image(selected_theme, g_basename(path));
	theme_ctrl_set_actual_theme_modified();
}

/*=================================================================
 
  Internal helper functions

  ---------------------------------------------------------------*/
gboolean theme_ctrl_copy_image_into_theme_dir(gchar *path, Theme *theme)
{
	gchar *command;
	gboolean result;

	g_return_val_if_fail(path != NULL, FALSE);
	g_return_val_if_fail(theme != NULL, FALSE);
	
	command = g_strconcat("cp ", path, " ", theme->path, NULL);

	result = exec_synchronous(NULL, command);
	g_free(command);

	return result;
}



void update_theme_item(Theme *theme, GtkCList *clist)
{
	gchar *text[1];
	gint clist_row;

	if(theme->modified)
	{
		text[0] = g_strconcat(theme->name, "*", NULL);
	}
	else
	{
		text[0] = g_strdup(theme->name);
	}
	
	clist_row = gtk_clist_append(clist, text);
	gtk_clist_set_row_data(clist, clist_row, theme);	
}

gint find_theme_equal_name(Theme *theme, const gchar *theme_name)
{
	return g_strcasecmp(theme_name, theme->name); 
}

gint find_theme_equal_dir(Theme *theme, const gchar *dir_name)
{
	gchar *dir;

	dir = g_basename(theme->path);

	return g_strcasecmp(dir_name, dir);
}

gboolean theme_ctrl_is_locale_theme(Theme *theme)
{
	gchar *locale_dir;
	guint str_len;

	g_return_val_if_fail(theme != NULL, FALSE);

	locale_dir = g_concat_dir_and_file(g_get_home_dir(), ".atomix/themes");	
	str_len = (guint) strlen(locale_dir);

	if(g_strncasecmp(locale_dir, theme->path, str_len) == 0)
	{
		g_free(locale_dir);
		return TRUE;
	}
	else
	{
		g_free(locale_dir);
		return FALSE;
	}
}

gboolean exec_synchronous(gchar *dir, gchar *command)
{
	gint result;
	gchar *full_command = NULL;

	if(dir != NULL)
	{
		
		full_command = g_strconcat("cd ", dir, "; ", command, NULL);
	}
	else
	{
		full_command = g_strdup(command);
	}

	// g_print("execute command: > %s\n", full_command);
	result = system(full_command); 
	g_free(full_command);

	return !((result == -1) || (result == 127));
}

gboolean theme_ctrl_make_theme_locale(Theme *theme)
{
	gchar *dir_name;
	gchar *locale_dir;
	gchar *theme_dir;
	gchar *command;
	gboolean result = FALSE;
		
	g_return_val_if_fail(theme!=NULL, FALSE);
	g_return_val_if_fail(theme->path!=NULL, FALSE);

	locale_dir = g_concat_dir_and_file(g_get_home_dir(), ".atomix/themes");	
	dir_name = g_basename(theme->path);
	theme_dir = g_concat_dir_and_file(locale_dir, dir_name);

	/* check if dir already exists */
	if(g_file_test(theme_dir, G_FILE_TEST_ISFILE | G_FILE_TEST_ISDIR | G_FILE_TEST_ISLINK))
	{
		command = g_strconcat("rmdir ", dir_name, NULL);
		exec_synchronous(locale_dir, command);
		g_free(command);
	}
	       
	command = g_strconcat("mkdir ", dir_name, NULL);
	if(exec_synchronous(locale_dir, command))
	{
		g_free(command);

		command = g_strconcat("cp * ", theme_dir, NULL);
		if(exec_synchronous(theme->path, command))
		{
			result = TRUE;
		}
	}

	if(result)
	{
		/* now, set the new path */
		if(theme->path)
		{
			g_free(theme->path);
		}
		theme->path = g_strdup(theme_dir);
	}

	g_free(command);
	g_free(locale_dir);
	g_free(theme_dir);


	return result;
}


/**********************************************************************/
/*                Test Suite functions                                */
/**********************************************************************/

void theme_ctrl_test_suite(void)
{
	test_existance("fabian", "fabian");    // t/t
	test_existance("default", "default");  // t/t
	test_existance("fabian", "fabian2");   // t/f
	test_existance("default", "default2"); // t/f
	test_existance("katomic2", "katomic"); // f/t
	test_existance("def", "katomic");      // f/t
	test_existance(NULL, NULL);            // f/f
	
	g_print("================================\n");	

	check_locale("fabian");
	check_locale("katomic");
	check_locale("default");
	check_locale("wer?");	
}

void check_locale(const gchar *theme_name)
{
	Theme *theme;
	theme = theme_ctrl_get_theme_by_name(theme_name);

	if(theme!= NULL)
	{
		gboolean result;

		result = theme_ctrl_is_locale_theme(theme);
		g_print("Theme %s is locale: %i\n\n", theme_name, result);

		if(!result)
		{
			gboolean locale_result;
			locale_result = theme_ctrl_make_theme_locale(theme);

			if(locale_result)
			{
				result = theme_ctrl_is_locale_theme(theme);
				g_print("    Theme %s is locale: %i\n\n", theme_name, result);
			}
			else
			{
				g_print("    ** Warning: error during make_theme_locale\n\n");
			}
		}
	}
	else
	{
		g_print(" ** Warning: Theme %s not found.\n\n", theme_name);
	}
}

void test_existance(const gchar *theme_name, const gchar *dir_name)
{
	gboolean dir_exists = 0;
	gboolean name_exists = 0;

	theme_ctrl_does_theme_exist(theme_name, dir_name, &name_exists, &dir_exists);
	
	g_print("  Theme with name %s: %i\n", theme_name, name_exists);
	g_print("  Dir with name %s: %i\n\n", dir_name, dir_exists);	
}	


/**********************************************************************/
