/*
 * Copyright 2000 Murray Cumming
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <bakery/App/App_WithDoc_Gtk.h>
#include <bakery/App/GtkDialogs.h>
#include <bakery/App/libegg/recent-files/egg-recent-view-gtk.h>
#include <bakery/App/libegg/recent-files/egg-recent-view.h>
#include <bakery/App/libegg/recent-files/egg-recent-model.h>
#include <libgnomevfsmm/utils.h> //For escape_path_string()
#include <libgnomevfsmm/mime-handlers.h> //For type_is_known(). 
#include <gtkmm/toolbutton.h>
#include <gtkmm/stock.h>
//#include <fstream>
#include <algorithm>
#include <libintl.h>

//#include <gtk/gtkfilesel.h>


namespace //anonymous
{

void c_callback_recent_files_activate(EggRecentViewGtk* view, EggRecentItem* item, void* user_data)
{
  if(user_data)
  {
    Bakery::App_WithDoc_Gtk::RecentFileSlot* slot = static_cast<Bakery::App_WithDoc_Gtk::RecentFileSlot*>(user_data);
    slot->operator()(item);    
  } 
}

} //anonymous namespace

namespace Bakery
{


//Initialize static member data:

App_WithDoc_Gtk::App_WithDoc_Gtk(const Glib::ustring& appname)
: App_WithDoc(appname),
  App_Gtk(appname),
  m_recent_files_model(0)
{
}

/// This constructor can be used with Gnome::Glade::Xml::get_derived_widget().
App_WithDoc_Gtk::App_WithDoc_Gtk(BaseObjectType* cobject, const Glib::ustring& appname)
: App_WithDoc(appname),
  App_Gtk(cobject, appname),
  Gtk::Window(cobject), //This is a virtual base class (not a direct base), so we must specify a constructor or the default constructor will be called, regardless of what the App_Gtk(cobject) constructor does. Derived classes must do this as well.
  m_recent_files_model(0)
{
  //TODO: appname.
}

  
App_WithDoc_Gtk::~App_WithDoc_Gtk()
{
  //TODO: Wrap it in a Glib::RefPtr<Glib::Object>?
  if(m_recent_files_model)
  {
    g_object_unref(G_OBJECT(m_recent_files_model));
    m_recent_files_model = 0;
  }
}


void App_WithDoc_Gtk::init()
{  
  App_WithDoc::init(); //Create document and ask to show it in the UI.
  
  init_layout();
    
  show();
}

void App_WithDoc_Gtk::init_toolbars()
{
  //Build part of the menu structure, to be merged in by using the "PH" placeholders:
  static const Glib::ustring ui_description =
    "<ui>"
    "  <toolbar name='Bakery_ToolBar'>"
    "    <placeholder name='Bakery_ToolBarItemsPH'>"
    "      <toolitem action='BakeryAction_File_New' />"
    "      <toolitem action='BakeryAction_File_Open' />"
    "      <toolitem action='BakeryAction_File_Save' />"
    "    </placeholder>"
    "  </toolbar>"
    "</ui>";

  add_ui_from_string(ui_description);
}

void App_WithDoc_Gtk::init_menus_file_recentfiles(const Glib::ustring& path)
{
  if(!m_mime_types.empty()) //"Recent-files" is useless unless it knows what documents (which MIME-types) to show.
  {
    //Add recent-files submenu:
    Gtk::MenuItem* pMenuItem = dynamic_cast<Gtk::MenuItem*>(m_refUIManager->get_widget(path));
    if(pMenuItem)
    {
      Gtk::Menu* pSubMenu_RecentFiles = pMenuItem->get_submenu();
      if(pSubMenu_RecentFiles)
      {
         m_recent_files_model = egg_recent_model_new(EGG_RECENT_MODEL_SORT_MRU);

         //Add the mime-types, so that it only shows those documents:
         for(type_list_strings::iterator iter = m_mime_types.begin(); iter != m_mime_types.end(); ++iter)
         {
           const Glib::ustring mime_type = *iter;
           if( Gnome::Vfs::Mime::type_is_known(mime_type) )
             egg_recent_model_add_filter_mime_type( m_recent_files_model, mime_type.c_str());
           else
           {
             g_warning("App_WithDoc_Gtk::init_menus_file_recentfiles(): MIME-type %s is not known to gnome-vfs", mime_type.c_str());
           }
         }

         //egg_recent_model_set_filter_mime_types (model, "", NULL);
         //egg_recent_model_set_filter_uri_schemes (model, "file", NULL);
         egg_recent_model_set_limit(m_recent_files_model, 10 /* this should be a global GNOME preference, I think. */);

         EggRecentViewGtk* view = egg_recent_view_gtk_new( GTK_WIDGET(pSubMenu_RecentFiles->gobj()), 0);
         egg_recent_view_set_model(EGG_RECENT_VIEW(view), m_recent_files_model);
         egg_recent_view_gtk_show_numbers(view, FALSE);

         m_slot_recent_files_activate = sigc::mem_fun(*this, &App_WithDoc_Gtk::on_recent_files_activate);

         //Connect the signal handler:
         g_signal_connect(G_OBJECT(view), "activate", G_CALLBACK (c_callback_recent_files_activate), &m_slot_recent_files_activate);
      }
    }
    else
    {
      std::cout << "debug: recent files menu not found" << std::endl;
    }
  }
}

void App_WithDoc_Gtk::init_menus_file()
{
  // File menu

  //Build actions:
  m_refFileActionGroup = Gtk::ActionGroup::create("BakeryFileActions");

  m_refFileActionGroup->add(Gtk::Action::create("BakeryAction_Menu_File", gettext("_File")));
  m_refFileActionGroup->add(Gtk::Action::create("BakeryAction_Menu_File_RecentFiles", gettext("_Recent Files")));

  //File actions
  m_refFileActionGroup->add(Gtk::Action::create("BakeryAction_File_New", Gtk::Stock::NEW),
                        sigc::mem_fun((App&)*this, &App::on_menu_file_new));
  m_refFileActionGroup->add(Gtk::Action::create("BakeryAction_File_Open", Gtk::Stock::OPEN),
                        sigc::mem_fun((App_WithDoc&)*this, &App_WithDoc::on_menu_file_open));

  //Remember thes ones for later, so we can disable Save menu and toolbar items:
  m_action_save = Gtk::Action::create("BakeryAction_File_Save", Gtk::Stock::SAVE);
  m_refFileActionGroup->add(m_action_save,
                        sigc::mem_fun((App_WithDoc&)*this, &App_WithDoc::on_menu_file_save));

  m_action_saveas = Gtk::Action::create("BakeryAction_File_SaveAs", Gtk::Stock::SAVE_AS);                   
  m_refFileActionGroup->add(m_action_saveas,
                        sigc::mem_fun((App_WithDoc&)*this, &App_WithDoc::on_menu_file_saveas));
                        
  m_refFileActionGroup->add(Gtk::Action::create("BakeryAction_File_Close", Gtk::Stock::CLOSE),
                        sigc::mem_fun((App_WithDoc&)*this, &App_WithDoc::on_menu_file_close));
  m_refFileActionGroup->add(Gtk::Action::create("BakeryAction_File_Exit", Gtk::Stock::QUIT),
                        sigc::mem_fun((App&)*this, &App::on_menu_file_exit));
                        
  m_refUIManager->insert_action_group(m_refFileActionGroup);

  //Build part of the menu structure, to be merged in by using the "PH" placeholders:
  static const Glib::ustring ui_description =
    "<ui>"
    "  <menubar name='Bakery_MainMenu'>"
    "    <placeholder name='Bakery_MenuPH_File'>"
    "      <menu action='BakeryAction_Menu_File'>"
    "        <menuitem action='BakeryAction_File_New' />"
    "        <menuitem action='BakeryAction_File_Open' />"
    "        <menu action='BakeryAction_Menu_File_RecentFiles'>"
    "        </menu>"
    "        <menuitem action='BakeryAction_File_Save' />"
    "        <menuitem action='BakeryAction_File_SaveAs' />"
    "        <separator/>"
    "        <menuitem action='BakeryAction_File_Close' />"
    "        <menuitem action='BakeryAction_File_Exit' />"
    "      </menu>"
    "    </placeholder>"
    "  </menubar>"
    "</ui>";
  
  //Add menu:
  add_ui_from_string(ui_description);
 
  //Add recent-files submenu:
  init_menus_file_recentfiles("/Bakery_MainMenu/Bakery_MenuPH_File/BakeryAction_Menu_File/BakeryAction_Menu_File_RecentFiles");
}


void App_WithDoc_Gtk::update_window_title()
{
  //Set application's main window title:

  Glib::ustring strTitle = m_strAppName;
  Document* pDoc = get_document();
  if(pDoc)
  {
    strTitle += " - " + pDoc->get_name();

    //Indicate unsaved changes:
    if(pDoc->get_modified())
      strTitle += " *";

    //Indicate read-only files:
    if(pDoc->get_read_only())
      strTitle += gettext(" (read-only)");

    set_title(strTitle);
  }
}

void App_WithDoc_Gtk::ui_warning(const Glib::ustring& text)
{
  GtkDialogs::ui_warning(text);
}

Glib::ustring App_WithDoc_Gtk::ui_file_select_open()
{
  return GtkDialogs::ui_file_select_open();
}

Glib::ustring App_WithDoc_Gtk::ui_file_select_save(const Glib::ustring& old_file_uri)
{
  return GtkDialogs::ui_file_select_save(old_file_uri);
}

void App_WithDoc_Gtk::ui_show_modification_status()
{
  bool modified = m_pDocument->get_modified();
  
  //Enable Save and SaveAs menu items:
  if(m_action_save)
    m_action_save->property_sensitive() = modified; //TODO: Use a set_sensitive(modified)?

  if(m_action_saveas)
    m_action_saveas->property_sensitive() = modified; //TODO: Use a set_sensitive(modified)?

}

App_WithDoc_Gtk::enumSaveChanges App_WithDoc_Gtk::ui_offer_to_save_changes()
{
  return GtkDialogs::ui_offer_to_save_changes(m_pDocument->get_file_uri());
}

bool App_WithDoc_Gtk::ui_ask_overwrite(const std::string& file_uri)
{
  return GtkDialogs::ui_ask_overwrite(m_pDocument->get_file_uri());
}

void App_WithDoc_Gtk::document_history_add(const Glib::ustring& file_uri)
{ 
  if(m_recent_files_model && !file_uri.empty())
  { 
    //TODO: Wrap gnome_vfs_escape_path_string() in gnome-vfsmm.
    //Glib::ustring filename_e = Gnome::Vfs::escape_path_string(file_uri);
    Glib::ustring uri = file_uri; // "file://" + filename_e;

    egg_recent_model_add(m_recent_files_model , uri.c_str());
  }
}

void App_WithDoc_Gtk::document_history_remove(const Glib::ustring& file_uri)
{
  if(m_recent_files_model && !file_uri.empty())
  {
    //Glib::ustring filename_e = Gnome::Vfs::escape_path_string(file_uri.c_str());
    Glib::ustring uri = file_uri; //"file://" + filename_e;

    egg_recent_model_delete(m_recent_files_model, uri.c_str());
  }
}


void App_WithDoc_Gtk::on_recent_files_activate(EggRecentItem* item)
{
  int prefix_len = strlen ("file://");
  char* uri = egg_recent_item_get_uri(item);

  if (strlen(uri) > prefix_len)
  {
    Glib::ustring file_uri = Gnome::Vfs::unescape_string(uri + prefix_len);
  
    bool bTest = open_document(file_uri);

    //Remove it from the recent documents list if it couldn't be opened.
    if(!bTest)
     document_history_remove(file_uri);
  }

  g_free (uri);
}
  

} //namespace
