//
// HitView.cs
//
// Copyright (C) 2004 Novell, Inc.
//

//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.
//


using System;
using System.Diagnostics;

using Gdk;
using Gtk;
using GtkSharp;

using Beagle;

namespace Best {

	public class HitView : Gtk.VBox {

		Hit hit;
		Box textBox;

		public HitView (Hit _hit)
		{
			hit = _hit;

			HBox hbox = new HBox (false, 0);
			
			Button iconButton = new Button ();
			iconButton.Add (BuildIcon ());
			iconButton.Clicked += new EventHandler (this.DoClicked);

			textBox = new VBox (false, 3);
			BuildText ();

			hbox.PackStart (iconButton, false, true, 3);
			hbox.PackStart (textBox, true, true, 3);

			PackStart (hbox, true, true, 3);
			hbox.ShowAll ();
		}

		private void DoClicked (object o, EventArgs args)
		{
			Launch ();
		}

		private void Launch ()
		{
			Console.WriteLine ("Loading {0}", hit.Uri);
			String command = null, arguments = null;

			if (hit.Uri.StartsWith ("file://")) {
				Beagle.Util.GnomeVFSMimeApplication app;
				app = Beagle.Util.GnomeIconLookup.GetDefaultAction (hit.MimeType);
				command = app.command;
				arguments = hit.Uri.Substring (7);
			} else if (hit.Uri.StartsWith ("http://")) {
				command = "epiphany";
				arguments = hit.Uri;
			} else if (hit.Uri.StartsWith ("email://")) {
				command = "evolution-1.5";
				arguments = hit.Uri;
			}
			
			if (command != null && arguments != null) {
				ProcessStartInfo psi = new ProcessStartInfo ();
				psi.FileName = command;
				psi.Arguments = arguments;
				Process.Start (psi);
			} else {
				Console.WriteLine ("Can't launch {0} (mime type: {1})", hit.Uri, hit.MimeType);
			}
		}

		private Widget BuildIcon ()
		{
			String iconPath = Beagle.Util.GnomeIconLookup.LookupMimeIcon (hit.MimeType, (Gtk.IconSize) 48);
			return new Gtk.Image (iconPath);
		}

		private void AddText (String text)
		{
			if (text == null || text == "")
				return;
			Label label = new Label (text);
			label.Xalign = 0;
			label.UseUnderline = false;
			textBox.PackStart (label, false, true, 1);
		}

		private void AddMarkup (String markup)
		{
			if (markup == null || markup == "")
				return;
			Label label = new Label ("");
			label.Xalign = 0;
			label.UseUnderline = false;
			label.Markup = markup;
			textBox.PackStart (label, false, true, 1);
		}

		private void BuildText ()
		{
			if (hit.Uri.StartsWith ("file:///")) {
				String path = hit.Uri.Substring (7);
				String dir = System.IO.Path.GetDirectoryName (path);
				String name = System.IO.Path.GetFileName (path);

				String home = Environment.GetEnvironmentVariable ("HOME");
				if (dir.StartsWith (home))
					dir = "~" + dir.Substring (home.Length);

				AddMarkup ("<i>" + dir + "</i>");
				AddMarkup (name);
				AddMarkup ("<small>Last modified " + hit.Timestamp.ToString ("g") + "</small>");
				return;
			}

			if (hit.Type == "WebLink") {
				AddText (hit.Uri);
				if (hit["title"] != null)
					AddMarkup ("<i>" + hit["title"] + "</i>");
				if (hit.ValidTimestamp)
					AddMarkup ("<small>Cached " + hit.Timestamp.ToString ("g") + "</small>");
				return;
			}

			if (hit.Type == "MailMessage") {
				AddText (hit ["Subject"]);
				AddText ("From: " + hit ["From"]);
				AddMarkup ("<small>" + hit.Timestamp + "</small>");
				return;
			}

			if (hit.Type == "Contact") {
				AddMarkup ("<b>" + hit ["Name"] + "</b>");
				if (hit ["Nickname"] != null)
					AddMarkup ("<i>\"" + hit ["Nickname"] + "</i>");
				AddText (hit ["Email1"]);
				AddText (hit ["Email2"]);
				AddText (hit ["Email3"]);
				AddText (hit ["HomePhone"]);
				AddText (hit ["MobilePhone"]);
			}

		}
	}
}
