// DasherView.h
//
// Copyright (c) 2001-2005 David Ward

#ifndef __DasherView_h_
#define __DasherView_h_

namespace Dasher {
  class CDasherScreen;
  class CDasherModel;
  class CDasherInput;
  class CDasherComponent;
  class CDasherView;
  class CDasherNode;
}

#include "DasherTypes.h"
#include "DasherComponent.h"


// CDasherView is an abstract view class
// See the CDasherViewSquare class for an example

// DJW 200504 - at the moment its quite hard work to plug in a new view

////// \brief View base class.
////// Dasher views represent the visualisation of a Dasher model on the screen.
////// Note that we really should aim to avoid having to try and keep
/// multiple pointers to the same object (model etc.) up-to-date at
/// once. We should be able to avoid the need for this just by being
/// sane about passing pointers as arguments to the relevant
/// functions, for example we could pass a pointer to the canvas every
/// time we call the render routine, rather than worrying about
/// notifying this object every time it changes. The same logic can be
/// applied in several other places.
///
/// We should also attempt to try and remove the need for this class
/// to know about the model. When we call render we should just pass a
/// pointer to the root node, which we can obtain elsewhere, and make
/// sure that the data structure contains all the info we need to do
/// the rendering (eg make sure it contains strings as well as symbol
/// IDs).
////// There are really three roles played by CDasherView: providing high
/// level drawing functions, providing a mapping between Dasher
/// co-ordinates and screen co-ordinates and providing a mapping
/// between true and effective Dasher co-ordinates (eg for eyetracking
/// mode). We should probably consider creating separate classes for
/// these.

class Dasher::CDasherView:public Dasher::CDasherComponent
{
public:

  /// 
  /// \param pEventHandler Pointer to the event handler
  /// \param pSettingsStore Pointer to the settings store
  /// \param DasherScreen Pointer to the CDasherScreen object used to do rendering
  /// \param DasherModel Reference to the CDasherModel which is to be represented

  CDasherView(CEventHandler * pEventHandler, CSettingsStore * pSettingsStore, CDasherScreen * DasherScreen);
  virtual ~ CDasherView() {
  }
  ////// Event handler
  /// \param pEvent Pointer to incoming event
  virtual void HandleEvent(Dasher::CEvent * pEvent);

  /// \deprecated Use parameter interface instead

  void ChangeOrientation(Dasher::Opts::ScreenOrientations Orientation);


  virtual bool IsNodeVisible(myint y1, myint y2) { return true; };

  //void SetDrawKeyboard(bool bDrawKeyboard);

  // 0 - no box, 1 - upper box, 2 - lower box
  //void SetDrawMousePosBox(int MousePosBox);

  /// Render the display

  virtual void Render(CDasherNode *pRoot, myint iRootMin, myint iRootMax, std::vector<CDasherNode *> &vNodeList, std::vector<CDasherNode *> &vDeleteList);

  /// Renders Dasher with mouse-dependent items
  /// \todo Clarify relationship between Render functions and probably only expose one

  virtual bool Render(CDasherNode *pRoot, myint iRootMin, myint iRootMax, std::vector<CDasherNode *> &vNodeList, std::vector<CDasherNode *> &vDeleteList, int iMouseX, int iMouseY, bool bRedrawDisplay);

  /// Renders the Dasher node structure
  /// \todo Shouldn't be public?

  virtual void RenderNodes(CDasherNode *pRoot, myint iRootMin, myint iRootMax, std::vector<CDasherNode *> &vNodeList, std::vector<CDasherNode *> &vDeleteList) = 0;

  /// Translates the screen coordinates to Dasher coordinates and calls
  /// dashermodel.TapOnDisplay

  virtual void TapOnDisplay(screenint mousex, screenint mousey, unsigned long Time, myint &iDasherX, myint &iDasherY, VECTOR_SYMBOL_PROB* pAdded = NULL, int* pNumDeleted = NULL)=0;

  /// Dasher Click Mode
  
  virtual void ClickTo(int x, int y, myint &dasherx, myint &dashery) = 0;

  /// Handles start-on-mouse behaviour - check whether we are in the box, and change box or start on timer,.
  /// \param iTime Current time in ms.

  virtual bool HandleStartOnMouse(int iTime) = 0;

  /// translates the screen coordinates to Dasher coordinates and calls
  /// dashermodel.GoTo

  virtual void GoTo(screenint mousex, screenint mousey) = 0;

  /// Change the screen - must be called if the Screen is replaced or resized
  /// \param NewScreen Pointer to the new CDasherScreen.

  virtual void ChangeScreen(CDasherScreen * NewScreen);

  /// Get autocallibration offset
  /// \retval Current offset

  virtual int GetAutoOffset() const {
    return 0;
  }
  ////// \todo Document this
  virtual void DrawGoTo(screenint mousex, screenint mousey) = 0;

  virtual void NewDrawGoTo(myint iDasherMin, myint iDasherMax, bool bActive) = 0;

  /// Draw the mouse cursor
  /// \todo Probably shouldn't be public

  virtual void DrawMouse(screenint mousex, screenint mousey) = 0;

  /// Draw the mouse line
  /// \todo Probably shouldn't be public

  virtual void DrawMouseLine(screenint mousex, screenint mousey) = 0;

  /// \todo Document this

  virtual void DrawKeyboard() = 0;

  /// \todo Document this

  virtual void DrawMousePosBox();

  /// Draw the game mode pointer

  virtual void DrawGameModePointer() = 0;

  virtual void Dasher2Screen(myint iDasherX, myint iDasherY, screenint &iScreenX, screenint &iScreenY) = 0;

  /// 
  /// Return a reference to the model

/*   CDasherModel * DasherModel() { */
/*     return m_pDasherModel; */
/*   } */

  /// \todo Erm...

/*   const CDasherModel * DasherModel() const { */
/*     return m_pDasherModel; */
/*   } */
 ////// Return a reference to the screen
  
  CDasherScreen * Screen() {
    return m_pScreen;
  }

  /// Request the Screen to copy its buffer to the Display
  /// \todo Shouldn't be public?

  void Display();

  /// \todo Document this

  virtual void ResetSum() {
  }

  /// \todo Document this

  virtual void ResetSumCounter() {
  }

  /// \todo Document this

  virtual void ResetYAutoOffset() {
  }

  /// Set the input device class. Note that this class will now assume ownership of the pointer, ie it will delete the object when it's done with it.
  /// \param _pInput Pointer to the new CDasherInput.

  void SetInput(CDasherInput * _pInput);

  /// Get the co-ordinates from the input device
  /// \todo This shouldn't be public?

  int GetCoordinates(int iN, myint * pCoordinates);

  /// Get the co-ordinate count from the input device

  int GetCoordinateCount();

  virtual void SpeedControl(myint iDasherX, myint iDasherY, double dFrameRate) {};

  virtual double xmap(double x) const {return 0.0;};  
  virtual double ymap(double x) const {return 0.0;};


protected:
  // Orientation of Dasher Screen
  inline void MapScreen(screenint * DrawX, screenint * DrawY);
  inline void UnMapScreen(screenint * DrawX, screenint * DrawY);

private:
  CDasherScreen * m_pScreen;    // provides the graphics (text, lines, rectangles):
  //  CDasherModel * m_pDasherModel; // Model view represents
  CDasherInput *m_pInput;       // Input device abstraction

  // Pure virtuals to implement
  virtual void Crosshair(myint sx) = 0; // Tells m_Screen to draw a crosshair - or other static decoration


};

#include "DasherView.inl"

#endif /* #ifndef __DasherView_h_ */
