/* Dia -- an diagram creation/manipulation program
 * Copyright (C) 1998 Alexander Larsson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <config.h>

#include <stdio.h>
#include <math.h>

#include "modify_tool.h"
#include "handle_ops.h"
#include "object_ops.h"
#include "connectionpoint_ops.h"
#include "message.h"
#include "properties.h"
#include "select.h"
#include "preferences.h"
#include "cursor.h"

static Object *click_select_object(DDisplay *ddisp, Point *clickedpoint,
				   GdkEventButton *event);
static int do_if_clicked_handle(DDisplay *ddisp, ModifyTool *tool,
				Point *clickedpoint,
				GdkEventButton *event);
static void modify_button_press(ModifyTool *tool, GdkEventButton *event,
				 DDisplay *ddisp);
static void modify_button_release(ModifyTool *tool, GdkEventButton *event,
				  DDisplay *ddisp);
static void modify_motion(ModifyTool *tool, GdkEventMotion *event,
			  DDisplay *ddisp);
static void modify_double_click(ModifyTool *tool, GdkEventButton *event,
				DDisplay *ddisp);

Tool *
create_modify_tool(void)
{
  ModifyTool *tool;

  tool = g_new0(ModifyTool, 1);
  tool->tool.type = MODIFY_TOOL;
  tool->tool.button_press_func = (ButtonPressFunc) &modify_button_press;
  tool->tool.button_release_func = (ButtonReleaseFunc) &modify_button_release;
  tool->tool.motion_func = (MotionFunc) &modify_motion;
  tool->tool.double_click_func = (DoubleClickFunc) &modify_double_click;
  tool->gc = NULL;
  tool->state = STATE_NONE;
  tool->break_connections = 0;
  tool->auto_scrolled = FALSE;

  tool->orig_pos = NULL;
  
  return (Tool *)tool;
}


void
free_modify_tool(Tool *tool)
{
  ModifyTool *mtool = (ModifyTool *)tool;
  if (mtool->gc)
    gdk_gc_unref(mtool->gc);
  g_free(mtool);
}

/*
  This function is buggy. Fix it later!
static void
transitive_select(DDisplay *ddisp, Point *clickedpoint, Object *obj)
{
  guint i;
  GList *j;
  Object *obj1;

  for(i = 0; i < obj->num_connections; i++) {
    printf("%d\n", i);
    j = obj->connections[i]->connected;
    while(j != NULL && (obj1 = (Object *)j->data) != NULL) {
      diagram_select(ddisp->diagram, obj1);
      obj1->ops->select(obj1, clickedpoint,
			(Renderer *)ddisp->renderer);
      transitive_select(ddisp, clickedpoint, obj1);
      j = g_list_next(j);
    }
  }
}
*/

static Object *
click_select_object(DDisplay *ddisp, Point *clickedpoint,
		    GdkEventButton *event)
{
  Diagram *diagram;
  real click_distance;
  Object *obj;
  
  diagram = ddisp->diagram;
  
  /* Find the closest object to select it: */

  click_distance = ddisplay_untransform_length(ddisp, 3.0);
  
  obj = diagram_find_clicked_object(diagram, clickedpoint,
				    click_distance);

  if (obj!=NULL) {
    /* Selected an object. */
    GList *already;
    /*printf("Selected object!\n");*/
      
    already = g_list_find(diagram->data->selected, obj);
    if (already == NULL) { /* Not already selected */
      /*printf("Not already selected\n");*/

      if (!(event->state & GDK_SHIFT_MASK)) {
	/* Not Multi-select => remove current selection */
	diagram_remove_all_selected(diagram, TRUE);
      }
      
      diagram_select(diagram, obj);
      obj->ops->selectf(obj, clickedpoint,
		        ddisp->renderer);

      /*
	This stuff is buggy, fix it later.
      if (event->state & GDK_CONTROL_MASK) {
	transitive_select(ddisp, clickedpoint, obj);
      }
      */

      ddisplay_do_update_menu_sensitivity(ddisp);
      object_add_updates_list(diagram->data->selected, diagram);
      diagram_flush(diagram);

      return obj;
    } else { /* Clicked on already selected. */
      /*printf("Already selected\n");*/
      obj->ops->selectf(obj, clickedpoint,
		       ddisp->renderer);
      object_add_updates_list(diagram->data->selected, diagram);
      diagram_flush(diagram);
      
      if (event->state & GDK_SHIFT_MASK) { /* Multi-select */
	/* Remove the selected selected */
	ddisplay_do_update_menu_sensitivity(ddisp);
	diagram_unselect_object(ddisp->diagram, (Object *)already->data);
	diagram_flush(ddisp->diagram);
      } else {
	return obj;
      }
    }
  } /* Else part moved to allow union/intersection select */

  return NULL;
}

static glong
time_micro()
{
  GTimeVal tv;

  g_get_current_time(&tv);
  return tv.tv_sec*G_USEC_PER_SEC+tv.tv_usec;
}

static int do_if_clicked_handle(DDisplay *ddisp, ModifyTool *tool,
				Point *clickedpoint, GdkEventButton *event)
{
  Object *obj;
  Handle *handle;
  real dist;
  
  handle = NULL;
  dist = diagram_find_closest_handle(ddisp->diagram, &handle,
				     &obj, clickedpoint);
  if  (handle_is_clicked(ddisp, handle, clickedpoint)) {
    tool->state = STATE_MOVE_HANDLE;
    tool->break_connections = TRUE;
    tool->last_to = handle->pos;
    tool->handle = handle;
    tool->object = obj;
    gdk_pointer_grab (ddisp->canvas->window, FALSE,
                      GDK_POINTER_MOTION_HINT_MASK | GDK_BUTTON1_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
                      NULL, NULL, event->time);
    tool->start_at = handle->pos;
    tool->start_time = time_micro();
    ddisplay_set_all_cursor(get_cursor(CURSOR_SCROLL));
    return TRUE;
  }
  return FALSE;
}

static void
modify_button_press(ModifyTool *tool, GdkEventButton *event,
		     DDisplay *ddisp)
{
  Point clickedpoint;
  Object *clicked_obj;
  
  ddisplay_untransform_coords(ddisp,
			      (int)event->x, (int)event->y,
			      &clickedpoint.x, &clickedpoint.y);


  if (do_if_clicked_handle(ddisp, tool, &clickedpoint, event))
    return;
  
  clicked_obj = click_select_object(ddisp, &clickedpoint, event);
  if (do_if_clicked_handle(ddisp, tool, &clickedpoint, event))
    return;

  if ( clicked_obj != NULL ) {
    tool->state = STATE_MOVE_OBJECT;
    tool->object = clicked_obj;
    tool->move_compensate = clicked_obj->position;
    point_sub(&tool->move_compensate, &clickedpoint);
    tool->break_connections = TRUE;
    gdk_pointer_grab (ddisp->canvas->window, FALSE,
                      GDK_POINTER_MOTION_HINT_MASK | GDK_BUTTON1_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
                      NULL, NULL, event->time);
    tool->start_at = clickedpoint;
    tool->start_time = time_micro();
    ddisplay_set_all_cursor(get_cursor(CURSOR_SCROLL));
  } else {
    tool->state = STATE_BOX_SELECT;
    tool->start_box = clickedpoint;
    tool->end_box = clickedpoint;
    tool->x1 = tool->x2 = (int) event->x;
    tool->y1 = tool->y2 = (int) event->y;

    if (tool->gc == NULL) {
      tool->gc = gdk_gc_new(ddisp->canvas->window);
      gdk_gc_set_line_attributes(tool->gc, 1, GDK_LINE_ON_OFF_DASH, 
				 GDK_CAP_BUTT, GDK_JOIN_MITER);
      gdk_gc_set_foreground(tool->gc, &color_gdk_white);
      gdk_gc_set_function(tool->gc, GDK_XOR);
    }

    gdk_draw_rectangle (ddisp->canvas->window, tool->gc, FALSE,
			tool->x1, tool->y1,
			tool->x2 - tool->x1, tool->y2 - tool->y1);

    gdk_pointer_grab (ddisp->canvas->window, FALSE,
                      GDK_POINTER_MOTION_HINT_MASK | GDK_BUTTON1_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
                      NULL, NULL, event->time);
  }
}


static void
modify_double_click(ModifyTool *tool, GdkEventButton *event,
		    DDisplay *ddisp)
{
  Point clickedpoint;
  Object *clicked_obj;
  
  ddisplay_untransform_coords(ddisp,
			      (int)event->x, (int)event->y,
			      &clickedpoint.x, &clickedpoint.y);

  clicked_obj = click_select_object(ddisp, &clickedpoint, event);
  
  if ( clicked_obj != NULL ) {
    properties_show(ddisp->diagram, clicked_obj);
  } else { /* No object selected */
    /*printf("didn't select object\n");*/
    if (!(event->state & GDK_SHIFT_MASK)) {
      /* Not Multi-select => Remove all selected */
      ddisplay_do_update_menu_sensitivity(ddisp);
      diagram_remove_all_selected(ddisp->diagram, TRUE);
      diagram_flush(ddisp->diagram);
    }
  }
}

#define MIN_PIXELS 10

/** Makes sure that objects aren't accidentally moved when double-clicking
 * for properties.  Objects do not move unless double click time has passed
 * or the move is 'significant'.  Allowing the 'significant' move makes a
 * regular grab-and-move less jerky.
 *
 * There's a slight chance that the user could move outside the
 * minimum movement limit and back in within the double click time
 * (normally .25 seconds), but that's very, very unlikely.  If
 * it happens, the cursor wouldn't reset just right.
 */

static gboolean
modify_move_already(ModifyTool *tool, DDisplay *ddisp, Point *to)
{
  static gboolean settings_taken = FALSE;
  static int double_click_time = 250;
  real dist;

  if (!settings_taken) {
    /* One could argue that if the settings were updated while running,
       we should re-read them.  But I don't see any way to get notified,
       and I don't want to do this whole thing for each bit of the
       move --Lars */
    GtkSettings *settings = gtk_settings_get_default();
    if (settings == NULL) {
      g_message(_("Couldn't get GTK settings"));
    } else {
      g_object_get(G_OBJECT(settings), 
		   "gtk-double-click-time", &double_click_time, NULL);
    }
    settings_taken = TRUE;
  }
  if (tool->start_time < time_micro()-double_click_time*1000) {
    return TRUE;
  }
  dist = distance_point_point_manhattan(&tool->start_at, to);
  if (ddisp->grid.snap) {
    real grid_x = ddisp->diagram->data->grid.width_x;
    real grid_y = ddisp->diagram->data->grid.width_y;
    if (dist > grid_x || dist > grid_y) {
      return TRUE;
    }
  }
  if (ddisplay_transform_length(ddisp, dist) > MIN_PIXELS) {
    return (ddisplay_transform_length(ddisp, dist) > MIN_PIXELS);
  } else {
    return FALSE;
  }
}


static void
modify_motion(ModifyTool *tool, GdkEventMotion *event,
	      DDisplay *ddisp)
{
  Point to;
  Point now, delta, full_delta;
  gboolean auto_scroll, vertical = FALSE;
  ConnectionPoint *connectionpoint;
  ObjectChange *objchange;

  if (tool->state==STATE_NONE)
    return; /* Fast path... */

  auto_scroll = ddisplay_autoscroll(ddisp, event->x, event->y);
  
  ddisplay_untransform_coords(ddisp, event->x, event->y, &to.x, &to.y);

  if (!modify_move_already(tool, ddisp, &to)) return;

  switch (tool->state) {
  case STATE_MOVE_OBJECT:

    if (tool->orig_pos == NULL) {
      GList *list;
      int i;
      Object *obj;

      /* consider non-selected children affected */
      list = parent_list_affected(ddisp->diagram->data->selected);
      tool->orig_pos = g_new(Point, g_list_length(list));
      i=0;
      while (list != NULL) {
	obj = (Object *)  list->data;
	tool->orig_pos[i] = obj->position;
	list = g_list_next(list); i++;
      }
    }
    
    if (tool->break_connections)
      diagram_unconnect_selected(ddisp->diagram); /* Pushes UNDO info */
  
    if (event->state & GDK_CONTROL_MASK) {
      full_delta = to;
      point_sub(&full_delta, &tool->start_at);
      vertical = (fabs(full_delta.x) < fabs(full_delta.y));
    }

    point_add(&to, &tool->move_compensate);
    snap_to_grid(ddisp, &to.x, &to.y);
  
    now = tool->object->position;
    
    delta = to;
    point_sub(&delta, &now);
    
    if (event->state & GDK_CONTROL_MASK) {
      /* Up-down or left-right */
      if (vertical) {
	delta.x = tool->start_at.x + tool->move_compensate.x - now.x;
      } else {
	delta.y = tool->start_at.y + tool->move_compensate.y - now.y;
      }
    }

    object_add_updates_list(ddisp->diagram->data->selected, ddisp->diagram);
    objchange = object_list_move_delta(ddisp->diagram->data->selected, &delta);
    if (objchange != NULL) {
      undo_object_change(ddisp->diagram, tool->object, objchange);
    }
    object_add_updates_list(ddisp->diagram->data->selected, ddisp->diagram);
  
    diagram_update_connections_selection(ddisp->diagram);
    diagram_flush(ddisp->diagram);
    break;
  case STATE_MOVE_HANDLE:
    full_delta = to;
    point_sub(&full_delta, &tool->start_at);

    /* make sure resizing is restricted to its parent */


    /* if resize was blocked by parent, that means the resizing was
      outward, thus it won't bother the children so we don't have to
      check the children */
    if (!parent_handle_move_out_check(tool->object, &to))
      parent_handle_move_in_check(tool->object, &to, &tool->start_at);

    /* Move to ConnectionPoint if near: */
    connectionpoint =
      object_find_connectpoint_display(ddisp, &to, tool->object);

    if (event->state & GDK_CONTROL_MASK)
      vertical = (fabs(full_delta.x) < fabs(full_delta.y));

    if ( (tool->handle->connect_type != HANDLE_NONCONNECTABLE) &&
	 (connectionpoint != NULL) ) {
      to = connectionpoint->pos;
      ddisplay_set_all_cursor(get_cursor(CURSOR_CONNECT));
    } else {
      /* No connectionopoint near, then snap to grid (if enabled) */
      snap_to_grid(ddisp, &to.x, &to.y);
      ddisplay_set_all_cursor(get_cursor(CURSOR_SCROLL));
    }

    if (tool->break_connections) {
      /* break connections to the handle currently selected. */
      if (tool->handle->connected_to!=NULL) {
	Change *change = undo_unconnect(ddisp->diagram, tool->object,
					tool->handle);
					
        (change->apply)(change, ddisp->diagram);
      }
    }

    if (event->state & GDK_CONTROL_MASK) {
      /* Up-down or left-right */
      if (vertical) {
	to.x = tool->start_at.x;
      } else {
	to.y = tool->start_at.y;
      }
    }

    if (tool->orig_pos == NULL) {
      tool->orig_pos = g_new(Point, 1);
      *tool->orig_pos = tool->handle->pos;
    }

    object_add_updates(tool->object, ddisp->diagram);
    objchange = tool->object->ops->move_handle(tool->object, tool->handle, 
					       &to, connectionpoint,
					       HANDLE_MOVE_USER, 0);
    if (objchange != NULL) {
      undo_object_change(ddisp->diagram, tool->object, objchange);
    }
    object_add_updates(tool->object, ddisp->diagram);
  
    diagram_update_connections_selection(ddisp->diagram);
    diagram_flush(ddisp->diagram);
    break;
  case STATE_BOX_SELECT:

    if (!auto_scroll && !tool->auto_scrolled)
    {
      gdk_draw_rectangle (ddisp->canvas->window, tool->gc, FALSE,
			  tool->x1, tool->y1,
			  tool->x2 - tool->x1, tool->y2 - tool->y1);
    }

    tool->end_box = to;

    ddisplay_transform_coords(ddisp,
			      MIN(tool->start_box.x, tool->end_box.x),
			      MIN(tool->start_box.y, tool->end_box.y),
			      &tool->x1, &tool->y1);
    ddisplay_transform_coords(ddisp,
			      MAX(tool->start_box.x, tool->end_box.x),
			      MAX(tool->start_box.y, tool->end_box.y),
			      &tool->x2, &tool->y2);

    gdk_draw_rectangle (ddisp->canvas->window, tool->gc, FALSE,
			tool->x1, tool->y1,
			tool->x2 - tool->x1, tool->y2 - tool->y1);
    break;
  case STATE_NONE:
    
    break;
  default:
    message_error("Internal error: Strange state in modify_tool\n");
  }

  tool->last_to = to;
  tool->auto_scrolled = auto_scroll;
}


static void
modify_button_release(ModifyTool *tool, GdkEventButton *event,
		      DDisplay *ddisp)
{
  Point *dest_pos, to;
  GList *list;
  int i;
  Object *obj;
  ObjectChange *objchange;
  
  tool->break_connections = FALSE;
  ddisplay_set_all_cursor(default_cursor);

  switch (tool->state) {
  case STATE_MOVE_OBJECT:
    /* Return to normal state */
    gdk_pointer_ungrab (event->time);

    ddisplay_untransform_coords(ddisp, event->x, event->y, &to.x, &to.y);
    if (!modify_move_already(tool, ddisp, &to)) {
      tool->orig_pos = NULL;
      tool->state = STATE_NONE;
      return;
    }

    diagram_update_connections_selection(ddisp->diagram);

    if (tool->orig_pos != NULL) {
      /* consider the non-selected children affected */
      list = parent_list_affected(ddisp->diagram->data->selected);
      dest_pos = g_new(Point, g_list_length(list));
      i=0;
      while (list != NULL) {
	obj = (Object *)  list->data;
	dest_pos[i] = obj->position;
	list = g_list_next(list); i++;
      }

      undo_move_objects(ddisp->diagram, tool->orig_pos, dest_pos,
			parent_list_affected(ddisp->diagram->data->selected));
    }
    
    ddisplay_connect_selected(ddisp); /* pushes UNDO info */
    diagram_update_extents(ddisp->diagram);
    diagram_modified(ddisp->diagram);
    diagram_flush(ddisp->diagram);

    undo_set_transactionpoint(ddisp->diagram->undo);

    tool->orig_pos = NULL;
    tool->state = STATE_NONE;
    break;
  case STATE_MOVE_HANDLE:
    gdk_pointer_ungrab (event->time);
    tool->state = STATE_NONE;

    if (tool->orig_pos != NULL) {
      undo_move_handle(ddisp->diagram, tool->handle, tool->object,
		       *tool->orig_pos, tool->last_to);
    }
    
    /* Final move: */
    object_add_updates(tool->object, ddisp->diagram);
    objchange = tool->object->ops->move_handle(tool->object, tool->handle,
					       &tool->last_to, NULL,
					       HANDLE_MOVE_USER_FINAL,0);
    if (objchange != NULL) {
      undo_object_change(ddisp->diagram, tool->object, objchange);
    }

    object_add_updates(tool->object, ddisp->diagram);

    /* Connect if possible: */
    if (tool->handle->connect_type != HANDLE_NONCONNECTABLE) {
      object_connect_display(ddisp, tool->object, tool->handle); /* pushes UNDO info */
      diagram_update_connections_selection(ddisp->diagram);
    }
    
    diagram_flush(ddisp->diagram);
    
    diagram_modified(ddisp->diagram);
    diagram_update_extents(ddisp->diagram);

    undo_set_transactionpoint(ddisp->diagram->undo);

    if (tool->orig_pos != NULL) {
      g_free(tool->orig_pos);
      tool->orig_pos = NULL;
    }

    break;
  case STATE_BOX_SELECT:
    gdk_pointer_ungrab (event->time);
    /* Remove last box: */
    if (!tool->auto_scrolled) {
      gdk_draw_rectangle (ddisp->canvas->window, tool->gc, FALSE,
			tool->x1, tool->y1,
			tool->x2 - tool->x1, tool->y2 - tool->y1);
    }

    {
      Rectangle r;
      GList *list, *list_to_free;
      Object *obj;

      r.left = MIN(tool->start_box.x, tool->end_box.x);
      r.right = MAX(tool->start_box.x, tool->end_box.x);
      r.top = MIN(tool->start_box.y, tool->end_box.y);
      r.bottom = MAX(tool->start_box.y, tool->end_box.y);

      if (prefs.reverse_rubberbanding_intersects) {
	if (tool->start_box.x > tool->end_box.x) {
	  list = list_to_free =
	    layer_find_objects_intersecting_rectangle(ddisp->diagram->data->active_layer, &r);
	} else {
	  list = list_to_free =
	    layer_find_objects_in_rectangle(ddisp->diagram->data->active_layer, &r);
	}
      } else {
	list = list_to_free =
	  layer_find_objects_in_rectangle(ddisp->diagram->data->active_layer, &r);
      }
      
      if (selection_style == SELECT_REPLACE &&
          !(event->state & GDK_SHIFT_MASK)) {
        /* Not Multi-select => Remove all selected */
        diagram_remove_all_selected(ddisp->diagram, TRUE);
      }

      if (selection_style == SELECT_INTERSECTION) {
        GList *intersection = NULL;

        while (list != NULL) {
          obj = (Object *)list->data;
          
          if (diagram_is_selected(ddisp->diagram, obj)) {
            intersection = g_list_append(intersection, obj);
          }
          
          list = g_list_next(list);
        }
        list = intersection;
        diagram_remove_all_selected(ddisp->diagram, TRUE);
        while (list != NULL) {
          obj = (Object *)list->data;

          diagram_select(ddisp->diagram, obj);

          list = g_list_next(list);
        }
        g_list_free(intersection);
      } else {
        while (list != NULL) {
          obj = (Object *)list->data;
          
          if (selection_style == SELECT_REMOVE) {
            if (diagram_is_selected(ddisp->diagram, obj))
              diagram_unselect_object(ddisp->diagram, obj);
          } else if (selection_style == SELECT_INVERT) { 
            if (diagram_is_selected(ddisp->diagram, obj))
              diagram_unselect_object(ddisp->diagram, obj);
            else
              diagram_select(ddisp->diagram, obj);
          } else {
            if (!diagram_is_selected(ddisp->diagram, obj))
              diagram_select(ddisp->diagram, obj);
          }
          
          list = g_list_next(list);
        }
      }

      g_list_free(list_to_free);
      
    }
    
    ddisplay_do_update_menu_sensitivity(ddisp);
    ddisplay_flush(ddisp);

    tool->state = STATE_NONE;
    break;
  case STATE_NONE:
    break;
  default:
    message_error("Internal error: Strange state in modify_tool\n");
      
  }
}






