/*
 *  Copyright (C) 2003 Marco Pesenti Gritti
 *  Copyright (C) 2004 Christian Persch
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  $Id: ephy-images-extension.c,v 1.3 2004/02/24 17:47:23 chpe Exp $
 */

#include "config.h"

#include "ephy-images-extension.h"
#include "mozilla-helpers.h"
#include "ephy-debug.h"

#include <epiphany/ephy-extension.h>

#include <gtk/gtkaction.h>
#include <gtk/gtktoggleaction.h>
#include <gtk/gtkactiongroup.h>
#include <gtk/gtkuimanager.h>
#include <gtk/gtkstock.h>
#include <gmodule.h>
#include <glib/gi18n-lib.h>

#define EPHY_IMAGES_EXTENSION_GET_PRIVATE(object)(G_TYPE_INSTANCE_GET_PRIVATE ((object), EPHY_TYPE_IMAGES_EXTENSION, EphyImagesExtensionPrivate))

struct EphyImagesExtensionPrivate
{
	gpointer dummy;
};

static void ephy_images_extension_class_init	(EphyImagesExtensionClass *klass);
static void ephy_images_extension_iface_init	(EphyExtensionIface *iface);
static void ephy_images_extension_init		(EphyImagesExtension *extension);

typedef struct
{
	GtkActionGroup *action_group;
	guint ui_id;
} WindowData;

#define DATA_KEY	"ephy-images-extension-window-data"

static GObjectClass *parent_class = NULL;

static GType type = 0;

GType
ephy_images_extension_get_type (void)
{
	return type;
}

GType
ephy_images_extension_register_type (GTypeModule *module)
{
	static const GTypeInfo our_info =
	{
		sizeof (EphyImagesExtensionClass),
		NULL, /* base_init */
		NULL, /* base_finalize */
		(GClassInitFunc) ephy_images_extension_class_init,
		NULL,
		NULL, /* class_data */
		sizeof (EphyImagesExtension),
		0, /* n_preallocs */
		(GInstanceInitFunc) ephy_images_extension_init
	};

	static const GInterfaceInfo extension_info =
	{
		(GInterfaceInitFunc) ephy_images_extension_iface_init,
		NULL,
		NULL
	};

	type = g_type_module_register_type (module,
					    G_TYPE_OBJECT,
					    "EphyImagesExtension",
					    &our_info, 0);

	g_type_module_add_interface (module,
				     type,
				     EPHY_TYPE_EXTENSION,
				     &extension_info);

	return type;
}

static void
ephy_images_extension_init (EphyImagesExtension *extension)
{
	extension->priv = EPHY_IMAGES_EXTENSION_GET_PRIVATE (extension);

	LOG ("EphyImagesExtension initialising");
}

static void
ephy_images_extension_finalize (GObject *object)
{
	LOG ("EphyImagesExtension finalising");

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
toggle_allow_images_cb (GtkAction *action,
			EphyWindow *window)
{
	EphyEmbed *embed;

	LOG ("toggle_allow_images_cb window %p", window);

	embed = ephy_window_get_active_embed (window);
	g_return_if_fail (EPHY_IS_EMBED (embed));

	ephy_embed_set_capability
		(embed, EMBED_CAP_IMAGES,
		 gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action)));
}

#if 0

static void
toggle_allow_javascript_cb (GtkAction *action,
			    EphyWindow *window)
{
	EphyEmbed *embed;

	LOG ("toggle_allow_javascript_cb window %p", window);

	embed = ephy_window_get_active_embed (window);
	g_return_if_fail (EPHY_IS_EMBED (embed));

	ephy_embed_set_capability
		(embed, EMBED_CAP_JAVASCRIPT,
		 gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action)));
}

static void
toggle_allow_plugins_cb (GtkAction *action,
			 EphyWindow *window)
{
	EphyEmbed *embed;

	LOG ("toggle_allow_plugins_cb window %p", window);

	embed = ephy_window_get_active_embed (window);
	g_return_if_fail (EPHY_IS_EMBED (embed));

	ephy_embed_set_capability
		(embed, EMBED_CAP_PLUGINS,
		 gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action)));
}

static void
toggle_allow_history_cb (GtkAction *action,
			 EphyWindow *window)
{
	EphyEmbed *embed;

	LOG ("toggle_allow_history_cb window %p", window);

	embed = ephy_window_get_active_embed (window);
	g_return_if_fail (EPHY_IS_EMBED (embed));

	ephy_embed_set_capability
		(embed, EMBED_CAP_HISTORY,
		 gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action)));
}

#endif /* 0 */

static void
load_images_cb (GtkAction *action,
		EphyWindow *window)
{
	EphyEmbed *embed;

	LOG ("load_images_cb window %p", window);

	embed = ephy_window_get_active_embed (window);
	g_return_if_fail (EPHY_IS_EMBED (embed));

	ephy_embed_load_images (embed);
}

static void
sync_active_tab_cb (EphyWindow *window,
		    GParamSpec *pspec,
		    gpointer dummy)
{
	WindowData *data;
	EphyEmbed *embed;
	GtkAction *action;
	gboolean enabled;

	LOG ("sync_active_tab_cb window %p", window);

	data = g_object_get_data (G_OBJECT (window), DATA_KEY);
	g_return_if_fail (data != NULL);

	embed = ephy_window_get_active_embed (window);
	g_return_if_fail (EPHY_IS_EMBED (embed));

	if (!GTK_WIDGET_REALIZED (GTK_WIDGET (embed))) return;

	enabled = ephy_embed_get_capability (embed, EMBED_CAP_IMAGES);
	action = gtk_action_group_get_action
		(data->action_group, "ImagesAllowImages");
	g_return_if_fail (action != NULL);
	g_signal_handlers_block_by_func
		(action, G_CALLBACK (toggle_allow_images_cb), window);
	gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action), enabled);
	g_signal_handlers_unblock_by_func
		(action, G_CALLBACK (toggle_allow_images_cb), window);

#if 0
	enabled = ephy_embed_get_capability (embed, EMBED_CAP_JAVASCRIPT);
	action = gtk_action_group_get_action
		(data->action_group, "ImagesAllowJavaScript");
	g_return_if_fail (action != NULL);
	g_signal_handlers_block_by_func
		(action, G_CALLBACK (toggle_allow_javascript_cb), window);
	gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action), enabled);
	g_signal_handlers_unblock_by_func
		(action, G_CALLBACK (toggle_allow_javascript_cb), window);

	enabled = ephy_embed_get_capability (embed, EMBED_CAP_PLUGINS);
	action = gtk_action_group_get_action
		(data->action_group, "ImagesAllowPlugins");
	g_return_if_fail (action != NULL);
	g_signal_handlers_block_by_func
		(action, G_CALLBACK (toggle_allow_plugins_cb), window);
	gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action), enabled);
	g_signal_handlers_unblock_by_func
		(action, G_CALLBACK (toggle_allow_plugins_cb), window);

	enabled = ephy_embed_get_capability (embed, EMBED_CAP_HISTORY);
	action = gtk_action_group_get_action
		(data->action_group, "ImagesAllowHistory");
	g_return_if_fail (action != NULL);
	g_signal_handlers_block_by_func
		(action, G_CALLBACK (toggle_allow_history_cb), window);
	gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action), enabled);
	g_signal_handlers_unblock_by_func
		(action, G_CALLBACK (toggle_allow_history_cb), window);
#endif /* 0 */
}

static GtkToggleActionEntry toggle_action_entries [] =
{
        { "ImagesAllowImages",
          NULL,
          N_("Allow _Images"),
          NULL, /* shortcut key */
          N_("Allow loading of images for this tab"),
          G_CALLBACK (toggle_allow_images_cb),
	  TRUE },
#if 0
        { "ImagesAllowJavaScript",
          NULL,
          N_("Allow Java_Script"),
          NULL, /* shortcut key */
          N_("Allow execution of JavaScript for this tab"),
          G_CALLBACK (toggle_allow_javascript_cb),
	  TRUE },
        { "ImagesAllowPlugins",
          NULL,
          N_("Allow _Plugins"),
          NULL, /* shortcut key */
          N_("Allow execution of plugins for this tab"),
          G_CALLBACK (toggle_allow_plugins_cb),
	  TRUE },
        { "ImagesAllowHistory",
          NULL,
          N_("Allow _History"),
          NULL, /* shortcut key */
          N_("Allow recording of history information for this tab"),
          G_CALLBACK (toggle_allow_history_cb),
	  TRUE },
#endif /* 0 */
};

static GtkActionEntry action_entries [] =
{
        { "ImagesLoadImages",
          GTK_STOCK_MISSING_IMAGE,
          N_("_Load Images"),
          "<images>I", /* Netscape 4.x had this shortcut */
          N_("Load all images in this page"),
          G_CALLBACK (load_images_cb) }
};

static void
impl_attach_window (EphyExtension *ext,
		    EphyWindow *window)
{
	WindowData *data;
	GtkUIManager *manager;
	GtkActionGroup *action_group;
	guint ui_id;

	LOG ("EphyImagesExtension attach_window");

	data = g_new (WindowData, 1);
	g_object_set_data_full
		(G_OBJECT (window), DATA_KEY, data, (GDestroyNotify) g_free);

	action_group = gtk_action_group_new ("EphyImagesActions");
	gtk_action_group_set_translation_domain (action_group, GETTEXT_PACKAGE);
	data->action_group = action_group;

	gtk_action_group_add_toggle_actions (action_group, toggle_action_entries,
					     G_N_ELEMENTS (toggle_action_entries),
					     window);
	gtk_action_group_add_actions (action_group, action_entries,
				      G_N_ELEMENTS (action_entries), window);
	manager = GTK_UI_MANAGER (ephy_window_get_ui_manager (window));
	gtk_ui_manager_insert_action_group (manager, action_group, -1);
	g_object_unref (action_group);

	ui_id = gtk_ui_manager_new_merge_id (manager);
	data->ui_id = ui_id;

	gtk_ui_manager_add_ui (manager, ui_id, "/menubar/ToolsMenu",
			       "ImagesExtSep0", "ImagesExtSep0",
			       GTK_UI_MANAGER_SEPARATOR, FALSE);
	gtk_ui_manager_add_ui (manager, ui_id, "/menubar/ToolsMenu",
			       "ImagesAllowImagesItem", "ImagesAllowImages",
			       GTK_UI_MANAGER_MENUITEM, FALSE);
#if 0
	gtk_ui_manager_add_ui (manager, ui_id, "/menubar/ToolsMenu",
			       "ImagesAllowJavaScriptItem", "ImagesAllowJavaScript",
			       GTK_UI_MANAGER_MENUITEM, FALSE);
	gtk_ui_manager_add_ui (manager, ui_id, "/menubar/ToolsMenu",
			       "ImagesAllowPluginsItem", "ImagesAllowPlugins",
			       GTK_UI_MANAGER_MENUITEM, FALSE);
	gtk_ui_manager_add_ui (manager, ui_id, "/menubar/ToolsMenu",
			       "ImagesAllowHistoryItem", "ImagesAllowHistory",
			       GTK_UI_MANAGER_MENUITEM, FALSE);
#endif /* 0 */
	gtk_ui_manager_add_ui (manager, ui_id, "/menubar/ToolsMenu",
			       "ImagesLoadImagesItem", "ImagesLoadImages",
			       GTK_UI_MANAGER_MENUITEM, FALSE);
	gtk_ui_manager_add_ui (manager, ui_id, "/menubar/ToolsMenu",
			       "ImagesExtSep1", "ImagesExtSep1",
			       GTK_UI_MANAGER_SEPARATOR, FALSE);

	g_signal_connect (window, "notify::active-tab",
			  G_CALLBACK (sync_active_tab_cb), NULL);
}

static void
impl_detach_window (EphyExtension *ext,
		    EphyWindow *window)
{
	//WindowData *data;
	//GtkUIManager *manager;

	LOG ("EphyImagesExtension detach_window");
/*
	data = g_object_get_data (G_OBJECT (window), DATA_KEY);
	g_return_if_fail (data != NULL);
	g_return_if_fail (data->action_group != NULL);
	g_return_if_fail (data->ui_id != 0);

	manager = GTK_UI_MANAGER (window->ui_merge);
	gtk_ui_manager_remove_ui (manager, data->ui_id);
	gtk_ui_manager_remove_action_group (manager, data->action_group);
*/
	g_signal_handlers_disconnect_by_func
		(window, G_CALLBACK (sync_active_tab_cb), NULL);

	g_object_set_data (G_OBJECT (window), DATA_KEY, NULL);
}

static void
ephy_images_extension_iface_init (EphyExtensionIface *iface)
{
	iface->attach_window = impl_attach_window;
	iface->detach_window = impl_detach_window;
}

static void
ephy_images_extension_class_init (EphyImagesExtensionClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = ephy_images_extension_finalize;

	g_type_class_add_private (object_class, sizeof (EphyImagesExtensionPrivate));
}
