/***************************************************************************
 *            camel-kolab-imapx-store.c
 *
 *  Fri Sep  3 12:48:31 2010
 *  Copyright  2010  Christian Hilberg
 *  <hilberg@kernelconcepts.de>
 ****************************************************************************/

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with main.c; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */
 
/*----------------------------------------------------------------------------*/

#include <camel/camel.h>

#include <camel/providers/imapx/camel-imapx-store-summary.h>
#include <camel/providers/imapx/camel-imapx-server.h>
#include <camel/providers/imapx/camel-imapx-metadata.h>
#include <camel/providers/imapx/camel-imapx-utils.h>

#include <libekolabutil/kolab-util-error.h>
#include <libekolabutil/kolab-util-folder.h>

#include "camel-kolab-imapx-store.h"

/*----------------------------------------------------------------------------*/

static CamelIMAPXStoreClass* parent_class = NULL;

/*----------------------------------------------------------------------------*/
/* Service class construction */

static void
kolab_imapx_construct (CamelService *service,
                       CamelSession *session,
                       CamelProvider *provider,
                       CamelURL *url,
                       CamelException *ex)
{	
	CamelKolabIMAPXStore *store = NULL;
	gboolean metadata_ok = FALSE;
	CamelException *tmp_ex = NULL;
	GError *tmp_err = NULL;

	g_assert (CAMEL_IS_SERVICE (service));
	g_assert (CAMEL_IS_SESSION (session));
	g_assert (provider != NULL);
	g_assert (url != NULL);
	g_assert (ex != NULL);

	tmp_ex = camel_exception_new ();

	(CAMEL_SERVICE_CLASS (parent_class))->construct (service,
	                                                 session,
	                                                 provider,
	                                                 url,
	                                                 tmp_ex);
	if (camel_exception_is_set (tmp_ex)) {
		camel_exception_xfer (ex, tmp_ex);
		return;
	}

	/*------- metadata db -------*/

	store = CAMEL_KOLAB_IMAPX_STORE (service);

	metadata_ok = kolab_imapx_meta_data_init (store->kmd,
	                                          service,
	                                          session,
	                                          &tmp_err);
	if (!metadata_ok) {
		kolab_camelexception_propagate_from_gerror (ex, tmp_err);
		return;
	}

	camel_exception_free (tmp_ex);

	/* default folder type to create (for use in Evo) */
	store->folder_create_type = KOLAB_FOLDER_TYPE_EMAIL;

	/* default folder context (for use in Evo) */
	store->folder_context = KOLAB_FOLDER_CONTEXT_EMAIL;
	
	g_debug ("%s: metadata db initialized", __func__);
}

static gboolean
kolab_imapx_connect (CamelService *service,
                     CamelException *ex)
{
	CamelException *tmp_ex = NULL;

	g_assert (CAMEL_IS_SERVICE (service));
	g_assert (ex != NULL);

	tmp_ex = camel_exception_new ();

	(CAMEL_SERVICE_CLASS (parent_class))->connect (service, tmp_ex);
	if (camel_exception_is_set (tmp_ex)) {
		g_warning ("%s: %s",
		           __func__,
		           camel_exception_get_description (tmp_ex));
		camel_exception_xfer (ex, tmp_ex);
		return FALSE;
	}

	camel_exception_free (tmp_ex);
	return TRUE;
}

static gboolean
kolab_imapx_disconnect (CamelService *service,
                        gboolean clean,
                        CamelException *ex)
{		
	CamelException *tmp_ex = NULL;
	gboolean parent_ok = FALSE;

	g_assert (CAMEL_IS_SERVICE (service));
	/* *ex may be NULL */

	tmp_ex = camel_exception_new ();
	parent_ok = (CAMEL_SERVICE_CLASS (parent_class))->disconnect (service,
	                                                              clean,
	                                                              tmp_ex);
	if (camel_exception_is_set (tmp_ex)) {
		g_warning ("%s: %s",
		           __func__,
		           camel_exception_get_description (tmp_ex));
		if (ex != NULL)
			camel_exception_xfer (ex, tmp_ex);
	}
	
	camel_exception_free (tmp_ex);
	return parent_ok;
}

static KolabFolderTypeID
kolab_imapx_folder_get_foldertype (CamelService *service,
                                   const CamelFolderInfo *fi,
                                   CamelKolabIMAPXMetaData *kmd)
{
	KolabFolderTypeID folder_type = KOLAB_FOLDER_TYPE_INVAL;
	GError *tmp_err = NULL;

	g_assert (CAMEL_IS_SERVICE (service));
	g_assert (fi != NULL);
	g_assert (kmd != NULL);

	folder_type = kolab_imapx_meta_data_get_foldertype (kmd,
	                                                    service,
	                                                    fi->full_name,
	                                                    TRUE,
	                                                    &tmp_err);
	if (tmp_err != NULL) {
		g_warning ("%s: error getting folder type: %s",
		           __func__,
		           tmp_err->message);
		g_error_free (tmp_err);
		return KOLAB_FOLDER_TYPE_INVAL;
	}
	return folder_type;
}

static CamelFolderInfo*
kolab_imapx_folder_info_build_restricted (CamelService *service, 
                                          const CamelFolderInfo *fi,
                                          CamelKolabIMAPXMetaData *kmd,
                                          const gboolean do_care[])
{
	CamelKolabIMAPXStore *self = NULL;
	KolabFolderTypeID folder_type = KOLAB_FOLDER_TYPE_INVAL;
	CamelFolderInfo *self_fi = NULL;
	CamelFolderInfo *next_fi = NULL;
	CamelFolderInfo *chld_fi = NULL;

	g_assert (CAMEL_IS_SERVICE (service));
	self = (CamelKolabIMAPXStore *)service;
	
	if (fi == NULL)
		return NULL;

	next_fi = kolab_imapx_folder_info_build_restricted (service,
	                                                    fi->next,
	                                                    kmd,
	                                                    do_care);

	chld_fi = kolab_imapx_folder_info_build_restricted (service,
	                                                    fi->child,
	                                                    kmd,
	                                                    do_care);

	folder_type = kolab_imapx_folder_get_foldertype (service, fi, kmd);

	if ((chld_fi != NULL) || (do_care[folder_type])) {
		self_fi = camel_folder_info_new ();
		if (fi->uri)
			self_fi->uri = g_strdup (fi->uri);
		if (fi->name)
			self_fi->name = g_strdup (fi->name);
		if (fi->full_name)
			self_fi->full_name = g_strdup (fi->full_name);
		self_fi->flags = fi->flags;
		self_fi->unread = fi->unread;
		self_fi->total = fi->total;
		self_fi->next = next_fi;
		self_fi->child = chld_fi;
		if (self_fi->child != NULL)
			self_fi->child->parent = self_fi;
		if ((self->folder_context != KOLAB_FOLDER_CONTEXT_EMAIL) &&
		    do_care[folder_type]) {
			    self->folder_names_do_care = g_list_prepend (self->folder_names_do_care,
			                                                 g_strdup (self_fi->full_name));
		}
		if (!do_care[folder_type]) {
			self_fi->flags |= CAMEL_FOLDER_NOSELECT;
			/* TODO check whether we need to (re)set more
			 *	(e.g. unread=0, total=0, ...)
			 */
		}
		return self_fi;
	}

	return next_fi;
}
                                          
static CamelFolderInfo*
kolab_imapx_get_folder_info (CamelStore *store,
                             const gchar *top,
                             guint32 flags,
                             CamelException *ex)
{
	CamelFolderInfo *fi = NULL;
	CamelFolderInfo *k_fi = NULL;
	CamelKolabIMAPXStore *ikstore = CAMEL_KOLAB_IMAPX_STORE (store);
	CamelService *service = CAMEL_SERVICE (store);
	CamelException *tmp_ex = NULL;

	g_assert (CAMEL_IS_STORE (store));
	/* 'top' may be NULL */
	g_assert (ex != NULL);

	tmp_ex = camel_exception_new ();

	g_mutex_lock (ikstore->kolab_finfo_lock);

	fi = (CAMEL_STORE_CLASS (parent_class))->get_folder_info (store,
	                                                          top,
	                                                          flags,
	                                                          tmp_ex);
	if (fi == NULL) {
		g_warning ("%s: aborted.", __func__);
		if (camel_exception_is_set (tmp_ex))
			camel_exception_xfer (ex, tmp_ex);
	    	else
			camel_exception_free (tmp_ex);
		g_mutex_unlock (ikstore->kolab_finfo_lock);
		return NULL;
	}

	k_fi = kolab_imapx_folder_info_build_restricted (service,
		                                         fi,
		                                         ikstore->kmd,
		                                         ikstore->folder_types_do_care);
	camel_store_free_folder_info (store, fi);

	g_mutex_unlock (ikstore->kolab_finfo_lock);

	camel_exception_free (tmp_ex);

	return k_fi;
}

static CamelFolderInfo*
kolab_imapx_create_folder (CamelStore *store,
                           const gchar *parent_name,
                           const gchar *folder_name,
                           CamelException *ex)
{
	CamelFolderInfo *fi = NULL;
	CamelFolderInfo *k_fi = NULL;
	CamelKolabIMAPXStore *ikstore = CAMEL_KOLAB_IMAPX_STORE (store);
	CamelService *service = CAMEL_SERVICE (store);
	GError *tmp_err = NULL;
	CamelException *tmp_ex = NULL;
	gboolean metadata_ok = FALSE;
	gchar *full_name = NULL;
	
	g_assert (CAMEL_IS_STORE (store));
	/* 'parent_name' may be NULL */
	g_assert (folder_name != NULL);
	g_assert (ex != NULL);

	tmp_ex = camel_exception_new ();

	fi = (CAMEL_STORE_CLASS (parent_class))->create_folder (store,
	                                                        parent_name,
	                                                        folder_name,
	                                                        tmp_ex);
	if (fi == NULL) {
		g_warning ("%s: create [%s/%s] on server failed.",
		           __func__, parent_name, folder_name);
		if (camel_exception_is_set (tmp_ex))
			camel_exception_xfer (ex, tmp_ex);
		return NULL;
	}

	camel_exception_free (tmp_ex);

	/* FIXME use Camel function(s) to create full_name */
	full_name = g_strdup_printf ("%s/%s", parent_name, folder_name);
	metadata_ok = kolab_imapx_meta_data_set_foldertype (ikstore->kmd,
	                                                    service,
	                                                    full_name,
	                                                    ikstore->folder_create_type,
	                                                    &tmp_err);
	g_free (full_name);
	
	if (!metadata_ok) {
		g_warning ("%s: setting type [%i] for [%s] on server failed.",
		           __func__, ikstore->folder_create_type, full_name);
		camel_store_free_folder_info (store, fi);
		kolab_camelexception_propagate_from_gerror (ex, tmp_err);
		return NULL;
	}
	
	k_fi = kolab_imapx_folder_info_build_restricted (service,
		                                         fi,
		                                         ikstore->kmd,
		                                         ikstore->folder_types_do_care);
	camel_store_free_folder_info (store, fi);
	return k_fi;
}

static void
kolab_imapx_delete_folder (CamelStore *store,
                           const gchar *folder_name,
                           CamelException *ex)
{
	CamelKolabIMAPXStore *ikstore = CAMEL_KOLAB_IMAPX_STORE (store);
	CamelException *tmp_ex = NULL;
	GError *tmp_err = NULL;
	gboolean metadata_ok = FALSE;
	
	g_assert (CAMEL_IS_STORE (store));
	g_assert (folder_name != NULL);
	g_assert (ex != NULL);

	tmp_ex = camel_exception_new ();

	(CAMEL_STORE_CLASS (parent_class))->delete_folder (store,
	                                                   folder_name,
	                                                   tmp_ex);
	if (camel_exception_is_set (tmp_ex)) {
		camel_exception_xfer (ex, tmp_ex);
		return;
	}
	camel_exception_free (tmp_ex);

	metadata_ok = kolab_imapx_meta_data_remove (ikstore->kmd,
	                                            folder_name,
	                                            &tmp_err);
	if (!metadata_ok) {
		kolab_camelexception_propagate_from_gerror (ex, tmp_err);
		return;
	}
}

/*----------------------------------------------------------------------------*/
/* CamelObject initialization and shutdown */

static void
camel_kolab_imapx_store_class_init (CamelKolabIMAPXStoreClass *klass)
{
	CamelServiceClass *camel_service_class = NULL;
	CamelServiceClass *parent_service_class = NULL;
	CamelStoreClass *camel_store_class = NULL;
	CamelStoreClass *parent_store_class = NULL;

	g_assert (klass != NULL);

	camel_service_class = CAMEL_SERVICE_CLASS (klass);
	camel_store_class = CAMEL_STORE_CLASS (klass);

	parent_class = CAMEL_IMAPX_STORE_CLASS (camel_type_get_global_classfuncs (__KOLAB_camel_imapx_store_get_type ()));
	parent_service_class = CAMEL_SERVICE_CLASS (parent_class);
	parent_store_class = CAMEL_STORE_CLASS (parent_class);
	
	/* camel_service_class->construct = parent_service_class->construct; */
	camel_service_class->construct = kolab_imapx_construct;
	
	camel_service_class->query_auth_types = parent_service_class->query_auth_types;
	camel_service_class->get_name = parent_service_class->get_name;

	/* camel_service_class->connect = parent_service_class->connect; */
	camel_service_class->connect = kolab_imapx_connect;

	/* camel_service_class->disconnect = parent_service_class->disconnect; */
	camel_service_class->disconnect = kolab_imapx_disconnect;

	camel_store_class->get_trash = parent_store_class->get_trash;
	camel_store_class->get_junk = parent_store_class->get_junk;
	camel_store_class->noop = parent_store_class->noop;
	camel_store_class->get_folder = parent_store_class->get_folder;
	camel_store_class->get_inbox = parent_store_class->get_inbox;
	camel_store_class->hash_folder_name = parent_store_class->hash_folder_name;
	camel_store_class->compare_folder_name = parent_store_class->compare_folder_name;

	/* camel_store_class->get_folder_info = parent_store_class->get_folder_info; */
	camel_store_class->get_folder_info = kolab_imapx_get_folder_info;

	camel_store_class->can_refresh_folder = parent_store_class->can_refresh_folder;
	
	/* camel_store_class->create_folder = parent_store_class->create_folder; */
	camel_store_class->create_folder = kolab_imapx_create_folder;
	
	camel_store_class->rename_folder = parent_store_class->rename_folder;

	/* camel_store_class->delete_folder = parent_store_class->delete_folder; */
	camel_store_class->delete_folder = kolab_imapx_delete_folder;
		
	camel_store_class->subscribe_folder = parent_store_class->subscribe_folder;
	camel_store_class->unsubscribe_folder = parent_store_class->unsubscribe_folder;
	camel_store_class->folder_subscribed = parent_store_class->folder_subscribed;
	camel_store_class->free_folder_info = camel_store_free_folder_info_full;

	camel_store_class->hash_folder_name = parent_store_class->hash_folder_name;
	camel_store_class->compare_folder_name = parent_store_class->compare_folder_name;
}

static void
camel_kolab_imapx_store_init (gpointer object,
                              gpointer klass)
{
	CamelKolabIMAPXStore *ikstore = NULL;
	gint ii = 0;

	g_assert (klass != NULL);
	
	ikstore = CAMEL_KOLAB_IMAPX_STORE (object);
	ikstore->kmd = kolab_imapx_meta_data_new ();

	ikstore->kolab_finfo_lock = g_mutex_new ();

	/* folder types to care for with this kolab_imapx instance
	 * Default: Email and unknown (so no need to reconfigure
	 * this instance for use in Evo). Needs to be reconfigured
	 * when used in ECal/EBook backends
	 */
	for (ii = 0; ii < KOLAB_FOLDER_LAST_TYPE; ii++)
		ikstore->folder_types_do_care[ii] = FALSE;
	ikstore->folder_types_do_care[KOLAB_FOLDER_TYPE_UNKNOWN] 	 = TRUE;
	ikstore->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL] 		 = TRUE;
	ikstore->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL_INBOX] 	 = TRUE;
	ikstore->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL_DRAFTS] 	 = TRUE;
	ikstore->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL_SENTITEMS] = TRUE;
	ikstore->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL_JUNKEMAIL] = TRUE;

	ikstore->folder_names_do_care = NULL;

	g_debug ("%s: done", __func__);
}

static void
camel_kolab_imapx_store_finalize (CamelObject* object)
{
	CamelKolabIMAPXStore *store = NULL;
	GError *tmp_err = NULL;
	gboolean ok = FALSE;

	store = CAMEL_KOLAB_IMAPX_STORE (object);

	/* finalize parent */

	/* finalize instance */
	while (! g_mutex_trylock (store->kolab_finfo_lock));
	g_mutex_unlock (store->kolab_finfo_lock);
	g_mutex_free (store->kolab_finfo_lock);
	
	/* TODO rework - there should not be I/O in finalize() */

	/*------- metadata db -------*/
	ok = kolab_imapx_meta_data_uninit (store->kmd, &tmp_err);
	if (!ok) {
		g_warning ("%s: metadata uninit error: %s",
		           __func__,
		           tmp_err->message);
		g_error_free (tmp_err);
		return;
	}
	kolab_imapx_meta_data_free (store->kmd);
	g_debug ("%s: metadata uninitialized",
	         __func__);

	if (store->folder_names_do_care != NULL) {
		GList *list_ptr = store->folder_names_do_care;
		while (list_ptr != NULL) {
			g_free (list_ptr);
			list_ptr = g_list_next (list_ptr);
		}
		g_free (store->folder_names_do_care);
        }

	g_debug ("%s: done", __func__);
}

CamelType
camel_kolab_imapx_store_get_type (void)
{
	static CamelType camel_kolab_imapx_store_type = CAMEL_INVALID_TYPE;

	if (camel_kolab_imapx_store_type == CAMEL_INVALID_TYPE) {
		camel_kolab_imapx_store_type = camel_type_register(__KOLAB_camel_imapx_store_get_type (),
		                                                   "CamelKolabIMAPXStore",
		                                                   sizeof (CamelKolabIMAPXStore),
		                                                   sizeof (CamelKolabIMAPXStoreClass),
		                                                   (CamelObjectClassInitFunc)camel_kolab_imapx_store_class_init,
		                                                   NULL,
		                                                   (CamelObjectInitFunc)camel_kolab_imapx_store_init,
		                                                   camel_kolab_imapx_store_finalize);
	}

	g_assert (camel_kolab_imapx_store_type != CAMEL_INVALID_TYPE);
	return camel_kolab_imapx_store_type;
}

gboolean
kolab_imapx_store_set_folder_creation_type (CamelKolabIMAPXStore *store,
                                            KolabFolderTypeID type_id)
{
	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (store));
	
	/* TODO error reporting */

	/* check that the given folder type id lies within the configured folder context */
	if (! kolab_util_folder_type_match_with_context_id (type_id, store->folder_context))
		return FALSE;
	
	store->folder_create_type = type_id;

	return TRUE;
}

gboolean
kolab_imapx_store_set_folder_context (CamelKolabIMAPXStore *store,
                                      KolabFolderContextID context)
{
	gint ii = 0;
	
	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (store));
	g_assert ((context > KOLAB_FOLDER_CONTEXT_INVAL) &&
	          (context < KOLAB_FOLDER_LAST_CONTEXT));
	
	for (ii = 0; ii < KOLAB_FOLDER_LAST_TYPE; ii++)
		store->folder_types_do_care[ii] = FALSE;

	store->folder_context = context;
	
	switch (context) {
		case KOLAB_FOLDER_CONTEXT_EMAIL:
			store->folder_types_do_care[KOLAB_FOLDER_TYPE_UNKNOWN]		= TRUE;
			store->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL]		= TRUE;
			store->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL_INBOX]	= TRUE;
			store->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL_DRAFTS]	= TRUE;
			store->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL_SENTITEMS]	= TRUE;
			store->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL_JUNKEMAIL]	= TRUE;
			break;
		case KOLAB_FOLDER_CONTEXT_CALENDAR:
			store->folder_types_do_care[KOLAB_FOLDER_TYPE_EVENT]		= TRUE;
			store->folder_types_do_care[KOLAB_FOLDER_TYPE_EVENT_DEFAULT]	= TRUE;
			store->folder_types_do_care[KOLAB_FOLDER_TYPE_JOURNAL]		= TRUE;
			store->folder_types_do_care[KOLAB_FOLDER_TYPE_JOURNAL_DEFAULT]	= TRUE;
			store->folder_types_do_care[KOLAB_FOLDER_TYPE_TASK]		= TRUE;
			store->folder_types_do_care[KOLAB_FOLDER_TYPE_TASK_DEFAULT]	= TRUE;
			store->folder_types_do_care[KOLAB_FOLDER_TYPE_NOTE]		= TRUE;
			store->folder_types_do_care[KOLAB_FOLDER_TYPE_NOTE_DEFAULT]	= TRUE;
			break;
		case KOLAB_FOLDER_CONTEXT_CONTACT:
			store->folder_types_do_care[KOLAB_FOLDER_TYPE_CONTACT]		= TRUE;
			store->folder_types_do_care[KOLAB_FOLDER_TYPE_CONTACT_DEFAULT]	= TRUE;
			break;
		default:
			/* can't happen */
			g_assert_not_reached ();
	}
	
	return TRUE;
}

KolabFolderTypeID
kolab_imapx_store_get_folder_type (CamelKolabIMAPXStore *store,
                                   const gchar *foldername,
                                   GError **err)
{
	/* TODO merge this with kolab_imapx_folder_get_foldertype() */
	
	KolabFolderTypeID folder_type = KOLAB_FOLDER_TYPE_INVAL;
	GError *tmp_err = NULL;

	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (store));
	g_assert (foldername != NULL);
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	folder_type = kolab_imapx_meta_data_get_foldertype (store->kmd,
	                                                    CAMEL_SERVICE (store),
	                                                    foldername,
	                                                    TRUE,
	                                                    &tmp_err);
	if (tmp_err != NULL) {
		g_propagate_error (err, tmp_err);
		return KOLAB_FOLDER_TYPE_INVAL;
	}
	
	return folder_type;
}

void
kolab_imapx_store_logout_sync (CamelKolabIMAPXStore *store)
{
	CamelIMAPXServer *iserver = NULL;
	gboolean ok = FALSE;
	CamelException *tmp_ex = NULL;
	
	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (store));

	tmp_ex = camel_exception_new ();
	iserver = __KOLAB_camel_imapx_store_get_server (CAMEL_IMAPX_STORE (store),
	                                                tmp_ex);
	if (camel_exception_is_set (tmp_ex)) {
		g_warning ("%s: could not get server: %s",
		           __func__,
		           camel_exception_get_description (tmp_ex));
		camel_exception_free (tmp_ex);
		if (iserver != NULL)
			camel_object_unref (iserver);
		return;
	}

	g_assert (CAMEL_IS_IMAPX_SERVER (iserver));

	ok = __KOLAB_imapx_server_logout_sync (iserver, tmp_ex);

	camel_object_unref (iserver);
	if (! ok) {
		g_debug ("%s: %s",
		         __func__,
		         camel_exception_get_description (tmp_ex));		
	}
	camel_exception_free (tmp_ex);
}

GList*
kolab_imapx_store_resect_folder_list (CamelKolabIMAPXStore *store)
{
	GList *folder_list = NULL;
	
	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (store));

	g_mutex_lock (store->kolab_finfo_lock);

	folder_list = store->folder_names_do_care;
	store->folder_names_do_care = NULL;
	
	g_mutex_unlock (store->kolab_finfo_lock);
	
	return folder_list;
}

/*----------------------------------------------------------------------------*/
