/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* camel-imapx-exception.c : imapx exceptions */

/*
 * Authors: ?
 *
 * Copyright (C) 1999-2008 Novell, Inc. (www.novell.com)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

/* This file has been borrowed from the original IMAPX implementation
 * of libcamel1.2-14, which is part of the Evolution Data Server.
 * Since the IMAPX API is not a public libcamel one, yet we need to
 * derive from (and extend) it, a local copy is kept here. The verbatim
 * copies are kept in a separate branch. 
 * This version of the file might have been extended by the
 * evolution-kolab project. Diff against the version in the verbatim branch
 * to see the changeset.
 *
 * changed by: Christian Hilberg <hilberg@kernelconcepts.de>
 * Changes (c) 2010 kernel concepts
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

#include <glib.h>

#include "camel-imapx-exception.h"

#include <pthread.h>

static pthread_key_t handler_key = 0;

void
__KOLAB_camel_exception_setup(void)
{
	pthread_key_create(&handler_key, NULL);
}

void
__KOLAB_camel_exception_try(struct _CamelExceptionEnv *env)
{
	struct _CamelExceptionEnv *handler;

	handler = pthread_getspecific(handler_key);
	env->parent = handler;
	handler = env;
	env->ex = NULL;

	pthread_setspecific(handler_key, handler);
}

void
__KOLAB_camel_exception_throw_ex(CamelException *ex)
{
	struct _CamelExceptionEnv *env;

	printf("throwing exception '%s'\n", ex->desc);

	env = pthread_getspecific(handler_key);
	if (env != NULL) {
		env->ex = ex;
		pthread_setspecific(handler_key, env->parent);
		longjmp(env->env, ex->id);
	} else {
		fprintf(stderr, "\nUncaught exception: %s\n", ex->desc);
		/* we just crash and burn, this is a code problem */
		/* we dont use g_assert_not_reached() since its not a noreturn function */
		abort();
	}
}

void
__KOLAB_camel_exception_throw(gint id, gchar *fmt, ...)
{
	CamelException *ex;
	va_list ap;

	ex = camel_exception_new();
	ex->id = id;
	va_start(ap, fmt);
	ex->desc = g_strdup_vprintf(fmt, ap);
	va_end(ap);

	__KOLAB_camel_exception_throw_ex(ex);
}

void
__KOLAB_camel_exception_drop(struct _CamelExceptionEnv *env)
{
	pthread_setspecific(handler_key, env->parent);
}

void
__KOLAB_camel_exception_done(struct _CamelExceptionEnv *env)
{
	pthread_setspecific(handler_key, env->parent);
	if (env->ex != NULL) {
		camel_exception_free(env->ex);
	}
}
