/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/***************************************************************************
 *            kolab-util-backend.c
 *
 *  Mon Jan 17 11:16:52 2011
 *  Copyright  2011  Christian Hilberg
 *  <hilberg@kernelconcepts.de>
 ****************************************************************************/

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */
 
/*----------------------------------------------------------------------------*/

#include "kolab-types.h"
#include "kolab-util-backend.h"
#include "kolab-mail-access.h"

#include <camel/camel-url.h>
#include <camel/camel-exception.h>
#include <string.h>
#include <time.h>
#include <libical/ical.h>
#include <libedataserver/e-source.h>

#include "kolab-util-backend.h"

const gchar *kolab_sync_strategy_desc[KOLAB_SYNC_LAST_STRATEGY] = {
	KOLAB_STRATEGY_DESC_NEWEST,
	KOLAB_STRATEGY_DESC_SERVER,
	KOLAB_STRATEGY_DESC_CLIENT,
	KOLAB_STRATEGY_DESC_DUPE
};

const gchar *kolab_tls_variant_desc[KOLAB_TLS_LAST_VARIANT] = {
	KOLAB_TLS_DESC_VARIANT_NONE,
	KOLAB_TLS_DESC_VARIANT_SSL,
	KOLAB_TLS_DESC_VARIANT_TLS
};

/*----------------------------------------------------------------------------*/
/* GError for libekolabbackend */

GQuark
kolab_util_backend_error_quark (void)
{
        static GQuark quark = 0;

        if (G_UNLIKELY (quark == 0)) {
                const gchar *string = "kolab-backend-error-quark";
                quark = g_quark_from_static_string (string);
        }

        return quark;
}

/*----------------------------------------------------------------------------*/
/* calendar name <-> folder name translation */

gchar*
kolab_util_backend_sourcename_new_from_foldername (const gchar *foldername,
                                                   GError **err)
{
	/* foldername can be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	/* FIXME do input sanitization here, provide real implementation */

	/* TODO
	 * - do we need a full mapping of calendar names to folder names?
	 * - can we use Camel infrastructure for checking of mailbox names?
	 */

	if (foldername == NULL)
		return NULL;

	return g_strdup (foldername);
}

gchar*
kolab_util_backend_foldername_new_from_sourcename (const gchar *sourcename,
                                                   GError **err)
{
	/* sourcename may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	if (sourcename == NULL)
		return NULL;

	return g_strdup (sourcename);
}

/**
 * kolab_util_backend_get_relative_path_from_uri:
 * %uri: An URI string
 *
 * Extracts the path from the given uri. Leading "/" are removed.
 *
 * Returns: A newly allocated string containing the uri, or NULL.
 */
gchar *
kolab_util_backend_get_relative_path_from_uri (const gchar *uri)
{
	CamelURL *c_url = NULL;
	CamelException *cex = NULL;
	gchar *tmp = NULL;
	gchar *path = NULL;

	cex = camel_exception_new ();
	c_url = camel_url_new (uri, cex);
	if (c_url == NULL) {
		g_warning ("%s()[%u] error: %s", __func__, __LINE__, camel_exception_get_description (cex));
		camel_exception_free (cex);
		return NULL;
	}
	camel_exception_free (cex);
	tmp = g_strdup (c_url->path);
	camel_url_free (c_url);
	if (tmp[0] == '/') {
		path = g_strdup (tmp+1);
		g_free (tmp);
	}
	else {
		path = tmp;
	}

	return path;
} /* kolab_util_backend_get_relative_path_from_uri () */

static gint
kolab_util_misc_generic_integer_from_property (const gchar *prop_str,
                                               gint DEFAULT_VALUE)
{
	guint64 tmp_value;
	gint prop_value;
	gchar *endptr = NULL;

	if (prop_str == NULL) /* Empty property -> use DEFAULT_VALUE. */
		return DEFAULT_VALUE;

	tmp_value = g_ascii_strtoull (prop_str, &endptr, 10);
	if ((tmp_value == 0) && (endptr == prop_str)) {
		/* This is a conversion error. Use DEFAULT_VALUE instead. */
		prop_value = DEFAULT_VALUE;
	} else if (tmp_value > G_MAXINT) {
		/* Overflow, handle gracefully: defaulting to DEFAULT_VALUE.
		 * Any other error leads to sync_value being zero which we
		 * assume to be the default value anyway */
		prop_value = DEFAULT_VALUE;
	} else
		prop_value = (gint) tmp_value;
	return prop_value;
} /* kolab_util_misc_generic_integer_from_property () */

KolabSyncStrategyID
kolab_util_misc_sync_value_from_property (const gchar *sync_prop)
{
	return kolab_util_misc_generic_integer_from_property (sync_prop, KOLAB_SYNC_STRATEGY_DEFAULT);
} /* kolab_util_misc_sync_value_from_property () */

KolabTLSVariantID
kolab_util_misc_tls_variant_from_property (const gchar *tls_variant)
{
	return kolab_util_misc_generic_integer_from_property (tls_variant, KOLAB_TLS_VARIANT_DEFAULT);
} /* kolab_util_misc_tls_variant_from_property () */

KolabReqPkcs11
kolab_util_misc_req_pkcs11_from_property (const gchar *req_pkcs11_prop)
{
	return kolab_util_misc_generic_integer_from_property (req_pkcs11_prop, KOLAB_PKCS11_INFRASTRUCTURE_DEFAULT);
}

void
kolab_util_backend_koma_table_cleanup_cb (gpointer data, 
                                          GObject *object,
                                          gboolean is_last_ref)
{
	KolabMailAccess *koma = KOLAB_MAIL_ACCESS (object);
	GHashTable *koma_table = (GHashTable *) data;
	const gchar *servername = NULL;
	const gchar *username = NULL;
	gchar *user_at_server = NULL;
	KolabSettingsHandler *ksettings = NULL;
	gboolean ok;
g_debug ("%s()[%u] called.", __func__, __LINE__);
	ksettings = kolab_mail_access_get_settings_handler (koma);
	servername = kolab_settings_handler_get_char_field (ksettings, KOLAB_SETTINGS_HANDLER_CHAR_FIELD_KOLAB_SERVER_NAME, NULL);
	username = kolab_settings_handler_get_char_field (ksettings, KOLAB_SETTINGS_HANDLER_CHAR_FIELD_KOLAB_USER_NAME, NULL);
	user_at_server = g_strdup_printf ("%s@%s", username, servername);
	ok = g_hash_table_remove (koma_table, user_at_server);
	
	g_free (user_at_server);
	g_object_remove_toggle_ref (object, kolab_util_backend_koma_table_cleanup_cb, data);
} /* kolab_util_backend_koma_table_cleanup_cb () */

void
kolab_util_backend_modtime_set_on_econtact (EContact *econtact)
{
	time_t rawtime;
	struct tm *ts = NULL;
	gchar *buf = NULL;
	
	g_assert (E_IS_CONTACT (econtact));

	time (&rawtime);
	ts = gmtime (&rawtime);
	buf = g_new0 (gchar, 21);
	strftime (buf, 21, "%Y-%m-%dT%H:%M:%SZ", ts); /* same as in contact-i-to-e.c */
	e_contact_set (econtact, E_CONTACT_REV, buf);
	g_free (buf);
} /* kolab_util_backend_modtime_set_on_econtact () */

void
kolab_util_backend_modtime_set_on_ecalcomp (ECalComponent *ecalcomp)
{
	struct icaltimetype itt;
	
	g_assert (E_IS_CAL_COMPONENT (ecalcomp));

	itt = icaltime_current_time_with_zone (NULL); /* need UTC here, hence NULL timezone */
	e_cal_component_set_last_modified (ecalcomp, &itt);
} /* kolab_util_backend_modtime_set_on_ecalcomp () */

void
kolab_util_backend_prepare_settings (KolabSettingsHandler *ksettings,
                                     ESource *esource,
                                     const gchar *servername,
                                     const gchar *username,
                                     const gchar *password,
                                     const gchar *sourcename,
                                     const KolabSyncStrategyID *sync_value)
{
	/* Note: Currently (2.30), there is no good way of returning errors from 
	 * the backend to the frontends. So this helper function simply notifies
	 * on stdout/stderr and proceeds.
	 */
	const gchar *prop_str = NULL;
	KolabTLSVariantID enc_value;
	KolabReqPkcs11 req_pkcs11_value;
	
	GError *error = NULL;
g_debug ("%s()[%u] called.", __func__, __LINE__);
	if (esource != NULL) {
		prop_str = e_source_get_uri (esource);
		(void) kolab_settings_handler_set_char_field (ksettings, KOLAB_SETTINGS_HANDLER_CHAR_FIELD_KOLAB_URI, g_strdup (prop_str), &error);
		if (error != NULL) {
			g_warning ("%s()[%u] error: %s", __func__, __LINE__, error->message);
			g_error_free (error);
			error = NULL;
		}
		prop_str = e_source_get_property (esource, KOLAB_TLS_VARIANT_PROP);
		enc_value = kolab_util_misc_tls_variant_from_property (prop_str);
		(void) kolab_settings_handler_set_uint_field (ksettings, KOLAB_SETTINGS_HANDLER_UINT_FIELD_TLS_VARIANT, enc_value, &error);
		if (error != NULL) {
			g_warning ("%s()[%u] error: %s", __func__, __LINE__, error->message);
			g_error_free (error);
			error = NULL;
		}
		prop_str = e_source_get_property (esource, KOLAB_REQ_PKCS11_PROP);
		req_pkcs11_value = kolab_util_misc_req_pkcs11_from_property (prop_str);
		if (req_pkcs11_value == KOLAB_PKCS11_INFRASTRUCTURE_ENABLE) {
			prop_str = e_source_get_property (esource, KOLAB_PKCS11_PIN_PROP);
		}
		else {
			prop_str = NULL;
		}
		(void) kolab_settings_handler_set_char_field (ksettings, KOLAB_SETTINGS_HANDLER_CHAR_FIELD_PKCS11_USER_PIN, g_strdup (prop_str), &error);
		if (error != NULL) {
			g_warning ("%s()[%u] error: %s", __func__, __LINE__, error->message);
			g_error_free (error);
			error = NULL;
		}
	}
	if (servername != NULL) {
		(void) kolab_settings_handler_set_char_field (ksettings, KOLAB_SETTINGS_HANDLER_CHAR_FIELD_KOLAB_SERVER_NAME, g_strdup (servername), &error);
		if (error != NULL) {
			g_warning ("%s()[%u] error: %s", __func__, __LINE__, error->message);
			g_error_free (error);
			error = NULL;
		}
	}
	if (username != NULL) {
		(void) kolab_settings_handler_set_char_field (ksettings, KOLAB_SETTINGS_HANDLER_CHAR_FIELD_KOLAB_USER_NAME, g_strdup (username), &error);
		if (error != NULL) {
			g_warning ("%s()[%u] error: %s", __func__, __LINE__, error->message);
			g_error_free (error);
			error = NULL;
		}
	}
	if (password != NULL) {
		(void) kolab_settings_handler_set_char_field (ksettings, KOLAB_SETTINGS_HANDLER_CHAR_FIELD_KOLAB_USER_PASSWORD, g_strdup (password), &error);
		if (error != NULL) {
			g_warning ("%s()[%u] error: %s", __func__, __LINE__, error->message);
			g_error_free (error);
			error = NULL;
		}
	}
	if (sync_value != NULL) {
		(void) kolab_settings_handler_set_value (ksettings, KOLAB_SETTINGS_HANDLER_TBL_SYNCSTRATEGY, sourcename, GUINT_TO_POINTER (*sync_value), &error);
		if (error != NULL) {
			g_warning ("%s()[%u] error: %s", __func__, __LINE__, error->message);
			g_error_free (error);
			error = NULL;
		}
	}
} /* kolab_util_backend_prepare_settings () */

gboolean
kolab_util_backend_sqlite_db_new_open (KolabUtilSqliteDb **kdb,
                                       KolabSettingsHandler *ksettings,
                                       const gchar *dbfilename,
                                       GError **err)
{
	const gchar *dbpath = NULL;
	gboolean ok = FALSE;
	GError *tmp_err = NULL;

	g_assert ((kdb != NULL) && (*kdb == NULL));
	g_assert (KOLAB_IS_SETTINGS_HANDLER (ksettings));
	g_assert (dbfilename != NULL);
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);	
	
	dbpath = kolab_settings_handler_get_char_field (ksettings,
	                                                KOLAB_SETTINGS_HANDLER_CHAR_FIELD_CAMEL_ACCOUNT_DIR,
	                                                &tmp_err);
	if (tmp_err != NULL) {
		g_propagate_error (err, tmp_err);
		return FALSE;
	}
	if (dbpath == NULL) {
		g_set_error (err,
		             KOLAB_BACKEND_ERROR,
		             KOLAB_BACKEND_ERROR_CAMEL,
		             "%s: could not get Camel storage path for InfoDb",
		             __func__);
		return FALSE;
	}
	*kdb = kolab_util_sqlite_db_new ();
	ok = kolab_util_sqlite_db_open (*kdb,
	                                dbpath,
	                                dbfilename,
	                                &tmp_err);
	if (! ok) {
		g_propagate_error (err, tmp_err);
		return FALSE;
	}
	
	return TRUE;
}
