/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/***************************************************************************
 *            e-cal-backend-kolab.c
 *
 *  Thu Jun 10 18:25:26 2010
 *  Copyright  2010  Christian Hilberg
 *  <hilberg@kernelconcepts.de>
 *  and Silvan Marco Fin <silvan@kernelconcepts.de> in 2011
 ****************************************************************************/

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

/*----------------------------------------------------------------------------*/

#include "kolab-util-calendar.h"
#include "kolab-util-calendar-cache.h"
#include "e-cal-backend-kolab.h"

#include <libekolab/kolab-types.h>
#include <libekolab/kolab-mail-access.h>
#include <libekolab/kolab-settings-handler.h>
#include <libekolab/kolab-mail-handle.h>
#include <libekolabutil/kolab-util-http.h>
#include <libekolabutil/kolab-http-job.h>
#include <libekolabutil/kolab-util-glib.h>
#include <libekolabutil/kolab-util-camel.h>
#include <libekolabutil/kolab-util-cal-freebusy.h>

#include <libedata-cal/e-cal-backend-cache.h>
#include <libecal/e-cal.h>
#include <camel/camel-exception.h>

#include <libical/ical.h>

/*----------------------------------------------------------------------------*/

#define E_CAL_BACKEND_KOLAB_PRIVATE(o)  (G_TYPE_INSTANCE_GET_PRIVATE ((o), E_TYPE_CAL_BACKEND_KOLAB, ECalBackendKolabPrivate))

G_DEFINE_TYPE (ECalBackendKolab, e_cal_backend_kolab, E_TYPE_CAL_BACKEND_SYNC)

/* private structures for Kolab calendar class ********************************/

static ECalBackendSyncStatus e_cal_backend_kolab_add_timezone (ECalBackendSync *backend, EDataCal *cal, const gchar *tzobj);

/* TODO check what is appropriate for Kolab here */

typedef struct {
	GCond *cond;
	GMutex *mutex;
	gboolean exit;
} SyncDelta;

/* Private part of the ECalBackendKolab structure */
typedef struct _ECalBackendKolabPrivate ECalBackendKolabPrivate;
struct _ECalBackendKolabPrivate {
	/*
	kolab_id_t 		fid;
	uint32_t 		olFolder;
	gchar 			*profile;
	*/
	GHashTable		*koma_table;
	CalMode			cal_mode;
	KolabMailAccess		*cal_koma;
	ECalBackendCache	*cal_cache;
	gchar			*user_email;
	ECalComponent		*default_zone;
	ECalSourceType		source_type;
	gboolean		already_opened;
	gchar *			cal_uri;
};

/* Forward declarations for internal use */

static void e_cal_backend_kolab_set_mode (ECalBackend *backend, CalMode mode);

/* implementation of parent class' virtual methods ****************************/

/**
 * e_cal_backend_kolab_is_read_only:
 * @backend: An ECalBackendSync object.
 * @cal: An EDataCal object.
 * @read_only: Return value for read-only status.
 *
 * Queries whether a calendar backend is read only or not.
 *
 * Returns: Status code.
 */
static ECalBackendSyncStatus
e_cal_backend_kolab_is_read_only (ECalBackendSync *backend,
                                  EDataCal *cal,
                                  gboolean *read_only)
{
	(void) cal;

	g_return_val_if_fail (backend && E_IS_CAL_BACKEND_KOLAB (backend), GNOME_Evolution_Calendar_OtherError);
	g_return_val_if_fail (read_only, GNOME_Evolution_Calendar_OtherError);

	*read_only = FALSE;
	
	return GNOME_Evolution_Calendar_Success;
} /* e_cal_backend_kolab_is_read_only () */

/**
 * e_cal_backend_kolab_get_cal_address:
 * @backend: An ECalBackendSync object.
 * @cal: An EDataCal object.
 * @address: Return value for the address.
 *
 * Queries the cal address associated with a calendar backend, which must
 * already have an open calendar.
 *
 * Returns: Status code.
 */
static ECalBackendSyncStatus
e_cal_backend_kolab_get_cal_address (ECalBackendSync *backend,
                                     EDataCal *cal,
                                     gchar **address)
{
	/* See comment at e_cal_backend_kolab_get_alarm_email_address (). */
	(void) backend;
	(void) cal;

	*address = NULL;

	return GNOME_Evolution_Calendar_Success;
} /* e_cal_backend_kolab_get_cal_address () */

/**
 * e_cal_backend_kolab_get_alarm_email_address:
 * @backend: An ECalBackendSync object.
 * @cal: An EDataCal object.
 * @address: Return value for the address.
 *
 * Queries the address to be used for alarms in a calendar client.
 *
 * Returns: Status code.
 */
static ECalBackendSyncStatus
e_cal_backend_kolab_get_alarm_email_address (ECalBackendSync *backend,
                                             EDataCal *cal,
                                             gchar **address)
{
	/* There is no obvious implementation for this feature in the context
	 * of kolab: The question to answer would be: how does a user supply, 
	 * to which e-mail address a notification about an alarm should be send.
	 */
	(void) backend;
	(void) cal;

	*address = NULL;

	return GNOME_Evolution_Calendar_Success;
} /* e_cal_backend_kolab_get_alarm_email_address () */

/**
 * e_cal_backend_kolab_get_ldap_attribute:
 * @backend: An ECalBackendSync object.
 * @cal: An EDataCal object.
 * @attribute: Return value for the LDAP attribute.
 *
 * Queries the LDAP attribute for a calendar client. This implementation 
 * always provides NULL via the LDAP attribute.
 *
 * Returns: ECalBackendSyncStatus code.
 */
static ECalBackendSyncStatus
e_cal_backend_kolab_get_ldap_attribute (ECalBackendSync *backend,
                                        EDataCal *cal,
                                        char **attribute)
{
	/* ..._get_ldap_attribute is only required by Sun ONE connector.
	 * Whatever that means, it seems sufficient to return NULL via *attribute.
	 */
	(void) backend;
	(void) cal;

	*attribute = NULL;

	return GNOME_Evolution_Calendar_Success;
} /* e_cal_backend_kolab_get_ldap_attribute () */

/**
 * e_cal_backend_kolab_get_static_capabilities:
 * @backend: An ECalBackendSync object.
 * @cal: An EDataCal object.
 * @capabilities: Return value for capabilities.
 *
 * Queries the calendar for static capabilities.
 *
 * Returns: Status code.
 */
static ECalBackendSyncStatus
e_cal_backend_kolab_get_static_capabilities (ECalBackendSync *backend,
                                             EDataCal *cal,
                                             gchar **capabilities)
{
	(void) backend;
	(void) cal;

	*capabilities = g_strdup ("");
	return GNOME_Evolution_Calendar_Success;
} /* e_cal_backend_kolab_get_static_capabilities () */

/**
 * e_cal_backend_kolab_open:
 * @backend: An ECalBackendSync object.
 * @cal: An EDataCal object.
 * @only_if_exists: Whether to open the calendar if and only if it already exists
 * or just create it when it does not exist.
 * @username: User name to use for authentication.
 * @password: Password to use for authentication.
 *
 * Opens a calendar backend with data from a calendar stored at the specified URI.
 *
 * Returns: Status code.
 */
static ECalBackendSyncStatus
e_cal_backend_kolab_open (ECalBackendSync *backend,
                          EDataCal *cal,
                          gboolean only_if_exists,
                          const gchar *username,
                          const gchar *password)
{
	ECalBackendKolab *kolab = E_CAL_BACKEND_KOLAB (backend);
	ECalBackendKolabPrivate *priv = E_CAL_BACKEND_KOLAB_PRIVATE (kolab);
	gchar *servername = NULL;
	gchar *user_at_server = NULL;
	CamelURL *c_url = NULL;
	CamelException *cex = NULL;
	GError *error = NULL;
	ESource *esource = NULL;
	icalcomponent_kind icalkind;
	ECalBackendSyncStatus status = GNOME_Evolution_Calendar_Success;
	gboolean ok;
	KolabMailAccess *tmp_koma = NULL;
	gchar *tmp_key = NULL;
	const gchar *prop_str = NULL;
	gchar *sourcename = NULL;
	KolabSyncStrategyID sync_value;
#if 0
	KolabTLSVariantID enc_value;
	KolabReqPkcs11 req_pkcs11_value;
#endif
	KolabSettingsHandler *ksettings = NULL;

	(void)cal;
	(void)only_if_exists;

	/* to trigger eds to pass along username and password, set the property 
	 * "auth" to "true" in the source setting:
	 * <property name="auth" value="true"/>
	 * <property name="username" value="..."/>
	 */

	if (priv->already_opened == TRUE) return status;
	
	esource = e_cal_backend_get_source (E_CAL_BACKEND (backend));
	/* Getting the esource from the cal-Object did not work as expected? */

	icalkind = e_cal_backend_get_kind (E_CAL_BACKEND (kolab));
	/* TODO: this has to be set according to the get_kind() method */
	switch (icalkind) {
	case ICAL_VEVENT_COMPONENT:
		priv->source_type = E_CAL_SOURCE_TYPE_EVENT;
		break;
	case ICAL_VTODO_COMPONENT:
		priv->source_type = E_CAL_SOURCE_TYPE_TODO;
		break;
	case ICAL_VJOURNAL_COMPONENT:
		priv->source_type = E_CAL_SOURCE_TYPE_JOURNAL;
		break;
	default:
		g_error ("%s()[%u]: Unknown Type used in e-cal-backend-kolab initialization!", __func__, __LINE__);
	}

	kolab_util_glib_init ();
	kolab_util_http_init ();
	/* libcamel
	 * Curl init may configure the underlying SSL lib,
	 * but as far as SSL goes, we want Camel to rule here 
	 * TODO check whether Camel session needs to be initialized before or after libcurl.
	 */
	ok = kolab_util_camel_init (&error);
	if (! ok) {
		status = kolab_util_calendar_map_error (error);
		g_warning ("%s()[%u]: %s", __func__, __LINE__, error->message);
		g_error_free (error);
		error = NULL;
	}

	priv->cal_uri = e_source_get_uri(esource);

	cex = camel_exception_new ();
	c_url = camel_url_new (priv->cal_uri, cex);
	if (c_url == NULL) {
		g_warning ("%s()[%u]: %s", __func__, __LINE__, camel_exception_get_description (cex));
		camel_exception_free (cex);
		status = GNOME_Evolution_Calendar_OtherError;
	}
	servername = g_strdup (c_url->host);
	camel_url_free (c_url);

	/* Initialize backend cache */
	if (priv->cal_cache != NULL) {
		g_object_unref (priv->cal_cache);
	}
	priv->cal_cache = e_cal_backend_cache_new (priv->cal_uri, priv->source_type);
	ok = e_file_cache_clean (E_FILE_CACHE (priv->cal_cache));

	/* Prepare data from sync strategy property */
	prop_str = e_source_get_property (esource, KOLAB_SYNC_STRATEGY_PROP);
	sync_value = kolab_util_misc_sync_value_from_property (prop_str);
	sourcename = kolab_util_backend_get_relative_path_from_uri (priv->cal_uri);

	/* This was the easy part. Now try to get the KoMA objects figured out */
	user_at_server = g_strdup_printf ("%s@%s", username, servername);

	ok = g_hash_table_lookup_extended (priv->koma_table, user_at_server, (gpointer *) &tmp_key, (gpointer *) &tmp_koma);
	if (ok) {
		/* There is already a KoMA instance for $servername. Use it and return */
		g_object_ref (tmp_koma);
		priv->cal_koma = tmp_koma;
		g_free (user_at_server);
		ksettings = kolab_mail_access_get_settings_handler (priv->cal_koma);
		kolab_util_backend_prepare_settings (ksettings, NULL, NULL, NULL, NULL, sourcename, &sync_value);
		goto notifications;
	}
	/* Otherwise we need to setup a new KoMA instance and a settings handler */

	ksettings = KOLAB_SETTINGS_HANDLER (g_object_new (KOLAB_TYPE_SETTINGS_HANDLER, NULL));
	(void) kolab_settings_handler_configure (ksettings, KOLAB_FOLDER_CONTEXT_CALENDAR, &error);
	if (error != NULL) {
		g_warning ("%s()[%u]: %s", __func__, __LINE__, error->message);
		g_error_free (error);
		error = NULL;
	}
	(void) kolab_settings_handler_bringup (ksettings, &error);
	if (error != NULL) {
		g_warning ("%s()[%u]: %s", __func__, __LINE__, error->message);
		g_error_free (error);
		error = NULL;
	}
	kolab_util_backend_prepare_settings (ksettings, esource, servername, username, password, sourcename, &sync_value);

	priv->cal_koma = KOLAB_MAIL_ACCESS (g_object_new (KOLAB_TYPE_MAIL_ACCESS, NULL));
	g_object_add_toggle_ref (G_OBJECT (priv->cal_koma), kolab_util_backend_koma_table_cleanup_cb, priv->koma_table);
	g_hash_table_insert (priv->koma_table, user_at_server, priv->cal_koma);

	kolab_mail_access_configure (priv->cal_koma, ksettings, &error);
	g_object_unref (ksettings);
	if (error != NULL) {
		status = kolab_util_calendar_map_error (error);
		g_warning ("%s()[%u]: %s", __func__, __LINE__, error->message);
		g_error_free (error);
		return status;
	}
	kolab_mail_access_bringup (priv->cal_koma, &error);
	if (error != NULL) {
		status = kolab_util_calendar_map_error (error);
		g_warning ("%s()[%u]: %s", __func__, __LINE__, error->message);
		g_error_free (error);
		return status;
	}
	
notifications:
	priv->already_opened = TRUE;
	e_cal_backend_kolab_set_mode (E_CAL_BACKEND (backend), priv->cal_mode);

	g_free (sourcename);
	g_free (servername);

	return status;
} /* e_cal_backend_kolab_open () */

static ECalBackendSyncStatus
e_cal_backend_kolab_refresh (ECalBackendSync *backend,
                             EDataCal *cal)
{
	/* TODO: implement me */
	(void) backend;
	(void) cal;

	g_warning ("Unimplemented method called: %s()", __func__);

	return GNOME_Evolution_Calendar_UnsupportedMethod;
} /* e_cal_backend_kolab_refresh () */

static ECalBackendSyncStatus
e_cal_backend_kolab_remove (ECalBackendSync *backend,
                            EDataCal *cal)
{
	ECalBackendKolab *kolab = E_CAL_BACKEND_KOLAB (backend);
	ECalBackendKolabPrivate *priv = E_CAL_BACKEND_KOLAB_PRIVATE (kolab);

	(void) cal;

	if (priv->already_opened != FALSE) {
		g_object_unref (priv->cal_koma);
		priv->cal_koma = NULL;
	}
	priv->already_opened = FALSE;

	/* Remove backend cache */
	if (priv->cal_cache != NULL) {
		(void) e_file_cache_remove (E_FILE_CACHE (priv->cal_cache));
		g_object_unref (priv->cal_cache);
		priv->cal_cache = NULL;
	}

	return GNOME_Evolution_Calendar_Success;
} /* e_cal_backend_kolab_remove () */

/**
 * e_cal_backend_kolab_get_default_object:
 * @backend: An ECalBackendSync object.
 * @cal: An EDataCal object.
 * @object: Placeholder for returned object.
 *
 * Returns the default object via object placeholder.
 *
 * Returns: Status code.
 */
static ECalBackendSyncStatus
e_cal_backend_kolab_get_default_object (ECalBackendSync *backend,
                                        EDataCal *cal,
                                        gchar **object)
{
	/* FIXME: Remove experimental code related to summary*/
	ECalComponent *comp;
	ECalComponentText *summary = NULL;
	ECalComponentText *desc = NULL;
	GSList *desclist = NULL;
	gchar *tmpsum = g_strdup ("");
	gchar *tmploc = g_strdup ("");
	gchar *tmpdesc1 = g_strdup ("");
	(void) cal;

	summary = g_new0 (ECalComponentText, 1);
	summary->value=tmpsum;
	desc = g_new0 (ECalComponentText, 1);
	desc->value=tmpdesc1;
	desclist = g_slist_append(desclist, desc);

	comp = e_cal_component_new ();

	switch (e_cal_backend_get_kind (E_CAL_BACKEND (backend))) {
	case ICAL_VEVENT_COMPONENT:
		e_cal_component_set_new_vtype (comp, E_CAL_COMPONENT_EVENT);
		e_cal_component_set_summary (comp, summary);
		e_cal_component_set_description_list (comp, desclist);
		e_cal_component_set_location (comp, tmploc);
		break;
	default:
		g_object_unref (comp);
		return GNOME_Evolution_Calendar_ObjectNotFound;
	}

	*object = e_cal_component_get_as_string (comp);

	if (comp)
		g_object_unref (comp);

	g_free (summary);
	g_free (tmploc);
	g_free (tmpdesc1);
	g_slist_foreach (desclist, (GFunc) g_free, NULL);
	g_slist_free (desclist);

	return GNOME_Evolution_Calendar_Success;
} /* e_cal_backend_kolab_get_default_object () */

/**
 * e_cal_backend_kolab_get_object:
 * @backend: An ECalBackendSync object.
 * @cal: An EDataCal object.
 * @uid: UID of the object to get.
 * @rid: Recurrence ID of the specific instance to get, or NULL if getting the
 * master object.
 * @object: Placeholder for returned object.
 *
 * Queries a calendar backend for a calendar object based on its unique
 * identifier and its recurrence ID (if a recurrent appointment).
 *
 * Returns: Status code.
 */
static ECalBackendSyncStatus
e_cal_backend_kolab_get_object (ECalBackendSync *backend,
                                EDataCal *cal,
                                const gchar *uid,
                                const gchar *rid,
                                gchar **object)
{
	ECalBackendKolab *kolab = E_CAL_BACKEND_KOLAB (backend);
	ECalBackendKolabPrivate *priv = E_CAL_BACKEND_KOLAB_PRIVATE (kolab);
	ECalComponent *ecalcomp = NULL;
#if 0
	ECalComponent *oldcomp = NULL;
	gchar *old_object = NULL;
#endif
	ECalBackendSyncStatus status = GNOME_Evolution_Calendar_Success;
	GError *error = NULL;

	(void) cal;
	(void) rid;
#if 0
	oldcomp = e_cal_backend_cache_get_component (priv->cal_cache, uid, rid);
	if (oldcomp == NULL)
		old_object = NULL;
	else {
		old_object = e_cal_component_get_as_string (oldcomp);
		g_object_unref (oldcomp);
	}
#endif
	ecalcomp = kolab_util_calendar_cache_get_object (priv->cal_cache, priv->cal_koma, priv->cal_uri, uid, FALSE, &error);
	if (error != NULL) {
		status = kolab_util_calendar_map_error (error);
		g_warning ("%s()[%u]: %s", __func__, __LINE__, error->message);
		g_error_free (error);
		return status;
	}
	if (ecalcomp != NULL) {
		*object = (gchar *) e_cal_component_get_as_string (ecalcomp);

		g_object_unref (ecalcomp);
	}
	else {
		*object = NULL;
	}

	return GNOME_Evolution_Calendar_Success;
} /* e_cal_backend_kolab_get_object () */

static ECalBackendSyncStatus
e_cal_backend_kolab_get_object_list (ECalBackendSync *backend,
                                     EDataCal *cal,
                                     const gchar *query,
                                     GList **objects)
{
	ECalBackendKolab *kolab = E_CAL_BACKEND_KOLAB (backend);
	ECalBackendKolabPrivate *priv = E_CAL_BACKEND_KOLAB_PRIVATE (kolab);
	gchar *sourcename = NULL;
	GList *uid_list = NULL;
	GList *it = NULL;
	GError *error = NULL;
	(void) cal;

	sourcename = kolab_util_backend_get_relative_path_from_uri (priv->cal_uri);

	kolab_util_calendar_cache_update_on_query (priv->cal_cache, priv->cal_koma, query, priv->cal_uri);
	uid_list = kolab_mail_access_query_uids (priv->cal_koma, sourcename, query, &error);
	if (error != NULL) {
		g_warning ("%s()[%u]: %s", __func__, __LINE__, error->message);
		g_error_free (error);
		error = NULL;
	}

	for (it = g_list_first (uid_list); it != NULL; it = g_list_next (it)) {
		gchar *uid = it->data;
		ECalComponent *ecalcomp = NULL;
		ecalcomp = kolab_util_calendar_cache_get_object (priv->cal_cache, priv->cal_koma, priv->cal_uri, uid, TRUE, &error);
		if (error != NULL) {
			g_warning ("%s()[%u]: %s", __func__, __LINE__, error->message);
			g_error_free (error);
			error = NULL;
		}
		/* [FIXME speed] ecalcomp = e_cal_backend_cache_get_component (priv->cal_cache, uid, NULL); */		
		if (ecalcomp != NULL) {
			*objects = g_list_append (*objects, e_cal_component_get_as_string (ecalcomp));
			g_object_unref (ecalcomp);
		}
	}
	g_free (sourcename);

	kolab_util_glib_glist_free (uid_list);

	return GNOME_Evolution_Calendar_Success;
} /* e_cal_backend_kolab_get_object_list () */

static ECalBackendSyncStatus
e_cal_backend_kolab_get_attachment_list (ECalBackendSync *backend,
                                         EDataCal *cal,
                                         const gchar *uid,
                                         const gchar *rid,
                                         GSList **list)
{
	/* currently not supported. */
	(void) backend;
	(void) cal;
	(void) uid;
	(void) rid;
	(void) list;
g_error ("%s()[%u]: called.", __func__, __LINE__);
	return GNOME_Evolution_Calendar_UnsupportedMethod;
} /* e_cal_backend_kolab_get_attachment_list () */

/**
 * e_cal_backend_kolab_create_object:
 * @backend: (sync) kolab calendar backend object.
 * @cal: An EDataCal object.
 * @calobj: Contains the event to be created in the calendar as iCalendar string.
 * @uid: Return value of the created events uid.
 *
 * Creates a new calendar entry in the Kolab calendar. This method gets called, 
 * when Evolution requests a new entry to be saved in the calendar.
 */
static ECalBackendSyncStatus
e_cal_backend_kolab_create_object (ECalBackendSync *backend,
                                   EDataCal *cal,
                                   gchar **calobj,
                                   gchar **uid)
{
	ECalBackendKolab *kolab = E_CAL_BACKEND_KOLAB (backend);
	ECalBackendKolabPrivate *priv = E_CAL_BACKEND_KOLAB_PRIVATE (kolab);
	ECalComponent *ecalcomp = NULL;
	ECalComponent *tzcomp = NULL;
	GError *error = NULL;
	const gchar *tmp_uid = NULL;
	ECalBackendSyncStatus status = GNOME_Evolution_Calendar_Success;
		
	(void) cal;

	/* What do we have here? There is an ECalComponent, unable to hold
	 * timezone information and an calendar entry at the same time.
	 * So Kolab glue code expects two ECalComponents, one for the calendar entry
	 * one for the timezone.
	 */
	ecalcomp = e_cal_component_new_from_string (*calobj);
	if (ecalcomp == NULL) {
		g_warning (" + **calobj could not be parsed into ECalComponent: %s", *calobj);
		return GNOME_Evolution_Calendar_InvalidObject;
	}
	tzcomp = kolab_util_calendar_cache_get_tz (priv->cal_cache, ecalcomp);

	if (tzcomp == NULL) {
		gchar *tzid = NULL;
		tzid = kolab_util_calendar_get_tzid (ecalcomp, E_CAL_COMPONENT_FIELD_DTSTART);
		/* If tzid is UTC, everything is fine */
		if (g_strcmp0 ("UTC", tzid) == 0) {
			g_free (tzid);
			tzid = NULL;
		}
		if (tzid != NULL) {
			/* TZID is set, but no timezone is available. Currently
			 * this may happen, if an object is copied to an yet unused/empty
			 * calendar. We do not want such crap in the kolab database.
			 */
			g_free (tzid);
			g_object_unref (ecalcomp);
			/* Don't ever expect to get the following message to see.
			 * Ignorance is bliss. 
			 */
			e_data_cal_notify_error (cal, "TZID is set, but no timezone is available. Currently this may happen, if an object is copied to an empty calendar.");
			return GNOME_Evolution_Calendar_InvalidObject;
		}
	}

	if (! kolab_util_calendar_cache_assure_uid_on_ecalcomponent (priv->cal_cache, priv->cal_koma, priv->cal_uri, ecalcomp, FALSE, &error)) {
		status = kolab_util_calendar_map_error (error);
		g_warning ("%s()[%u]: %s", __func__, __LINE__, error->message);
		g_error_free (error);
		g_object_unref (ecalcomp);
		if (tzcomp != NULL)
			g_object_unref (tzcomp);
		return status;
	}

	kolab_util_calendar_store (ecalcomp, tzcomp, priv->default_zone, priv->cal_koma, priv->cal_uri, &error);
	if (error != NULL) {
		status = kolab_util_calendar_map_error (error);
		g_warning ("%s()[%u]: %s", __func__, __LINE__, error->message);
		g_error_free (error);
		return status;
	}	
	/* [FIXME speed] e_cal_backend_cache_put_component (priv->cal_cache, ecalcomp); */
	e_cal_component_get_uid (ecalcomp, &tmp_uid);
	*uid = g_strdup (tmp_uid);
	*calobj = e_cal_component_get_as_string (ecalcomp);

	if (tzcomp != NULL)
		g_object_unref (tzcomp);
	g_object_unref (ecalcomp);

	return GNOME_Evolution_Calendar_Success;
} /* e_cal_backend_kolab_create_object () */

/**
 * e_cal_backend_kolab_modify_object:
 * @backend: An ECalBackendSync object.
 * @cal: An EDataCal object.
 * @calobj: Object to be modified.
 * @mod: Type of modification to be done.
 * @old_object: Placeholder for returning the old object as it was stored on the
 * backend.
 * @new_object: Placeholder for returning the new object as it has been stored
 * on the backend.
 *
 * Requests the calendar backend to modify an existing object. If the object 
 * does not exist on the calendar, it will be issued as new event.
 *
 * Returns: Status code.
 */
static ECalBackendSyncStatus
e_cal_backend_kolab_modify_object (ECalBackendSync *backend,
                                   EDataCal *cal,
                                   const gchar *calobj,
                                   CalObjModType mod,
                                   gchar **old_object,
                                   gchar **new_object)
{
	ECalBackendKolab *kolab = E_CAL_BACKEND_KOLAB (backend);
	ECalBackendKolabPrivate *priv = E_CAL_BACKEND_KOLAB_PRIVATE (kolab);
	ECalComponent *ecalcomp = NULL;
	ECalComponent *oldcomp = NULL;
	ECalComponent *ecaltz = NULL;
	GError *error = NULL;
	const gchar *uid = NULL;
	gboolean is_instance = FALSE;
	ECalBackendSyncStatus status = GNOME_Evolution_Calendar_Success;
	
	(void) mod;
	(void) cal;

	/* calobj contains the modifications to be performed.
	 * ecalcomp is the ECalComponent created from calobj.
	 * scenarios:
	 *  a) ecalcomp is no part of a recurring event:
	 *     read oldobj by means of ecalcomps uid
	 *     store ecalcomp and we are done.
	 *  b) ecalcomp is a modification to all instances of a recurring event:
	 *     should be handled like (a)
	 *  c) ecalcomp is a single instance of a recurring event:
	 *     Kolab can not handle detached recurrences. Remove the instance
	 *     from the recurrence and create a new single event für ecalcomp.
	 */

	ecalcomp = e_cal_component_new_from_string (calobj);
	if (ecalcomp == NULL)
		return GNOME_Evolution_Calendar_InvalidObject;
	e_cal_component_get_uid (ecalcomp, &uid);
	
	oldcomp = kolab_util_calendar_cache_get_object (priv->cal_cache, priv->cal_koma, priv->cal_uri, uid, FALSE, &error);
	if (error != NULL) {
		/* TODO: how to handle request to modify non existing objects? */
		g_warning ("%s()[%u]: %s", __func__, __LINE__, error->message);
		g_error_free (error);
		g_object_unref (ecalcomp);
		g_object_unref (oldcomp);
		return GNOME_Evolution_Calendar_ObjectNotFound;
	}

	*old_object = e_cal_component_get_as_string (oldcomp);
	/* Is this correct? Perhaps _get_tz (..., oldcomp) is needed.
	 * This depends on wether evolution sends the timezone, if modification
	 * to the timezone has been made. In any other case the timezone
	 * information from ecalcomp and oldcomp should be all the same.
	 */
	ecaltz = kolab_util_calendar_cache_get_tz (priv->cal_cache, ecalcomp);
	is_instance = e_cal_component_is_instance (ecalcomp);
	if (is_instance) {
		/* Handling of (c) */
		ECalComponent *newcomp = NULL;
		ECalComponent *oldtz = NULL;
		gchar *rid = NULL;
		gchar *notify_object = NULL;

		oldtz = kolab_util_calendar_cache_get_tz (priv->cal_cache, oldcomp);
		rid = e_cal_component_get_recurid_as_string (ecalcomp);
		/* This removes the recurrence from ecalcomp. */
		e_cal_component_set_recurid (ecalcomp, NULL);
		/* This removes ecalcomp from the recurrence. */
		newcomp = kolab_util_calendar_cache_remove_instance (priv->cal_cache, CALOBJ_MOD_THIS, oldcomp, uid, rid);
		if (! kolab_util_calendar_cache_assure_uid_on_ecalcomponent (priv->cal_cache, priv->cal_koma, priv->cal_uri, ecalcomp, TRUE, &error)) {
			status = kolab_util_calendar_map_error (error);
			g_warning ("%s()[%u]: %s", __func__, __LINE__, error->message);
			g_error_free (error);
			if (oldtz != NULL)
				g_object_unref (oldtz);
			g_object_unref (ecalcomp);
			if (ecaltz != NULL)
				g_object_unref (ecaltz);
			g_object_unref (newcomp);
			g_object_unref (oldcomp);
			g_free (*old_object);
			old_object = NULL;
			return status;
		}

		kolab_util_calendar_store (newcomp, oldtz, priv->default_zone, priv->cal_koma, priv->cal_uri, &error);
		if (error != NULL) {
			status = kolab_util_calendar_map_error (error);
			g_warning ("%s()[%u]: error : %s", __func__, __LINE__, error->message);
			g_error_free (error);
			if (oldtz != NULL)
				g_object_unref (oldtz);
			g_object_unref (ecalcomp);
			if (ecaltz != NULL)
				g_object_unref (ecaltz);
			g_object_unref (newcomp);
			g_object_unref (oldcomp);
			g_free (*old_object);
			old_object = NULL;
			return status;
		}
		notify_object = e_cal_component_get_as_string (newcomp);
		e_cal_backend_notify_object_modified (E_CAL_BACKEND (kolab), *old_object, notify_object);
		g_free (notify_object);
		g_object_unref (newcomp);
		if (oldtz != NULL)
			g_object_unref (oldtz);
	}
	g_object_unref (oldcomp);
	kolab_util_calendar_store (ecalcomp, ecaltz, priv->default_zone, priv->cal_koma, priv->cal_uri, &error);
	if (error != NULL) {
		status = kolab_util_calendar_map_error (error);
		g_warning ("%s()[%u]: %s", __func__, __LINE__, error->message);
		g_error_free (error);
		g_object_unref (ecalcomp);
		return status;
	}
	/* [FIXME speed] e_cal_backend_cache_put_component (priv->cal_cache, ecalcomp); */

	/* g_free (*new_object); */
	*new_object = g_strdup (e_cal_component_get_as_string (ecalcomp));
	if (is_instance) {
		e_cal_backend_notify_object_created (E_CAL_BACKEND (backend), *new_object);
	}

	g_object_unref (ecalcomp);
	if (ecaltz != NULL)
		g_object_unref (ecaltz);

	return GNOME_Evolution_Calendar_Success;
} /* e_cal_backend_kolab_modify_object () */

/**
 * e_cal_backend_kolab_remove_object:
 * @backend: An ECalBackendSync object.
 * @cal: An EDataCal object.
 * @uid: UID of the object to remove.
 * @rid: Recurrence ID of the instance to remove, or NULL if removing the
 * whole object.
 * @mod: Type of removal.
 * @old_object: Placeholder for returning the old object as it was stored on the
 * backend.
 * @object: Placeholder for returning the object after it has been modified (when
 * removing individual instances). If removing the whole object, this will be
 * NULL.
 *
 * Removes an (a part of a [recurring] ) object from a calendar backend. The
 * backend will notify all of its clients about the change.
 *
 * Returns: Status code.
 */
static ECalBackendSyncStatus
e_cal_backend_kolab_remove_object (ECalBackendSync *backend,
                                   EDataCal *cal,
                                   const gchar *uid,
                                   const gchar *rid,
                                   CalObjModType mod,
                                   gchar **old_object,
                                   gchar **object)
{
	ECalBackendKolab *kolab = E_CAL_BACKEND_KOLAB (backend);
	ECalBackendKolabPrivate *priv = E_CAL_BACKEND_KOLAB_PRIVATE (kolab);
	KolabSettingsHandler *ksettings = NULL;
	ECalBackendSyncStatus status = GNOME_Evolution_Calendar_Success;
	gchar *sourcename = NULL;
	ECalComponent *oldcomp = NULL;
	ECalComponent *newcomp = NULL;
	ECalComponent *oldtz = NULL;
	ECalComponentId *id = NULL;
	GError *error = NULL;
	gboolean ok;

	(void) cal;

	if (rid == NULL)
		mod = CALOBJ_MOD_ALL;

	oldcomp = kolab_util_calendar_cache_get_object (priv->cal_cache, priv->cal_koma, priv->cal_uri, uid, FALSE, &error);
	if (error != NULL) {
		switch (error->code) {
		case KOLAB_BACKEND_ERROR_NOTFOUND:
			(void) e_cal_backend_cache_remove_component (priv->cal_cache, uid, NULL);
			/* Don't bother if this worked */
			id = g_new0 (ECalComponentId, 1);
			id->uid=g_strdup (uid);
			id->rid=g_strdup (rid);
			e_cal_backend_notify_object_removed (E_CAL_BACKEND (backend), id, NULL, NULL);
			g_free (id);
			status = GNOME_Evolution_Calendar_Success;
			break;
		default:
			status = kolab_util_calendar_map_error (error);
			g_warning ("%s()[%u]: error (%s) code (%i)", __func__, __LINE__, error->message, error->code);
		}
		g_error_free (error);
		return status;
	}

	*old_object = e_cal_component_get_as_string (oldcomp);
	id = e_cal_component_get_id (oldcomp);
	switch (mod) {
	case CALOBJ_MOD_THIS:
		oldtz = kolab_util_calendar_cache_get_tz (priv->cal_cache, oldcomp);
		newcomp = kolab_util_calendar_cache_remove_instance (priv->cal_cache, CALOBJ_MOD_THIS, oldcomp, uid, rid);
		/* newcomp now contains an exception rule for the rid */
		kolab_util_calendar_store (newcomp, oldtz, priv->default_zone, priv->cal_koma, priv->cal_uri, &error);
		if (error != NULL) {
			status = kolab_util_calendar_map_error (error);
			g_warning ("%s()[%u]: error %s", __func__, __LINE__, error->message);
			g_error_free (error);
			g_object_unref (newcomp);
			g_object_unref (oldcomp);
			if (oldtz != NULL)
				g_object_unref (oldtz);
			return status;
		}
		*object = e_cal_component_get_as_string (newcomp);
		g_object_unref (newcomp);
		break;
	case CALOBJ_MOD_THISANDPRIOR: /* not supported by ui */
		g_warning ("%s()[%u]: removing this and prior not supported.", __func__, __LINE__);
		status = GNOME_Evolution_Calendar_UnsupportedMethod;
		break;
	case CALOBJ_MOD_THISANDFUTURE: /* not supported by ui */
		g_warning ("%s()[%u]: removing this and future not supported.", __func__, __LINE__);
		status = GNOME_Evolution_Calendar_UnsupportedMethod;
		break;
	case CALOBJ_MOD_ALL:
	default:
		sourcename = kolab_util_backend_get_relative_path_from_uri (priv->cal_uri);
		ok = kolab_mail_access_delete_by_uid (priv->cal_koma, uid, sourcename, &error);
		if (error != NULL) {
			status = kolab_util_calendar_map_error (error);
			g_warning ("%s()[%u]: error %s", __func__, __LINE__, error->message);
			g_error_free (error);
			error = NULL;
		} else {
			/* Notification is not necessary. (?) */
			status = GNOME_Evolution_Calendar_Success;
		}
		(void) e_cal_backend_cache_remove_component (priv->cal_cache, uid, NULL);

		ksettings = kolab_mail_access_get_settings_handler (priv->cal_koma);
		g_object_unref (ksettings);
		*object = NULL;
	}
	g_object_unref (oldcomp);
	/* Do we need this? */

	ok = kolab_mail_access_source_fbtrigger_needed (priv->cal_koma, sourcename, &error);
	if (error != NULL)
	{
		status = kolab_util_calendar_map_error (error);
		g_warning ("%s()[%u]: %s", __func__, __LINE__, error->message);
		g_error_free (error);
		error = NULL;
		goto cleanup;
	}
	if (ok) {
		ksettings = kolab_mail_access_get_settings_handler (priv->cal_koma);
		kolab_util_calendar_toggle_pfb_trigger (ksettings, sourcename);
		g_object_unref (ksettings);
	}

	e_cal_backend_notify_object_removed (E_CAL_BACKEND (kolab), id, *old_object, *object);
	e_cal_component_free_id (id);

cleanup:
	g_free (sourcename);

	return status;
} /* e_cal_backend_kolab_remove_object () */

static ECalBackendSyncStatus
e_cal_backend_kolab_discard_alarm (ECalBackendSync *backend,
                                   EDataCal *cal,
                                   const gchar *uid,
                                   const gchar *auid)
{
	/* TODO implement me */
	(void) backend;
	(void) cal;
	(void) uid;
	(void) auid;

	g_warning ("Unimplemented method called: %s()", __func__);

	return GNOME_Evolution_Calendar_UnsupportedMethod;
} /* e_cal_backend_kolab_discard_alarm () */

/**
 * e_cal_backend_kolab_receive_objects:
 * @backend: An ECalBackendSync object.
 * @cal: An EDataCal object.
 * @calobj: iCalendar object to receive.
 *
 * Submits a (series of) calendar event(s) through iCalendar object(s).
 *
 * Returns: Status code.
 */
static ECalBackendSyncStatus
e_cal_backend_kolab_receive_objects (ECalBackendSync *backend,
                                     EDataCal *cal,
                                     const gchar *calobj)
{
	ECalBackendKolab *kolab = E_CAL_BACKEND_KOLAB (backend);
	ECalBackendKolabPrivate *priv = E_CAL_BACKEND_KOLAB_PRIVATE (kolab);
	ECalBackendSyncStatus status = GNOME_Evolution_Calendar_Success;
	icalcomponent *icalcomp = NULL;
	ECalComponent *ecalcomp = NULL;
	ECalComponent *tzcomp = NULL;
	gchar *tzstr;
	GError *error = NULL;
	(void) cal;

	/* Workflow:
	 * Incoming calobj is supposed to be a vcalendar string.
	 * KolabMailHandle is the data structure we want to 
	 * generate and KolabMailHandle takes an ECalComponent and a timezone 
	 * (as ECalComponent). The task at hand: convert calobj into an 
	 * icalcomponent ( #1 ). Afterwards process this icalcomponent and extract
	 * the relevant information an ECalComponent can handle ( #2 ). Then
	 * create the ECalComponent, generate a timezone and stuff it
	 * into the KolabMailAccess object. ( #3 )
	 */

	icalcomp = icalparser_parse_string (calobj);
	if (icalcomp == NULL) {
		g_warning (" + *calobj could not be parsed into an icalcomponent: %s", calobj);
		return GNOME_Evolution_Calendar_InvalidObject;
	}
	/* now this better should be an vcalendar entry */
	if (icalcomponent_isa (icalcomp) != ICAL_VCALENDAR_COMPONENT) {
		g_warning (" + **calobj does not represent an vcalendar entry: %s", calobj);
		icalcomponent_free (icalcomp);
		return GNOME_Evolution_Calendar_InvalidObject;
	}
	/* ( #1 ) check */

	status = kolab_util_calendar_extract (icalcomp, priv->source_type, &ecalcomp, &tzcomp);
	if (status != GNOME_Evolution_Calendar_Success) {
		icalcomponent_free (icalcomp);
		return status;
	}
	icalcomponent_free (icalcomp);
	/* ( #2 ) check */
	/* _receive_object () is not supposed to generate new uids. If an object
	 * with the same uid is already in the database, then it will be
	 * replaced by the new object.
	 * if (! kolab_util_calendar_cache_assure_uid_on_ecalcomponent (priv->cal_cache, priv->cal_koma, priv->cal_uri, ecalcomp, &error)) {
	 * 	g_debug ("%s()[%u]: %s", __func__, __LINE__, error->message);
	 * 	g_error_free (error);
	 * 	g_object_unref (ecalcomp);
	 * 	if (tzcomp != NULL)
	 * 		g_object_unref (tzcomp);
	 * 	return GNOME_Evolution_Calendar_OtherError;
	 * }
	 */

	kolab_util_calendar_store (ecalcomp, tzcomp, priv->default_zone, priv->cal_koma, priv->cal_uri, &error);
	if (error != NULL) {
		status = kolab_util_calendar_map_error (error);
		g_warning ("Error during KolabMailHandle creation: %s", error->message);
		g_error_free (error);
		g_object_unref (ecalcomp);
		if (tzcomp != NULL)
			g_object_unref (tzcomp);
		return status;
	}
	/* ( #3 ) check */

	if (tzcomp != NULL) {
		tzstr = e_cal_component_get_as_string (tzcomp);
		status = e_cal_backend_kolab_add_timezone (backend, cal, tzstr);
	}
	if (status != GNOME_Evolution_Calendar_Success)
		return status;
#if 0 /* [FIXME speed] */
	if (e_cal_backend_cache_put_component (priv->cal_cache, ecalcomp) == FALSE)
		status = GNOME_Evolution_Calendar_OtherError;
#endif
	e_cal_backend_notify_object_created (E_CAL_BACKEND (kolab), e_cal_component_get_as_string (ecalcomp));
	g_object_unref (ecalcomp);
	if (tzcomp != NULL)
		g_object_unref (tzcomp);

	return status;
} /* e_cal_backend_kolab_receive_objects () */

static ECalBackendSyncStatus
e_cal_backend_kolab_send_objects (ECalBackendSync *backend,
                                  EDataCal *cal,
                                  const gchar *calobj,
                                  GList **users,
                                  gchar **modified_calobj)
{

	(void) backend;
	(void) cal;

	*users = NULL;

	*modified_calobj = g_strdup (calobj);

	return GNOME_Evolution_Calendar_Success;
} /* e_cal_backend_kolab_send_objects () */

/**
 * e_cal_backend_kolab_add_timezone:
 * @backend: An ECalBackendSync object.
 * @cal: An EDataCal object.
 * @tzobj: VTIMEZONE object to be added.
 *
 * Add a timezone object to the given backend.
 *
 * Returns: Status code.
 */
static ECalBackendSyncStatus
e_cal_backend_kolab_add_timezone (ECalBackendSync *backend,
                                  EDataCal *cal,
                                  const gchar *tzobj)
{
	ECalBackendKolab *kolab = E_CAL_BACKEND_KOLAB (backend);
	ECalBackendKolabPrivate *priv = E_CAL_BACKEND_KOLAB_PRIVATE (kolab);
	icalcomponent *icalcomp = NULL;
	icaltimezone *icaltz = NULL;
	(void) cal;

	icalcomp = icalparser_parse_string (tzobj);
	if (icalcomp == NULL)
		return GNOME_Evolution_Calendar_InvalidObject;

	if (icalcomponent_isa (icalcomp) == ICAL_VTIMEZONE_COMPONENT) {

		icaltz = icaltimezone_new ();
		icaltimezone_set_component (icaltz, icalcomp);

		if (e_cal_backend_cache_put_timezone (priv->cal_cache, icaltz) == FALSE) {
			g_warning ("%s()[%u]: Putting timezone in cache did not succeed.",__func__, __LINE__);
			icaltimezone_free (icaltz, 1);
			return GNOME_Evolution_Calendar_OtherError;
		}
		icaltimezone_free (icaltz, 1);
	}
	return GNOME_Evolution_Calendar_Success;
} /* e_cal_backend_kolab_add_timezone () */

/**
 * e_cal_backend_kolab_set_default_zone:
 * @backend: An ECalBackendSync object.
 * @cal: An EDataCal object.
 * @tz: Timezone object as string.
 *
 * Sets the default timezone for the calendar, which is used to resolve DATE and
 * floating DATE-TIME values.
 *
 * Returns: Status code.
 */
static ECalBackendSyncStatus
e_cal_backend_kolab_set_default_zone (ECalBackendSync *backend,
                                      EDataCal *cal,
                                      const gchar *tzobj)
{
	ECalBackendKolab *kolab = E_CAL_BACKEND_KOLAB (backend);
	ECalBackendKolabPrivate *priv = E_CAL_BACKEND_KOLAB_PRIVATE (kolab);
	ECalComponent *ecaltz = NULL;
	(void) cal;

	ecaltz = e_cal_component_new_from_string (tzobj);
	if (ecaltz == NULL)
		return GNOME_Evolution_Calendar_InvalidObject;

	if (priv->default_zone != NULL)
		g_object_unref (priv->default_zone);
	priv->default_zone = ecaltz;
	return GNOME_Evolution_Calendar_Success;
} /* e_cal_backend_kolab_set_default_zone () */

static ECalBackendSyncStatus
e_cal_backend_kolab_get_free_busy (ECalBackendSync *backend,
                                   EDataCal *cal,
                                   GList *users,
                                   time_t start,
                                   time_t end,
                                   GList **freebusy)
{
	ECalBackendKolab *kolab = E_CAL_BACKEND_KOLAB (backend);
	ECalBackendKolabPrivate *priv = E_CAL_BACKEND_KOLAB_PRIVATE (kolab);
	GList *it;
	GError *error = NULL;
	gboolean internal_state = TRUE;
	KolabSettingsHandler *ksettings = NULL;

	(void) cal;
	(void) start;
	(void) end;

	ksettings = kolab_mail_access_get_settings_handler (priv->cal_koma);
	if (ksettings == NULL)
		return GNOME_Evolution_Calendar_OtherError;

	/* receive the xfb information */
	for (it = g_list_first(users); it != NULL; it = g_list_next (it)) {
		KolabHttpJob *job = NULL;
		ECalComponent *ecalcomp = NULL;

		job = kolab_util_calendar_retrieve_xfb (ksettings, (gchar *) it->data);
		if (job != NULL) {
			ecalcomp = kolabconv_cal_util_freebusy_ecalcomp_new_from_ics ((gchar *) (job->buffer->data), job->nbytes, &error);
			if (error != NULL) {
				g_warning ("%s", error->message);
				g_error_free (error);
				error = NULL;
				internal_state = FALSE;
			}
			kolab_http_job_free (job);
		}
		if (ecalcomp == NULL) {
			/* Strictly, this is not an error, but there's no F/B
			 * information available.
			 */
			g_object_unref (ksettings);
			return GNOME_Evolution_Calendar_ObjectNotFound;
		}
		else {
			*freebusy = g_list_append (*freebusy, g_strdup (e_cal_component_get_as_string (ecalcomp)));
			g_object_unref (ecalcomp);
		}
	}
	g_object_unref (ksettings);
	if (internal_state == TRUE)
		return GNOME_Evolution_Calendar_Success;
	else
		return GNOME_Evolution_Calendar_OtherError;
} /* e_cal_backend_kolab_get_free_busy () */

static ECalBackendSyncStatus
e_cal_backend_kolab_get_changes (ECalBackendSync *backend,
                                 EDataCal *cal,
                                 const gchar *change_id,
                                 GList **adds,
                                 GList **modifies,
                                 GList **deletes)
{
	/* TODO implement me */
	(void) backend;
	(void) cal;
	(void) change_id;
	(void) adds;
	(void) modifies;
	(void) deletes;

	g_warning ("Unimplemented method called: %s()", __func__);

	return GNOME_Evolution_Calendar_Success;
} /* e_cal_backend_kolab_get_changes () */

/**
 * e_cal_backend_kolab_start_query:
 * @backend: A calendar backend.
 * @query: The query to be started.
 *
 * Starts a new live query on the given backend.
 */
static void
e_cal_backend_kolab_start_query (ECalBackend *backend,
                                 EDataCalView *cal_view)
{
	ECalBackendKolab *kolab = E_CAL_BACKEND_KOLAB (backend);
	ECalBackendKolabPrivate *priv = E_CAL_BACKEND_KOLAB_PRIVATE (kolab);
	GError *error = NULL;
	const gchar *query = NULL;
	ECalBackendSExp *ksexp = NULL;
	gchar *sourcename = NULL;
	GList *iCal_objects = NULL; /* list of gchar * (iCal) */
	GList *uid_list = NULL;
	GList *it = NULL;
	ECalBackendSyncStatus status = GNOME_Evolution_Calendar_Success;

	query = e_data_cal_view_get_text (cal_view);
	ksexp = e_cal_backend_sexp_new (query);

	sourcename = kolab_util_backend_get_relative_path_from_uri (priv->cal_uri);

	kolab_util_calendar_cache_update_on_query (priv->cal_cache, priv->cal_koma, query, priv->cal_uri);
	uid_list = kolab_mail_access_query_uids (priv->cal_koma, sourcename, query, &error);
	if (error != NULL) {
		g_warning ("%s()[%u]: %s", __func__, __LINE__, error->message);
		g_error_free (error);
		error = NULL;
	}

	for (it = g_list_first (uid_list); it != NULL; it = g_list_next (it)) {
		gchar *uid = it->data;
		ECalComponent *ecalcomp = NULL;
		ecalcomp = kolab_util_calendar_cache_get_object (priv->cal_cache, priv->cal_koma, priv->cal_uri, uid, TRUE, &error);
		if (error != NULL) {
			g_warning ("%s()[%u]: %s", __func__, __LINE__, error->message);
			g_error_free (error);
			error = NULL;
		}
		/* [FIXME speed] ecalcomp = e_cal_backend_cache_get_component (priv->cal_cache, uid, NULL); */		
		if (ecalcomp != NULL) {
			iCal_objects = g_list_append (iCal_objects, e_cal_component_get_as_string (ecalcomp));
			g_object_unref (ecalcomp);
		}
	}

	e_data_cal_view_notify_objects_added (cal_view, iCal_objects);

	g_list_foreach (iCal_objects, (GFunc) g_free, NULL);
	g_list_free (iCal_objects);
	g_object_unref (ksexp);
	g_free (sourcename);

	kolab_util_glib_glist_free (uid_list);

	e_data_cal_view_notify_done (cal_view, status);
} /* e_cal_backend_kolab_start_query () */

/**
 * e_cal_backend_internal_get_default_timezone:
 * @backend: A calendar backend.
 *
 * Requests the default timezone from the backend.
 *
 * Returns: The icaltimezone representation of the default timezone.
 */
static icaltimezone*
e_cal_backend_kolab_internal_get_default_timezone (ECalBackend *backend)
{
	ECalBackendKolab *kolab = E_CAL_BACKEND_KOLAB (backend);
	ECalBackendKolabPrivate *priv = E_CAL_BACKEND_KOLAB_PRIVATE (kolab);
	icaltimezone *icaltz = NULL;
	icalcomponent *icalcomp = NULL;

	icaltz = icaltimezone_new ();
	icalcomp = e_cal_component_get_icalcomponent (priv->default_zone);
	if (icalcomp == NULL)
		return NULL;
	icaltimezone_set_component (icaltz, icalcomp);

	return icaltz;
} /* e_cal_backend_kolab_internal_get_default_timezone () */

/**
 * e_cal_backend_kolab_get_mode:
 * @backend: A Kolab calendar backend.
 *
 * Queries the current calendar backend mode (local/remote).
 *
 * Returns: The current mode, the calendar is in.
 */
static CalMode
e_cal_backend_kolab_get_mode (ECalBackend *backend)
{
	ECalBackendKolab *kolab = E_CAL_BACKEND_KOLAB (backend);
	ECalBackendKolabPrivate *priv = E_CAL_BACKEND_KOLAB_PRIVATE (kolab);
	KolabMailAccessOpmodeID koma_mode = KOLAB_MAIL_ACCESS_OPMODE_INVAL;
	GError *error = NULL;

	if (priv->cal_koma == NULL) {
		g_warning ("%s()[%u]: KolabMailAccess object not existent.", __func__, __LINE__);
		return CAL_MODE_INVALID;
	}
	koma_mode = kolab_mail_access_get_opmode (priv->cal_koma, &error);
	if (error != NULL) {
		g_warning ("%s()[%u]: %s", __func__, __LINE__, error->message);
		g_error_free (error);
		return CAL_MODE_INVALID;
	}

	switch (koma_mode) {
	case KOLAB_MAIL_ACCESS_OPMODE_OFFLINE:
		return CAL_MODE_LOCAL;
		break;
	case KOLAB_MAIL_ACCESS_OPMODE_ONLINE:
		return CAL_MODE_REMOTE;
		break;
	case KOLAB_MAIL_ACCESS_OPMODE_SHUTDOWN:
	case KOLAB_MAIL_ACCESS_OPMODE_NEW:
	case KOLAB_MAIL_ACCESS_OPMODE_INVAL:
	case KOLAB_MAIL_ACCESS_OPMODE_CONFIGURED:
	default:
		g_warning ("%s()[%u]: KolabMailAccess object's state unknown: %u.", __func__, __LINE__, koma_mode);
		return CAL_MODE_INVALID;
	}
} /* e_cal_backend_kolab_get_mode () */

/**
 * e_cal_backend_kolab_set_mode:
 * @backend: A Kolab calendar backend.
 * @mode: Mode to change to.
 * 
 * Sets the mode of the calendar backend to the given mode.
 */
static void
e_cal_backend_kolab_set_mode (ECalBackend *backend,
                              CalMode mode)
{
	ECalBackendKolab *kolab = E_CAL_BACKEND_KOLAB (backend);
	ECalBackendKolabPrivate *priv = E_CAL_BACKEND_KOLAB_PRIVATE (kolab);
	GError *error = NULL;
	KolabMailAccessOpmodeID koma_mode = KOLAB_MAIL_ACCESS_OPMODE_INVAL;
	KolabMailAccessOpmodeID tmp_mode = KOLAB_MAIL_ACCESS_OPMODE_INVAL;
	gchar *sourcename = NULL;
	KolabSettingsHandler *ksettings = NULL;
	gboolean ok;

	switch (mode) {
	case CAL_MODE_LOCAL:
		priv->cal_mode = CAL_MODE_LOCAL;
		koma_mode = KOLAB_MAIL_ACCESS_OPMODE_OFFLINE;
		break;
	case CAL_MODE_REMOTE:
		priv->cal_mode = CAL_MODE_REMOTE;
		koma_mode = KOLAB_MAIL_ACCESS_OPMODE_ONLINE;
		break;
	case CAL_MODE_ANY:
	case CAL_MODE_INVALID:
	default:
		g_assert_not_reached();
	}
	if (priv->cal_koma == NULL) {
		g_warning ("%s()[%u]: KolabMailAccess object not existent.", __func__, __LINE__);
		return;
	}
	tmp_mode = kolab_mail_access_get_opmode (priv->cal_koma, &error);
	if (error != NULL)
	{
		g_warning ("%s()[%u]: error getting mode: %s", __func__, __LINE__, error->message);
		g_error_free (error);
		return;
	}

	if (tmp_mode < KOLAB_MAIL_ACCESS_OPMODE_OFFLINE) {
		return;
	}
	ok = kolab_mail_access_set_opmode (priv->cal_koma, koma_mode, &error);
	if (error != NULL)
	{
		g_warning ("%s()[%u]: error setting mode: %s", __func__, __LINE__, error->message);
		g_error_free (error);
		error = NULL;
		return;
	}
	
	if (ok && (koma_mode < KOLAB_MAIL_ACCESS_OPMODE_ONLINE))
		return;
	
	sourcename = kolab_util_backend_get_relative_path_from_uri (priv->cal_uri);
	ok = kolab_mail_access_source_fbtrigger_needed (priv->cal_koma, sourcename, &error);
	if (error != NULL)
	{
		g_warning ("%s()[%u]: %s", __func__, __LINE__, error->message);
		g_error_free (error);
		error = NULL;
		goto cleanup;
	}
	if (ok) {
		ksettings = kolab_mail_access_get_settings_handler (priv->cal_koma);
		kolab_util_calendar_toggle_pfb_trigger (ksettings, sourcename);
		g_object_unref (ksettings);
	}
cleanup:
	g_free (sourcename);
} /* e_cal_backend_kolab_set_mode () */

/**
 * e_cal_backend_kolab_set_koma_table:
 * @kolab: An ECalBackendKolab object.
 * @koma_objects: A GHashTable to contain the required KoMA instances.
 *
 * This method has to be called before any other method which accesses
 * Kolab infrastructure. In this case it should be called from
 * e-cal-backend-kolab-factory during creation of a new ECalBackendKolab 
 * instance.
 */
void
e_cal_backend_kolab_set_koma_table (ECalBackendKolab *kolab,
                                    GHashTable *koma_objects)
{
	ECalBackendKolabPrivate *priv = E_CAL_BACKEND_KOLAB_PRIVATE (kolab);

	g_hash_table_ref (koma_objects);
	priv->koma_table = koma_objects;
}


/* init and teardown **********************************************************/

static void
e_cal_backend_kolab_init (ECalBackendKolab *kolab)
{
	ECalBackendKolabPrivate *priv = E_CAL_BACKEND_KOLAB_PRIVATE (kolab);

	priv->koma_table = NULL;
	priv->cal_mode = CAL_MODE_LOCAL; /* Set mode to local at initialization. */
	priv->user_email = NULL;
	priv->default_zone = NULL;
	priv->cal_koma = NULL;
	priv->cal_cache = NULL;
	priv->already_opened = FALSE;
	priv->cal_uri = NULL;

} /* e_cal_backend_kolab_init () */

static void
e_cal_backend_kolab_dispose (GObject *object)
{
	ECalBackendKolabPrivate *priv = E_CAL_BACKEND_KOLAB_PRIVATE (object);
	
	if (priv->already_opened != FALSE) {
		priv->already_opened = FALSE;
		g_object_unref (priv->cal_koma);
		priv->cal_koma = NULL;
	}
	if (priv->user_email != NULL) {
		g_free (priv->user_email);
		priv->user_email = NULL;
	}
	if (priv->default_zone != NULL) {
		g_object_unref (priv->default_zone);
		priv->default_zone = NULL;
	}
	if (priv->cal_cache != NULL) {
		g_object_unref (priv->cal_cache);
		priv->cal_cache = NULL;
	}
	if (priv->cal_uri != NULL) {
		g_free (priv->cal_uri);
		priv->cal_uri = NULL;
	}
	/* chain up dispose */
	if (G_OBJECT_CLASS (e_cal_backend_kolab_parent_class)->dispose)
		(* G_OBJECT_CLASS (e_cal_backend_kolab_parent_class)->dispose) (object);
} /* e_cal_backend_kolab_dispose () */

static void
e_cal_backend_kolab_finalize (GObject *object)
{
	/* TODO: Add deinitalization code here */

	G_OBJECT_CLASS (e_cal_backend_kolab_parent_class)->finalize (object);
}

/**
 * e_cal_backend_kolab_class_init:
 * @klass: An ECalBackendKolabClass type.
 *
 * Class initialisation function for the Kolab backend.
 */
static void
e_cal_backend_kolab_class_init (ECalBackendKolabClass *klass)
{
	GObjectClass* object_class = G_OBJECT_CLASS (klass);
	ECalBackendClass *backend_class = E_CAL_BACKEND_CLASS (klass);
	ECalBackendSyncClass* sync_class = E_CAL_BACKEND_SYNC_CLASS (klass);

	g_type_class_add_private (klass, sizeof (ECalBackendKolabPrivate));

	object_class->dispose = e_cal_backend_kolab_dispose;
	object_class->finalize = e_cal_backend_kolab_finalize;
	
	klass->set_koma_table = e_cal_backend_kolab_set_koma_table;

	/* Assignment of virtual methods - see e-cal-backend.h for reference */
	sync_class->is_read_only_sync = e_cal_backend_kolab_is_read_only;
	sync_class->get_cal_address_sync = e_cal_backend_kolab_get_cal_address;
	sync_class->get_alarm_email_address_sync = e_cal_backend_kolab_get_alarm_email_address;
	sync_class->get_ldap_attribute_sync = e_cal_backend_kolab_get_ldap_attribute;
	sync_class->get_static_capabilities_sync = e_cal_backend_kolab_get_static_capabilities;

	sync_class->open_sync = e_cal_backend_kolab_open;
	sync_class->refresh_sync = e_cal_backend_kolab_refresh;
	sync_class->remove_sync = e_cal_backend_kolab_remove;

	/* Assignment of object related virtual methods - see e-cal-backend.h for reference */
	sync_class->create_object_sync = e_cal_backend_kolab_create_object;
	sync_class->modify_object_sync = e_cal_backend_kolab_modify_object;
	sync_class->remove_object_sync = e_cal_backend_kolab_remove_object;

	sync_class->discard_alarm_sync = e_cal_backend_kolab_discard_alarm;

	sync_class->receive_objects_sync = e_cal_backend_kolab_receive_objects;
	sync_class->send_objects_sync = e_cal_backend_kolab_send_objects;

	sync_class->get_default_object_sync = e_cal_backend_kolab_get_default_object;
	sync_class->get_object_sync = e_cal_backend_kolab_get_object;
	sync_class->get_object_list_sync = e_cal_backend_kolab_get_object_list;

	sync_class->get_attachment_list_sync = e_cal_backend_kolab_get_attachment_list;

	/* Assignment of timezone related virtual methods - see e-cal-backend.h for reference */
	/* optional: any object deriving from ECalBackendSync can implement only internal_get_timezone 
	 * sync_class->get_timezone_sync = e_cal_backend_kolab_get_timezone; */

	sync_class->add_timezone_sync = e_cal_backend_kolab_add_timezone;
	sync_class->set_default_zone_sync = e_cal_backend_kolab_set_default_zone;
	/* deprecated: do not use: sync_class->set_default_timezone_sync = e_cal_backend_kolab_set_default_timezone; */

	backend_class->start_query = e_cal_backend_kolab_start_query;
	backend_class->internal_get_default_timezone = e_cal_backend_kolab_internal_get_default_timezone;

	/* Assignment of mode relate virtual methods - see e-cal-backend.h for reference */
	backend_class->get_mode = e_cal_backend_kolab_get_mode;
	backend_class->set_mode = e_cal_backend_kolab_set_mode;

	sync_class->get_freebusy_sync = e_cal_backend_kolab_get_free_busy;
	sync_class->get_changes_sync = e_cal_backend_kolab_get_changes;

} /* e_cal_backend_kolab_class_init () */

