/***************************************************************************
 *            camel-kolab-imapx-metadata-db.c
 *
 *  Mon Oct 11 12:37:05 2010
 *  Copyright  2010  Christian Hilberg
 *  <hilberg@kernelconcepts.de>
 ****************************************************************************/

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with main.c; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */
 
/*----------------------------------------------------------------------------*/

/* TODO There is code duplication regarding SQLite in KolabImapxMetadataDb,
 *	KolabMailInfoDb and KolabMailSideCache.
 *
 *	libekolabutil/kolab-util-sqlite.[hc] has the needed functions,
 *	which can be used here instead of the local dupes.
 *
 *	A generalized approach could be to use some sort of persistent
 *	GHashTable, which would be mapped transparently onto a SQLite DB
 *	(or an object oriented DB right away), with some constraints 
 *	applying to the hash table keys and values (as with marshalling
 *	of serializable objects). A GPersistentHashTable could be nested,
 *	binary data stored as GByteArray values and basic data types carry
 *	with them some type annotations (maybe through GType)
 */

/*----------------------------------------------------------------------------*/

#include <string.h>
#include <sqlite3.h>

/* Kolab error reporting */
#include <libekolabutil/kolab-util-error.h>

#include <camel/providers/imapx/camel-imapx-utils.h>

#include "camel-kolab-imapx-metadata.h"
#include "camel-kolab-imapx-metadata-db.h"

/*----------------------------------------------------------------------------*/

#define CAMEL_KOLAB_IMAPX_METADATA_DB_FILE "folders_metadata.db"
#define CAMEL_KOLAB_IMAPX_SQLITE_DB_MASTER "sqlite_master"
#define CAMEL_KOLAB_IMAPX_SQLITE_DB_TBL_FOLDERS "folders"

#define CAMEL_KOLAB_IMAPX_SQLITE_COL_FOLDERNAME "folder_name"
#define CAMEL_KOLAB_IMAPX_SQLITE_COL_FOLDERTYPE "folder_type"


typedef enum {
	CAMEL_KOLAB_IMAPX_METADATA_TABLE_TYPE_PATHS,
	CAMEL_KOLAB_IMAPX_METADATA_TABLE_TYPE_ENTRIES
} CamelKolabIMAPXMetaDataTableType;

enum {
	CAMEL_KOLAB_IMAPX_METADATA_TABLE_COL_FOLDERNAME = 0,
	CAMEL_KOLAB_IMAPX_METADATA_TABLE_COL_FOLDERTYPE,
	CAMEL_KOLAB_IMAPX_METADATA_TABLE_LAST_COL
};

/*----------------------------------------------------------------------------*/

static sqlite3*
kolab_db_open_trycreate (const gchar* path)
{
	gint sql3_err;
	sqlite3 *db;

	g_assert (path != NULL);

	sql3_err = sqlite3_open_v2 (path,
	                            &db,
	                            SQLITE_OPEN_READWRITE | SQLITE_OPEN_CREATE,
	                            NULL /* default sqlite3 vfs module */);

	if (sql3_err != SQLITE_OK) {
		g_warning ("%s: %s", __func__, sqlite3_errmsg(db));
		(void)sqlite3_close (db);
		return NULL;
	}

	return db;
}

static gint
kolab_db_table_exists_cb (void *data, gint ncols, gchar **coltext, gchar **colname)
{
	CamelKolabIMAPXMetaDataDB *mdb = (CamelKolabIMAPXMetaDataDB *)data;
	
	(void)ncols;
	(void)coltext;
	(void)colname;

	mdb->ctr++;
	
	return SQLITE_OK;
}

static gboolean
kolab_db_table_exists (CamelKolabIMAPXMetaDataDB *mdb,
                       const gchar *name,
                       gboolean *exists,
                       GError **err)
{
	gchar* sql_str;
	gint sql_errno;
	
	g_assert (mdb != NULL);
	g_assert (mdb->db != NULL);
	g_assert (name != NULL);
	g_assert (exists != NULL);
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	/* SELECT name FROM sqlite_master WHERE type='table' AND name='table_name'; */
	sql_str = sqlite3_mprintf ("SELECT name FROM %Q WHERE type='table' AND name=%Q;",
	                           CAMEL_KOLAB_IMAPX_SQLITE_DB_MASTER,
	                           name);
	mdb->ctr = 0;
	sql_errno = sqlite3_exec (mdb->db,
	                          sql_str,
	                          kolab_db_table_exists_cb,
	                          (void*)mdb,
	                          NULL);
	sqlite3_free (sql_str);
	
	if (sql_errno != SQLITE_OK) {
		g_set_error (err,
			     KOLAB_CAMEL_KOLAB_ERROR,
			     KOLAB_CAMEL_KOLAB_ERROR_DB,
			     "%s: %s",
		             __func__, sqlite3_errmsg (mdb->db));
		return FALSE;	
	}

	if (mdb->ctr > 1) {
		/* each table must exist only once (or not at all) */
		g_set_error (err,
			     KOLAB_CAMEL_KOLAB_ERROR,
			     KOLAB_CAMEL_KOLAB_ERROR_DB,
			     "%s: multiple table [%s], corrupted [%s]",
		             __func__, name, mdb->path);	
		return FALSE;
	}

	if (mdb->ctr == 0)
		*exists = FALSE;
	else
		*exists = TRUE;
	
	return TRUE;
}

static gboolean
kolab_db_table_create (CamelKolabIMAPXMetaDataDB *mdb,
                       const gchar *name,
                       GError **err)
{
	gchar* sql_str = NULL;
	gint sql_errno;

	g_assert (mdb != NULL);
	g_assert (mdb->db != NULL);
	g_assert (name != NULL);
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	/* primary key (1st column) is folder names,
	 * rest of the row maps to CamelKolabFolderMetaData */
	sql_str = sqlite3_mprintf ("CREATE TABLE IF NOT EXISTS %Q ( %Q TEXT PRIMARY KEY, %Q INTEGER );",
			           name,
	                           CAMEL_KOLAB_IMAPX_SQLITE_COL_FOLDERNAME,
	                           CAMEL_KOLAB_IMAPX_SQLITE_COL_FOLDERTYPE);
	
	sql_errno = sqlite3_exec (mdb->db, sql_str, NULL, NULL, NULL);
	sqlite3_free (sql_str);
	
	if (sql_errno != SQLITE_OK) {
		g_set_error (err,
			     KOLAB_CAMEL_KOLAB_ERROR,
			     KOLAB_CAMEL_KOLAB_ERROR_DB,
			     "%s: %s",
		             __func__, sqlite3_errmsg (mdb->db));
		return FALSE;	
	}

	return TRUE;	
}

/*----------------------------------------------------------------------------*/

CamelKolabIMAPXMetaDataDB*
kolab_imapx_meta_data_db_new (void)
{
	CamelKolabIMAPXMetaDataDB *mdb;
	mdb = g_new0 (CamelKolabIMAPXMetaDataDB, 1);
	mdb->db   = NULL;
	mdb->path = NULL;
	mdb->ctr  = 0;
	return mdb;
}

gboolean
kolab_imapx_meta_data_db_free (CamelKolabIMAPXMetaDataDB *mdb,
                               GError **err)
{
	return kolab_imapx_meta_data_db_close (mdb, err);
}


gboolean
kolab_imapx_meta_data_db_open (CamelKolabIMAPXMetaDataDB *mdb,
                               CamelService *service,
                               CamelSession *session,
                               GError **err)
{
	GError  *tmp_err;
	gchar   *store_db_path;
	sqlite3	*tmp_db;

	g_assert (mdb != NULL);
	g_assert (service != NULL);
	g_assert (session != NULL);
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);
	
	if (mdb->db != NULL)
		return TRUE;

	/* inspired by construct() from camel-store.c */
	
	store_db_path = g_build_filename (service->url->path,
	                                  CAMEL_KOLAB_IMAPX_METADATA_DB_FILE,
	                                  NULL);

	if (!service->url->path || strlen (store_db_path) < 2) {
		CamelException *ex;
		gchar *store_path;

		g_free (store_db_path);
		ex = camel_exception_new ();
		store_path = camel_session_get_storage_path (session,
		                                             service,
		                                             ex);
		store_db_path = g_build_filename (store_path,
		                                  CAMEL_KOLAB_IMAPX_METADATA_DB_FILE,
		                                  NULL);
		g_free (store_path);

		if (camel_exception_is_set (ex)) {
			g_free (store_db_path);
			tmp_err = kolab_gerror_new_from_camelexception (ex, KOLAB_CAMEL_KOLAB_ERROR);
			camel_exception_free (ex);
			g_propagate_error (err, tmp_err);
			return FALSE;
		}
		camel_exception_free (ex);
	}

	/* if store_db_path is still invalid, something's really hosed -
	 * IMAPX should have taken care already via construct() in
	 * camel-store.c (via camel-imapx-store.c)
	 */

	tmp_db = kolab_db_open_trycreate (store_db_path);

	if (tmp_db == NULL) {
		gchar *store_path;
		gchar *store_db_path_prev;

		store_db_path_prev = store_db_path;
		store_path = camel_session_get_storage_path (session,
		                                             service,
		                                             NULL);
		store_db_path = g_build_filename (store_path,
		                                  CAMEL_KOLAB_IMAPX_METADATA_DB_FILE,
		                                  NULL);
		g_free (store_path);

		tmp_db = kolab_db_open_trycreate (store_db_path);
		
		if (tmp_db == NULL) {
			g_set_error (err,
				     KOLAB_CAMEL_KOLAB_ERROR,
				     KOLAB_CAMEL_KOLAB_ERROR_DB,
				     "%s: could not open/create sqlite db for any of [%s] and [%s]",
				     __func__, store_db_path_prev, store_db_path);
			g_free (store_db_path_prev);
			g_free (store_db_path);
			return FALSE;
		}
		
		g_free (store_db_path_prev);
	}
	
	mdb->db = tmp_db;
	mdb->path = store_db_path;
	mdb->ctr = 0;

	return TRUE;
}

gboolean
kolab_imapx_meta_data_db_init (CamelKolabIMAPXMetaDataDB *mdb, GError **err)
{
	GError *tmp_err = NULL;
	gboolean sql_ok;
	gboolean exists;
	
	g_assert (mdb != NULL);
	g_assert (mdb->db != NULL);
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	/* make sure paths table is there */
	sql_ok = kolab_db_table_exists (mdb,
	                                CAMEL_KOLAB_IMAPX_SQLITE_DB_TBL_FOLDERS,
	                                &exists,
	                                &tmp_err);
	if (!sql_ok) {
		g_propagate_error (err, tmp_err);
		return FALSE;
	}

	if (!exists)
		sql_ok = kolab_db_table_create (mdb,
		                                CAMEL_KOLAB_IMAPX_SQLITE_DB_TBL_FOLDERS,
		                                &tmp_err);
	if (!sql_ok) {
		g_propagate_error (err, tmp_err);
		return FALSE;
	}
	
	return TRUE;
}

gboolean
kolab_imapx_meta_data_db_close (CamelKolabIMAPXMetaDataDB *mdb, GError **err)
{
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	if (mdb == NULL)
		return TRUE;

	if (mdb->db) {
		if (sqlite3_close (mdb->db) != SQLITE_OK) {
			g_set_error (err,
				     KOLAB_CAMEL_KOLAB_ERROR,
				     KOLAB_CAMEL_KOLAB_ERROR_DB,
				     "%s: %s",
				     __func__, sqlite3_errmsg (mdb->db));
			return FALSE;
		}
	}
	
	if (mdb->path)
		g_free (mdb->path);

	g_free (mdb);
	return TRUE;
}

gboolean
kolab_imapx_meta_data_db_folder_update (CamelKolabIMAPXMetaDataDB *mdb,
                                        const gchar *folder_name,
                                        const CamelKolabFolderMetaData *kfmd,
                                        GError **err)
{
	gchar *sql_str;
	gint sql_errno;
	
	g_assert (mdb != NULL);
	g_assert (mdb->db != NULL);
	g_assert (folder_name != NULL);
	g_assert (kfmd != NULL);
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	sql_str = sqlite3_mprintf ("INSERT OR REPLACE INTO %Q VALUES ( %Q, %i );",
	                           CAMEL_KOLAB_IMAPX_SQLITE_DB_TBL_FOLDERS,
	                           folder_name,
	                           kfmd->folder_type);

	sql_errno = sqlite3_exec (mdb->db, sql_str, NULL, NULL, NULL);
	sqlite3_free (sql_str);
	if (sql_errno != SQLITE_OK) {
		g_set_error (err,
			     KOLAB_CAMEL_KOLAB_ERROR,
			     KOLAB_CAMEL_KOLAB_ERROR_DB,
			     "%s: %s",
		             __func__, sqlite3_errmsg (mdb->db));
		return FALSE;	
	}
	
	return TRUE;
}

gboolean
kolab_imapx_meta_data_db_update (CamelKolabIMAPXMetaDataDB *mdb,
                                 GHashTable *kolab_metadata,
                                 GError **err)
{
	/* TODO extend this to true 'update' or
	 *	create a separate db cleansing function
	 */

	GError *tmp_err = NULL;
	GHashTableIter kmd_iter;
	gpointer kmd_key;
	gpointer kmd_value;
	gchar *folder_name;
	CamelKolabFolderMetaData *kfmd;
	gboolean row_ok;
	
	g_assert (mdb != NULL);
	g_assert (mdb->db != NULL);
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	if (kolab_metadata == NULL)
		return TRUE;

	g_hash_table_iter_init (&kmd_iter, kolab_metadata);
	while (g_hash_table_iter_next (&kmd_iter, &kmd_key, &kmd_value)) {
		folder_name = (gchar*) kmd_key;
		kfmd = (CamelKolabFolderMetaData *) kmd_value;
		row_ok = kolab_imapx_meta_data_db_folder_update (mdb,
		                                                 folder_name,
		                                                 kfmd,
		                                                 &tmp_err);
		if (!row_ok) {
			g_propagate_error (err, tmp_err);
			return FALSE;
		}
	}	

	return TRUE;
}

CamelKolabFolderMetaData*
kolab_imapx_meta_data_db_lookup (CamelKolabIMAPXMetaDataDB *mdb,
                                 const gchar *folder_name,
                                 GError **err)
{
	CamelKolabFolderMetaData *kfmd;
	gchar *sql_str;
	gint sql_errno;
	sqlite3_stmt *sql_stmt;
	
	g_assert (mdb != NULL);
	g_assert (mdb->db != NULL);
	g_assert (folder_name != NULL);
	g_return_val_if_fail (err == NULL || *err == NULL, NULL);

	sql_str = sqlite3_mprintf ("SELECT * FROM %Q WHERE %q=%Q;",
	                           CAMEL_KOLAB_IMAPX_SQLITE_DB_TBL_FOLDERS,
	                           CAMEL_KOLAB_IMAPX_SQLITE_COL_FOLDERNAME,
	                           folder_name);

	sql_errno = sqlite3_prepare_v2 (mdb->db, sql_str, -1, &sql_stmt, NULL);
	g_assert ((sql_errno == SQLITE_OK) && (sql_stmt != NULL));

	sql_errno = sqlite3_step (sql_stmt);
	if (sql_errno != SQLITE_ROW) {
		if (sql_errno != SQLITE_DONE) {
			g_set_error (err,
				     KOLAB_CAMEL_KOLAB_ERROR,
				     KOLAB_CAMEL_KOLAB_ERROR_DB,
				     "%s: %s",
				     __func__, sqlite3_errmsg (mdb->db));
		}
		sql_errno = sqlite3_finalize (sql_stmt);
		sqlite3_free (sql_str);
		return NULL;
	}

	kfmd = kolab_folder_meta_data_new ();
	kfmd->folder_type = sqlite3_column_int (sql_stmt,
	                                        CAMEL_KOLAB_IMAPX_METADATA_TABLE_COL_FOLDERTYPE);

	sql_errno = sqlite3_finalize (sql_stmt);
	sqlite3_free (sql_str);
	if (sql_errno != SQLITE_OK) {
		g_set_error (err,
			     KOLAB_CAMEL_KOLAB_ERROR,
			     KOLAB_CAMEL_KOLAB_ERROR_DB,
			     "%s: %s",
		             __func__, sqlite3_errmsg (mdb->db));
		kolab_folder_meta_data_free (kfmd);
		return NULL;	
	}

	return kfmd;
}

gboolean
kolab_imapx_meta_data_db_remove_folder (CamelKolabIMAPXMetaDataDB *mdb,
                                        const gchar *folder_name,
                                        GError **err)
{
	gchar *sql_str;
	gint sql_errno;

	g_assert (mdb != NULL);
	g_assert (mdb->db != NULL);
	g_assert (folder_name != NULL);
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	sql_str = sqlite3_mprintf ("DELETE FROM %Q WHERE %q=%Q;",
	                           CAMEL_KOLAB_IMAPX_SQLITE_DB_TBL_FOLDERS,
	                           CAMEL_KOLAB_IMAPX_SQLITE_COL_FOLDERNAME,
	                           folder_name);

	sql_errno = sqlite3_exec (mdb->db, sql_str, NULL, NULL, NULL);
	sqlite3_free (sql_str);

	/* the SQLite3-Docs state that SQLITE_NOTFOUND is an unused
	 * error code. Since the WHERE clause should return an empty
	 * match for non-existent rows, DELETE should just take no action
	 * and return successfully, if folder_name was not found in the
	 * table
	 */
#if 0
	if (sql_errno == SQLITE_NOTFOUND)
		return TRUE;
#endif

	if (sql_errno != SQLITE_OK) {
		g_set_error (err,
			     KOLAB_CAMEL_KOLAB_ERROR,
			     KOLAB_CAMEL_KOLAB_ERROR_DB,
			     "%s: %s",
		             __func__, sqlite3_errmsg (mdb->db));
		return FALSE;	
	}

	return TRUE;
}

/*----------------------------------------------------------------------------*/
