/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/***************************************************************************
 *            e-book-backend-kolab.c
 *
 *  2010
 *  Copyright  2010  Christian Hilberg
 *  <hilberg@kernelconcepts.de>
 * and Silvan Marco Fin <silvan@kernelconcepts.de> in 2011
 ****************************************************************************/

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

/*----------------------------------------------------------------------------*/

#include <glib.h>
#include <glib-object.h>

#include <libedata-book/e-data-book-view.h>
#include <libedata-book/e-book-backend-sync.h>
#include <libedata-book/e-book-backend-cache.h>

#include <libekolabutil/camel-system-headers.h>
#include <libekolabutil/kolab-util-camel.h>
#include <libekolabutil/kolab-util-http.h>
#include <libekolabutil/kolab-util-glib.h>

#include <libekolab/kolab-types.h>
#include <libekolab/kolab-mail-access.h>
#include <libekolab/kolab-settings-handler.h>

#include "kolab-util-contact.h"
#include "kolab-util-contact-cache.h"

#include "e-book-backend-kolab.h"

/*----------------------------------------------------------------------------*/
/* table of KolabMailAccess objects */

/* static GHashTable *koma_objects = NULL; */

/*----------------------------------------------------------------------------*/

typedef struct _EBookBackendKolabPrivate EBookBackendKolabPrivate;
struct _EBookBackendKolabPrivate
{
	/* GNOME_Evolution_Addressbook_BookMode	book_mode; */
	EBookBackendCache			*book_cache;
	KolabMailAccess				*book_koma;
	gchar					*book_uri;
	GHashTable				*koma_table;
	gboolean				auth_received;
};

#define E_BOOK_BACKEND_KOLAB_PRIVATE(o)  (G_TYPE_INSTANCE_GET_PRIVATE ((o), E_TYPE_BOOK_BACKEND_KOLAB, EBookBackendKolabPrivate))


G_DEFINE_TYPE (EBookBackendKolab, e_book_backend_kolab, E_TYPE_BOOK_BACKEND_SYNC)

/*----------------------------------------------------------------------------*/
/* internal statics */

static void
e_book_backend_kolab_open (EBookBackendSync *backend,
                           EDataBook *book,
                           GCancellable *cancellable,
                           gboolean only_if_exists,
                           GError **err)
{
	EBookBackendKolab *self = E_BOOK_BACKEND_KOLAB (backend);
	EBookBackendKolabPrivate *priv = E_BOOK_BACKEND_KOLAB_PRIVATE (self);

	g_assert (E_IS_DATA_BOOK (book));
	(void)cancellable; /* FIXME */ /* cancellable may be NULL */
	(void)only_if_exists; /* FIXME */
	g_return_if_fail (err == NULL || *err == NULL);

	(void)self;
	(void)priv;
	g_error ("%s: FIXME implement me", __func__);

#if 0 /* FIXME old*/
	g_debug ("%s()[%u] called.", __func__, __LINE__);

	kolab_util_glib_init ();
	kolab_util_http_init ();
	/* libcamel
	 * Curl init may configure the underlying SSL lib,
	 * but as far as SSL goes, we want Camel to rule here
	 * TODO check whether Camel session needs to be initialized before or after libcurl.
	 */
	ok = kolab_util_camel_init (&error);
	if (! ok) {
		status = kolab_util_contact_map_error (error);
		g_warning ("%s()[%u]: %s",
		           __func__, __LINE__, error->message);
		g_error_free (error);
		error = NULL;
	}

	priv->book_uri = e_source_get_uri(source);
	g_debug ("%s()[%u] uri = %s",
	         __func__, __LINE__, priv->book_uri);

	cex = camel_exception_new ();
	c_url = camel_url_new (priv->book_uri, cex);
	if (c_url == NULL) {
		g_warning ("%s()[%u]: %s",
		           __func__, __LINE__, camel_exception_get_description (cex));
		camel_exception_free (cex);
		status = GNOME_Evolution_Addressbook_OtherError;
	}
	servername = g_strdup (c_url->host);
	username = g_strdup (c_url->user);
	camel_url_free (c_url);
	g_debug ("%s()[%u] servername = %s",
	         __func__, __LINE__, servername);
	g_debug ("%s()[%u]   username = %s",
	         __func__, __LINE__, username);

	/* Initialize backend cache */
	if (priv->book_cache != NULL) {
		g_object_unref (priv->book_cache);
	}
	priv->book_cache = e_book_backend_cache_new (priv->book_uri);
	ok = e_file_cache_clean (E_FILE_CACHE (priv->book_cache));
	g_debug (" + Book cache cleaning %s.", ok ? "was successful" : "FAILED");

	/* Prepare data from sync strategy property */
	sync_prop = e_source_get_property (source, KOLAB_SYNC_STRATEGY_PROP);
	sync_value = kolab_util_misc_sync_value_from_property (sync_prop);
	sourcename = kolab_util_backend_get_relative_path_from_uri (priv->book_uri);

	user_at_server = g_strdup_printf ("%s@%s",
	                                  username, servername);

	ok = g_hash_table_lookup_extended (priv->koma_table,
	                                   user_at_server,
	                                   (gpointer *) &tmp_key,
	                                   (gpointer *) &tmp_koma);
	if (ok) {
		/* There is already a KoMA instance for $servername. Use it and return */
		g_object_ref (tmp_koma);
		priv->book_koma = tmp_koma;
		g_free (servername);
		g_free (username);
		g_free (user_at_server);
		ksettings = kolab_mail_access_get_settings_handler (priv->book_koma);
		kolab_util_backend_prepare_settings (ksettings,
		                                     NULL,
		                                     NULL,
		                                     NULL,
		                                     NULL,
		                                     sourcename,
		                                     &sync_value);
		goto notifications;
	}
	/* Otherwise we need to setup a new KoMA instance and a settings handler */

	/* Configure settings handler */
	ksettings = KOLAB_SETTINGS_HANDLER (g_object_new (KOLAB_TYPE_SETTINGS_HANDLER, NULL));
	(void) kolab_settings_handler_configure (ksettings,
	                                         KOLAB_FOLDER_CONTEXT_CONTACT,
	                                         &error);
	if (error != NULL) {
		g_warning ("%s()[%u]: %s",
		           __func__, __LINE__, error->message);
		g_error_free (error);
		error = NULL;
	}
	(void) kolab_settings_handler_bringup (ksettings,
	                                       &error);
	if (error != NULL) {
		g_warning ("%s()[%u]: %s",
		           __func__, __LINE__, error->message);
		g_error_free (error);
		error = NULL;
	}
	kolab_util_backend_prepare_settings (ksettings,
	                                     source,
	                                     servername,
	                                     username,
	                                     NULL,
	                                     sourcename,
	                                     &sync_value);

	g_free (servername);
	g_free (username);
	g_free (sourcename);

	priv->book_koma = KOLAB_MAIL_ACCESS (g_object_new (KOLAB_TYPE_MAIL_ACCESS, NULL));
	g_object_add_toggle_ref (G_OBJECT (priv->book_koma),
	                         kolab_util_backend_koma_table_cleanup_cb,
	                         priv->koma_table);
	g_hash_table_insert (priv->koma_table,
	                     user_at_server,
	                     priv->book_koma);

	kolab_mail_access_configure (priv->book_koma,
	                             ksettings,
	                             &error);
	g_object_unref (ksettings);
	if (error != NULL) {
		status = kolab_util_contact_map_error (error);
		g_warning ("%s()[%u]: %s",
		           __func__, __LINE__, error->message);
		g_error_free (error);
		return status;
	}
	kolab_mail_access_bringup (priv->book_koma,
	                           &error);
	if (error != NULL) {
		status = kolab_util_contact_map_error (error);
		g_warning ("%s()[%u]: %s",
		           __func__, __LINE__, error->message);
		g_error_free (error);
		return status;
	}

 notifications:
	/* e_book_backend_kolab_set_mode (backend, priv->book_mode); */
	tmp_mode = kolab_mail_access_get_opmode (priv->book_koma, &error);
	if (error != NULL) {
		g_warning ("%s()[%u]: %s",
		           __func__, __LINE__, error->message);
		g_error_free (error);
		error = NULL;
	}

	e_book_backend_notify_auth_required (backend);
	e_book_backend_set_is_loaded (backend, TRUE);
	e_book_backend_notify_connection_status (backend,
	                                         tmp_mode = KOLAB_MAIL_ACCESS_OPMODE_ONLINE ? TRUE : FALSE);
	e_book_backend_set_is_writable (backend, TRUE);
	e_book_backend_notify_writable (backend, TRUE);

	return GNOME_Evolution_Addressbook_Success;
#endif /* FIXME */
}

static void
e_book_backend_kolab_remove (EBookBackendSync *backend,
                             EDataBook *book,
                             GCancellable *cancellable,
                             GError **err)
{
	EBookBackendKolab *self = E_BOOK_BACKEND_KOLAB (backend);
	EBookBackendKolabPrivate *priv = E_BOOK_BACKEND_KOLAB_PRIVATE (self);

	g_assert (E_IS_DATA_BOOK (book));
	(void)cancellable; /* FIXME */ /* cancellable may be NULL */
	g_return_if_fail (err == NULL || *err == NULL);

	(void)self;
	(void)priv;
	g_error ("%s: FIXME implement me", __func__);

#if 0 /* FIXME old */
	g_debug ("%s()[%u] called.", __func__, __LINE__);
	if (e_book_backend_is_loaded (E_BOOK_BACKEND (backend)) != FALSE) {
		g_object_unref (priv->book_koma);
		priv->book_koma = NULL;
	}

	/* Removing backend cache */
	if (priv->book_cache != NULL) {
		(void) e_file_cache_remove (E_FILE_CACHE (priv->book_cache));
		g_object_unref (priv->book_cache);
		priv->book_cache = NULL;
	}

	e_book_backend_set_is_writable (E_BOOK_BACKEND (backend), FALSE);
	e_book_backend_notify_writable (E_BOOK_BACKEND (backend), FALSE);
	e_book_backend_notify_connection_status (E_BOOK_BACKEND (backend), FALSE);
	e_book_backend_set_is_loaded (E_BOOK_BACKEND (backend), FALSE);

	return GNOME_Evolution_Addressbook_Success;
#endif /* FIXME*/

}

static void
e_book_backend_kolab_refresh (EBookBackendSync *backend,
                              EDataBook *book,
                              GCancellable *cancellable,
                              GError **err)
{
	EBookBackendKolab *self = E_BOOK_BACKEND_KOLAB (backend);
	EBookBackendKolabPrivate *priv = E_BOOK_BACKEND_KOLAB_PRIVATE (self);

	g_assert (E_IS_DATA_BOOK (book));
	(void)cancellable; /* FIXME */ /* cancellable may be NULL */
	g_return_if_fail (err == NULL || *err == NULL);

	(void)self;
	(void)priv;
	g_error ("%s: FIXME implement me", __func__);
}

static gboolean
e_book_backend_kolab_get_backend_property (EBookBackendSync *backend,
                                           EDataBook *book,
                                           GCancellable *cancellable,
                                           const gchar *prop_name,
                                           gchar **prop_value,
                                           GError **err)
{
	EBookBackendKolab *self = E_BOOK_BACKEND_KOLAB (backend);
	EBookBackendKolabPrivate *priv = E_BOOK_BACKEND_KOLAB_PRIVATE (self);

	g_assert (E_IS_DATA_BOOK (book));
	(void)cancellable; /* FIXME */ /* cancellable may be NULL */
	(void)prop_name; /* FIXME */
	(void)prop_value; /* FIXME */
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	(void)self;
	(void)priv;
	g_error ("%s: FIXME implement me", __func__);
}

static gboolean
e_book_backend_kolab_set_backend_property (EBookBackendSync *backend,
                                           EDataBook *book,
                                           GCancellable *cancellable,
                                           const gchar *prop_name,
                                           const gchar *prop_value,
                                           GError **err)
{
	EBookBackendKolab *self = E_BOOK_BACKEND_KOLAB (backend);
	EBookBackendKolabPrivate *priv = E_BOOK_BACKEND_KOLAB_PRIVATE (self);

	g_assert (E_IS_DATA_BOOK (book));
	(void)cancellable; /* FIXME */ /* cancellable may be NULL */
	(void)prop_name; /* FIXME */
	(void)prop_value; /* FIXME */
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	(void)self;
	(void)priv;
	g_error ("%s: FIXME implement me", __func__);
}

static void
e_book_backend_kolab_create_contacts (EBookBackendSync *backend,
                                      EDataBook *book,
                                      GCancellable *cancellable,
                                      const GSList *vcards,
                                      GSList **added_contacts,
                                      GError **err)
{
	EBookBackendKolab *self = E_BOOK_BACKEND_KOLAB (backend);
	EBookBackendKolabPrivate *priv = E_BOOK_BACKEND_KOLAB_PRIVATE (self);

	g_assert (E_IS_DATA_BOOK (book));
	(void)cancellable; /* FIXME */ /* cancellable may be NULL */
	(void)vcards; /* FIXME */
	(void)added_contacts; /* FIXME */
	g_return_if_fail (err == NULL || *err == NULL);

	(void)self;
	(void)priv;
	g_error ("%s: FIXME implement me", __func__);

#if 0 /* FIXME old */
	g_debug ("%s()[%u] called.", __func__, __LINE__);

	*contact = e_contact_new_from_vcard (vcard);
	if (*contact == NULL) {
		g_warning ("%s()[%u] error creating contact from vcard:\n%s",
		           __func__, __LINE__, vcard);
		return GNOME_Evolution_Addressbook_OtherError;
	}

	kolab_util_contact_cache_assure_uid_on_econtact (priv->book_cache,
	                                                 priv->book_koma,
	                                                 priv->book_uri,
	                                                 *contact,
	                                                 FALSE,
	                                                 &error);
	if (error != NULL) {
		status = kolab_util_contact_map_error (error);
		g_warning ("%s()[%u]: %s.",
		           __func__, __LINE__, error->message);
		g_error_free (error);
		g_object_unref (*contact);
		*contact = NULL;
		return status;
	}

	kolab_util_contact_store (*contact,
	                          priv->book_koma,
	                          priv->book_uri,
	                          &error);
	if (error != NULL) {
		status = kolab_util_contact_map_error (error);
		g_warning ("%s()[%u]: %s",
		           __func__, __LINE__, error->message);
		g_error_free (error);
		g_object_unref (*contact);
		*contact = NULL;
		return status;
	}

	return GNOME_Evolution_Addressbook_Success;
#endif /* FIXME */
}

static void
e_book_backend_kolab_remove_contacts (EBookBackendSync *backend,
                                      EDataBook *book,
                                      GCancellable *cancellable,
                                      const GSList *id_list,
                                      GSList **removed_ids,
                                      GError **err)
{
	EBookBackendKolab *self = E_BOOK_BACKEND_KOLAB (backend);
	EBookBackendKolabPrivate *priv = E_BOOK_BACKEND_KOLAB_PRIVATE (self);

	g_assert (E_IS_DATA_BOOK (book));
	(void)cancellable; /* FIXME */ /* cancellable may be NULL */
	(void)id_list; /* FIXME */
	(void)removed_ids; /* FIXME */
	g_return_if_fail (err == NULL || *err == NULL);

	(void)self;
	(void)priv;
	g_error ("%s: FIXME implement me", __func__);

#if 0 /* FIXME old */
	g_debug ("%s()[%u] called.", __func__, __LINE__);
	sourcename = kolab_util_backend_get_relative_path_from_uri (priv->book_uri);
	for (it = g_list_first (id_list); it != NULL; it = g_list_next (it)) {
		gchar *uid = it->data;
		ok = kolab_mail_access_delete_by_uid (priv->book_koma, uid, sourcename, &error);
		if (error != NULL) {
			g_warning ("%s()[%u]: %s", __func__, __LINE__, error->message);
			g_error_free (error);
			error = NULL;
		} else {
			*removed_ids = g_list_append (*removed_ids, g_strdup(uid));
		}
	}
	g_free (sourcename);

	return GNOME_Evolution_Addressbook_Success;
#endif /* FIXME */
}

static void
e_book_backend_kolab_modify_contacts (EBookBackendSync *backend,
                                      EDataBook *book,
                                      GCancellable *cancellable,
                                      const GSList *vcards,
                                      GSList **modified_contacts,
                                      GError **err)
{
	EBookBackendKolab *self = E_BOOK_BACKEND_KOLAB (backend);
	EBookBackendKolabPrivate *priv = E_BOOK_BACKEND_KOLAB_PRIVATE (self);

	g_assert (E_IS_DATA_BOOK (book));
	(void)cancellable; /* FIXME */ /* cancellable may be NULL */
	(void)vcards; /* FIXME */
	(void)modified_contacts; /* FIXME */
	g_return_if_fail (err == NULL || *err == NULL);

	(void)self;
	(void)priv;
	g_error ("%s: FIXME implement me", __func__);

#if 0 /* FIXME old */
	g_debug ("%s()[%u] called.", __func__, __LINE__);

	*contact = e_contact_new_from_vcard (vcard);
	if (*contact == NULL) {
		g_warning ("%s()[%u] error creating contact from vcard:\n%s", __func__, __LINE__, vcard);
		return GNOME_Evolution_Addressbook_OtherError;
	}

	kolab_util_contact_store (*contact, priv->book_koma, priv->book_uri, &error);
	if (error != NULL) {
		status = kolab_util_contact_map_error (error);
		g_warning ("%s()[%u]: %s", __func__, __LINE__, error->message);
		g_error_free (error);
		g_object_unref (*contact);
		*contact = NULL;
		return status;
	}

	return GNOME_Evolution_Addressbook_Success;
#endif /* FIXME */
}

static void
e_book_backend_kolab_get_contact (EBookBackendSync *backend,
                                  EDataBook *book,
                                  GCancellable *cancellable,
                                  const gchar *id,
                                  gchar **vcard,
                                  GError **err)
{
	EBookBackendKolab *self = E_BOOK_BACKEND_KOLAB (backend);
	EBookBackendKolabPrivate *priv = E_BOOK_BACKEND_KOLAB_PRIVATE (self);

	g_assert (E_IS_DATA_BOOK (book));
	(void)cancellable; /* FIXME */ /* cancellable may be NULL */
	(void)id; /* FIXME */
	(void)vcard; /* FIXME */
	g_return_if_fail (err == NULL || *err == NULL);

	(void)self;
	(void)priv;
	g_error ("%s: FIXME implement me", __func__);

#if 0 /* FIXME old */
	g_debug ("%s()[%u] called.", __func__, __LINE__);

	econtact = kolab_util_contact_cache_get_object (priv->book_cache,
	                                                priv->book_koma,
	                                                priv->book_uri,
	                                                id, FALSE,
	                                                &error);
	if (error != NULL) {
		status = kolab_util_contact_map_error (error);
		g_warning ("%s()[%u]: %s",
		           __func__, __LINE__, error->message);
		g_error_free (error);
		*vcard = NULL;
		return status;
	}
	*vcard = e_vcard_to_string (E_VCARD (econtact),
	                            EVC_FORMAT_VCARD_30);

	g_object_unref (econtact);

	return GNOME_Evolution_Addressbook_Success;
#endif /* FIXME */
}

static void
e_book_backend_kolab_get_contact_list (EBookBackendSync *backend,
                                       EDataBook *book,
                                       GCancellable *cancellable,
                                       const gchar *query,
                                       GSList **contacts,
                                       GError **err)
{
	EBookBackendKolab *self = E_BOOK_BACKEND_KOLAB (backend);
	EBookBackendKolabPrivate *priv = E_BOOK_BACKEND_KOLAB_PRIVATE (self);

	g_assert (E_IS_DATA_BOOK (book));
	(void)cancellable; /* FIXME */ /* cancellable may be NULL */
	(void)query; /* FIXME */
	(void)contacts; /* FIXME */
	g_return_if_fail (err == NULL || *err == NULL);

	(void)self;
	(void)priv;
	g_error ("%s: FIXME implement me", __func__);

#if 0 /* FIXME old */
	g_debug ("%s()[%u] called.", __func__, __LINE__);
	g_debug (" + query: %s", query);

	/* Update the cache to contain all data requested. */
	kolab_util_contact_cache_update_on_query (priv->book_cache,
	                                          priv->book_koma,
	                                          query,
	                                          priv->book_uri);
	/* Fetch information from BackendCache */;
	econtact_list = kolab_util_contact_cache_get_contacts (priv->book_cache,
	                                                       priv->book_koma,
	                                                       query,
	                                                       priv->book_uri,
	                                                       &error);
	if (error != NULL) {
		status = kolab_util_contact_map_error (error);
		g_warning ("%s()[%u]: %s",
		           __func__, __LINE__, error->message);
		g_error_free (error);
		*contacts = NULL;
		return status;
	}
	for (it = g_list_first (econtact_list); it != NULL; it = g_list_next (it)) {
		EContact *econtact = it->data;
		gchar *vcard = e_vcard_to_string (E_VCARD (econtact),
		                                  EVC_FORMAT_VCARD_30);
		*contacts = g_list_append (*contacts, vcard);
		g_object_unref (econtact);
	}

	g_list_free (econtact_list);

	return GNOME_Evolution_Addressbook_Success;
#endif /* FIXME */
}

static void
e_book_backend_kolab_get_contact_list_uids (EBookBackendSync *backend,
                                            EDataBook *book,
                                            GCancellable *cancellable,
                                            const gchar *query,
                                            GSList **contacts_uids,
                                            GError **err)
{
	EBookBackendKolab *self = E_BOOK_BACKEND_KOLAB (backend);
	EBookBackendKolabPrivate *priv = E_BOOK_BACKEND_KOLAB_PRIVATE (self);

	g_assert (E_IS_DATA_BOOK (book));
	(void)cancellable; /* FIXME */ /* cancellable may be NULL */
	(void)query; /* FIXME */
	(void)contacts_uids; /* FIXME */
	g_return_if_fail (err == NULL || *err == NULL);

	(void)self;
	(void)priv;
	g_error ("%s: FIXME implement me", __func__);
}

static void
e_book_backend_kolab_authenticate_user (EBookBackendSync *backend,
                                        GCancellable *cancellable,
                                        ECredentials *credentials,
                                        GError **err)
{
	EBookBackendKolab *self = E_BOOK_BACKEND_KOLAB (backend);
	EBookBackendKolabPrivate *priv = E_BOOK_BACKEND_KOLAB_PRIVATE (self);

	(void)cancellable; /* FIXME */ /* cancellable may be NULL */
	(void)credentials; /* FIXME */
	g_return_if_fail (err == NULL || *err == NULL);

	(void)self;
	(void)priv;
	g_error ("%s: FIXME implement me", __func__);

#if 0 /* FIXME old */
	g_debug ("%s()[%u] called.",
	         __func__, __LINE__);
	g_debug (" + user: %s\n + auth_method: %s",
	         user, auth_method);
	g_debug ("%s()[%u]    username = %s",
	         __func__, __LINE__, user);
	g_debug ("%s()[%u] auth_method = %s",
	         __func__, __LINE__, auth_method);
	password_placeholder = g_strdup (passwd);
	password_placeholder = g_strcanon (password_placeholder, "", '*');
	g_debug ("%s()[%u]    password = %s",
	         __func__, __LINE__, password_placeholder);
	g_free (password_placeholder);

	ksettings = kolab_mail_access_get_settings_handler (priv->book_koma);
	tmp_user = kolab_settings_handler_get_char_field (ksettings,
	                                                  KOLAB_SETTINGS_HANDLER_CHAR_FIELD_KOLAB_USER_NAME,
	                                                  &error);
	if (error != NULL) {
		g_warning ("%s()[%u]: %s",
		           __func__, __LINE__, error->message);
		g_error_free (error);
		error = NULL;
	}
	if (g_strcmp0 (tmp_user, user) != 0)
		g_warning (" + username from argument and username in "
		           "KolabSettingsHandler do not match: %s vs. %s",
		           tmp_user, user);
	sourcename = kolab_util_backend_get_relative_path_from_uri (priv->book_uri);

	kolab_util_backend_prepare_settings (ksettings,
	                                     NULL,
	                                     NULL,
	                                     NULL,
	                                     passwd,
	                                     NULL,
	                                     NULL);

	priv->auth_received = TRUE;
	ok = kolab_util_contact_deploy_mode_by_backend (priv->book_koma,
	                                                priv->book_mode);

	return GNOME_Evolution_Addressbook_Success;
#endif /* FIXME */
}

static void
e_book_backend_kolab_start_book_view (EBookBackend *backend,
                                      EDataBookView *book_view)
{
	EBookBackendKolab *self = E_BOOK_BACKEND_KOLAB (backend);
	EBookBackendKolabPrivate *priv = E_BOOK_BACKEND_KOLAB_PRIVATE (self);

	g_assert (E_IS_DATA_BOOK_VIEW (book_view));

	(void)self;
	(void)priv;
	g_error ("%s: FIXME implement me", __func__);

#if 0 /* FIXME old */
	g_debug ("%s()[%u] called.", __func__, __LINE__);
	/* This should not be necessary, if everything works as expected,
	 * ..._start_book_view () will not be issued, if the KoMA was not
	 * configured correctly.
	 */
	koma_mode = kolab_mail_access_get_opmode (priv->book_koma, &error);
	if (error != NULL) {
		g_warning ("%s()[%u]: %s", __func__, __LINE__, error->message);
		g_error_free (error);
		error = NULL;
	}
	if (koma_mode < KOLAB_MAIL_ACCESS_OPMODE_OFFLINE) {
		g_error ("%s()[%u]: %s called while the KolabMailAccess was "
		         "not ready! Should not happen, forcing exit.",
		         __func__, __LINE__, __func__);
	}
	if ((koma_mode < KOLAB_MAIL_ACCESS_OPMODE_ONLINE) &&
	    (priv->book_mode == GNOME_Evolution_Addressbook_MODE_REMOTE)) {
		g_warning ("%s()[%u]: %s called before authentication took place. "
		           "Requesting auth now.",
		           __func__, __LINE__, __func__);
		e_book_backend_notify_auth_required (backend);
		return;
	}

	query = e_data_book_view_get_card_query (book_view);
	/* First update the BackendCache to the current situation. */
	kolab_util_contact_cache_update_on_query (priv->book_cache,
	                                          priv->book_koma,
	                                          query,
	                                          priv->book_uri);
	/* econtact_list = e_book_backend_cache_get_contacts (priv->book_cache, query); */
	econtact_list = kolab_util_contact_cache_get_contacts (priv->book_cache,
	                                                       priv->book_koma,
	                                                       query,
	                                                       priv->book_uri,
	                                                       &error);
	if (error != NULL) {
		g_warning ("%s()[%u]: %s",
		           __func__, __LINE__, error->message);
		g_error_free (error);
		error = NULL;
	}
	/* Then deliver requests from there. */
	for (it = g_list_first (econtact_list); it != NULL; it = g_list_next (it)) {
		EContact *econtact = it->data;
		e_data_book_view_notify_update (book_view, econtact);
		g_object_unref (econtact);
	}

	g_list_free (econtact_list);
	g_free (sourcename);

	e_data_book_view_notify_complete (book_view,
	                                  GNOME_Evolution_Addressbook_Success);
#endif /* FIXME */
}

static void
e_book_backend_kolab_stop_book_view (EBookBackend *backend,
                                     EDataBookView *book_view)
{
	EBookBackendKolab *self = E_BOOK_BACKEND_KOLAB (backend);
	EBookBackendKolabPrivate *priv = E_BOOK_BACKEND_KOLAB_PRIVATE (self);

	g_assert (E_IS_DATA_BOOK_VIEW (book_view));

	(void)self;
	(void)priv;
	g_error ("%s: FIXME implement me", __func__);

}

static void
e_book_backend_kolab_notify_update (EBookBackend *backend,
                                    const EContact *contact)
{
	EBookBackendKolab *self = E_BOOK_BACKEND_KOLAB (backend);
	EBookBackendKolabPrivate *priv = E_BOOK_BACKEND_KOLAB_PRIVATE (self);

	g_assert (E_IS_CONTACT (contact));

	(void)self;
	(void)priv;
	g_error ("%s: FIXME implement me", __func__);
}

/*----------------------------------------------------------------------------*/
/* object/class init */

static void
e_book_backend_kolab_init (EBookBackendKolab *backend)
{
	EBookBackendKolabPrivate *priv = E_BOOK_BACKEND_KOLAB_PRIVATE (backend);

	g_debug ("%s()[%u] called.", __func__, __LINE__);

	/* priv->book_mode = E_DATA_BOOK_MODE_LOCAL; */ /* Start in local mode for now. */
	priv->book_cache = NULL;
	priv->book_uri = NULL;
	priv->koma_table = NULL;
	priv->book_koma = NULL;
	priv->auth_received = FALSE;
} /* e_book_backend_kolab_init () */

static void
e_book_backend_kolab_dispose (GObject *object)
{
	EBookBackend *self = E_BOOK_BACKEND (object);
	EBookBackendKolabPrivate *priv = E_BOOK_BACKEND_KOLAB_PRIVATE (self);

	g_debug ("%s()[%u] called.", __func__, __LINE__);
#if 0 /* FIXME */
	if (e_book_backend_is_loaded (self)) {
		e_book_backend_set_is_writable (self, FALSE);
		e_book_backend_notify_writable (self, FALSE);
		e_book_backend_notify_connection_status (self, FALSE);
		e_book_backend_set_is_loaded (self, FALSE);
		g_object_unref (priv->book_koma);
		priv->book_koma = NULL;
	}
#endif
	if (priv->book_cache != NULL) {
		(void) e_file_cache_remove (E_FILE_CACHE (priv->book_cache));
		g_object_unref (priv->book_cache);
		priv->book_cache = NULL;
	}
}

static void
e_book_backend_kolab_finalize (GObject *object)
{
	EBookBackendKolabPrivate *priv = E_BOOK_BACKEND_KOLAB_PRIVATE (object);

	g_debug ("%s()[%u] called.", __func__, __LINE__);

	if (priv->book_uri != NULL)
		g_free (priv->book_uri);

	G_OBJECT_CLASS (e_book_backend_kolab_parent_class)->finalize (object);
} /* e_book_backend_kolab_finalize () */

static void
e_book_backend_kolab_class_init (EBookBackendKolabClass *klass)
{
	GObjectClass* object_class = G_OBJECT_CLASS (klass);
	EBookBackendClass *backend_class = E_BOOK_BACKEND_CLASS (klass);
	EBookBackendSyncClass *sync_class = E_BOOK_BACKEND_SYNC_CLASS (klass);

	g_debug ("%s()[%u] called.", __func__, __LINE__);

	g_type_class_add_private (klass, sizeof (EBookBackendKolabPrivate));

	/* Sync backend class functions */
	sync_class->open_sync = e_book_backend_kolab_open;
	sync_class->remove_sync = e_book_backend_kolab_remove;
	sync_class->refresh_sync = e_book_backend_kolab_refresh;
	sync_class->get_backend_property_sync = e_book_backend_kolab_get_backend_property;
	sync_class->set_backend_property_sync = e_book_backend_kolab_set_backend_property;
	sync_class->create_contacts_sync = e_book_backend_kolab_create_contacts;
	sync_class->remove_contacts_sync = e_book_backend_kolab_remove_contacts;
	sync_class->modify_contacts_sync = e_book_backend_kolab_modify_contacts;
	sync_class->get_contact_sync = e_book_backend_kolab_get_contact;
	sync_class->get_contact_list_sync = e_book_backend_kolab_get_contact_list;
	sync_class->get_contact_list_uids_sync = e_book_backend_kolab_get_contact_list_uids;
	sync_class->authenticate_user_sync = e_book_backend_kolab_authenticate_user;

	/* Backend parent class methods methods not covered in the sync backend part */
	backend_class->start_book_view = e_book_backend_kolab_start_book_view;
	backend_class->stop_book_view = e_book_backend_kolab_stop_book_view;
	backend_class->notify_update = e_book_backend_kolab_notify_update;

	object_class->dispose = e_book_backend_kolab_dispose;
	object_class->finalize = e_book_backend_kolab_finalize;
} /* e_book_backend_kolab_class_init () */


/*----------------------------------------------------------------------------*/
/* public API */


/*----------------------------------------------------------------------------*/

#if 0 /* FIXME old */

static EBookBackendSyncStatus
e_book_backend_kolab_get_required_fields (EBookBackendSync *backend,
                                          EDataBook *book,
                                          guint32 opid,
                                          GList **fields)
{
	(void) backend;
	(void) book;
	(void) opid;
	g_debug ("%s()[%u] called.", __func__, __LINE__);
	*fields = NULL;
	return GNOME_Evolution_Addressbook_Success;
} /* e_book_backend_kolab_get_contact_list () */

static EBookBackendSyncStatus
e_book_backend_kolab_get_supported_fields (EBookBackendSync *backend,
                                           EDataBook *book,
                                           guint32 opid,
                                           GList **fields)
{
	int i;
	(void) backend;
	(void) book;
	(void) opid;
	g_debug ("%s()[%u] called.", __func__, __LINE__);
	for (i=E_CONTACT_FIELD_FIRST; i<E_CONTACT_FIELD_LAST; i++)
		*fields = g_list_append (*fields, g_strdup (e_contact_field_name (i)));

	if (*fields == NULL)
		return GNOME_Evolution_Addressbook_OtherError;

	return GNOME_Evolution_Addressbook_Success;
} /* e_book_backend_kolab_get_supported_fields () */

static EBookBackendSyncStatus
e_book_backend_kolab_get_supported_auth_methods (EBookBackendSync *backend,
                                                 EDataBook *book,
                                                 guint32 opid,
                                                 GList **methods)
{
	gchar *auth_method = g_strdup ("plain/password");
	(void) backend;
	(void) book;
	(void) opid;
	g_debug ("%s()[%u] called.", __func__, __LINE__);
	*methods = g_list_append (*methods, auth_method);

	return GNOME_Evolution_Addressbook_Success;
} /* e_book_backend_kolab_get_supported_auth_methods () */

static gchar *
e_book_backend_kolab_get_static_capabilities (EBookBackend *backend)
{
	(void) backend;
	g_debug ("%s()[%u] called.", __func__, __LINE__);
	return g_strdup ("kolab,do-initial-query,contact-lists");
} /* e_book_backend_kolab_get_static_capabilities () */

/**
 * e_book_backend_kolab_set_mode:
 * @backend: A Kolab addressbook backend.
 * @mode: Mode to change to.
 *
 * Sets the mode of the addressbook backend to the given mode.
 */
static void
e_book_backend_kolab_set_mode (EBookBackend *backend,
                               GNOME_Evolution_Addressbook_BookMode mode)
{
	EBookBackendKolab *kolab = E_BOOK_BACKEND_KOLAB (backend);
	EBookBackendKolabPrivate *priv = E_BOOK_BACKEND_KOLAB_PRIVATE (kolab);
	KolabMailAccessOpmodeID koma_mode;
	KolabMailAccessOpmodeID tmp_mode;
	g_debug ("%s()[%u] called with mode: %u.", __func__, __LINE__, mode);
	switch (mode) {
	case GNOME_Evolution_Addressbook_MODE_ANY:
	case GNOME_Evolution_Addressbook_MODE_REMOTE:
		g_debug (" + got argument E_DATA_BOOK_MODE_ANY/E_DATA_BOOK_MODE_REMOTE, "
		         "switching backend_mode to E_DATA_BOOK_MODE_REMOTE.");
		priv->book_mode = GNOME_Evolution_Addressbook_MODE_REMOTE;
		koma_mode = KOLAB_MAIL_ACCESS_OPMODE_ONLINE;
		break;
	case GNOME_Evolution_Addressbook_MODE_LOCAL:
		g_debug (" + got argument E_DATA_BOOK_MODE_LOCAL, switching backend_mode.");
		priv->book_mode = GNOME_Evolution_Addressbook_MODE_LOCAL;
		koma_mode = KOLAB_MAIL_ACCESS_OPMODE_OFFLINE;
		break;
	default:
		g_error ("%s() ran into an unknown arguments for mode, value %u. "
		         "This should not happen.", __func__, mode);
	}

	if (priv->auth_received == TRUE)
		tmp_mode = kolab_util_contact_deploy_mode_by_koma (priv->book_koma,
		                                                   koma_mode);
	else
		tmp_mode = KOLAB_MAIL_ACCESS_OPMODE_OFFLINE;

	e_book_backend_notify_connection_status (backend,
	                                         tmp_mode = KOLAB_MAIL_ACCESS_OPMODE_ONLINE ? TRUE : FALSE);
} /* e_book_backend_kolab_set_mode () */

/**
 * e_book_backend_kolab_set_koma_table:
 * @kolab: An EBookBackendKolab object.
 * @koma_objects: A GHashTable to contain the required KoMA instances.
 *
 * This method has to be called before any other method which accesses
 * Kolab infrastructure. In this case it should be called from
 * e-book-backend-kolab-factory during creation of a new EBookBackendKolab
 * instance.
 */
void
e_book_backend_kolab_set_koma_table (EBookBackendKolab *kolab,
                                     GHashTable *koma_objects)
{
	EBookBackendKolabPrivate *priv = E_BOOK_BACKEND_KOLAB_PRIVATE (kolab);

	g_hash_table_ref (koma_objects);
	priv->koma_table = koma_objects;
} /* e_book_backend_kolab_set_koma_table () */

EBookBackend *
e_book_backend_kolab_new ()
{
	EBookBackendKolab *kolab = NULL;
	g_debug ("%s()[%u] called.", __func__, __LINE__);
	kolab = g_object_new (E_TYPE_BOOK_BACKEND_KOLAB, NULL);

	/* At this point there may be some more initialization before returning
	 * kolab.
	 */

	return E_BOOK_BACKEND (kolab);
} /* e_book_backend_kolab_new () */

#endif /* FIXME old */
