/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/***************************************************************************
 *            kolab-util-contact.c
 *
 *  2011
 *  Copyright  2011  Silvan Marco Fin
 *  <silvan@kernelconcepts.de>
 ****************************************************************************/

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

/*----------------------------------------------------------------------------*/

#include <libekolab/kolab-mail-access.h>

#include <libebook/e-contact.h>
#include <libecal/e-cal.h>

#include "kolab-util-contact.h"

/*----------------------------------------------------------------------------*/

gboolean
kolab_util_contact_has_id (EContact *contact) 
{
	gchar *uid = NULL;

	g_assert (E_IS_CONTACT (contact));

	uid = (gchar *) e_contact_get (contact, E_CONTACT_UID);

	if (uid == NULL)
		return FALSE;

	g_free (uid);

	return TRUE;
}

void
kolab_util_contact_gen_uid_if_none (EContact **contact)
{
	gchar *uid = NULL;

	g_assert (E_IS_CONTACT (*contact));

	if (! kolab_util_contact_has_id (*contact)) {
		/* no uid yet */
		uid = e_cal_component_gen_uid ();
		e_contact_set (*contact, E_CONTACT_UID, (gconstpointer) uid);
	}

	g_free (uid);

}

KolabMailAccessOpmodeID
kolab_util_contact_deploy_mode_by_koma (KolabMailAccess *koma,
                                        KolabMailAccessOpmodeID koma_mode,
                                        GCancellable *cancellable,
                                        GError **error)
{
	KolabMailAccessOpmodeID tmp_mode;
	GError *tmp_error = NULL;
	gboolean ok;

	g_debug ("%s()[%u] called.", __func__, __LINE__);

	if (koma == NULL) {
		g_debug ("%s()[%u] error: KolabMailAccess object not existent.",
		         __func__, __LINE__);
		/* FIXME set GError */
		return KOLAB_MAIL_ACCESS_OPMODE_INVAL;
	}
	tmp_mode = kolab_mail_access_get_opmode (koma,
	                                         &tmp_error);
	if (error != NULL) {
		g_propagate_error (error, tmp_error);
		return KOLAB_MAIL_ACCESS_OPMODE_INVAL;
	}
	if (tmp_mode < KOLAB_MAIL_ACCESS_OPMODE_OFFLINE) {
		g_debug ("%s()[%u] KolabMailAccess object not ready, deferring.",
		         __func__, __LINE__);
		return tmp_mode;
	}
	ok = kolab_mail_access_set_opmode (koma,
	                                   koma_mode,
	                                   cancellable,
	                                   &tmp_error);
	if (! ok) {
		g_propagate_error (error, tmp_error);
		return KOLAB_MAIL_ACCESS_OPMODE_INVAL;
	}
	return koma_mode;
} /* kolab_util_contacts_deploy_by_koma_mode () */

#if 0
gboolean
kolab_util_contact_deploy_mode_by_backend (KolabMailAccess *koma, 
                                           GNOME_Evolution_Addressbook_BookMode backend_mode)
{
	KolabMailAccessOpmodeID koma_mode;
	KolabMailAccessOpmodeID tmp_mode;

	switch (backend_mode) {
	case GNOME_Evolution_Addressbook_MODE_LOCAL:
		koma_mode = KOLAB_MAIL_ACCESS_OPMODE_OFFLINE;
		break;
	default:
		koma_mode = KOLAB_MAIL_ACCESS_OPMODE_ONLINE;
	}
	tmp_mode = kolab_util_contact_deploy_mode_by_koma (koma, koma_mode);

	return tmp_mode == koma_mode;
}
#endif

gboolean
kolab_util_contact_store (EContact *econtact,
                          KolabMailAccess *koma,
                          const gchar *uri,
                          GCancellable *cancellable,
                          GError **error)
{
	const gchar *sourcename = NULL;
	KolabMailHandle *kmh = NULL;
	GError *tmp_error = NULL;
	gboolean ok = FALSE;

	g_assert (E_IS_CONTACT (econtact));
	g_assert (KOLAB_IS_MAIL_ACCESS (koma));
	g_assert (uri != NULL);
	/* cancellable may be NULL */
	g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

	sourcename = kolab_util_backend_get_relative_path_from_uri (uri);

	kolab_util_backend_modtime_set_on_econtact (econtact);
	kmh = kolab_mail_handle_new_from_econtact (econtact);

	ok =  kolab_mail_access_store_handle (koma,
	                                      kmh,
	                                      sourcename,
	                                      cancellable,
	                                      &tmp_error);
	if (! ok) {
		g_propagate_error (error, tmp_error);
		return FALSE;
	}
	
	return TRUE;
} /* kolab_util_contact_store () */

#if 0
EBookBackendSyncStatus
kolab_util_contact_map_error (GError *error)
{
	EBookBackendSyncStatus status = GNOME_Evolution_Addressbook_Success;

	if (error == NULL)
		return GNOME_Evolution_Addressbook_Success;
	
	switch (error->code) {
	case KOLAB_BACKEND_ERROR_SYNC_NOTSTORED:
		status = GNOME_Evolution_Addressbook_Success;
		break;
	case KOLAB_BACKEND_ERROR_NOTFOUND:
		status = GNOME_Evolution_Addressbook_ContactNotFound;
		break;
	case KOLAB_BACKEND_ERROR_CONTEXT_MISUSE:
	case KOLAB_BACKEND_ERROR_INFODB_NOFOLDER:
		status = GNOME_Evolution_Addressbook_NoSuchBook;
		break;
	default:
		status = GNOME_Evolution_Addressbook_OtherError;
	}

	g_debug ("%s()[%u] EBookBackendSyncStatus: %i", __func__, __LINE__, status);
	return status;
}
#endif

/*----------------------------------------------------------------------------*/
