/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/***************************************************************************
 *            camel-kolab-imapx-store.c
 *
 *  Fri Sep  3 12:48:31 2010
 *  Copyright  2010  Christian Hilberg
 *  <hilberg@kernelconcepts.de>
 ****************************************************************************/

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with main.c; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

/*----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib/gi18n-lib.h>

#include <libekolabutil/camel-system-headers.h>

#include <camel/providers/imapx/camel-imapx-store-summary.h>
#include <camel/providers/imapx/camel-imapx-extd-server.h>
#include <camel/providers/imapx/camel-imapx-metadata.h>
#include <camel/providers/imapx/camel-imapx-utils.h>

#include <libekolabutil/kolab-util-error.h>

#include "camel-kolab-imapx-conn-manager.h"
#include "camel-kolab-imapx-settings.h"
#include "camel-kolab-imapx-store.h"

/*----------------------------------------------------------------------------*/

static GInitableIface *parent_initable_iface = NULL;
static CamelNetworkServiceInterface *parent_service_iface = NULL;
static CamelSubscribableInterface *parent_subscribable_iface = NULL;

static CamelServiceClass *parent_service_class = NULL;
static CamelStoreClass *parent_store_class = NULL;

/*----------------------------------------------------------------------------*/

/* forward declarations */
static void kolab_imapx_store_initable_init (GInitableIface *interface);
static void kolab_imapx_store_network_service_init (CamelNetworkServiceInterface *interface);
static void kolab_imapx_store_subscribable_init (CamelSubscribableInterface *interface);

typedef struct _CamelKolabIMAPXStorePrivate CamelKolabIMAPXStorePrivate;
struct _CamelKolabIMAPXStorePrivate {
	CamelKolabIMAPXServer *server;

	/* Used for syncronizing get_folder_info.
	 * TODO check whether we can re-use any other lock
	 */
	GMutex *kolab_finfo_lock;

	CamelKolabImapxMetadata *kmd;
	KolabFolderTypeID folder_create_type;
	KolabFolderContextID folder_context;
	gboolean folder_types_do_care[KOLAB_FOLDER_LAST_TYPE];
	GList *folder_names_do_care;
};

#define CAMEL_KOLAB_IMAPX_STORE_PRIVATE(obj)  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), CAMEL_TYPE_KOLAB_IMAPX_STORE, CamelKolabIMAPXStorePrivate))

G_DEFINE_TYPE_WITH_CODE (CamelKolabIMAPXStore,
                         camel_kolab_imapx_store,
                         CAMEL_TYPE_IMAPX_EXTD_STORE,
                         G_IMPLEMENT_INTERFACE (G_TYPE_INITABLE,
                                                kolab_imapx_store_initable_init)
                         G_IMPLEMENT_INTERFACE (CAMEL_TYPE_NETWORK_SERVICE,
                                                kolab_imapx_store_network_service_init)
                         G_IMPLEMENT_INTERFACE (CAMEL_TYPE_SUBSCRIBABLE,
                                                kolab_imapx_store_subscribable_init))

/*----------------------------------------------------------------------------*/
/* object/class init */

static void
camel_kolab_imapx_store_init (CamelKolabIMAPXStore *self)
{
	CamelIMAPXStore *istore = NULL;
	CamelKolabIMAPXConnManager *cm = NULL;
	CamelKolabIMAPXStorePrivate *priv = NULL;
	gint ii = 0;

	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (self));
	priv = CAMEL_KOLAB_IMAPX_STORE_PRIVATE (self);

	/* replace existing connection manager with
	 * our own Kolab version. in case we need
	 * to explicitly reference the Kolab
	 * version of the connection manager, we will need
	 * to override the respective parent class functions
	 */
	istore = CAMEL_IMAPX_STORE (self);
	if (istore->con_man != NULL) {
		camel_imapx_extd_conn_manager_close_connections (istore->con_man);
		g_object_unref (istore->con_man);
	}
	cm = camel_kolab_imapx_conn_manager_new (self);
	istore->con_man = CAMEL_IMAPX_CONN_MANAGER (cm);

	/* folder info lock */
	priv->kolab_finfo_lock = g_mutex_new ();

	/* metadata db and lookup table */
	priv->kmd = camel_kolab_imapx_metadata_new ();

	/* default folder type to create (for use in Evo) */
	priv->folder_create_type = KOLAB_FOLDER_TYPE_EMAIL;

	/* default folder context (for use in Evo) */
	priv->folder_context = KOLAB_FOLDER_CONTEXT_EMAIL;

	/* folder types to care for with this CamelKolabIMAPXStore
	 * instance.
	 * Default: Email and unknown (so no need to reconfigure
	 * this instance for use in Evo). Needs to be reconfigured
	 * when used in ECal/EBook backends
	 */
	for (ii = 0; ii < KOLAB_FOLDER_LAST_TYPE; ii++)
		priv->folder_types_do_care[ii] = FALSE;
	priv->folder_types_do_care[KOLAB_FOLDER_TYPE_UNKNOWN]         = TRUE;
	priv->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL]           = TRUE;
	priv->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL_INBOX]     = TRUE;
	priv->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL_DRAFTS]    = TRUE;
	priv->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL_SENTITEMS] = TRUE;
	priv->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL_JUNKEMAIL] = TRUE;

	priv->folder_names_do_care = NULL;
}

static void
camel_kolab_imapx_store_dispose (GObject *object)
{
	CamelKolabIMAPXStore *self = NULL;
	CamelIMAPXStore *istore = CAMEL_IMAPX_STORE (self);

	self = CAMEL_KOLAB_IMAPX_STORE (object);
	istore = CAMEL_IMAPX_STORE (self);

	/* disconnect service and unref the connection manager.
	 * see imapx_store_dispose() in camel-imapx-store.c
	 */
	if (istore->con_man != NULL) {
		camel_service_disconnect_sync (CAMEL_SERVICE (self),
		                               TRUE,
		                               NULL);
		g_object_unref (istore->con_man);
		istore->con_man = NULL;
		/* this part will now be skipped
		 * in the parent's dispose() function
		 */
	}

	/* Chain up to parent's dispose() method. */
	G_OBJECT_CLASS (camel_kolab_imapx_store_parent_class)->dispose (object);
}

static void
camel_kolab_imapx_store_finalize (GObject *object)
{
	CamelKolabIMAPXStore *self = NULL;
	CamelKolabIMAPXStorePrivate *priv = NULL;

	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (object));

	self = CAMEL_KOLAB_IMAPX_STORE (object);
	priv = CAMEL_KOLAB_IMAPX_STORE_PRIVATE (self);

	while (! g_mutex_trylock (priv->kolab_finfo_lock));
	g_mutex_unlock (priv->kolab_finfo_lock);
	g_mutex_free (priv->kolab_finfo_lock);

	camel_kolab_imapx_metadata_free (priv->kmd);

	if (priv->folder_names_do_care != NULL) {
		GList *list_ptr = priv->folder_names_do_care;
		while (list_ptr != NULL) {
			g_free (list_ptr);
			list_ptr = g_list_next (list_ptr);
		}
		g_free (priv->folder_names_do_care);
	}

	/* Chain up to parent's dispose() method. */
	G_OBJECT_CLASS (camel_kolab_imapx_store_parent_class)->finalize (object);
}

/*----------------------------------------------------------------------------*/
/* internal statics */

static CamelFolder*
imapx_store_get_folder_offline (CamelStore *self,
                                const gchar *folder_name,
                                guint32 flags,
                                GError **err)
{
	/* This function is a dupe of extd_store_get_folder_offline() in
	 * CamelIMAPXExtdStore, which is a dupe of get_folder_offline()
	 * in CamelIMAPXStore.
	 * We need to dupe it in order to return a CamelKolabIMAPXFolder
	 * (disguised as a CamelFolder). Upstream fixes need to be applied
	 * here, too.
	 */

	CamelKolabIMAPXStore *myself = CAMEL_KOLAB_IMAPX_STORE (self);
	CamelIMAPXStore *imapx_store = CAMEL_IMAPX_STORE (self);
	CamelService *service = CAMEL_SERVICE (self);
	CamelFolder *new_folder = NULL;
	CamelStoreInfo *si = NULL;
	const gchar *user_cache_dir = NULL;

	g_assert (folder_name != NULL);
	g_return_val_if_fail (err == NULL || *err == NULL, NULL);

	user_cache_dir = camel_service_get_user_cache_dir (service);

	si = camel_store_summary_path (CAMEL_STORE_SUMMARY (imapx_store->summary), folder_name);
	if (si) {
		gchar *folder_dir, *storage_path;

		/* Note: Although the INBOX is defined to be case-insensitive in the IMAP RFC
		 * it is still up to the server how to acutally name it in a LIST response. Since
		 * we stored the name as the server provided it us in the summary we take that name
		 * to look up the folder.
		 * But for the on-disk cache we do always capitalize the Inbox no matter what the
		 * server provided.
		 */
		if (!g_ascii_strcasecmp (folder_name, "INBOX"))
			folder_name = "INBOX";

		storage_path = g_build_filename (user_cache_dir, "folders", NULL);
		folder_dir = imapx_path_to_physical (storage_path, folder_name);
		g_free (storage_path);
		/* FIXME */ /* (create CamelKolabIMAPXFolder) */
		new_folder = CAMEL_FOLDER (camel_kolab_imapx_folder_new (myself,
		                                                         folder_dir,
		                                                         folder_name,
		                                                         err));
		g_free (folder_dir);
		camel_store_summary_info_free (CAMEL_STORE_SUMMARY (imapx_store->summary), si);
	} else {
		g_set_error (err,
		             CAMEL_STORE_ERROR,
		             CAMEL_STORE_ERROR_NO_FOLDER,
		             _("No such folder %s"), folder_name);
	}

	return new_folder;
}

/*----------------------------------------------------------------------------*/
/* class functions */

static gchar*
kolab_imapx_store_get_name (CamelService *service,
                            gboolean brief)
{
	/* This is an adjusted dupe of the CamelIMAPXStore
	 * imapx_get_name() function. Upstream fixes need
	 * to be applied here, too
	 */

	CamelNetworkSettings *network_settings;
	CamelSettings *settings;
	gchar *host;
	gchar *user;
	gchar *name;

	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (service));

	settings = camel_service_get_settings (service);

	network_settings = CAMEL_NETWORK_SETTINGS (settings);
	host = camel_network_settings_dup_host (network_settings);
	user = camel_network_settings_dup_user (network_settings);

	if (brief)
		name = g_strdup_printf (
			_("Kolab server %s"), host);
	else
		name = g_strdup_printf (
			_("Kolab service for %s on %s"), user, host);

	g_free (host);
	g_free (user);

	return name;
}

static gboolean
kolab_imapx_store_connect_sync (CamelService *service,
                                GCancellable *cancellable,
                                GError **err)
{
	gboolean ok = FALSE;

	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (service));
	/* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	ok = parent_service_class->connect_sync (service,
	                                         cancellable,
	                                         err);
	return ok;
}

static gboolean
kolab_imapx_store_disconnect_sync (CamelService *service,
                                   gboolean clean,
                                   GCancellable *cancellable,
                                   GError **err)
{
	gboolean ok = FALSE;

	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (service));
	/* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	ok = parent_service_class->disconnect_sync (service,
	                                            clean,
	                                            cancellable,
	                                            err);
	return ok;
}

static CamelAuthenticationResult
kolab_imapx_store_authenticate_sync (CamelService *service,
                                     const gchar *mechanism,
                                     GCancellable *cancellable,
                                     GError **err)
{
	CamelAuthenticationResult result = CAMEL_AUTHENTICATION_ERROR;

	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (service));
	/* mechanism may be NULL */
	/* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, CAMEL_AUTHENTICATION_REJECTED);

	result = parent_service_class->authenticate_sync (service,
	                                                  mechanism,
	                                                  cancellable,
	                                                  err);
	return result;
}

static GList*
kolab_imapx_store_query_auth_types_sync (CamelService *service,
                                         GCancellable *cancellable,
                                         GError **err)
{
	GList *auth_types = NULL;

	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (service));
	/* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, NULL);

	auth_types = parent_service_class->query_auth_types_sync (service,
	                                                          cancellable,
	                                                          err);
	return auth_types;
}

static guint
kolab_imapx_store_name_hash (gconstpointer key)
{
	if (g_ascii_strcasecmp (key, "INBOX") == 0)
		return g_str_hash ("INBOX");
	else
		return g_str_hash (key);
}

static gint
kolab_imapx_store_name_equal (gconstpointer a,
                              gconstpointer b)
{
	gconstpointer aname = a, bname = b;

	if (g_ascii_strcasecmp(a, "INBOX") == 0)
		aname = "INBOX";
	if (g_ascii_strcasecmp(b, "INBOX") == 0)
		bname = "INBOX";
	return g_str_equal (aname, bname);
}

static gboolean
kolab_imapx_store_can_refresh_folder (CamelStore *self,
                                      CamelFolderInfo *finfo,
                                      GError **err)
{
	gboolean can = FALSE;

	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (self));
	g_assert (finfo != NULL);
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	can = parent_store_class->can_refresh_folder (self,
	                                              finfo,
	                                              err);
	return can;
}

static CamelFolder*
kolab_imapx_store_get_folder_sync (CamelStore *self,
                                   const gchar *foldername,
                                   CamelStoreGetFolderFlags flags,
                                   GCancellable *cancellable,
                                   GError **err)
{
	CamelFolder *folder = NULL;

	g_assert (CAMEL_IS_IMAPX_EXTD_STORE (self));
	g_assert (foldername != NULL);
	(void)cancellable; /* FIXME */
	g_return_val_if_fail (err == NULL || *err == NULL, NULL);

	folder = imapx_store_get_folder_offline (self,
	                                         foldername,
	                                         flags,
	                                         err);
	/* FIXME */
	g_warning ("%s: FIXME getting CamelIMAPXFolder, expected CamelKolabIMAPXFolder",
	           __func__);
#if 0
	if (folder != NULL)
		g_assert (CAMEL_IS_KOLAB_IMAPX_FOLDER (folder));
#endif
	return folder;
}

static CamelFolderInfo*
kolab_imapx_store_get_folder_info_sync (CamelStore *self,
                                        const gchar *top,
                                        CamelStoreGetFolderInfoFlags flags,
                                        GCancellable *cancellable,
                                        GError **err)
{
	CamelFolderInfo *finfo = NULL;

	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (self));
	/* top may be NULL */ /* FIXME correct? */
	/* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, NULL);

	finfo = parent_store_class->get_folder_info_sync (self,
	                                                  top,
	                                                  flags,
	                                                  cancellable,
	                                                  err);
	return finfo;
}

static CamelFolder*
kolab_imapx_store_get_junk_folder_sync (CamelStore *self,
                                        GCancellable *cancellable,
                                        GError **err)
{
	CamelFolder *folder = NULL;

	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (self));
	/* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, NULL);

	folder = parent_store_class->get_junk_folder_sync (self,
	                                                   cancellable,
	                                                   err);
	/* CamelIMAPXFolder gets the junk folder from
	 * its parent class, CamelFolder, and does not
	 * change it to a CamelIMAPXFolder. That means,
	 * if we really need a CamelKolabIMAPXFolder
	 * here, we need to clone the CamelFolder into
	 * a locally created instance
	 */
#if 0
	if (folder != NULL)
		g_assert (CAMEL_IS_KOLAB_IMAPX_FOLDER (folder));
#endif
	return folder;
}

static CamelFolder*
kolab_imapx_store_get_trash_folder_sync (CamelStore *self,
                                         GCancellable *cancellable,
                                         GError **err)
{
	CamelFolder *folder = NULL;

	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (self));
	/* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, NULL);

	folder = parent_store_class->get_trash_folder_sync (self,
	                                                    cancellable,
	                                                    err);
	/* see kolab_imapx_store_get_junk_folder_sync()
	 * why we do not have a CamelIMAPXExtdFolder
	 * here
	 */
#if 0
	if (folder != NULL)
		g_assert (CAMEL_IS_KOLAB_IMAPX_FOLDER (folder));
#endif
	return folder;
}

static CamelFolderInfo*
kolab_imapx_store_create_folder_sync (CamelStore *self,
                                      const gchar *parentname,
                                      const gchar *foldername,
                                      GCancellable *cancellable,
                                      GError **err)
{
	CamelFolderInfo *finfo = NULL;

	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (self));
	/* parentname may be NULL */ /* FIXME correct? */
	g_assert (foldername != NULL);
	/* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, NULL);

	finfo = parent_store_class->create_folder_sync (self,
	                                                parentname,
	                                                foldername,
	                                                cancellable,
	                                                err);
	return finfo;
}

static gboolean
kolab_imapx_store_delete_folder_sync (CamelStore *self,
                                      const gchar *foldername,
                                      GCancellable *cancellable,
                                      GError **err)
{
	gboolean ok = FALSE;

	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (self));
	/* parentname may be NULL */ /* correct? */
	g_assert (foldername != NULL);
	/* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	ok = parent_store_class->delete_folder_sync (self,
	                                             foldername,
	                                             cancellable,
	                                             err);
	return ok;
}

static gboolean
kolab_imapx_store_rename_folder_sync (CamelStore *self,
                                      const gchar *foldername_old,
                                      const gchar *foldername_new,
                                      GCancellable *cancellable,
                                      GError **err)
{
	gboolean ok = FALSE;

	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (self));
	g_assert (foldername_old != NULL);
	g_assert (foldername_new != NULL);
	/* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	ok = parent_store_class->rename_folder_sync (self,
	                                             foldername_old,
	                                             foldername_new,
	                                             cancellable,
	                                             err);
	return ok;
}

static gboolean
kolab_imapx_store_noop_sync (CamelStore *self,
                             GCancellable *cancellable,
                             GError **err)
{
	gboolean ok = FALSE;

	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (self));
	/* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	ok = parent_store_class->noop_sync (self,
	                                    cancellable,
	                                    err);
	return ok;
}

static gboolean
kolab_imapx_store_set_folder_creation_type (CamelKolabIMAPXStore *self,
                                            KolabFolderTypeID type_id)
{
	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (self));
	g_assert ((type_id > KOLAB_FOLDER_TYPE_UNKNOWN) &&
	          (type_id < KOLAB_FOLDER_LAST_TYPE));

	/* FIXME implement me */
	g_error ("%s: FIXME implement me", __func__);

	return FALSE;
}

static gboolean
kolab_imapx_store_set_folder_context (CamelKolabIMAPXStore *self,
                                      KolabFolderContextID context)
{
	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (self));
	g_assert ((context > KOLAB_FOLDER_CONTEXT_INVAL) &&
	          (context < KOLAB_FOLDER_LAST_CONTEXT));

	/* FIXME implement me */
	g_error ("%s: FIXME implement me", __func__);

	return FALSE;
}

static KolabFolderTypeID
kolab_imapx_store_get_folder_type (CamelKolabIMAPXStore *self,
                                   const gchar *foldername,
                                   GCancellable *cancellable,
                                   GError **err)
{
	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (self));
	(void)foldername; /* FIXME */
	(void)cancellable; /* FIXME */ /* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, KOLAB_FOLDER_TYPE_INVAL);

	/* FIXME implement me */
	g_error ("%s: FIXME implement me", __func__);

	return KOLAB_FOLDER_TYPE_INVAL;
}

static GList*
kolab_imapx_store_resect_folder_list (CamelKolabIMAPXStore *self)
{
	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (self));

	/* FIXME remove me */
	g_error ("%s: FIXME implement me", __func__);

	return NULL;
}

/*----------------------------------------------------------------------------*/
/* interface functions */

static gboolean
kolab_imapx_store_initable_initialize (GInitable *initable,
                                       GCancellable *cancellable,
                                       GError **err)
{
	gboolean ok = FALSE;

	g_assert (G_IS_INITABLE (initable));
	/* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	/* chain up to parent interface's init() method. */
	ok = parent_initable_iface->init (initable,
	                                  cancellable,
	                                  err);
	return ok;
}

static const gchar*
kolab_imapx_store_get_service_name (CamelNetworkService *service,
                                    CamelNetworkSecurityMethod method)
{
	const gchar *sn = NULL;

	g_assert (CAMEL_IS_NETWORK_SERVICE (service));

	/* use parent function for now */
	sn = parent_service_iface->get_service_name (service,
	                                             method);

	return sn;
}

static guint16
kolab_imapx_store_get_default_port (CamelNetworkService *service,
                                    CamelNetworkSecurityMethod method)
{
	guint16 port = 0;

	g_assert (CAMEL_IS_NETWORK_SERVICE (service));

	/* use parent function for now */
	port = parent_service_iface->get_default_port (service,
	                                               method);

	return port;
}

static gboolean
kolab_imapx_store_folder_is_subscribed (CamelSubscribable *subscribable,
                                        const gchar *foldername)
{
	gboolean subscribed = FALSE;

	g_assert (CAMEL_IS_SUBSCRIBABLE (subscribable));
	g_assert (foldername != NULL);

	/* use parent function for now */
	subscribed = parent_subscribable_iface->folder_is_subscribed (subscribable,
	                                                              foldername);

	return subscribed;
}

static gboolean
kolab_imapx_store_subscribe_folder_sync (CamelSubscribable *subscribable,
                                         const gchar *foldername,
                                         GCancellable *cancellable,
                                         GError **err)
{
	gboolean ok = FALSE;

	g_assert (CAMEL_IS_SUBSCRIBABLE (subscribable));
	g_assert (foldername != NULL);
	/* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	/* use parent function for now */
	ok = parent_subscribable_iface->subscribe_folder_sync (subscribable,
	                                                       foldername,
	                                                       cancellable,
	                                                       err);
	return ok;
}

static gboolean
kolab_imapx_store_unsubscribe_folder_sync (CamelSubscribable *subscribable,
                                           const gchar *foldername,
                                           GCancellable *cancellable,
                                           GError **err)
{
	gboolean ok = FALSE;

	g_assert (CAMEL_IS_SUBSCRIBABLE (subscribable));
	g_assert (foldername != NULL);
	/* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	/* use parent function for now */
	ok = parent_subscribable_iface->unsubscribe_folder_sync (subscribable,
	                                                         foldername,
	                                                         cancellable,
	                                                         err);
	return ok;
}

static void
kolab_imapx_store_initable_init (GInitableIface *interface)
{
	parent_initable_iface = g_type_interface_peek_parent (interface);
	interface->init = kolab_imapx_store_initable_initialize;
}

static void
kolab_imapx_store_network_service_init (CamelNetworkServiceInterface *interface)
{
	g_assert (CAMEL_IS_NETWORK_SERVICE_INTERFACE (interface));

	parent_service_iface = g_type_interface_peek_parent (interface);
	interface->get_service_name = kolab_imapx_store_get_service_name;
	interface->get_default_port = kolab_imapx_store_get_default_port;
}

static void
kolab_imapx_store_subscribable_init (CamelSubscribableInterface *interface)
{
	g_assert (CAMEL_IS_SUBSCRIBABLE_INTERFACE (interface));

	parent_subscribable_iface = g_type_interface_peek_parent (interface);
	interface->folder_is_subscribed = kolab_imapx_store_folder_is_subscribed;
	interface->subscribe_folder_sync = kolab_imapx_store_subscribe_folder_sync;
	interface->unsubscribe_folder_sync = kolab_imapx_store_unsubscribe_folder_sync;
}

/*----------------------------------------------------------------------------*/
/* class initialization */

static void
camel_kolab_imapx_store_class_init (CamelKolabIMAPXStoreClass *klass)
{
	GObjectClass *object_class;
	CamelServiceClass *service_class;
	CamelStoreClass *store_class;

	parent_service_class = CAMEL_SERVICE_CLASS (camel_kolab_imapx_store_parent_class);
	parent_store_class = CAMEL_STORE_CLASS (camel_kolab_imapx_store_parent_class);

	g_type_class_add_private (klass, sizeof (CamelKolabIMAPXStorePrivate));

	object_class = G_OBJECT_CLASS (klass);
	object_class->dispose = camel_kolab_imapx_store_dispose;
	object_class->finalize = camel_kolab_imapx_store_finalize;

	service_class = CAMEL_SERVICE_CLASS (klass);
	service_class->settings_type = CAMEL_TYPE_KOLAB_IMAPX_SETTINGS;
	service_class->get_name = kolab_imapx_store_get_name;
	service_class->connect_sync = kolab_imapx_store_connect_sync;
	service_class->disconnect_sync = kolab_imapx_store_disconnect_sync;
	service_class->authenticate_sync = kolab_imapx_store_authenticate_sync;
	service_class->query_auth_types_sync = kolab_imapx_store_query_auth_types_sync;

	store_class = CAMEL_STORE_CLASS (klass);
	store_class->hash_folder_name = kolab_imapx_store_name_hash;
	store_class->compare_folder_name = kolab_imapx_store_name_equal;
	store_class->can_refresh_folder = kolab_imapx_store_can_refresh_folder;
	store_class->free_folder_info = camel_store_free_folder_info_full;
	store_class->get_folder_sync = kolab_imapx_store_get_folder_sync;
	store_class->get_folder_info_sync = kolab_imapx_store_get_folder_info_sync;
	store_class->get_junk_folder_sync = kolab_imapx_store_get_junk_folder_sync;
	store_class->get_trash_folder_sync = kolab_imapx_store_get_trash_folder_sync;
	store_class->create_folder_sync = kolab_imapx_store_create_folder_sync;
	store_class->delete_folder_sync = kolab_imapx_store_delete_folder_sync;
	store_class->rename_folder_sync = kolab_imapx_store_rename_folder_sync;
	store_class->noop_sync = kolab_imapx_store_noop_sync;

	klass->set_folder_creation_type = kolab_imapx_store_set_folder_creation_type;
	klass->set_folder_context = kolab_imapx_store_set_folder_context;
	klass->get_folder_type = kolab_imapx_store_get_folder_type;
	klass->resect_folder_list = kolab_imapx_store_resect_folder_list;
}

/*----------------------------------------------------------------------------*/
/* API functions */

gboolean
camel_kolab_imapx_store_set_folder_creation_type (CamelKolabIMAPXStore *self,
                                                  KolabFolderTypeID type_id)
{
	CamelKolabIMAPXStoreClass *klass = NULL;
	gboolean ok = FALSE;

	g_return_val_if_fail (CAMEL_IS_KOLAB_IMAPX_STORE (self), FALSE);

	klass = CAMEL_KOLAB_IMAPX_STORE_GET_CLASS (self);
	ok = klass->set_folder_creation_type (self, type_id);

	return ok;
}

gboolean
camel_kolab_imapx_store_set_folder_context (CamelKolabIMAPXStore *self,
                                            KolabFolderContextID context)
{
	CamelKolabIMAPXStoreClass *klass = NULL;
	gboolean ok = FALSE;

	g_return_val_if_fail (CAMEL_IS_KOLAB_IMAPX_STORE (self), FALSE);

	klass = CAMEL_KOLAB_IMAPX_STORE_GET_CLASS (self);
	ok = klass->set_folder_context (self, context);

	return ok;
}

KolabFolderTypeID
camel_kolab_imapx_store_get_folder_type (CamelKolabIMAPXStore *self,
                                         const gchar *foldername,
                                         GCancellable *cancellable,
                                         GError **err)
{
	CamelKolabIMAPXStoreClass *klass = NULL;
	KolabFolderTypeID foldertype = KOLAB_FOLDER_TYPE_INVAL;

	g_return_val_if_fail (CAMEL_IS_KOLAB_IMAPX_STORE (self), KOLAB_FOLDER_TYPE_INVAL);

	klass = CAMEL_KOLAB_IMAPX_STORE_GET_CLASS (self);
	foldertype = klass->get_folder_type (self,
	                                     foldername,
	                                     cancellable,
	                                     err);
	return foldertype;
}

GList*
camel_kolab_imapx_store_resect_folder_list (CamelKolabIMAPXStore *self)
{
	CamelKolabIMAPXStoreClass *klass = NULL;
	GList *list = NULL;

	g_return_val_if_fail (CAMEL_IS_KOLAB_IMAPX_STORE (self), NULL);

	klass = CAMEL_KOLAB_IMAPX_STORE_GET_CLASS (self);
	list = klass->resect_folder_list (self);

	return list;
}

gboolean
camel_kolab_imapx_store_logout_sync (CamelKolabIMAPXStore *self,
                                     GCancellable *cancellable,
                                     GError **err)
{
	/* TODO check whether this function is still needed */

	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (self));
	(void)cancellable; /* FIXME */ /* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	/* FIXME implement me */
	g_error ("%s: FIXME implement me", __func__);

	return FALSE;
}

/*----------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/



#if 0

static void
kolab_imapx_construct (CamelService *service,
                       CamelSession *session,
                       CamelProvider *provider,
                       CamelURL *url,
                       GError **err)
{
	CamelKolabIMAPXStore *store = NULL;
	gboolean metadata_ok = FALSE;
	GError *tmp_err = NULL;

	g_assert (CAMEL_IS_SERVICE (service));
	g_assert (CAMEL_IS_SESSION (session));
	g_assert (provider != NULL);
	g_assert (url != NULL);
	g_return_if_fail (err == NULL || *err == NULL);

	(CAMEL_SERVICE_CLASS (parent_class))->construct (service,
	                                                 session,
	                                                 provider,
	                                                 url,
	                                                 tmp_err);
	if (tmp_err != NULL) {
		g_propagate_error (err, tmp_err);
		return;
	}

	/*------- metadata db -------*/

	store = CAMEL_KOLAB_IMAPX_STORE (service);

	metadata_ok = kolab_imapx_meta_data_init (store->kmd,
	                                          service,
	                                          session,
	                                          &tmp_err);
	if (!metadata_ok) {
		kolab_camelexception_propagate_from_gerror (ex, tmp_err);
		return;
	}

	camel_exception_free (tmp_ex);

	/* default folder type to create (for use in Evo) */
	store->folder_create_type = KOLAB_FOLDER_TYPE_EMAIL;

	/* default folder context (for use in Evo) */
	store->folder_context = KOLAB_FOLDER_CONTEXT_EMAIL;

	g_debug ("%s: metadata db initialized", __func__);
}

static gboolean
kolab_imapx_connect (CamelService *service,
                     CamelException *ex)
{
	CamelException *tmp_ex = NULL;

	g_assert (CAMEL_IS_SERVICE (service));
	g_assert (ex != NULL);

	tmp_ex = camel_exception_new ();

	(CAMEL_SERVICE_CLASS (parent_class))->connect (service, tmp_ex);
	if (camel_exception_is_set (tmp_ex)) {
		g_warning ("%s: %s",
		           __func__,
		           camel_exception_get_description (tmp_ex));
		camel_exception_xfer (ex, tmp_ex);
		return FALSE;
	}

	camel_exception_free (tmp_ex);
	return TRUE;
}

static gboolean
kolab_imapx_disconnect (CamelService *service,
                        gboolean clean,
                        CamelException *ex)
{
	CamelException *tmp_ex = NULL;
	gboolean parent_ok = FALSE;

	g_assert (CAMEL_IS_SERVICE (service));
	/* *ex may be NULL */

	tmp_ex = camel_exception_new ();
	parent_ok = (CAMEL_SERVICE_CLASS (parent_class))->disconnect (service,
	                                                              clean,
	                                                              tmp_ex);
	if (camel_exception_is_set (tmp_ex)) {
		g_warning ("%s: %s",
		           __func__,
		           camel_exception_get_description (tmp_ex));
		if (ex != NULL)
			camel_exception_xfer (ex, tmp_ex);
	}

	camel_exception_free (tmp_ex);
	return parent_ok;
}

static KolabFolderTypeID
kolab_imapx_folder_get_foldertype (CamelService *service,
                                   const CamelFolderInfo *fi,
                                   CamelKolabIMAPXMetaData *kmd)
{
	KolabFolderTypeID folder_type = KOLAB_FOLDER_TYPE_INVAL;
	GError *tmp_err = NULL;

	g_assert (CAMEL_IS_SERVICE (service));
	g_assert (fi != NULL);
	g_assert (kmd != NULL);

	folder_type = kolab_imapx_meta_data_get_foldertype (kmd,
	                                                    service,
	                                                    fi->full_name,
	                                                    TRUE,
	                                                    &tmp_err);
	if (tmp_err != NULL) {
		g_warning ("%s: error getting folder type: %s",
		           __func__,
		           tmp_err->message);
		g_error_free (tmp_err);
		return KOLAB_FOLDER_TYPE_INVAL;
	}
	return folder_type;
}

static CamelFolderInfo*
kolab_imapx_folder_info_build_restricted (CamelService *service,
                                          const CamelFolderInfo *fi,
                                          CamelKolabIMAPXMetaData *kmd,
                                          const gboolean do_care[])
{
	CamelKolabIMAPXStore *self = NULL;
	KolabFolderTypeID folder_type = KOLAB_FOLDER_TYPE_INVAL;
	CamelFolderInfo *self_fi = NULL;
	CamelFolderInfo *next_fi = NULL;
	CamelFolderInfo *chld_fi = NULL;

	g_assert (CAMEL_IS_SERVICE (service));
	self = (CamelKolabIMAPXStore *)service;

	if (fi == NULL)
		return NULL;

	next_fi = kolab_imapx_folder_info_build_restricted (service,
	                                                    fi->next,
	                                                    kmd,
	                                                    do_care);

	chld_fi = kolab_imapx_folder_info_build_restricted (service,
	                                                    fi->child,
	                                                    kmd,
	                                                    do_care);

	folder_type = kolab_imapx_folder_get_foldertype (service, fi, kmd);

	if ((chld_fi != NULL) || (do_care[folder_type])) {
		self_fi = camel_folder_info_new ();
		if (fi->uri)
			self_fi->uri = g_strdup (fi->uri);
		if (fi->name)
			self_fi->name = g_strdup (fi->name);
		if (fi->full_name)
			self_fi->full_name = g_strdup (fi->full_name);
		self_fi->flags = fi->flags;
		self_fi->unread = fi->unread;
		self_fi->total = fi->total;
		self_fi->next = next_fi;
		self_fi->child = chld_fi;
		if (self_fi->child != NULL)
			self_fi->child->parent = self_fi;
		if ((self->folder_context != KOLAB_FOLDER_CONTEXT_EMAIL) &&
		    do_care[folder_type]) {
			self->folder_names_do_care = g_list_prepend (self->folder_names_do_care,
			                                             g_strdup (self_fi->full_name));
		}
		if (!do_care[folder_type]) {
			self_fi->flags |= CAMEL_FOLDER_NOSELECT;
			/* TODO check whether we need to (re)set more
			 *	(e.g. unread=0, total=0, ...)
			 */
		}
		return self_fi;
	}

	return next_fi;
}

static CamelFolderInfo*
kolab_imapx_get_folder_info (CamelStore *store,
                             const gchar *top,
                             guint32 flags,
                             CamelException *ex)
{
	CamelFolderInfo *fi = NULL;
	CamelFolderInfo *k_fi = NULL;
	CamelKolabIMAPXStore *ikstore = CAMEL_KOLAB_IMAPX_STORE (store);
	CamelService *service = CAMEL_SERVICE (store);
	CamelException *tmp_ex = NULL;

	g_assert (CAMEL_IS_STORE (store));
	/* 'top' may be NULL */
	g_assert (ex != NULL);

	tmp_ex = camel_exception_new ();

	g_mutex_lock (ikstore->kolab_finfo_lock);

	fi = (CAMEL_STORE_CLASS (parent_class))->get_folder_info (store,
	                                                          top,
	                                                          flags,
	                                                          tmp_ex);
	if (fi == NULL) {
		g_warning ("%s: aborted.", __func__);
		if (camel_exception_is_set (tmp_ex))
			camel_exception_xfer (ex, tmp_ex);
		else
			camel_exception_free (tmp_ex);
		g_mutex_unlock (ikstore->kolab_finfo_lock);
		return NULL;
	}

	k_fi = kolab_imapx_folder_info_build_restricted (service,
	                                                 fi,
	                                                 ikstore->kmd,
	                                                 ikstore->folder_types_do_care);
	camel_store_free_folder_info (store, fi);

	g_mutex_unlock (ikstore->kolab_finfo_lock);

	camel_exception_free (tmp_ex);

	return k_fi;
}

static CamelFolderInfo*
kolab_imapx_create_folder (CamelStore *store,
                           const gchar *parent_name,
                           const gchar *folder_name,
                           CamelException *ex)
{
	CamelFolderInfo *fi = NULL;
	CamelFolderInfo *k_fi = NULL;
	CamelKolabIMAPXStore *ikstore = CAMEL_KOLAB_IMAPX_STORE (store);
	CamelService *service = CAMEL_SERVICE (store);
	GError *tmp_err = NULL;
	CamelException *tmp_ex = NULL;
	gboolean metadata_ok = FALSE;
	gchar *full_name = NULL;

	g_assert (CAMEL_IS_STORE (store));
	/* 'parent_name' may be NULL */
	g_assert (folder_name != NULL);
	g_assert (ex != NULL);

	tmp_ex = camel_exception_new ();

	fi = (CAMEL_STORE_CLASS (parent_class))->create_folder (store,
	                                                        parent_name,
	                                                        folder_name,
	                                                        tmp_ex);
	if (fi == NULL) {
		g_warning ("%s: create [%s/%s] on server failed.",
		           __func__, parent_name, folder_name);
		if (camel_exception_is_set (tmp_ex))
			camel_exception_xfer (ex, tmp_ex);
		return NULL;
	}

	camel_exception_free (tmp_ex);

	/* FIXME use Camel function(s) to create full_name */
	full_name = g_strdup_printf ("%s/%s", parent_name, folder_name);
	metadata_ok = kolab_imapx_meta_data_set_foldertype (ikstore->kmd,
	                                                    service,
	                                                    full_name,
	                                                    ikstore->folder_create_type,
	                                                    &tmp_err);
	g_free (full_name);

	if (!metadata_ok) {
		g_warning ("%s: setting type [%i] for [%s] on server failed.",
		           __func__, ikstore->folder_create_type, full_name);
		camel_store_free_folder_info (store, fi);
		kolab_camelexception_propagate_from_gerror (ex, tmp_err);
		return NULL;
	}

	k_fi = kolab_imapx_folder_info_build_restricted (service,
	                                                 fi,
	                                                 ikstore->kmd,
	                                                 ikstore->folder_types_do_care);
	camel_store_free_folder_info (store, fi);
	return k_fi;
}

static void
kolab_imapx_delete_folder (CamelStore *store,
                           const gchar *folder_name,
                           CamelException *ex)
{
	CamelKolabIMAPXStore *ikstore = CAMEL_KOLAB_IMAPX_STORE (store);
	CamelException *tmp_ex = NULL;
	GError *tmp_err = NULL;
	gboolean metadata_ok = FALSE;

	g_assert (CAMEL_IS_STORE (store));
	g_assert (folder_name != NULL);
	g_assert (ex != NULL);

	tmp_ex = camel_exception_new ();

	(CAMEL_STORE_CLASS (parent_class))->delete_folder (store,
	                                                   folder_name,
	                                                   tmp_ex);
	if (camel_exception_is_set (tmp_ex)) {
		camel_exception_xfer (ex, tmp_ex);
		return;
	}
	camel_exception_free (tmp_ex);

	metadata_ok = kolab_imapx_meta_data_remove (ikstore->kmd,
	                                            folder_name,
	                                            &tmp_err);
	if (!metadata_ok) {
		kolab_camelexception_propagate_from_gerror (ex, tmp_err);
		return;
	}
}

/*----------------------------------------------------------------------------*/
/* CamelObject initialization and shutdown */

static void
camel_kolab_imapx_store_class_init (CamelKolabIMAPXStoreClass *klass)
{
	CamelServiceClass *camel_service_class = NULL;
	CamelServiceClass *parent_service_class = NULL;
	CamelStoreClass *camel_store_class = NULL;
	CamelStoreClass *parent_store_class = NULL;

	g_assert (klass != NULL);

	camel_service_class = CAMEL_SERVICE_CLASS (klass);
	camel_store_class = CAMEL_STORE_CLASS (klass);

	parent_class = CAMEL_IMAPX_STORE_CLASS (camel_type_get_global_classfuncs (__KOLAB_camel_imapx_store_get_type ()));
	parent_service_class = CAMEL_SERVICE_CLASS (parent_class);
	parent_store_class = CAMEL_STORE_CLASS (parent_class);

	/* camel_service_class->construct = parent_service_class->construct; */
	camel_service_class->construct = kolab_imapx_construct;

	camel_service_class->query_auth_types = parent_service_class->query_auth_types;
	camel_service_class->get_name = parent_service_class->get_name;

	/* camel_service_class->connect = parent_service_class->connect; */
	camel_service_class->connect = kolab_imapx_connect;

	/* camel_service_class->disconnect = parent_service_class->disconnect; */
	camel_service_class->disconnect = kolab_imapx_disconnect;

	camel_store_class->get_trash = parent_store_class->get_trash;
	camel_store_class->get_junk = parent_store_class->get_junk;
	camel_store_class->noop = parent_store_class->noop;
	camel_store_class->get_folder = parent_store_class->get_folder;
	camel_store_class->get_inbox = parent_store_class->get_inbox;
	camel_store_class->hash_folder_name = parent_store_class->hash_folder_name;
	camel_store_class->compare_folder_name = parent_store_class->compare_folder_name;

	/* camel_store_class->get_folder_info = parent_store_class->get_folder_info; */
	camel_store_class->get_folder_info = kolab_imapx_get_folder_info;

	camel_store_class->can_refresh_folder = parent_store_class->can_refresh_folder;

	/* camel_store_class->create_folder = parent_store_class->create_folder; */
	camel_store_class->create_folder = kolab_imapx_create_folder;

	camel_store_class->rename_folder = parent_store_class->rename_folder;

	/* camel_store_class->delete_folder = parent_store_class->delete_folder; */
	camel_store_class->delete_folder = kolab_imapx_delete_folder;

	camel_store_class->subscribe_folder = parent_store_class->subscribe_folder;
	camel_store_class->unsubscribe_folder = parent_store_class->unsubscribe_folder;
	camel_store_class->folder_subscribed = parent_store_class->folder_subscribed;
	camel_store_class->free_folder_info = camel_store_free_folder_info_full;

	camel_store_class->hash_folder_name = parent_store_class->hash_folder_name;
	camel_store_class->compare_folder_name = parent_store_class->compare_folder_name;
}

static void
camel_kolab_imapx_store_init (gpointer object,
                              gpointer klass)
{
	CamelKolabIMAPXStore *ikstore = NULL;
	gint ii = 0;

	g_assert (klass != NULL);

	ikstore = CAMEL_KOLAB_IMAPX_STORE (object);
	ikstore->kmd = kolab_imapx_meta_data_new ();

	ikstore->kolab_finfo_lock = g_mutex_new ();

	/* folder types to care for with this kolab_imapx instance
	 * Default: Email and unknown (so no need to reconfigure
	 * this instance for use in Evo). Needs to be reconfigured
	 * when used in ECal/EBook backends
	 */
	for (ii = 0; ii < KOLAB_FOLDER_LAST_TYPE; ii++)
		ikstore->folder_types_do_care[ii] = FALSE;
	ikstore->folder_types_do_care[KOLAB_FOLDER_TYPE_UNKNOWN] 	 = TRUE;
	ikstore->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL] 		 = TRUE;
	ikstore->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL_INBOX] 	 = TRUE;
	ikstore->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL_DRAFTS] 	 = TRUE;
	ikstore->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL_SENTITEMS] = TRUE;
	ikstore->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL_JUNKEMAIL] = TRUE;

	ikstore->folder_names_do_care = NULL;

	g_debug ("%s: done", __func__);
}

static void
camel_kolab_imapx_store_finalize (CamelObject* object)
{
	CamelKolabIMAPXStore *store = NULL;
	GError *tmp_err = NULL;
	gboolean ok = FALSE;

	store = CAMEL_KOLAB_IMAPX_STORE (object);

	/* finalize parent */

	/* finalize instance */
	while (! g_mutex_trylock (store->kolab_finfo_lock));
	g_mutex_unlock (store->kolab_finfo_lock);
	g_mutex_free (store->kolab_finfo_lock);

	/* TODO rework - there should not be I/O in finalize() */

	/*------- metadata db -------*/
	ok = kolab_imapx_meta_data_uninit (store->kmd, &tmp_err);
	if (!ok) {
		g_warning ("%s: metadata uninit error: %s",
		           __func__,
		           tmp_err->message);
		g_error_free (tmp_err);
		return;
	}
	kolab_imapx_meta_data_free (store->kmd);
	g_debug ("%s: metadata uninitialized",
	         __func__);

	if (store->folder_names_do_care != NULL) {
		GList *list_ptr = store->folder_names_do_care;
		while (list_ptr != NULL) {
			g_free (list_ptr);
			list_ptr = g_list_next (list_ptr);
		}
		g_free (store->folder_names_do_care);
	}

	g_debug ("%s: done", __func__);
}

CamelType
camel_kolab_imapx_store_get_type (void)
{
	static CamelType camel_kolab_imapx_store_type = CAMEL_INVALID_TYPE;

	if (camel_kolab_imapx_store_type == CAMEL_INVALID_TYPE) {
		camel_kolab_imapx_store_type = camel_type_register(__KOLAB_camel_imapx_store_get_type (),
		                                                   "CamelKolabIMAPXStore",
		                                                   sizeof (CamelKolabIMAPXStore),
		                                                   sizeof (CamelKolabIMAPXStoreClass),
		                                                   (CamelObjectClassInitFunc)camel_kolab_imapx_store_class_init,
		                                                   NULL,
		                                                   (CamelObjectInitFunc)camel_kolab_imapx_store_init,
		                                                   camel_kolab_imapx_store_finalize);
	}

	g_assert (camel_kolab_imapx_store_type != CAMEL_INVALID_TYPE);
	return camel_kolab_imapx_store_type;
}

gboolean
kolab_imapx_store_set_folder_creation_type (CamelKolabIMAPXStore *store,
                                            KolabFolderTypeID type_id)
{
	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (store));

	/* TODO error reporting */

	/* check that the given folder type id lies within the configured folder context */
	if (! kolab_util_folder_type_match_with_context_id (type_id, store->folder_context))
		return FALSE;

	store->folder_create_type = type_id;

	return TRUE;
}

gboolean
kolab_imapx_store_set_folder_context (CamelKolabIMAPXStore *store,
                                      KolabFolderContextID context)
{
	gint ii = 0;

	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (store));
	g_assert ((context > KOLAB_FOLDER_CONTEXT_INVAL) &&
	          (context < KOLAB_FOLDER_LAST_CONTEXT));

	for (ii = 0; ii < KOLAB_FOLDER_LAST_TYPE; ii++)
		store->folder_types_do_care[ii] = FALSE;

	store->folder_context = context;

	switch (context) {
	case KOLAB_FOLDER_CONTEXT_EMAIL:
		store->folder_types_do_care[KOLAB_FOLDER_TYPE_UNKNOWN]		= TRUE;
		store->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL]		= TRUE;
		store->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL_INBOX]	= TRUE;
		store->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL_DRAFTS]	= TRUE;
		store->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL_SENTITEMS]	= TRUE;
		store->folder_types_do_care[KOLAB_FOLDER_TYPE_EMAIL_JUNKEMAIL]	= TRUE;
		break;
	case KOLAB_FOLDER_CONTEXT_CALENDAR:
		store->folder_types_do_care[KOLAB_FOLDER_TYPE_EVENT]		= TRUE;
		store->folder_types_do_care[KOLAB_FOLDER_TYPE_EVENT_DEFAULT]	= TRUE;
		store->folder_types_do_care[KOLAB_FOLDER_TYPE_JOURNAL]		= TRUE;
		store->folder_types_do_care[KOLAB_FOLDER_TYPE_JOURNAL_DEFAULT]	= TRUE;
		store->folder_types_do_care[KOLAB_FOLDER_TYPE_TASK]		= TRUE;
		store->folder_types_do_care[KOLAB_FOLDER_TYPE_TASK_DEFAULT]	= TRUE;
		store->folder_types_do_care[KOLAB_FOLDER_TYPE_NOTE]		= TRUE;
		store->folder_types_do_care[KOLAB_FOLDER_TYPE_NOTE_DEFAULT]	= TRUE;
		break;
	case KOLAB_FOLDER_CONTEXT_CONTACT:
		store->folder_types_do_care[KOLAB_FOLDER_TYPE_CONTACT]		= TRUE;
		store->folder_types_do_care[KOLAB_FOLDER_TYPE_CONTACT_DEFAULT]	= TRUE;
		break;
	default:
		/* can't happen */
		g_assert_not_reached ();
	}

	return TRUE;
}

KolabFolderTypeID
kolab_imapx_store_get_folder_type (CamelKolabIMAPXStore *store,
                                   const gchar *foldername,
                                   GError **err)
{
	/* TODO merge this with kolab_imapx_folder_get_foldertype() */

	KolabFolderTypeID folder_type = KOLAB_FOLDER_TYPE_INVAL;
	GError *tmp_err = NULL;

	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (store));
	g_assert (foldername != NULL);
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	folder_type = kolab_imapx_meta_data_get_foldertype (store->kmd,
	                                                    CAMEL_SERVICE (store),
	                                                    foldername,
	                                                    TRUE,
	                                                    &tmp_err);
	if (tmp_err != NULL) {
		g_propagate_error (err, tmp_err);
		return KOLAB_FOLDER_TYPE_INVAL;
	}

	return folder_type;
}

void
kolab_imapx_store_logout_sync (CamelKolabIMAPXStore *store)
{
	CamelIMAPXServer *iserver = NULL;
	gboolean ok = FALSE;
	CamelException *tmp_ex = NULL;

	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (store));

	tmp_ex = camel_exception_new ();
	iserver = __KOLAB_camel_imapx_store_get_server (CAMEL_IMAPX_STORE (store),
	                                                tmp_ex);
	if (camel_exception_is_set (tmp_ex)) {
		g_warning ("%s: could not get server: %s",
		           __func__,
		           camel_exception_get_description (tmp_ex));
		camel_exception_free (tmp_ex);
		if (iserver != NULL)
			camel_object_unref (iserver);
		return;
	}

	g_assert (CAMEL_IS_IMAPX_SERVER (iserver));

	ok = __KOLAB_imapx_server_logout_sync (iserver, tmp_ex);

	camel_object_unref (iserver);
	if (! ok) {
		g_debug ("%s: %s",
		         __func__,
		         camel_exception_get_description (tmp_ex));
	}
	camel_exception_free (tmp_ex);
}

GList*
kolab_imapx_store_resect_folder_list (CamelKolabIMAPXStore *store)
{
	GList *folder_list = NULL;

	g_assert (CAMEL_IS_KOLAB_IMAPX_STORE (store));

	g_mutex_lock (store->kolab_finfo_lock);

	folder_list = store->folder_names_do_care;
	store->folder_names_do_care = NULL;

	g_mutex_unlock (store->kolab_finfo_lock);

	return folder_list;
}

#endif
