/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/***************************************************************************
 *            camel-kolab-session.c
 *
 *  Tue Aug 10 15:04:38 2010
 *  Copyright  2010  Christian Hilberg
 *  <hilberg@kernelconcepts.de>
 ****************************************************************************/

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with main.c; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

/*----------------------------------------------------------------------------*/

/*
 * The CamelSession class for Kolab access. To be instantiated once for
 * each IMAPX Camel.Provider we have. Within EDS, this is one for address book
 * and one for calendar access. Within Evolution, there should be one more
 * for email.
 */

/*----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib/gi18n-lib.h>

#include <pk11func.h>
#include <prtypes.h>

#include <libedataserver/e-account.h>
#include <libedataserverui/e-passwords.h>

#include <libekolabutil/camel-system-headers.h>

/* Kolab error reporting */
#include <libekolabutil/kolab-util-error.h>

#include "camel-kolab-session.h"

/*----------------------------------------------------------------------------*/

static gchar *nss_tok_pin = NULL; /* FIXME better solution for this */

/*----------------------------------------------------------------------------*/

typedef struct _CamelKolabSessionPrivate CamelKolabSessionPrivate;
struct _CamelKolabSessionPrivate {
	/* TODO these should be retrieved
	 * from CamelKolabSettings instead
	 */
	gchar *data_dir;
	gchar *config_dir;
	gchar *passwd;

	/* TODO get rid of this workaround */
	gchar *nss_tok_pwd;

	gboolean is_initialized;
};

#define CAMEL_KOLAB_SESSION_PRIVATE(obj)  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), CAMEL_TYPE_KOLAB_SESSION, CamelKolabSessionPrivate))

G_DEFINE_TYPE (CamelKolabSession, camel_kolab_session, CAMEL_TYPE_SESSION)

/*----------------------------------------------------------------------------*/

static void
camel_kolab_session_init (CamelKolabSession *self)
{
	CamelKolabSessionPrivate *priv = NULL;

	g_assert (CAMEL_IS_KOLAB_SESSION (self));
	priv = CAMEL_KOLAB_SESSION_PRIVATE (self);

	priv->data_dir = NULL;
	priv->config_dir = NULL;
	priv->nss_tok_pwd = NULL;
	priv->passwd = NULL;
	priv->is_initialized = FALSE;
}

static void
camel_kolab_session_dispose (GObject *object)
{
	g_assert (CAMEL_IS_KOLAB_SESSION (object));

	/* Chain up to parent's dispose() method. */
	G_OBJECT_CLASS (camel_kolab_session_parent_class)->dispose (object);
}

static void
camel_kolab_session_finalize (GObject *object)
{
	CamelKolabSession *self = NULL;
	CamelKolabSessionPrivate *priv = NULL;

	g_assert (CAMEL_IS_KOLAB_SESSION (self));

	self = CAMEL_KOLAB_SESSION (object);
	priv = CAMEL_KOLAB_SESSION_PRIVATE (self);

	if (priv->data_dir != NULL)
		g_free (priv->data_dir);
	if (priv->config_dir != NULL)
		g_free (priv->config_dir);
	if (priv->nss_tok_pwd != NULL)
		g_free (priv->nss_tok_pwd);
	if (priv->passwd)
		g_free (priv->passwd);

	/* Chain up to parent's finalize() method. */
	G_OBJECT_CLASS (camel_kolab_session_parent_class)->finalize (object);
}

static void
camel_kolab_session_class_init (CamelKolabSessionClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	CamelSessionClass *session_class = CAMEL_SESSION_CLASS (klass);
	CamelSessionClass *parent_class = CAMEL_SESSION_CLASS (camel_kolab_session_parent_class);

	g_type_class_add_private (klass, sizeof (CamelKolabSessionPrivate));

	object_class->set_property = G_OBJECT_CLASS (parent_class)->set_property;
	object_class->get_property = G_OBJECT_CLASS (parent_class)->get_property;
	object_class->dispose = camel_kolab_session_dispose;
	object_class->finalize = camel_kolab_session_finalize;

	session_class->add_service = parent_class->add_service; /* TODO need to override this */

	session_class->authenticate_sync = parent_class->authenticate_sync;
	session_class->authenticate = parent_class->authenticate;
	session_class->authenticate_finish = parent_class->authenticate_finish;
}

/*----------------------------------------------------------------------------*/
/* password callbacks */

/* NSS token pin callback */
static gchar* PR_CALLBACK
pk11_password (PK11SlotInfo *slot,
               PRBool retry,
               gpointer arg)
{
	(void)slot;
	(void)arg;

	/* as recommended by PK11_SetPasswordFunc docs */
	if (retry)
		return NULL;

	return PL_strdup (nss_tok_pin);
}

/*----------------------------------------------------------------------------*/

gboolean
camel_kolab_session_bringup (CamelKolabSession *self,
                             GCancellable *cancellable,
                             GError **err)
{
	/* TODO rework to use GInitable */

	CamelKolabSessionPrivate *priv = NULL;

	g_assert (CAMEL_IS_KOLAB_SESSION (self));
	(void)cancellable; /* FIXME */ /* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	priv = CAMEL_KOLAB_SESSION_PRIVATE (self);

	if (priv->is_initialized)
		return TRUE;

	/* TODO need to supply (parts of) storage paths here so the imap
	 *	objects for the backends will store their files in their
	 *	own respective directories
	 */

	/* TODO check whether this is The Right Way to
	 *	initialize the session parent
	 */
	if (priv->data_dir == NULL) {
		g_set_error (err,
		             KOLAB_CAMEL_KOLAB_ERROR,
		             KOLAB_CAMEL_KOLAB_ERROR_GENERIC,
		             _("Camel data directory not configured"));
		return FALSE;
	}

	/* TODO junk settings */

	/* setup further NSS bits here */
	PK11_SetPasswordFunc (pk11_password);

	priv->is_initialized = TRUE;
	g_debug ("%s: camel session initialized",
	         __func__);

	return TRUE;
}

gboolean
camel_kolab_session_shutdown (CamelKolabSession *self,
                              GCancellable *cancellable,
                              GError **err)
{
	CamelKolabSessionPrivate *priv = NULL;

	g_assert (CAMEL_IS_KOLAB_SESSION (self));
	(void)cancellable; /* FIXME */ /* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	priv = CAMEL_KOLAB_SESSION_PRIVATE (self);

	if (! priv->is_initialized)
		return TRUE;

	g_debug ("%s: camel session shut down",
	         __func__);

	return TRUE;
}

/*----------------------------------------------------------------------------*/

void
camel_kolab_session_set_data_dir (CamelKolabSession *self,
                                  gchar *datadir)
{
	/* TODO rework to use CamelSettings */

	CamelKolabSessionPrivate *priv = NULL;

	g_assert (CAMEL_IS_KOLAB_SESSION (self));
	g_assert (datadir != NULL);

	priv = CAMEL_KOLAB_SESSION_PRIVATE (self);

	if (priv->data_dir != NULL)
		g_free (priv->data_dir);

	priv->data_dir = datadir;
}

const gchar*
camel_kolab_session_get_data_dir (CamelKolabSession *self)
{
	/* TODO rework to use CamelSettings */

	CamelKolabSessionPrivate *priv = NULL;

	g_assert (CAMEL_IS_KOLAB_SESSION (self));
	priv = CAMEL_KOLAB_SESSION_PRIVATE (self);

	return priv->data_dir;
}

void
camel_kolab_session_set_config_dir (CamelKolabSession *self,
                                    const gchar *configdir)
{
	/* TODO rework to use CamelSettings */

	CamelKolabSessionPrivate *priv = NULL;

	g_assert (CAMEL_IS_KOLAB_SESSION (self));
	g_assert (configdir != NULL);

	priv = CAMEL_KOLAB_SESSION_PRIVATE (self);

	if (priv->config_dir != NULL)
		g_free (priv->config_dir);

	priv->config_dir = g_strdup (configdir);
}

const gchar*
camel_kolab_session_get_config_dir (CamelKolabSession *self)
{
	/* TODO rework to use CamelSettings */

	CamelKolabSessionPrivate *priv = NULL;

	g_assert (CAMEL_IS_KOLAB_SESSION (self));
	priv = CAMEL_KOLAB_SESSION_PRIVATE (self);

	return priv->config_dir;
}

const gchar*
camel_kolab_session_get_password (CamelKolabSession *self)
{
	/* TODO rework to CamelSettings */

	CamelKolabSessionPrivate *priv = NULL;

	g_assert (CAMEL_IS_KOLAB_SESSION (self));
	priv = CAMEL_KOLAB_SESSION_PRIVATE (self);

	return priv->passwd;
}

void
camel_kolab_session_forget_password (CamelKolabSession *self)
{
	/* TODO rework to CamelSettings */

	CamelKolabSessionPrivate *priv = NULL;

	g_assert (CAMEL_IS_KOLAB_SESSION (self));
	priv = CAMEL_KOLAB_SESSION_PRIVATE (self);

	if (priv->passwd != NULL) {
		g_free (priv->passwd);
		priv->passwd = NULL;
	}
}

void
camel_kolab_session_set_password (CamelKolabSession *self,
                                  const gchar *passwd)
{
	/* TODO rework to CamelSettings */

	CamelKolabSessionPrivate *priv = NULL;

	g_assert (CAMEL_IS_KOLAB_SESSION (self));
	priv = CAMEL_KOLAB_SESSION_PRIVATE (self);

	if (priv->passwd != NULL)
		g_free (priv->passwd);

	priv->passwd = g_strdup (passwd);
}

void
camel_kolab_session_set_token_pin (CamelKolabSession *self,
                                   const gchar *pin)
{
	/* TODO rework
	 *
	 * The entire NSS token handling needs
	 * to be rewritten (it is a mere demonstrator
	 * presently, and should not be relied upon
	 * in critical environments
	 *
	 */

	g_assert (CAMEL_IS_KOLAB_SESSION (self));

	if (nss_tok_pin != NULL)
		g_free (nss_tok_pin);

	nss_tok_pin = g_strdup (pin);
}

/*----------------------------------------------------------------------------*/
