 /*---[ firestarter.c ]-----------------------------------------------
 * Copyright (C) 2000 Tomas Junnonen (majix@sci.fi)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * The main application file
 *--------------------------------------------------------------------*/

#include <config.h>
#include <gnome.h>

#include <status-docklet.h>

#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>

#include "firestarter.h"
#include "menus.h"
#include "parse.h"
#include "druid.h"
#include "preferences.h"
#include "modrules.h"

#include "xpm/statusrunning.xpm"
#include "xpm/statuscaught.xpm"
#include "xpm/statustopped.xpm"

#include <libgnomeui/gnome-window-icon.h>

gboolean NETFILTER;
gboolean running = FALSE;

GtkWidget *window;
GtkWidget *panelvbox;
GtkWidget *panelpic;

void set_panel_pic (gchar *pixmap[]);

/* [ check_version ]
 * Check version on the script file. If it was created for the wrong
 * kernel type or the version was < program version, start fresh.
 */
static void
check_version (void)
{
	FILE *f;
	gchar *version = "0.0.0";
	gchar buf[512];
	gint wrong_kernel = FALSE;

	f = fopen (FIRESTARTER_RULES_DIR "/firestarter/firewall.sh", "r");

	if (!f) {
		gnome_config_set_bool ("/firestarter/Druid/firsttime", TRUE);
		gnome_config_sync ();
		return;
	}

	while (fgets (buf, 512, f) != NULL) {
		if (strstr (buf, "by Firestarter "))  {
			version = get_text_between (buf, "by Firestarter ", ",");
			if (strstr (buf, "NETFILTER")) {
				if (!NETFILTER)
					wrong_kernel = TRUE;
			} else
				if (NETFILTER)
					wrong_kernel = TRUE;
			break;
		}
	}

	fclose (f);

	if ((g_strcasecmp (version, VERSION) < 0) || wrong_kernel) {
		GtkWidget *dialog;
		GtkWidget *label;
		int button;

		mkdir (FIRESTARTER_RULES_DIR "/firestarter", 00700);

		dialog = gnome_dialog_new (_("Old version detected"),
			GNOME_STOCK_BUTTON_YES, GNOME_STOCK_BUTTON_NO, NULL);

		label = gtk_label_new (_(
			"\nA firewall created by an older version of\n"
			"Firestarter has been detected.\n\n"
			"OK to remove old firewall (recommended)?\n"));
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox),
			label, FALSE, FALSE, 0);
		gtk_widget_show (label);

		button = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

		switch (button) {
			case 0:
				remove (FIRESTARTER_RULES_DIR "/firestarter/firewall.sh");
				remove (FIRESTARTER_RULES_DIR "/firestarter/allow-all");
				remove (FIRESTARTER_RULES_DIR "/firestarter/deny-all");
				remove (FIRESTARTER_RULES_DIR "/firestarter/allow-service-all");
				remove (FIRESTARTER_RULES_DIR "/firestarter/allow-service-machine");
				remove (FIRESTARTER_RULES_DIR "/firestarter/do-not-log-ports");
				gnome_config_set_bool ("/firestarter/Druid/firsttime", TRUE);
				gnome_config_sync ();
				printf ("Scripts removed");
				break;

			case 1:
			case -1:
				printf ("Scripts not removed");
				break;
		}
	}
}

/* [ appbar_update ]
 * Updates the application status bar with the state of the firewall
 */
static void
appbar_update (char *state)
{
	if (state == "stop")
		gnome_appbar_set_default (GNOME_APPBAR (appbar),
					_("Firewall stopped"));
	if (state == "start")
		gnome_appbar_set_default (GNOME_APPBAR (appbar),
					 _("Firewall running"));
	if (state == "halt")
		gnome_appbar_set_default (GNOME_APPBAR (appbar),
					_("All traffic halted"));
	if (state == "unknown") {
		if (running)
			gnome_appbar_set_default (GNOME_APPBAR (appbar),
						 _("Firewall running"));
		else
			gnome_appbar_set_default (GNOME_APPBAR (appbar),
						_("Firewall stopped"));
	}
}

static void
detect_netfilter (void)
{
	FILE *f;
	gchar buf[512];

	f = fopen ("/proc/version", "r");
	if (!f) {
		g_print ("Could not determine kernel version,"
			 "defaulting to 2.2/ipchains");
		return;
	}

	while (fgets (buf, 512, f) != NULL) {
		if (strstr (buf, " 2.4.") != NULL) {
			g_print ("NETFILTER detected\n");
			system ("/sbin/insmod ip_tables");
			NETFILTER = TRUE;
			fclose (f);
			return;
		}
	}
	g_print ("IPCHAINS detected\n");
	NETFILTER = FALSE;
	fclose (f);
}

/* [ clist_row_select ]
 * Attach current line data to a clist. Uggly.
 */
static void
clist_row_select (GtkWidget *clist, gint row, gint column,
		  GdkEventButton *event, gpointer data)
{
	gint *clistrow = GINT_TO_POINTER (row+1);
	gint *clistcol = GINT_TO_POINTER (column+1);

	gtk_object_set_data (GTK_OBJECT (clist), "row", clistrow);
	gtk_object_set_data (GTK_OBJECT (clist), "col", clistcol);

	if (clist != hitlogclist)
		dynclist =  (gpointer *) clist;
	else return;

	if (clist != denyallclist)
		gtk_clist_unselect_all (GTK_CLIST (denyallclist));
	if (clist != allowallclist)
		gtk_clist_unselect_all (GTK_CLIST (allowallclist));
	if (clist != allowsmclist)
		gtk_clist_unselect_all (GTK_CLIST (allowsmclist));
	if (clist != allowsaclist)
		gtk_clist_unselect_all (GTK_CLIST (allowsaclist));
}

/* [ clist_row_unselect ]
 * Clear current line data from a clist
 */
static void
clist_row_unselect (GtkWidget *clist)
{
	if (clist == (GtkWidget *)dynclist)
		dynclist = NULL;

	gtk_object_set_data (GTK_OBJECT (clist), "row", NULL);
	gtk_object_set_data (GTK_OBJECT (clist), "col", NULL);
}

/* [ hide_window_cb ]
 * Hides the main window, either when "destroyed" or docklet clicked.
 */
static void
hide_window_cb (GtkWidget *window, GdkEventButton* bevent)
{
	if (bevent->button != 3) {
		if (GTK_WIDGET_VISIBLE (window))
			gtk_widget_hide (window);
		else
			if (GTK_WIDGET_VISIBLE (hitlogclist))
				gtk_widget_show (window);
			else
				gtk_widget_show_all (window);

		if (running)
			set_panel_pic (statusrunning_xpm);
	}
}

/* [ stop_firewall ]
 * Flushes, zeroes and sets the default policy of the chains
 */
void
stop_firewall (void)
{
	gchar *soundfile;
	gchar *sbin;
	gchar *command = NULL;

	if (geteuid ()) {
		GtkWidget *dialog=gnome_error_dialog (_(
			"You must be root to manipulate the firewall."));
		gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
		return;
	}

	if (gnome_config_get_bool ("/firestarter/Druid/locatesbins=FALSE"))
		if (NETFILTER)
			sbin = g_strdup ("`which iptables`");
		else
			sbin = g_strdup ("`which ipchains`");
	else
		if (NETFILTER)
			sbin = g_strdup ("/sbin/iptables");
		else
			sbin = g_strdup ("/sbin/ipchains");

	if (NETFILTER) {
		command = g_strconcat (sbin, " -F;",
				       sbin, " -X;",
				       sbin, " -Z;",
			     	       sbin, " -P INPUT ACCEPT;",
				       sbin, " -P FORWARD ACCEPT;",
			               sbin, " -P OUTPUT ACCEPT;", NULL);
		system (command);
	}
	else {
		command = g_strconcat (sbin, " -X;",
				       sbin, " -F;",
				       sbin, " -Z;",
				       sbin, " -P input ACCEPT;",
				       sbin, " -P forward ACCEPT;",
				       sbin, " -P output ACCEPT;", NULL);
		system (command);
	}

	g_free (sbin);
	g_free (command);

	g_print (_("Firewall stopped, network traffic is now flowing freely\n"));
	running = FALSE;
	set_panel_pic (statustopped_xpm);
	appbar_update ("stop");

	if (gnome_config_get_bool ("/firestarter/Sounds/enablesounds=TRUE")) {
		soundfile = gnome_config_get_string (
			"/firestarter/Sounds/stopsoundfile=");
		gnome_sound_play (soundfile);
		g_free (soundfile);
	}
}

/* [ restart_firewall ]
 * Executes the firewall script
 */
void
restart_firewall (void)
{
	gchar *commandline = g_strjoin (NULL, "/bin/sh ",
		FIRESTARTER_RULES_DIR "/firestarter/firewall.sh",
		NULL);
	gchar *soundfile;

	if (geteuid ()) {
		GtkWidget *dialog=gnome_error_dialog (_(
			"You must be root to manipulate the firewall."));
		gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
		return;
	}

	system (commandline);
	g_print (_("Firewall script restarted\n"));
	running = TRUE;

	set_panel_pic (statusrunning_xpm);
	g_free (commandline);
	appbar_update ("start");

	if (gnome_config_get_bool ("/firestarter/Sounds/enablesounds=TRUE")) {
		soundfile = gnome_config_get_string (
			"/firestarter/Sounds/restartsoundfile");
		gnome_sound_play (soundfile);
		g_free (soundfile);
	}
}

/* [ halt_traffic ]
 * Flushes and sets all policys to deny
 */
void
halt_traffic (void)
{
	gchar *sbin;
	gchar *command;

	if (geteuid ()) {
		GtkWidget *dialog=gnome_error_dialog (_(
			"You must be root to manipulate the firewall."));
		gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
		return;
	}

	if (gnome_config_get_bool ("/firestarter/Druid/locatesbins=FALSE"))
		if (NETFILTER)
			sbin = g_strdup ("`which iptables`");
		else
			sbin = g_strdup ("`which ipchains`");
	else
		if (NETFILTER)
			sbin = g_strdup ("/sbin/iptables");
		else
			sbin = g_strdup ("/sbin/ipchains");

	if (NETFILTER) {
		command = g_strconcat (sbin, " -P INPUT DROP;",
				       sbin, " -P FORWARD DROP;",
				       sbin, " -P OUTPUT DROP;",
				       sbin, " -F;",
				       sbin, " -X;",
				       sbin, " -Z;", NULL);
		system (command);
	}
	else {
		command = g_strconcat (sbin, " -P input DENY;",
				       sbin, " -P forward DENY;",
				       sbin, " -P output DENY;",
				       sbin, " -X;",
				       sbin, " -F;",
				       sbin, " -Z;", NULL);
		system (command);
	}

	g_free (sbin);
	g_free (command);

	g_print (_("All traffic halted\n"));
	running = FALSE;
	set_panel_pic (statustopped_xpm);
	appbar_update ("halt");
}

/* [ clear_hitlist ]
 * Clears the log CList
 */
void
clear_hitlist (void)
{
	if (running)
		set_panel_pic (statusrunning_xpm);
	gtk_clist_clear (GTK_CLIST (hitlogclist));
	clist_row_unselect (hitlogclist);
}

static void
tail_and_resize (GtkWidget *clist)
{
	gint i, width;

	/* Optimize the clists column widths */
	for (i = 0; i < 4; i++) {
		width = gtk_clist_optimal_column_width (GTK_CLIST (hitlogclist), i);
		gtk_clist_set_column_width (GTK_CLIST (hitlogclist), i, width);
	}

	/* Move to the end of the clist */
	while (gtk_clist_get_text (GTK_CLIST (hitlogclist), i, 0, NULL))
		i++;
	gtk_clist_moveto (GTK_CLIST (hitlogclist), i-1, 0, 0.0, 0.0);
	gtk_widget_queue_resize	(window);
}

/* [ reload_hitlist ]
 * Reads the entire log file into the hitlogclist
 */
void
reload_hitlist (gpointer data)
{
	gchar buf[512];
	gchar *logpath = gnome_config_get_string (
		"/firestarter/Files/logfile=/var/log/messages");

	FILE *f;

	f = fopen (logpath, "r");
	if (!f) {
		g_print ("Error opening log file");
		return;
	}

	gtk_clist_freeze (GTK_CLIST (hitlogclist));
	gtk_clist_clear (GTK_CLIST (hitlogclist));
	while (fgets (buf, 512, f) != NULL) {
		g_print ("%s", buf);

		if (NETFILTER) {
			if (strstr (buf, "kernel: IN"))
				parse_hit_log (buf);
		} else
			if (strstr (buf, "DENY") || strstr (buf, "REJECT"))
				parse_hit_log (buf);
	}
	gtk_clist_thaw (GTK_CLIST (hitlogclist));
	tail_and_resize (hitlogclist);

	fclose (f);
}

/* [ poll_log_timeout ]
 * Polls the logfile every 700ms for change, if change, parse lines
 */
static int
poll_log_timeout (gpointer file)
{
	gchar buf[512];
	static gchar *key, *key2;
	gboolean sound = gnome_config_get_bool (
		"/firestarter/Sounds/enablesounds=TRUE");
	gchar *soundfile = gnome_config_get_string (
		"/firestarter/Sounds/hitsoundfile=");

	if (!key) {
		if (NETFILTER)
			key = g_strdup ("kernel: IN=");
		else {
			key = g_strdup ("DENY");
			if (!key2)
				key2 = g_strdup ("REJECT");
		}		
	}

	if (fgets (buf, 512, file) != NULL) {
		if ( (strstr (buf, key) || (!NETFILTER && strstr (buf, key2))) && parse_hit_log (buf)) {
			set_panel_pic (statuscaught_xpm);
			tail_and_resize (hitlogclist);

			if (sound) {
				gnome_sound_play (soundfile);
				g_free (soundfile);
			}
		}
	}
	return TRUE;
}

/* [ show_about ]
 * Creates the about dialog
 */
void
show_about (GtkWidget *widget, gpointer data)
{
	static GtkWidget *dialog = NULL;

	if (dialog != NULL) {
		g_assert (GTK_WIDGET_REALIZED (dialog));
		gdk_window_show (dialog->window);
		gdk_window_raise (dialog->window);
	}
	else {
		const gchar *authors[] = {"Tomas Junnonen <majix@sci.fi> - Main Developer, Maintainer",
			"Paul Drain <pd@cipherfunk.org> - Developer",
			NULL};

		dialog = gnome_about_new (
			"Firestarter", VERSION,
			"(C) 2000 Tomas Junnonen", authors,
			_("An all-in-one Linux firewall utility for GNOME.\n"
			"http://firestarter.sourceforge.net"),
			NULL);

		gtk_signal_connect (GTK_OBJECT (dialog), "destroy",
			GTK_SIGNAL_FUNC (gtk_widget_destroyed), &dialog);

		gtk_widget_show (dialog);
	}
}

/* [ set_panel_pic ]
 * Sets the picture on the docklet to *pixmap
 */
void
set_panel_pic (gchar *pixmap[])
{
	gtk_container_remove (GTK_CONTAINER (panelvbox), panelpic);
	panelpic = gnome_pixmap_new_from_xpm_d (pixmap);
	gtk_box_pack_end (GTK_BOX (panelvbox), panelpic, FALSE, FALSE, 2);
	gtk_widget_show (panelpic);
}

/* [ status_docklet_build_plug ]
 * Builds the docklet plug
 */
static void
status_docklet_build_plug (StatusDocklet *docklet, GtkWidget *plug)
{
	GtkWidget *eventbox;

	g_return_if_fail (docklet != NULL);
	g_return_if_fail (IS_STATUS_DOCKLET (docklet));

	eventbox = gtk_event_box_new ();
	gtk_widget_set_events (eventbox, GDK_BUTTON_PRESS_MASK);
	gtk_signal_connect_object (GTK_OBJECT (eventbox), "button_press_event",
		GTK_SIGNAL_FUNC (hide_window_cb), GTK_OBJECT (window));
	gtk_container_add (GTK_CONTAINER (plug), eventbox);

	panelvbox = gtk_vbox_new (FALSE, 5);
	gtk_container_add (GTK_CONTAINER (eventbox), panelvbox);

	if (gnome_config_get_bool ("/firestarter/Program/shutdown=FALSE")) {
		panelpic = gnome_pixmap_new_from_xpm_d (statustopped_xpm);
		running = FALSE;
	}
	else {
		panelpic = gnome_pixmap_new_from_xpm_d (statusrunning_xpm);
		running = TRUE;
	}

	gtk_box_pack_start (GTK_BOX (panelvbox), panelpic, FALSE, FALSE, 2);

	install_docklet_menu (eventbox);

	gtk_widget_show_all (plug);
}

/* [ pack_clist ]
 * Convenience function to pack a clist into a scrolled window and box
 */
static void
pack_clist (GtkWidget *clist, GtkWidget *box)
{
	GtkWidget *scrolled_win = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_win),
		GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);

	gtk_container_add (GTK_CONTAINER (scrolled_win), clist);

	gtk_box_pack_start (GTK_BOX (box), scrolled_win, TRUE, TRUE, 0);
}

/* [ prepare_clist ]
 * Convinience function that installs popups, signals and packs the clists
 */
static void
prepare_clist (GtkWidget *clist, GtkWidget *vbox)
{
	install_clist_popup (clist);

	gtk_signal_connect (GTK_OBJECT (clist), "select-row",
			    GTK_SIGNAL_FUNC (clist_row_select), NULL);
	gtk_signal_connect (GTK_OBJECT (clist), "unselect-row",
			    GTK_SIGNAL_FUNC (clist_row_unselect), NULL);

	if (clist == hitlogclist)
		gtk_signal_connect (GTK_OBJECT (clist), "button_press_event",
				    GTK_SIGNAL_FUNC (popup_menu), hitpopup);
	else {
		gtk_signal_connect (GTK_OBJECT (clist), "button_press_event",
				    GTK_SIGNAL_FUNC (popup_menu), dynlistpopup);
		pack_clist (clist, vbox);
	}
}

/* [ exit_firestarter ]
 * Quit firestater
 */
void
exit_firestarter (void)
{
	if (gnome_config_get_bool ("/firestarter/Program/shutdown=FALSE"))
		stop_firewall ();
	gtk_main_quit ();
}

/* [ closewindow ]
 * Quit or hide, determined by the config files
 */
void
closewindow (void)
{
	if (gnome_config_get_bool ("/firestarter/Program/exitonclose=FALSE"))
		exit_firestarter ();
	else
		gtk_widget_hide (window);
}

/* [ main ]
 * The main function, this is where it all begins
 */
int
main (int argc, char* argv[])
{
	GtkObject *status_dock;

	GtkWidget *notebook;
	GtkWidget *scrolled_win;
	GtkWidget *tablabel;

	gchar *hitlogtitles[4];

	GtkWidget *dynamicvbox;
	gchar *denyalltitles[1];
	gchar *allowtitles[1];
	gchar *allowsmtitles[1];
	gchar *allowsatitles[1];

	gchar *logpath;
	gint i;
	FILE *file;
	CORBA_Environment ev;
	CORBA_ORB orb;

	bindtextdomain (PACKAGE, GNOMELOCALEDIR);
	textdomain (PACKAGE);

	hitlogtitles[0] = _("Port");
	hitlogtitles[1] = _("Sent from");
	hitlogtitles[2] = _("Service");
	hitlogtitles[3] = _("Time");

	denyalltitles[0] = _("Deny all connections from");
	allowtitles[0] = _("Allow all connections from");
	allowsmtitles[0] = _("Open service to machine");
	allowsatitles[0] = _("Open service to anyone");

	CORBA_exception_init (&ev);
	orb = gnome_CORBA_init (PACKAGE, VERSION, &argc, argv, 0, &ev);

	window = gnome_app_new (PACKAGE, "Firestarter");
	detect_netfilter ();

	ttips = gtk_tooltips_new ();

/* Set up the docklet */
	status_dock = status_docklet_new ();
	gtk_signal_connect (GTK_OBJECT (status_dock), "build_plug",
			    GTK_SIGNAL_FUNC (status_docklet_build_plug), NULL);
	status_docklet_run (STATUS_DOCKLET (status_dock));

/* Set up the window */
	gtk_signal_connect (GTK_OBJECT (window), "delete_event",
			    GTK_SIGNAL_FUNC (closewindow), NULL);

	gnome_window_icon_set_from_file (GTK_WINDOW (window),
					 "/usr/share/pixmaps/firestarter.png");

	install_menus_and_toolbar (window);

	notebook = gtk_notebook_new ();
	gnome_app_set_contents (GNOME_APP (window), notebook);

/* set up hitlogclist */
	scrolled_win = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_win),
					GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);

	tablabel = gtk_label_new (_("Firewall hits"));
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), scrolled_win,
				  tablabel);

	for (i = 0; i < 4;i++)
		 hitlogtitles[i] = _(hitlogtitles[i]);
	hitlogclist = gtk_clist_new_with_titles (4, hitlogtitles);
	prepare_clist (hitlogclist, NULL);
	gtk_container_add (GTK_CONTAINER (scrolled_win), hitlogclist);

/* set up the clists on the dynamic rules page */
	tablabel = gtk_label_new (_("Dynamic rules"));
	dynamicvbox = gtk_vbox_new (FALSE, 0);
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), dynamicvbox,
				  tablabel);

	denyalltitles[0] = N_(denyalltitles[0]);
	denyallclist = gtk_clist_new_with_titles (1, denyalltitles);
	gtk_object_set_data (GTK_OBJECT (denyallclist), "path",
		FIRESTARTER_RULES_DIR "/firestarter/deny-all");
	prepare_clist (denyallclist, dynamicvbox);

	allowtitles[0] = N_(allowtitles[0]);
	allowallclist = gtk_clist_new_with_titles (1, allowtitles);
	gtk_object_set_data (GTK_OBJECT (allowallclist), "path",
		FIRESTARTER_RULES_DIR "/firestarter/allow-all");
	prepare_clist (allowallclist, dynamicvbox);

	allowsmtitles[0] = N_(allowsmtitles[0]);
	allowsmclist = gtk_clist_new_with_titles (1, allowsmtitles);
	gtk_object_set_data (GTK_OBJECT (allowsmclist), "path",
		FIRESTARTER_RULES_DIR "/firestarter/allow-service-machine");
	prepare_clist (allowsmclist, dynamicvbox);

	allowsatitles[0] = N_(allowsatitles[0]);
	allowsaclist = gtk_clist_new_with_titles (1, allowsatitles);
	gtk_object_set_data (GTK_OBJECT (allowsaclist), "path",
		FIRESTARTER_RULES_DIR "/firestarter/allow-service-all");
	prepare_clist (allowsaclist, dynamicvbox);

	logpath = gnome_config_get_string (
		"/firestarter/Files/logfile=/var/log/messages");

	/* Check if user has read access to the log file */
	if (fopen (logpath, "r") == NULL) {
		gchar *string;
		GtkWidget *dialog;

		string = g_strjoin (NULL, _(
			"Log file not found or access denied.\n"
			"Firewall log monitoring disabled.\n\n"
			"You can configure the path to the log file in the "
			"\"preferences\" section."), NULL);
		dialog = gnome_error_dialog (string);
		gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
		gtk_widget_set_sensitive (GTK_WIDGET (hitlogclist), FALSE);

		g_free (string);
	}
	else {
		gchar buf[512];

		file = fopen (logpath, "r");
		/* Move to the end of the file */
		while (fgets (buf, 512, file) != NULL) continue;
		gtk_timeout_add (500, poll_log_timeout, file);
	}

	mkdir (FIRESTARTER_RULES_DIR "/firestarter", 00700);

	/* populate the clists on the dynamic rules notebookpage*/
	parse_allow_and_deny_all ();
	parse_allow_service_machine ();
	parse_allow_service_all ();

	appbar_update ("unknown");

	if (!gnome_config_get_bool ("/firestarter/Program/starthidden=FALSE"))
		gtk_widget_show_all (window);

	check_version ();

	if (gnome_config_get_bool ("/firestarter/Druid/firsttime=TRUE")) {
		stop_firewall ();
		rundruid ();
	}

	if (gnome_config_get_bool ("/firestarter/Program/startup=TRUE"))
		restart_firewall ();

	gtk_main ();

	CORBA_exception_free (&ev);

	return 0;
}
