/*---[ ipchains-script.c ]-----------------------------------------------
 * Copyright (C) 2000 Tomas Junnonen (majix@sci.fi)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Functions to write the ipchains shell scripts
 *--------------------------------------------------------------------*/

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "druid.h"
#include "ipchains-script.h"

/* [ write_ipchains_script ]
 * Creates the ipchains shell script
 */
void
write_ipchains_script (Druid *data)
{
	gboolean masq = GTK_WIDGET_IS_SENSITIVE (GTK_WIDGET (data->masq));
	gboolean tos = GTK_WIDGET_IS_SENSITIVE (GTK_WIDGET (data->tos));

	gchar *scriptpath = FIRESTARTER_RULES_DIR "/firestarter/firewall.sh";

	FILE *script = fopen (scriptpath, "w");

	chmod (scriptpath, 00700);

/* Init variables and external files */
	fprintf (script, "#!/bin/sh\n");
	fprintf (script, "#Generated by Firestarter " VERSION ", IPCHAINS in use\n\n");

	if (gnome_config_get_bool ("/firestarter/Druid/locatesbins=FALSE"))
		fprintf (script, "IPC=`which ipchains`\nMPB=`which modprobe`\nLSM=`which lsmod`\n\n");
	else
		fprintf (script, "IPC=/sbin/ipchains\nMPB=/sbin/modprobe\nLSM=/sbin/lsmod\n\n");

	/* The network interface */
	fprintf (script, "IF=%s\n", gtk_entry_get_text (GTK_ENTRY (data->extdevice)));
	/* The internal network interface */
	if (masq)
		fprintf (script, "INIF=%s\n", gtk_entry_get_text (GTK_ENTRY (data->intdevice)));

	/* IP address of interface */
	fprintf (script, "IP=`/sbin/ifconfig $IF | grep inet | cut -d : -f 2 | cut -d \\  -f 1`\n");
	/* Netmask/net of external interface */
	fprintf (script, "MASK=`/sbin/ifconfig $IF | grep Mas | cut -d : -f 4`\n");
	fprintf (script, "NET=$IP/$MASK\n");
	/* Internal network*/
	if (masq) {
		if (GTK_WIDGET_IS_SENSITIVE (GTK_WIDGET (data->auto_int_ip)))
			fprintf (script, "INNET=%s\n\n", gtk_entry_get_text (GTK_ENTRY (data->intrange)));
		else {
			fprintf (script, "INIP=`/sbin/ifconfig $INIF | grep inet | cut -d : -f 2 | cut -d \\  -f 1`\n");
			fprintf (script, "INMASK=`/sbin/ifconfig $INIF | grep Mas | cut -d : -f 4`\n");
			fprintf (script, "INNET=$INIP/$INMASK\n");
		}
	}

	if (gnome_config_get_bool ("/firestarter/Druid/selectdeny=TRUE"))
		fprintf (script, "STOP=DENY\n\n");
	else
		fprintf (script, "STOP=REJECT\n\n");
		
	if (gtk_toggle_button_get_active (GTK_RADIO_BUTTON (data->tos_option_throughput)))
			fprintf (script, "TOSOPT=\"0x01 0x08\"\n\n");
	if (gtk_toggle_button_get_active (GTK_RADIO_BUTTON (data->tos_option_reliability)))
			fprintf (script, "TOSOPT=\"0x01 0x04\"\n\n");
	if (gtk_toggle_button_get_active (GTK_RADIO_BUTTON (data->tos_option_delay)))
			fprintf (script, "TOSOPT=\"0x01 0x10\"\n\n");
			

	fprintf (script, "#Delete user made chains. Flush and zero any existing chains.\n");
	fprintf (script, "$IPC -X\n$IPC -F\n$IPC -Z\n\n");
	
	fprintf (script, "#Set default chain policy to $STOP.\n");
	fprintf (script, "$IPC -P input $STOP\n$IPC -P output $STOP\n$IPC -P forward $STOP\n\n");

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->pppcheck))) {
		fprintf (script, "#Modules: Loading bsdcomp module\n");
        fprintf (script, "if ! ( $LSM | /bin/grep bsd_comp > /dev/null ); then\n");
		fprintf (script, "$MPB bsd_comp\n");
	}

	/* Pipe in the modrules files */
	fprintf (script, "#Deny all traffic on these ports, without logging\n");
	fprintf (script, "if [ -e "FIRESTARTER_RULES_DIR "/firestarter/do-not-log-ports ]\n then\n"
		"source " FIRESTARTER_RULES_DIR "/firestarter/do-not-log-ports\nfi\n");

	fprintf (script, "#Deny all traffic from these machines\n");
	fprintf (script, "source " FIRESTARTER_RULES_DIR "/firestarter/deny-all\n");
	fprintf (script, "#Allow all traffic from these machines\n");
	fprintf (script, "source " FIRESTARTER_RULES_DIR "/firestarter/allow-all\n");
	fprintf (script, "#Allow a specific service to a specific machine\n");
	fprintf (script, "source " FIRESTARTER_RULES_DIR "/firestarter/allow-service-machine\n");
	fprintf (script, "#Allow a specific service to everyone\n");
	fprintf (script, "source " FIRESTARTER_RULES_DIR "/firestarter/allow-service-all\n\n");

/* GOOD IDEA(tm) RULES */
	fprintf (script, "#Allow all traffic on the loopback interface\n");
	fprintf (script, "$IPC -A input -i lo -s 0/0 -d 0/0 -j ACCEPT\n");
	fprintf (script, "$IPC -A output -i lo -s 0/0 -d 0/0 -j ACCEPT\n\n");

	fprintf (script, "#Turn on source address verification in kernel\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/conf/all/rp_filter ]\n then\n"
	"  for f in /proc/sys/net/ipv4/conf/*/rp_filter\n  do\n   echo 2 > $f\n  done\nfi\n\n");

	fprintf (script, "#Turn on syn cookies protection in kernel\n");	
	fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_syncookies ]\n then\n"
	"  echo 1 > /proc/sys/net/ipv4/tcp_syncookies\nfi\n\n");

	fprintf (script, "#ICMP Dead Error Messages protection\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/icmp_ignore_bogus_error_responses ]\n then\n"
	"  echo 1 > /proc/sys/net/ipv4/icmp_ignore_bogus_error_responses\nfi\n\n");

	fprintf (script, "#ICMP Broadcasting protection\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/icmp_echo_ignore_broadcasts ]\n then\n"
	"  echo 1 > /proc/sys/net/ipv4/icmp_echo_ignore_broadcasts\nfi\n\n");

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->pppcheck))) {
		fprintf (script, "#Turn on dynamic TCP/IP address hacking\n");
		fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_dynaddr ]\n then\n"
		"  echo 1 > /proc/sys/net/ipv4/ip_dynaddr\nfi\n\n");
	} else {
		fprintf (script, "#Turn off dynamic TCP/IP address hacking\n");
		fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_dynaddr ]\n then\n"
		"  echo 0 > /proc/sys/net/ipv4/ip_dynaddr\nfi\n\n");	
	}

/* ICMP RULES */

if GTK_WIDGET_IS_SENSITIVE (data->icmp)
{
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_echo))) {
		fprintf (script, "#ICMP: Ping Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type echo-request -j REJECT\n");
	}
		else {
		fprintf (script, "#ICMP: Ping Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type echo-request -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type echo-reply -j ACCEPT\n");
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_traceroute))) {
		fprintf (script, "#ICMP: Traceroute Requests\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 33434 -j REJECT -l\n");
	}
		else {
		fprintf (script, "#ICMP: Traceroute Requests\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 33434 -j ACCEPT -l\n");
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_tracert))) {
		fprintf (script, "#ICMP: MS Traceroute Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type destination-unreachable -j REJECT -l\n");
	}
		else {
		fprintf (script, "#ICMP: MS Traceroute Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type destination-unreachable -j ACCEPT -l\n");
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_unreach))) {
		fprintf (script, "#ICMP: Unreachable Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type host-unreachable -j REJECT -l\n");
	}
		else {
		fprintf (script, "#ICMP: Unreachable Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type host-unreachable -j ACCEPT -l\n");
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_timestamp))) {
		fprintf (script, "#ICMP: Timestamping Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type timestamp-request -j REJECT -l\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type timestamp-reply -j REJECT -l\n");
	}
		else {
		fprintf (script, "#ICMP: Timestamping Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type timestamp-request -j ACCEPT -l\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type timestamp-reply -j ACCEPT -l\n");
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_masking))) {
		fprintf (script, "#ICMP: Address Masking\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type address-mask-request -j REJECT -l\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type address-mask-reply -j REJECT -l\n");
	}
		else {
		fprintf (script, "#ICMP: Address Masking\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type address-mask-request -j ACCEPT -l\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type address-mask-reply -j ACCEPT -l\n");
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_redir))) {
		fprintf (script, "#ICMP: Redirection Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type redirect -j REJECT -l\n");
	}
		else {
		fprintf (script, "#ICMP: Redirection Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type redirect -j ACCEPT -l\n");
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_quench))) {
		fprintf (script, "#ICMP: Source Quench Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type source-quench -j REJECT -l\n");
	}
		else {
		fprintf (script, "#ICMP: Source Quench Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type source-quench -j ACCEPT -l\n");
	}
}
else {
		fprintf (script, "#ICMP: Ping Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type echo-request -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type echo-reply -j ACCEPT\n");
		fprintf (script, "#ICMP: Traceroute Requests\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 33434 -j REJECT -l\n");
		fprintf (script, "#ICMP: MS Traceroute Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type destination-unreachable -j ACCEPT -l\n");
		fprintf (script, "#ICMP: Unreachable Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type host-unreachable -j ACCEPT -l\n");
		fprintf (script, "#ICMP: Timestamping Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type timestamp-request -j ACCEPT -l\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type timestamp-reply -j ACCEPT -l\n");
		fprintf (script, "#ICMP: Address Masking\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type address-mask-request -j ACCEPT -l\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type address-mask-reply -j ACCEPT -l\n");
		fprintf (script, "#ICMP: Redirection Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type redirect -j ACCEPT -l\n");
		fprintf (script, "#ICMP: Source Quench Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type source-quench -j ACCEPT -l\n\n");
}

/* MASQ RULES */
	if (masq) {

		fprintf (script, "#Loading MASQ modules\n");
		fprintf (script, "if ! ( $LSM | /bin/grep masq_ftp > /dev/null ); then\n");
		fprintf (script, "$MPB ip_masq_ftp\nfi\n");
		fprintf (script, "if ! ( $LSM | /bin/grep masq_irc > /dev/null ); then\n");
		fprintf (script, "$MPB ip_masq_irc\nfi\n");
		fprintf (script, "if ! ( $LSM | /bin/grep masq_quake > /dev/null ); then\n");
		fprintf (script, "$MPB ip_masq_quake\nfi\n");
		fprintf (script, "if ! ( $LSM | /bin/grep masq_raudio > /dev/null ); then\n");
		fprintf (script, "$MPB ip_masq_raudio\nfi\n\n");

		fprintf (script, "#Initialize MASQ timeout\n");
		fprintf (script, "$IPC -M -S 14400 60 600\n\n");

		fprintf (script, "#Turn on IP forwarding\n");
		fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_forward ]\n then\n"
		"  echo 1 > /proc/sys/net/ipv4/ip_forward\nfi\n\n");

		fprintf (script, "#Handle dynamic IP masquerading\n");
		fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_dynaddr ]\n then\n"
		"  echo 1 > /proc/sys/net/ipv4/ip_dynaddr\nfi\n\n");

		fprintf (script, "#Don't masq internal-internal traffic\n");
		fprintf (script, "$IPC -A forward -s $INNET -d $INNET -j ACCEPT\n\n");

		fprintf (script, "#Don't masq external interface traffic\n");
		fprintf (script, "$IPC -A forward -s $NET -d 0/0 -j ACCEPT\n\n");

		fprintf (script, "#Masquerade local net traffic\n");
		fprintf (script, "$IPC -A forward -s $INNET -d 0/0 -j MASQ\n\n");

		fprintf (script, "#Set forward policy to deny\n");
		fprintf (script, "$IPC -P forward $STOP\n\n");

		fprintf (script, "#Allow traffic from internal network going anywhere\n");
		fprintf (script, "$IPC -A input -s $INNET -d 0/0 -j ACCEPT\n");
		fprintf (script, "$IPC -A output -s $INNET -d 0/0 -j ACCEPT\n");
		fprintf (script, "$IPC -A output -p icmp -s $INNET -d 0/0 -j ACCEPT\n\n");
	}

	
/* TOS RULES */
if (tos) {

	fprintf (script, "#Altering Type of Service (ToS) flags\n\n");
	
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->tos_client))) {
	fprintf (script, "#ToS: Client Applications\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 20:21 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 22 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 68 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 80 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 443 -t $TOSOPT\n\n");
}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->tos_server))) {
	fprintf (script, "#ToS: Server Applications\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 20:21 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 22 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 25 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 53 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 67 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 80 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 110 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 143 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 443 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 1812 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 1813 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 2401 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 8080 -t $TOSOPT\n");
}
		
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->tos_X))) {
	fprintf (script, "#ToS: The X Window System\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 22 -t 0x01 0x10\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 6000:6015 -t 0x01 0x08\n\n");
 }
}
	fprintf (script, "#Block nonroutable IPs\n");	
	fprintf (script, "$IPC -A input -s 127.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 128.66.0.0/16 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 172.16.0.0/12 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 192.168.0.0/16 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 197.0.0.0/16 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 201.0.0.0/8 -d $NET -i $IF -j $STOP\n");

	fprintf (script, "#Block Back Orifice\n");	
	fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 31337 -j $STOP -l\n");
	fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 31337 -j $STOP -l\n\n");

	fprintf (script, "#Block NetBus\n");	
	fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 12345:12346 -j $STOP -l\n");
	fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 12345:12346 -j $STOP -l\n\n");

	fprintf (script, "#Block Trin00\n");
	fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 1524 -j $STOP -l\n");
	fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 27665 -j $STOP -l\n");
	fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 27444 -j $STOP -l\n");
	fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 31335 -j $STOP -l\n\n");

	fprintf (script, "#Block Multicast\n");	
	fprintf (script, "$IPC -A input -s 224.0.0.0/8 -d 0/0 -j $STOP -l\n");
	fprintf (script, "$IPC -A input -s 0/0 -d 224.0.0.0/8 -j $STOP -l\n\n");

/* RULES BASED ON WIZARD SELECTIONS */

	/* Allow DHCP */
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->dhcpcheck))) {
		fprintf (script, "#DHCP\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 67:68 -i $IF -j ACCEPT\n\n");
	}

if GTK_WIDGET_IS_SENSITIVE (data->services)
{
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->ftp))) {
		fprintf (script, "#FTP\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 20 -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 21 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->ssh))) {
		fprintf (script, "#SSH\n");		
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 22 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->telnet))) {
		fprintf (script, "#Telnet\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 23 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->smtp))) {
		fprintf (script, "#SMTP\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 25 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->dns))) {
		fprintf (script, "#DNS\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 53 -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 53 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->finger))) {
		fprintf (script, "#Finger\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 79 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->www))) {
		fprintf (script, "#HTTP\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 80 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->sslweb))) {
		fprintf (script, "#SSL HTTP\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 443 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->pop))) {
		fprintf (script, "#POP\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 110 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->imap))) {
		fprintf (script, "#IMAP\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 143 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->ident))) {
		fprintf (script, "#IDENT\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 113 -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 113 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->nntp))) {
		fprintf (script, "#NNTP\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 119 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->ntp))) {
		fprintf (script, "#NTP\n");		
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 123 -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 123 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->samba))) {
		fprintf (script, "#SAMBA\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 137:139 -i $IF -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 137:139 -i $IF -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 445 -i $IF -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 445 -i $IF -j ACCEPT\n\n");
 	}
	else {
		fprintf (script, "#Block SAMBA (Silently Rejected)\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 137:139 -i $IF -j REJECT -l\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 137:139 -i $IF -j REJECT -l\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 445 -i $IF -j REJECT -l\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 445 -i $IF -j REJECT -l\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->routed))) {
		fprintf (script, "#Routed\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 520 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->nfs))) {
		fprintf (script, "#NFS\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 2049 -i $IF -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 2049 -i $IF -j ACCEPT\n\n");
	}
	else {
		fprintf (script, "#Block NFS\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 2049 -i $IF -j $STOP -l\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 2049 -i $IF -j $STOP -l\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->x))) {
		fprintf (script, "#Xwindows\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 6000:6015 -i $IF -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 6000:6015 -i $IF -j ACCEPT\n\n");
	}
	else {
		fprintf (script, "#Block Xwindows\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 6000:6015 -i $IF -j $STOP -l\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 6000:6015 -i $IF -j $STOP -l\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->dhcp))) {
		fprintf (script, "#DHCP\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 67:68 -i $IF -j ACCEPT\n\n");
	}
}
else {
	fprintf (script, "#Block NFS\n");
	fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 2049 -i $IF -j $STOP -l\n");
	fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 2049 -i $IF -j $STOP -l\n\n");
	
	fprintf (script, "#Block SAMBA (Silently Rejected)\n");
	fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 137:139 -i $IF -j REJECT -l\n");
	fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 137:139 -i $IF -j REJECT -l\n");
	fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 445 -i $IF -j REJECT -l\n");
	fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 445 -i $IF -j REJECT -l\n\n");
}


/* REQUIRED RULES */
	fprintf (script, "#Allow ICMP Output\n");
	fprintf (script, "$IPC -A output -p icmp -s $NET -d 0/0 -j ACCEPT\n\n");

	fprintf (script, "#Open ports for inbound established connections\n");
	fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 20 -j ACCEPT\n");
	fprintf (script, "$IPC -A input ! -y -p tcp -s 0/0 -d $NET 1023:65535 -j ACCEPT\n");
	fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 1023:65535 -j ACCEPT\n\n");
	
	if (gnome_config_get_bool ("/firestarter/Druid/outfilter=TRUE")) {
		fprintf (script, "#Open ports for outbound established connections\n");
		fprintf (script, "$IPC -A output -p tcp -s $NET -d 0/0 1023:65535 -j ACCEPT\n");
		fprintf (script, "$IPC -A output -p udp -s $NET -d 0/0 1023:65535 -j ACCEPT\n\n");
	}
	else {
		fprintf (script, "$IPC -A output -j ACCEPT\n\n");
	}
	
	fprintf (script, "#Deny everything not let through earlier\n");	
	fprintf (script, "$IPC -A input -j $STOP -l\n");

	fclose (script);

	g_print (_("Firewall script saved as %s\n"), scriptpath);
}
