/* gasql.c
 *
 * Copyright (C) 1999-2001 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <gnome.h>
#include <tree.h>		/* libxml/tree.h ? */
#include <parser.h>		/* libxml/tree.h ? */
#include <liboaf/liboaf.h>
#include <bonobo.h>

#include "session.h"
#include "sqlaccess.h"
#include "sqlqueryres.h"
#include "sqldb.h"
#include "sqlwiddbtree.h"
#include "mainpagetable.h"
#include "mainpageseq.h"
#include "mainpagequery.h"
#include "sqlquery.h"
#include "sqlqueryedit.h"

/* basic and simple configuration of the general data */
#include "gasql_conf.h"

/* main interface callbacks */
#include "interface_cb.h"
#include "integrity.h"

#include <gnome-db.h>

static void todo_cb (GtkWidget * widget, gASQL_Main_Config * conf);
static void about_cb (GtkWidget * widget, gASQL_Main_Config * conf);
static gint delete_event_cb (GtkWidget * wid, GdkEvent * event,
			     gASQL_Main_Config * conf);
static void add_toolbar (GnomeApp * app, gASQL_Main_Config * conf);
static void add_menus (GnomeApp * app, gASQL_Main_Config * conf);
static gASQL_Main_Config *prepare_conf ();
static gint prepare_app (gASQL_Main_Config * conf);
static gint register_icon (gchar * name, gchar * filename);

extern int xmlDoValidityCheckingDefaultValue;

#ifdef debug
/* DEBUG FUNCTIONS */
static void
debug_test_cb (GtkWidget * widget, gASQL_Main_Config * conf)
{

}

void
debug_dump_cb (GtkWidget * widget, gASQL_Main_Config * conf)
{
	sql_db_dump_tables (conf->db);
	sql_db_dump_links (conf->db);
	sql_data_type_show_types (conf->srv->data_types);
	sql_data_function_show_functions (conf->srv->data_functions);
	sql_data_aggregate_show_aggregates (conf->srv->data_aggregates);
}

void
debug_mark_cb (GtkWidget * widget, gASQL_Main_Config * conf)
{
	static int mark = 0;

	g_print ("\n\n------------------------------------------------------- %d\n\n", mark++);
}

/* END OF DEBUG FUNCTIONS */
#endif

static void
todo_cb (GtkWidget * widget, gASQL_Main_Config * conf)
{
	gnome_ok_dialog_parented (_("Function still non implemented\n"),
				  GTK_WINDOW (conf->app));
}

static void
about_cb (GtkWidget * widget, gASQL_Main_Config * conf)
{
	static GtkWidget *dialog = NULL;

	if (dialog != NULL) {
		g_assert (GTK_WIDGET_REALIZED (dialog));
		gdk_window_show (dialog->window);
		gdk_window_raise (dialog->window);
	}
	else {
		const gchar *authors[] = {
			"Vivien Malerba <malerba@gnome-db.org>",
			"Gerhard Dieringer <DieringG@eba-haus.de> German translations",
			"Mauro Colorio <linuxbox@interfree.it> Italian translations",
			"Ali Pakkan <apakkan@hotmail.com> Turk translations",
			NULL
		};
		dialog = gnome_about_new (_("gASQL"), VERSION,
					  "(C) 1999-2001 Vivien Malerba",
					  authors,
					  _
					  ("A Database admin tool for any SQL database "
					   "accessible with the gnome-db module."),
					  PIXMAPDIR "/gasql.png");
		gtk_signal_connect (GTK_OBJECT (dialog), "destroy",
				    GTK_SIGNAL_FUNC (gtk_widget_destroyed),
				    &dialog);
		gnome_dialog_set_parent (GNOME_DIALOG (dialog),
					 GTK_WINDOW (conf->app));
		gtk_widget_show (dialog);
	}
}


static gint
delete_event_cb (GtkWidget * wid, GdkEvent * event, gASQL_Main_Config * conf)
{
	/* quit with confirmation */
	quit_cb (NULL, conf);

	/* we don't want the "destroy" signal to be emitted now */
	return TRUE;
}

static void
add_toolbar (GnomeApp * app, gASQL_Main_Config * conf)
{
	GnomeUIInfo toolbar[] = {
		{
		 GNOME_APP_UI_ITEM, N_("Connect"),
		 N_("Opens the SQL server connection"),
		 sql_conn_open_cb, conf, NULL,
		 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_EXEC,
		 0, 0, NULL},
		{
		 GNOME_APP_UI_ITEM, N_("Disconnect"),
		 N_("Closes the SQL server connection"),
		 sql_conn_close_cb, conf, NULL,
		 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_EXEC,
		 0, 0, NULL},
		{
		 GNOME_APP_UI_ITEM, N_("Relations"), N_("Relations Scheme"),
		 sql_show_relations_cb, conf, NULL,
		 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_ABOUT,
		 0, 0, NULL},
		GNOMEUIINFO_END
	};

	gnome_app_create_toolbar (app, toolbar);
	/* widgets to be shown or hidden depending on the connexion state */
	gasql_conf_register_show_on_connect (conf, (toolbar[1]).widget);
	gasql_conf_register_show_on_disconnect (conf, (toolbar[0]).widget);
	/*gasql_conf_register_sensitive_on_disconnect(conf, (toolbar[2]).widget); */
	gasql_conf_register_sensitive_on_connect (conf, (toolbar[2]).widget);	/*3! */
}



static void
add_menus (GnomeApp * app, gASQL_Main_Config * conf)
{
	/* file menu */
	GnomeUIInfo file_menu[] = {
		GNOMEUIINFO_MENU_NEW_ITEM (N_("_New Workspace"),
					   N_
					   ("Switch to a new workspace to work on "
					    "another database"), file_new_cb,
					   conf),
		GNOMEUIINFO_MENU_OPEN_ITEM (file_open_cb, conf),
		GNOMEUIINFO_MENU_CLOSE_ITEM (file_close_cb, conf),
		GNOMEUIINFO_MENU_SAVE_ITEM (file_save_cb, conf),
		GNOMEUIINFO_MENU_SAVE_AS_ITEM (file_save_as_cb, conf),
		{
		 GNOME_APP_UI_ITEM, N_("Export"), NULL, todo_cb,
		 conf, NULL,
		 GNOME_APP_PIXMAP_NONE, NULL,
		 0, 0, NULL},
		GNOMEUIINFO_MENU_PRINT_SETUP_ITEM (printer_setup_cb, conf),
		GNOMEUIINFO_MENU_EXIT_ITEM (quit_cb, conf),
		GNOMEUIINFO_END
	};

	/* plugins menu */
	GnomeUIInfo plugins_menu[] = {
		{
		 GNOME_APP_UI_ITEM, N_("Rescan list"), NULL,
		 rescan_display_plugins_cb,
		 conf, NULL,
		 GNOME_APP_PIXMAP_NONE, NULL,
		 0, 0, NULL},
		GNOMEUIINFO_MENU_PREFERENCES_ITEM (config_display_plugins_cb,
						   conf),
		GNOMEUIINFO_END
	};

	/* users management menu */
	GnomeUIInfo users_menu[] = {
		{
		 GNOME_APP_UI_ITEM, N_("Users"), NULL,
		 users_settings_cb,
		 conf, NULL,
		 GNOME_APP_PIXMAP_NONE, NULL,
		 0, 0, NULL},
		{
		 GNOME_APP_UI_ITEM, N_("Users groups"), NULL,
		 users_groups_cb,
		 conf, NULL,
		 GNOME_APP_PIXMAP_NONE, NULL,
		 0, 0, NULL},
		{
		 GNOME_APP_UI_ITEM, N_("Users access settings"), NULL,
		 users_access_cb,
		 conf, NULL,
		 GNOME_APP_PIXMAP_NONE, NULL,
		 0, 0, NULL},
		GNOMEUIINFO_END
	};


	/* settings menu */
	GnomeUIInfo settings_menu[] = {
		{
		 GNOME_APP_UI_ITEM, N_("Database Manager"),
		 N_("The GNOME-DB manager"),
		 run_gnomedb_manager_cb, conf,
		 NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_INDEX,
		 0, 0, NULL},
		GNOMEUIINFO_SUBTREE (_("Users management"), &users_menu),
		GNOMEUIINFO_SUBTREE (_("Plugins"), &plugins_menu),
		GNOMEUIINFO_END
	};

	/* help menu */
	GnomeUIInfo help_menu[] = {
		GNOMEUIINFO_HELP (_("gASQL")),
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_MENU_ABOUT_ITEM (about_cb, conf),
		GNOMEUIINFO_END
	};

	/* connection menu */
	GnomeUIInfo conn_menu[] = {
		{
		 GNOME_APP_UI_ITEM, N_("Connection preferences"), NULL,
		 options_config_cb,
		 conf, NULL,
		 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_PREFERENCES,
		 0, 0, NULL},
		{
		 GNOME_APP_UI_ITEM, N_("Connect Sql server"), NULL,
		 sql_conn_open_cb,
		 conf, NULL,
		 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_EXEC,
		 0, 0, NULL},
		{
		 GNOME_APP_UI_ITEM, N_("Disconnect Sql server"), NULL,
		 sql_conn_close_cb,
		 conf, NULL,
		 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_EXEC,
		 0, 0, NULL},
		GNOMEUIINFO_END
	};

	/* database menu */
	GnomeUIInfo db_menu[] = {
		{
		 GNOME_APP_UI_ITEM, N_("Relations Scheme"), NULL,
		 sql_show_relations_cb,
		 conf, NULL,
		 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_ABOUT,
		 0, 0, NULL},
		GNOMEUIINFO_SEPARATOR,
		{
		 GNOME_APP_UI_ITEM, N_("Integrity check"), NULL,
		 integrity_check_cb,
		 conf, NULL,
		 GNOME_APP_PIXMAP_NONE, NULL,
		 0, 0, NULL},
		GNOMEUIINFO_SEPARATOR,
		{
		 GNOME_APP_UI_ITEM, N_("System informations"),
		 N_("View the data types, functions, "
		    "aggregates,..."), sql_data_view_cb,
		 conf, NULL,
		 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_PROPERTIES,
		 0, 0, NULL},
		{
		 GNOME_APP_UI_ITEM, N_("Refresh memory structure"),
		 N_("Updates the memory "
		    "representation\nof the database's structure"),
		 sql_mem_update_cb,
		 conf, NULL,
		 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_REFRESH,
		 0, 0, NULL},
		{
		 GNOME_APP_UI_ITEM, N_("Create a new DB"),
		 N_("Creates a new database"), todo_cb,
		 conf, NULL,
		 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_NEW,
		 0, 0, NULL},
		GNOMEUIINFO_END
	};

#ifdef debug
	/* debug menu */
	GnomeUIInfo debug_menu[] = {
		{
		 GNOME_APP_UI_ITEM, "Dump DB Structure", NULL, debug_dump_cb,
		 conf, NULL,
		 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_CONVERT,
		 0, 0, NULL},
		{
		 GNOME_APP_UI_ITEM, "Insert Mark in output", NULL,
		 debug_mark_cb,
		 conf, NULL,
		 GNOME_APP_PIXMAP_NONE, NULL,
		 0, 0, NULL},
		{
		 GNOME_APP_UI_ITEM, "Current test", NULL, debug_test_cb,
		 conf, NULL,
		 GNOME_APP_PIXMAP_NONE, NULL,
		 0, 0, NULL},
		GNOMEUIINFO_END
	};
#endif

	/* ALL menus */
	GnomeUIInfo main_menu[] = {
		GNOMEUIINFO_MENU_FILE_TREE (file_menu),
		GNOMEUIINFO_SUBTREE (N_("Connection"), &conn_menu),
		GNOMEUIINFO_SUBTREE (N_("Data Base"), &db_menu),
		GNOMEUIINFO_MENU_SETTINGS_TREE (&settings_menu),
#ifdef debug
		GNOMEUIINFO_SUBTREE ("Debug", &debug_menu),
#endif
		GNOMEUIINFO_MENU_HELP_TREE (help_menu),
		GNOMEUIINFO_END
	};

	gnome_app_create_menus (app, main_menu);

	/* widgets to be shown or hidden depending on the connexion state */
	gasql_conf_register_show_on_connect (conf, (db_menu[0]).widget);
	gasql_conf_register_show_on_connect (conf, (db_menu[1]).widget);
	gasql_conf_register_show_on_connect (conf, (db_menu[2]).widget);
	gasql_conf_register_show_on_connect (conf, (db_menu[3]).widget);
	gasql_conf_register_show_on_connect (conf, (db_menu[4]).widget);
	gasql_conf_register_show_on_connect (conf, (db_menu[5]).widget);
	gasql_conf_register_show_on_disconnect (conf, (db_menu[6]).widget);

	gasql_conf_register_sensitive_on_disconnect (conf,
						     (conn_menu[0]).widget);
	gasql_conf_register_sensitive_on_disconnect (conf,
						     (conn_menu[1]).widget);
	gasql_conf_register_sensitive_on_connect (conf,
						  (conn_menu[2]).widget);
	gasql_conf_register_sensitive_on_disconnect (conf,
						     (file_menu[1]).widget);
	gasql_conf_register_sensitive_on_disconnect (conf,
						     (file_menu[2]).widget);
	gasql_conf_register_sensitive_on_connect (conf,
						  (file_menu[5]).widget);
}



static gint
prepare_app (gASQL_Main_Config * conf)
{
	GnomeClient *client;
	GtkWidget *vb, *wid, *owid, *hb, *sc;
	gchar *str;
	CORBA_ORB orb;

	/*******************/
	/* background part */
	/*******************/

	/* SqlAccess object */
	orb = gda_corba_get_orb ();
	conf->srv = SQL_ACCESS (sql_access_new (orb));
	conf->srv->description = g_strdup ("Main SQL server");
	/* those 2 CBs are to show/hide and order sensitiveness of objects */
	gtk_signal_connect (GTK_OBJECT (conf->srv), "conn_opened",
			    GTK_SIGNAL_FUNC (gasql_widgets_conn_cb), conf);
	gtk_signal_connect (GTK_OBJECT (conf->srv), "conn_closed",
			    GTK_SIGNAL_FUNC (gasql_widgets_conn_cb), conf);

	gtk_signal_connect (GTK_OBJECT (conf->srv), "conn_opened",
			    GTK_SIGNAL_FUNC (sql_server_conn_open_cb), conf);
	gtk_signal_connect (GTK_OBJECT (conf->srv), "conn_to_close",
			    GTK_SIGNAL_FUNC (sql_server_conn_to_close_cb),
			    conf);
	gtk_signal_connect (GTK_OBJECT (conf->srv), "conn_closed",
			    GTK_SIGNAL_FUNC (sql_server_conn_close_cb), conf);
	gtk_signal_connect (GTK_OBJECT (conf->srv), "error",
			    GTK_SIGNAL_FUNC (sql_server_catch_errors_cb),
			    conf);
	gtk_signal_connect (GTK_OBJECT (conf->srv), "progress",
			    GTK_SIGNAL_FUNC (progress_cb), conf);

	/* the SqlDb object */
	conf->db = SQL_DB (sql_db_new (conf->srv));
	gtk_signal_connect (GTK_OBJECT (conf->db), "fault",
			    GTK_SIGNAL_FUNC (access_db_link_cb), conf);
	gtk_signal_connect (GTK_OBJECT (conf->db), "updated",
			    GTK_SIGNAL_FUNC (sql_conf_update_save_flag),
			    conf);
	gtk_signal_connect (GTK_OBJECT (conf->db), "progress",
			    GTK_SIGNAL_FUNC (progress_cb), conf);

	/* cleaning of the WidRelData list in conf for the items which do
	   not exist anymore */
	gtk_signal_connect (GTK_OBJECT (conf->db), "table_dropped",
			    GTK_SIGNAL_FUNC (wid_db_rel_list_update), conf);
	gtk_signal_connect (GTK_OBJECT (conf->db), "seq_dropped",
			    GTK_SIGNAL_FUNC (wid_db_rel_list_update), conf);

	/***************/
	/* visual part */
	/***************/
	str = gasql_get_title (conf);
	conf->app = gnome_app_new ("gASQL", str);
	g_free (str);
	gtk_widget_set_usize (conf->app, 600, 350);
	conf->appbar = gnome_appbar_new (TRUE, TRUE, GNOME_PREFERENCES_USER);
	gnome_appbar_set_default (GNOME_APPBAR (conf->appbar), _("Ready."));
	gnome_app_set_statusbar (GNOME_APP (conf->app), conf->appbar);
	gtk_widget_show (conf->appbar);

	/* vbox as the contents of the app */
	vb = gtk_vbox_new (TRUE, 0);
	gnome_app_set_contents (GNOME_APP (conf->app), vb);
	gtk_widget_show (vb);

	/* welcome statement if no connection opened */
	conf->welcome = gtk_packer_new ();
	gtk_packer_set_default_border_width (GTK_PACKER (conf->welcome),
					     GNOME_PAD * 2);
	wid = gtk_label_new (_("You need to configure and open the "
			       "connection\n" "before doing anything else."));
	gtk_packer_add_defaults (GTK_PACKER (conf->welcome), wid,
				 GTK_SIDE_TOP, GTK_ANCHOR_CENTER, 0);
	wid = gnome_pixmap_new_from_file (PIXMAPDIR "/gasql_foot.png");
	gtk_packer_add_defaults (GTK_PACKER (conf->welcome), wid,
				 GTK_SIDE_RIGHT, GTK_ANCHOR_SOUTH, 0);

	gtk_box_pack_start (GTK_BOX (vb), conf->welcome, TRUE, TRUE, 0);
	gtk_widget_show_all (conf->welcome);

	/* hb for the shortcut and the notebook */
	hb = gtk_hbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (hb), 0);
	conf->working_box = hb;
	gtk_box_pack_start (GTK_BOX (vb), hb, TRUE, TRUE, 0);

	/* shortcut part */
	sc = gnome_db_shortcut_new ();
	if (register_icon ("gasql_tables", "gasql_tables.png"))
		str = "gasql_tables";
	else
		str = GNOME_STOCK_PIXMAP_BOOK_RED;
	gnome_db_shortcut_append (GNOME_DB_SHORTCUT (sc),
				  _("Tables & Views"),
				  _("Work on tables and views"),
				  str,
				  (GtkSignalFunc) show_tables_page_cb, conf);

	if (register_icon ("gasql_sequences", "gasql_sequences.png"))
		str = "gasql_sequences";
	else
		str = GNOME_STOCK_PIXMAP_BOOK_BLUE;
	gnome_db_shortcut_append (GNOME_DB_SHORTCUT (sc),
				  _("Sequences"),
				  _("Work on sequences"),
				  str,
				  (GtkSignalFunc) show_seqs_page_cb, conf);

	if (register_icon ("gasql_queries", "gasql_queries.png"))
		str = "gasql_queries";
	else
		str = GNOME_STOCK_PIXMAP_BOOK_GREEN;
	gnome_db_shortcut_append (GNOME_DB_SHORTCUT (sc),
				  _("Queries"),
				  _("Manage queries and forms"),
				  str,
				  (GtkSignalFunc) show_queries_page_cb, conf);

	gtk_box_pack_start (GTK_BOX (hb), sc, FALSE, FALSE, 0);
	gtk_widget_show (sc);

	/* notebook */
	conf->nb = gtk_notebook_new ();
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (conf->nb), FALSE);
	gtk_box_pack_start (GTK_BOX (hb), conf->nb, TRUE, TRUE, 0);
	gtk_widget_show (conf->nb);
	gtk_widget_set_usize (GTK_WIDGET (conf->nb), 420, 270);

	/* Tables */
	owid = main_page_table_new (conf);
	gtk_widget_show (owid);

	wid = gtk_label_new (_("Tables & Views"));
	gtk_widget_show (owid);
	gtk_notebook_append_page (GTK_NOTEBOOK (conf->nb), owid, wid);
	conf->tables_page = owid;

	/* Sequences */
	wid = gtk_label_new (_("Sequences"));
	gtk_widget_show (wid);
	owid = main_page_seq_new (conf);
	gtk_widget_show (owid);
	gtk_notebook_append_page (GTK_NOTEBOOK (conf->nb), owid, wid);
	conf->sequences_page = owid;


	/* Queries */
	wid = gtk_label_new (_("Queries"));
	gtk_widget_show (wid);
	owid = main_page_query_new (conf);
	conf->queries_page = owid;
	gtk_widget_show (owid);
	gtk_notebook_append_page (GTK_NOTEBOOK (conf->nb), owid, wid);
	gtk_widget_show (conf->app);

	add_menus (GNOME_APP (conf->app), conf);
	add_toolbar (GNOME_APP (conf->app), conf);
	gtk_widget_show (conf->app);
	gtk_signal_connect (GTK_OBJECT (conf->app), "delete_event",
			    GTK_SIGNAL_FUNC (delete_event_cb), conf);

	gasql_widgets_conn_cb (NULL, conf);

	/* session management */
	client = gnome_master_client ();
	gtk_signal_connect (GTK_OBJECT (client), "save_yourself",
			    GTK_SIGNAL_FUNC (session_save_state_cb), conf);
	gtk_signal_connect (GTK_OBJECT (client), "die",
			    GTK_SIGNAL_FUNC (session_die_cb), NULL);


	/* getting the previously saved parameters: TO BE REPLACED BY GCONF  */
	gnome_config_push_prefix ("/gASQL/SqlServer/");
	str = gnome_config_get_string ("plugins_dir");
	if (str)
		conf->plugins_dir = str;
	else
		conf->plugins_dir = g_strdup (PLUGINSINSTALLDIR);
	gnome_config_pop_prefix ();
	sql_access_rescan_display_plugins (conf->srv, conf->plugins_dir);


	/* analysis of command line parameters */
	if (conf->datasource)
		g_string_assign (conf->srv->gda_datasource, conf->datasource);
	if (conf->user_name)
		g_string_assign (conf->srv->user_name, conf->user_name);
	if (conf->user_passwd)
		g_string_assign (conf->srv->password, conf->user_passwd);

	if (conf->file_to_open)
		set_opened_file (conf, conf->file_to_open);

	return 0;		/* => everything OK */
}


int
main (int argc, char *argv[])
{
	gASQL_Main_Config *conf;
	/* Command line options */
	poptContext return_ctx;
	const char **args;
	gchar *ds = NULL, *user = NULL, *pass = NULL, *str;
	gboolean def;
	struct poptOption options[] = {
		{"datasource", 'd', POPT_ARG_STRING, &ds, 0,
		 N_("datasource name"),
		 N_("SOURCE")},
		{"user", 'u', POPT_ARG_STRING, &user, 0, N_("user name"),
		 N_("USER-NAME")},
		{"password", 'p', POPT_ARG_STRING, &pass, 0, N_("password"),
		 N_("PASSWORD")},
		{NULL, '\0', 0, NULL, 0}	/* end the list */
	};

	/* i18n and i10n */
	bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
	textdomain (PACKAGE);

	/* CORBA and GNOME init */
	gda_init (PACKAGE, VERSION, argc, argv);
	gnome_init_with_popt_table (PACKAGE,	/* app_id */
				    VERSION,	/* app_version */
				    argc, argv, options,	/* options */
				    0,	/* popt_flags */
				    &return_ctx);	/* return_ctx */

	/*if (bonobo_init (gda_corba_get_orb (), NULL, NULL) == FALSE)
		g_error (_("Could not initialize bonobo"));
	bonobo_activate();*/
	
	/* to make gASQL work with libxml >= V1.8.8 and 2.x */
	LIBXML_TEST_VERSION conf = prepare_conf ();

	/* taking care of the command line arguments */
	if (ds)
		conf->datasource = g_strdup (ds);
	if (user)
		conf->user_name = g_strdup (user);
	if (pass)
		conf->user_passwd = g_strdup (pass);

	/* remaining arguments on the command line */
	args = poptGetArgs (return_ctx);
	if (args) {
		if (args[0]) {	/* opens the specified file */
			conf->file_to_open = g_strdup (args[0]);
		}
	}

	/* XML DTD checking */
	xmlDoValidityCheckingDefaultValue = 0;


	/*
	 * We can't make any CORBA calls unless we're in the main
	 * loop. So we delay creating the container here.
	 */
	gtk_idle_add ((GtkFunction) prepare_app, conf);
	bonobo_main ();

	return 0;
}


static gASQL_Main_Config *
prepare_conf ()
{
	gASQL_Main_Config *conf;

	conf = (gASQL_Main_Config *) g_malloc (sizeof (gASQL_Main_Config));
	gasql_conf_init (conf);

}


static gint
register_icon (gchar * name, gchar * filename)
{
	GnomeStockPixmapEntryPath *new_entry;
	gchar *str;

	new_entry = g_malloc (sizeof (GnomeStockPixmapEntryPath));
	new_entry->type = GNOME_STOCK_PIXMAP_TYPE_PATH;
	new_entry->label = NULL;
	str = gnome_pixmap_file (filename);
	if (!str)
		/* try to add the Pixmap install path to the file */
		str = g_strdup_printf ("%s/%s", PIXMAPDIR, filename);
	new_entry->pathname = str;
	new_entry->width = 0;
	new_entry->height = 0;
	return gnome_stock_pixmap_register (name, GNOME_STOCK_PIXMAP_REGULAR,
					    (GnomeStockPixmapEntry *)
					    new_entry);
}
