from utils.Observable import Observable
from utils.TargetSettings import TargetSettings
from utils.TypeConverter import TypeConverter
from utils import i18n
from SensorConfigurator import SensorConfigurator
from Menu import Menu
from main import _

import gtk
import thread
import Queue



#
# Abstract class for all Sensors. Every Sensor has to inherit from this class.
#
class Sensor(Observable):

    OBS_OUTPUT = 0
    OBS_CMD_CONFIGURE = 1
    OBS_CMD_REMOVE = 2


    def __init__(self):

        # configuration manager for sensor properties
        self.__config_manager = None
        self.__config_id = ""

        # flag for stopping the sensor
        self.__stop_flag = 0
        

        # converter for data types
        self.__type_converter = TypeConverter()
        self.__defaults = {}



    #
    # Returns a new empty TargetSettings object.
    #
    def _new_output(self): return TargetSettings()



    #
    # Sends the given TargetSettings object to the Display.
    #
    def _send_output(self, output):

        self.update_observer(self.OBS_OUTPUT, output)



    #
    # Adds a timeout function with the given interval in ms.
    # The callback function must return TRUE in order to be called again.
    #
    def _add_timer(self, interval, callback, *args):

        def __wrap(self, callback, args):
            if (not self.__stop_flag):
                return apply(callback, args)
            else:
                return gtk.FALSE
            
        return gtk.timeout_add(interval, __wrap, self, callback, args)



    #
    # Adds and runs a new thread.
    #
    def _add_thread(self, threadfunction, *args):

        # the thread should not start before setup is complete, therefore
        # we are using the GTK idle handler
        def run_thread(threadfunction, args):
            thread.start_new_thread(threadfunction, args)

        gtk.idle_add(run_thread, threadfunction, args)
        


    def send_action(self, call, path, args = []):

        if (not self.__stop_flag):
            self.call_action(call, path, args)



    def call_action(self, call, path, args = []): raise NotImplementedError



    #
    # Stops this sensor.
    #
    def stop(self):

        self.__stop_flag = 1

        # clear GConf
        self.__config_manager.clear(self.__config_id)



    #
    # Returns whether this sensor is stopped. Use this method to check if your
    # threads have to terminate.
    #
    def _is_stopped(self): return self.__stop_flag



    def set_config_manager(self, cm, config_id):

        self.__config_id = config_id
        self.__config_manager = cm



    def _set_config(self, key, value):
        assert(self.__config_manager)

        if (not self.__stop_flag):
            value = self.__type_converter.type2str(key, value)
            self.__config_manager.set(self.__config_id, key, value)



    def _get_config(self, key):
        assert(self.__config_manager)

        value = self.__config_manager.get(self.__config_id, key)            
        if (value == ""): value = self.__defaults.get(key, "")
        
        return self.__type_converter.str2type(key, value)
    


    #
    # Sets the data types to be used for the configuration values.
    #
    def _set_config_type(self, key, dtype, default):

        self.__type_converter.add_type(key, dtype)
        self.__defaults[key] = default




    #
    # Creates and returns an empty new sensor configurator.
    #
    def _new_configurator(self): return SensorConfigurator(self._set_config,
                                                           self._get_config)


    #
    # Returns the configurator of this sensor.
    #
    def get_configurator(self): return None



    #
    # Opens the given menu.
    #
    def _open_menu(self, menu):

        if (menu): menu += [()]
        menu += [(_("_Configure display"), 1, None, self.__on_configure, []),
                 (_("_Remove display"), 1, None, self.__on_remove, []),
                 (),
                 (_("_About"), 1, None, self.__on_about, []),
                 (),
                 (_("_Quit"), 1, None, self.__on_quit, [])
                 ]
        menu = Menu(menu)
        menu.popup()



    def __on_configure(self, src):

        self.update_observer(self.OBS_CMD_CONFIGURE, None)



    def __on_remove(self, src):

        self.update_observer(self.OBS_CMD_REMOVE, None)



    def __on_about(self, src):

        from main import NAME, VERSION, COPYRIGHT, DESCRIPTION
        from main import AUTHORS, DOCUMENTERS, ICON
        
        import gnome.ui

        tcredits = _("translator_credits")
        if (tcredits == "translator_credits"): tcredits = ""
        pixbuf = gtk.gdk.pixbuf_new_from_file(ICON)
        about = gnome.ui.About(NAME, VERSION, COPYRIGHT, DESCRIPTION,
                               AUTHORS, DOCUMENTERS, tcredits,
                               pixbuf)
        # set the icon
        about.set_icon(gtk.gdk.pixbuf_new_from_file(ICON))

        about.show()



    def __on_quit(self, src):

        gtk.main_quit()
