/* GAIL - The GNOME Accessibility Implementation Library
 * Copyright 2001 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <atk/atk.h>
#include <gtk/gtk.h>
#include "gail.h"

static gboolean gail_focus_watcher      (GSignalInvocationHint *ihint,
                                         guint                  n_param_values,
                                         const GValue          *param_values,
                                         gpointer               data);
static gboolean gail_select_watcher     (GSignalInvocationHint *ihint,
                                         guint                  n_param_values,
                                         const GValue          *param_values,
                                         gpointer               data);
static void     gail_focus_tracker_init ();

static GtkWidget* focus_widget = NULL;

static gboolean
gail_focus_watcher (GSignalInvocationHint *ihint,
                    guint                  n_param_values,
                    const GValue          *param_values,
                    gpointer               data)
{
  GObject *object;
  AtkObject *aobject;
  GtkWidget *widget;
  GdkEvent *event;

  object = g_value_get_object (param_values + 0);
  g_return_val_if_fail (GTK_IS_WIDGET(object), FALSE);

  event = g_value_get_boxed (param_values + 1);
  widget = GTK_WIDGET (object);

  if (event->type == GDK_FOCUS_CHANGE) 
  {
    if (event->focus_change.in)
    {
      if (GTK_IS_WINDOW (widget) && GTK_WINDOW (widget)->focus_widget)
      {
        /*
         * Found focus widget so use that
         */
        widget = GTK_WINDOW (widget)->focus_widget;
      }
    }
    else
    {
      /* Ignore focus out */
      return TRUE;
    }
  }
  else
  {
    if (event->type == GDK_MOTION_NOTIFY && GTK_WIDGET_HAS_FOCUS (widget))
    {
      if (widget == focus_widget)
        return TRUE;
    }
    else
    {
      return TRUE;
    }
  }
  focus_widget = widget;

  if (GTK_IS_ENTRY (widget))
  {
    GtkWidget *other_widget = widget->parent;
    if (GTK_IS_COMBO (other_widget))
      widget = other_widget;
  } 

  aobject = gtk_widget_get_accessible (widget);

  if (GTK_IS_NOTEBOOK (widget)) 
  {
    gint page_num;

    page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (widget));
    if (page_num != -1)
    {
      aobject = atk_object_ref_accessible_child (aobject, page_num);
      g_object_unref (aobject);
    }
  }
  atk_focus_tracker_notify (aobject);

  return TRUE; 
}

static gboolean
gail_select_watcher (GSignalInvocationHint *ihint,
                     guint                  n_param_values,
                     const GValue          *param_values,
                     gpointer               data)
{
  GObject *object;
  AtkObject *aobject;
  GtkWidget *widget;

  object = g_value_get_object (param_values + 0);
  g_return_val_if_fail (GTK_IS_WIDGET(object), FALSE);

  widget = GTK_WIDGET (object);

  if (!GTK_WIDGET_MAPPED (widget))
  /*
   * select signal on non-mapped widget
   * so ignore
   */
  {
    return TRUE;
  }

  focus_widget = widget;
  aobject = gtk_widget_get_accessible (widget);

  atk_focus_tracker_notify (aobject);

  return TRUE; 
}

static void
gail_focus_tracker_init ()
{
  static gboolean  emission_hooks_added = FALSE;

  if (!emission_hooks_added)
  {
    /*
     * We cannot be sure that the classes exist so we make sure that they do.
     */
    gtk_type_class (GTK_TYPE_WIDGET);
    gtk_type_class (GTK_TYPE_ITEM);

    /*
     * We listen for event_after signal and then check that the
     * event was a focus in event so we get called after the event.
     */
    g_signal_add_emission_hook (
             g_signal_lookup ("event-after", GTK_TYPE_WIDGET), 0,
             gail_focus_watcher, NULL, (GDestroyNotify) NULL);
    /*
     * A "select" signal is emitted when arrow key is used to
     * move to a list item in the popup window of a GtkCombo or
     * a menu item in a menu.
     */
    g_signal_add_emission_hook (
             g_signal_lookup ("select", GTK_TYPE_ITEM), 0,
             gail_select_watcher, NULL, (GDestroyNotify) NULL);
    /*
     * Activating a menu item means that it is first selected
     * so we omit these.
     */
#if 0
    signal_id = g_signal_lookup ("activate_item", GTK_TYPE_MENU_ITEM);
    g_signal_add_emission_hook (signal_id, 0,
                                gail_event_watcher, NULL, (GDestroyNotify) NULL);
    signal_id = g_signal_lookup ("activate", GTK_TYPE_MENU_ITEM);
    g_signal_add_emission_hook (signal_id, 0,
                                gail_event_watcher, NULL, (GDestroyNotify) NULL);
#endif
    emission_hooks_added = TRUE;
  }
}

int
gtk_module_init(gint *argc, char** argv[])
{
  AtkRegistry* default_registry;
  gpointer data;

  g_print("GTK Accessibilty Module loaded\n");

  default_registry = atk_get_default_registry();
  atk_registry_set_factory_type (default_registry, GTK_TYPE_WIDGET, 
                                 GAIL_TYPE_WIDGET_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_CONTAINER, 
                                 GAIL_TYPE_CONTAINER_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_BUTTON, 
                                 GAIL_TYPE_BUTTON_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_ITEM, 
                                 GAIL_TYPE_ITEM_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_MENU_ITEM, 
                                 GAIL_TYPE_MENU_ITEM_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_TOGGLE_BUTTON, 
                                 GAIL_TYPE_TOGGLE_BUTTON_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_IMAGE, 
                                 GAIL_TYPE_IMAGE_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_TEXT_VIEW, 
                                 GAIL_TYPE_TEXT_VIEW_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_COMBO, 
                                 GAIL_TYPE_COMBO_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_ENTRY, 
                                 GAIL_TYPE_ENTRY_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_MENU_BAR, 
                                 GAIL_TYPE_MENU_SHELL_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_MENU,
                                 GAIL_TYPE_MENU_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_WINDOW, 
                                 GAIL_TYPE_WINDOW_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_HANDLE_BOX, 
                                 GAIL_TYPE_WINDOW_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_RANGE,
                                 GAIL_TYPE_RANGE_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_CLIST, 
                                 GAIL_TYPE_CLIST_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_LABEL, 
                                 GAIL_TYPE_LABEL_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_STATUSBAR, 
                                 GAIL_TYPE_STATUSBAR_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_NOTEBOOK, 
                                 GAIL_TYPE_NOTEBOOK_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_CALENDAR, 
                                 GAIL_TYPE_CALENDAR_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_PROGRESS_BAR,
                                 GAIL_TYPE_PROGRESS_BAR_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_SPIN_BUTTON,
                                 GAIL_TYPE_SPIN_BUTTON_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_TREE_VIEW,
                                 GAIL_TYPE_TREE_VIEW_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_FRAME,
                                 GAIL_TYPE_FRAME_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_CELL_RENDERER_TEXT,
                                 GAIL_TYPE_TEXT_CELL_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_CELL_RENDERER_TOGGLE,
                                 GAIL_TYPE_BOOLEAN_CELL_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_CELL_RENDERER_PIXBUF,
                                 GAIL_TYPE_IMAGE_CELL_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_RADIO_BUTTON,
                                 GAIL_TYPE_RADIO_BUTTON_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_ARROW,
                                 GAIL_TYPE_ARROW_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_PIXMAP,
                                 GAIL_TYPE_PIXMAP_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_SEPARATOR,
                                 GAIL_TYPE_SEPARATOR_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_BOX,
                                 GAIL_TYPE_BOX_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_SCROLLED_WINDOW,
                                 GAIL_TYPE_SCROLLED_WINDOW_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_LIST,
                                 GAIL_TYPE_LIST_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_PANED,
                                 GAIL_TYPE_PANED_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_SCROLLBAR,
                                 GAIL_TYPE_SCROLLBAR_FACTORY);
  atk_registry_set_factory_type (default_registry, GTK_TYPE_OPTION_MENU,
                                 GAIL_TYPE_OPTION_MENU_FACTORY);

  /* LIBGNOMECANVAS SUPPORT */
  atk_registry_set_factory_type (default_registry, GNOME_TYPE_CANVAS,
				GAIL_TYPE_CANVAS_FACTORY);
  atk_registry_set_factory_type (default_registry, GNOME_TYPE_CANVAS_GROUP,
				 GAIL_TYPE_CANVAS_GROUP_FACTORY);
  atk_registry_set_factory_type (default_registry, GNOME_TYPE_CANVAS_TEXT,
				 GAIL_TYPE_CANVAS_TEXT_FACTORY);
  atk_registry_set_factory_type (default_registry, GNOME_TYPE_CANVAS_RICH_TEXT,
				 GAIL_TYPE_CANVAS_TEXT_FACTORY);
  atk_registry_set_factory_type (default_registry, GNOME_TYPE_CANVAS_WIDGET,
				 GAIL_TYPE_CANVAS_WIDGET_FACTORY);

    atk_focus_tracker_init (gail_focus_tracker_init);

  /* Initialize the GailUtility class */
  data = g_type_class_ref (GAIL_TYPE_UTIL);
  g_type_class_unref (data);

  return 0;
}
