/* GAIL - The GNOME Accessibility Implementation Library
 * Copyright 2001 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include "gailutil.h"
#include "gailtoplevel.h"

static void		gail_util_class_init			(GailUtilClass		*klass);

/* atkutil.h */

static guint		gail_util_add_global_event_listener	(GSignalEmissionHook	listener,
							         const gchar*           event_type);
static void 		gail_util_remove_global_event_listener	(guint			remove_listener);
static guint		gail_util_add_key_event_listener	(AtkKeySnoopFunc	listener,
								 gpointer               data);
static void 		gail_util_remove_key_event_listener	(guint			remove_listener);
static AtkObject*	gail_util_get_root			(void);
static G_CONST_RETURN gchar *gail_util_get_toolkit_name		(void);
static G_CONST_RETURN gchar *gail_util_get_toolkit_version      (void);

/* Misc */

static void		_listener_info_destroy			(gpointer		data);

static AtkObject* root = NULL;
static gpointer parent_class = NULL;
static GHashTable *listener_list = NULL;
static gint listener_idx = 1;
static GHashTable *key_listener_list = NULL;
static guint key_snooper_id = 0;

typedef struct _GailUtilListenerInfo GailUtilListenerInfo;
typedef struct _GailKeyEventInfo GailKeyEventInfo;

struct _GailUtilListenerInfo
{
   gint key;
   guint signal_id;
   gulong hook_id;
};

struct _GailKeyEventInfo
{
  AtkKeyEventStruct *key_event;
  gpointer func_data;
};

GType
gail_util_get_type (void)
{
  static GType type = 0;

  if (!type)
  {
    static const GTypeInfo tinfo =
    {
      sizeof (GailUtilClass),
      (GBaseInitFunc) NULL, /* base init */
      (GBaseFinalizeFunc) NULL, /* base finalize */
      (GClassInitFunc) gail_util_class_init, /* class init */
      (GClassFinalizeFunc) NULL, /* class finalize */
      NULL, /* class data */
      sizeof (GailUtil), /* instance size */
      0, /* nb preallocs */
      (GInstanceInitFunc) NULL, /* instance init */
      NULL /* value table */
    };

    type = g_type_register_static (ATK_TYPE_UTIL,
                                   "GailUtil", &tinfo, 0);
  }
  return type;
}

static void	 
gail_util_class_init (GailUtilClass *klass)
{
  AtkUtilClass *atk_class;
  gpointer data;

  parent_class = g_type_class_ref (ATK_TYPE_UTIL);

  data = g_type_class_peek (ATK_TYPE_UTIL);
  atk_class = ATK_UTIL_CLASS (data);

  atk_class->add_global_event_listener =
    gail_util_add_global_event_listener;
  atk_class->remove_global_event_listener =
    gail_util_remove_global_event_listener;
  atk_class->add_key_event_listener =
    gail_util_add_key_event_listener;
  atk_class->remove_key_event_listener =
    gail_util_remove_key_event_listener;
  atk_class->get_root = gail_util_get_root;
  atk_class->get_toolkit_name = gail_util_get_toolkit_name;
  atk_class->get_toolkit_version = gail_util_get_toolkit_version;

  listener_list = g_hash_table_new_full(g_int_hash, g_int_equal, NULL, 
     _listener_info_destroy);
}

static guint
gail_util_add_global_event_listener (GSignalEmissionHook listener,
				     const gchar *event_type)
{
  GType type;
  guint signal_id;
  gchar **split_string;
  gint  rc = 0;

  split_string = g_strsplit(event_type, ":", 3);

  type = g_type_from_name(split_string[1]);
  if (type > 0)
    {
      signal_id  = g_signal_lookup (split_string[2], type);
      if (signal_id > 0)
        {
          GailUtilListenerInfo *listener_info;

          rc = listener_idx;

          listener_info = g_malloc(sizeof(GailUtilListenerInfo));
          listener_info->key = listener_idx;
          listener_info->hook_id =
		  g_signal_add_emission_hook (signal_id, 0, listener,
					      g_strdup (event_type),
					      (GDestroyNotify) g_free);
          listener_info->signal_id = signal_id;

	  g_hash_table_insert(listener_list, &(listener_info->key),
            listener_info);
          listener_idx++;
        }
      else
        {
          g_warning("Invalid signal type %s\n", split_string[2]);
        }
    }
  else
    {
      g_warning("Invalid object type %s\n", split_string[1]);
    }

  if (split_string != NULL)
    g_strfreev(split_string);

  return rc;
}

static void
gail_util_remove_global_event_listener (guint remove_listener)
{
  if (remove_listener > 0)
  {
    GailUtilListenerInfo *listener_info;
    gint tmp_idx = remove_listener;

    listener_info = (GailUtilListenerInfo *)
      g_hash_table_lookup(listener_list, &tmp_idx);

    if (listener_info != NULL)
      {
        /* Hook id of 0 and signal id of 0 are invalid */
        if (listener_info->hook_id != 0 && listener_info->signal_id != 0)
          {
            /* Remove the emission hook */
            g_signal_remove_emission_hook(listener_info->signal_id,
              listener_info->hook_id);

            /* Remove the element from the hash */
            g_hash_table_remove(listener_list, &tmp_idx);
          }
        else
          {
            g_warning("Invalid listener hook_id %ld or signal_id %d\n",
              listener_info->hook_id, listener_info->signal_id);
          }
      }
    else
      {
        g_warning("No listener with the specified listener id %d", 
          remove_listener);
      }
  }
  else
  {
    g_warning("Invalid listener_id %d", remove_listener);
  }
}


static
AtkKeyEventStruct *
atk_key_event_from_gdk_event_key (GdkEventKey *key)
{
  AtkKeyEventStruct *event = g_new0 (AtkKeyEventStruct, 1);
  switch (key->type)
    {
    case GDK_KEY_PRESS:
	    event->type = ATK_KEY_EVENT_PRESS;
	    break;
    case GDK_KEY_RELEASE:
	    event->type = ATK_KEY_EVENT_RELEASE;
	    break;
    default:
    }
  event->state = key->state;
  event->keyval = key->keyval;
  event->length = key->length;
  event->string = key->string;
  event->keycode = key->hardware_keycode;
  event->timestamp = key->time;
#ifdef GAIL_DEBUG  
  g_print ("GailKey:\tsym %u\n\tmods %x\n\tcode %u\n\ttime %lx\n",
	   (unsigned int) event->keyval,
	   (unsigned int) event->state,
	   (unsigned int) event->keycode,
	   (unsigned long int) event->timestamp);
#endif
  return event;
}

static void
notify_hf (gpointer key, gpointer value, gpointer data)
{
  GailKeyEventInfo *info = (GailKeyEventInfo *) data;
  (*(AtkKeySnoopFunc) value) (info->key_event, info->func_data);
}

static gint
gail_key_snooper (GtkWidget *the_widget, GdkEventKey *event, gpointer func_data)
{
  /* notify each AtkKeySnoopFunc in turn... */
  GailKeyEventInfo *info = g_new0 (GailKeyEventInfo, 1);
  if (key_listener_list)
    {
      info->key_event = atk_key_event_from_gdk_event_key (event);
      info->func_data = func_data;
      g_hash_table_foreach (key_listener_list, notify_hf, info);
    }
  g_free (info->key_event);
  g_free (info);
  return 0;
}

static guint
gail_util_add_key_event_listener (AtkKeySnoopFunc  listener,
				  gpointer         data)
{
  static guint key=0;
  if (!key_listener_list)
  {
    key_listener_list = g_hash_table_new (NULL, NULL);	  
    key_snooper_id = gtk_key_snooper_install (gail_key_snooper, data);
  }
  /* XXX: slightly fishy use of int as a 'pointer', is this OK? */
  g_hash_table_insert (key_listener_list, (gpointer) key++, listener);
  /* XXX: we don't check to see if n_listeners > MAXUINT */
  return key;
}

static void
gail_util_remove_key_event_listener (guint remove_listener)
{
  g_hash_table_remove (key_listener_list, GUINT_TO_POINTER (remove_listener));
  if (g_hash_table_size (key_listener_list) == 0)
    {
      gtk_key_snooper_remove (key_snooper_id);
    }
}

static AtkObject*
gail_util_get_root (void)
{
  if (!root)
    root = gail_toplevel_new();

  return root;
}

static G_CONST_RETURN gchar *
gail_util_get_toolkit_name (void)
{
  return "GAIL";
}

static G_CONST_RETURN gchar *
gail_util_get_toolkit_version (void)
{
 /*
  * Version is passed in as a -D flag when this file is
  * compiled.
  */
  return VERSION;
}

static void
_listener_info_destroy (gpointer data)
{
   free(data);
}

